/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>

#ifdef WIN32
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_unicode.h"
#include "udm_url.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_hash.h"
#include "udm_xmalloc.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_searchcache.h"
#include "udm_server.h"
#include "udm_stopwords.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_vars.h"
#include "udm_agent.h"
#include "udm_store.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_match.h"
#include "udm_indexer.h"

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif

static const char *BuildWhere(UDM_ENV * Conf,UDM_DB *db)
{
  size_t  i;
  char    *urlstr;
  char    *tagstr;
  char    *statusstr;
  char    *catstr;
  char    *langstr;
  char    *typestr;
  char    *fromstr;
  char    *serverstr;
  char    *sitestr;
  char    *timestr;
  char    *urlinfostr;
  int     fromserver = 1, fromurlinfo_lang = 1, fromurlinfo_type = 1, fromurlinfo = 1;
  int     dt = UDM_DT_UNKNOWN, dx = 1, dm = 0, dy = 1970, dd = 1;
  time_t  dp = 0, DB = 0, DE = time(NULL), dstmp= 0;
  struct tm tm;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(db->where)return(db->where);
  
  bzero((void*)&tm, sizeof(struct tm));
  urlstr = (char*)UdmStrdup("");
  tagstr = (char*)UdmStrdup("");
  statusstr = (char*)UdmStrdup("");
  catstr = (char*)UdmStrdup("");
  langstr = (char*)UdmStrdup("");
  typestr = (char*)UdmStrdup("");
  fromstr = (char*)UdmStrdup("");
  serverstr = (char*)UdmStrdup("");
  sitestr = (char*)UdmStrdup("");
  timestr = (char*)UdmStrdup("");
  urlinfostr = (char*)UdmStrdup("");
  
  
  for(i=0;i<Conf->Vars.nvars;i++)
  {
    const char *var=Conf->Vars.Var[i].name?Conf->Vars.Var[i].name:"";
    const char *val=Conf->Vars.Var[i].val?Conf->Vars.Var[i].val:"";
    int intval=atoi(val);
    int longval= atol(val);
    
    if(!val[0])continue;
    
    if(!strcmp(var,"tag") || !strcmp(var,"t"))
    {
      tagstr=(char*)UdmRealloc(tagstr,strlen(tagstr)+strlen(val)+50);
      if(tagstr[0])strcpy(UDM_STREND(tagstr)-1," OR ");
      else  strcat(tagstr,"(");

      if(db->DBType==UDM_DB_PGSQL)
        sprintf(UDM_STREND(tagstr),"(s.tag || '') LIKE '%s')",val);
      else
        sprintf(UDM_STREND(tagstr),"s.tag LIKE '%s')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 12);
        sprintf(UDM_STREND(fromstr), ", server s");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id");
      }
    }
    
    if(!strcmp(var,"status"))
    {
      statusstr=(char*)UdmRealloc(statusstr,strlen(statusstr)+strlen(val)+50);
      
      if(db->DBSQL_IN)
      {
        if(statusstr[0])sprintf(UDM_STREND(statusstr)-1,",%d)",intval);
        else  sprintf(statusstr," url.status IN (%d)",intval);
      }
      else
      {
        if(statusstr[0])strcpy(UDM_STREND(statusstr)-1," OR ");
        else  strcat(statusstr,"(");
        sprintf(UDM_STREND(statusstr),"url.status=%d)",intval);
      }
    }

    if(!strcmp(var,"ul"))
    {
      UDM_URL  URL;
      const char *first = "%";

      UdmURLInit(&URL);
      UdmURLParse(&URL,val);
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);

      if((URL.schema != NULL) && (URL.hostinfo != NULL))
      {
        first = "";
      }
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      if(db->DBType==UDM_DB_PGSQL)
        sprintf(UDM_STREND(urlstr),"(url.url || '') LIKE '%s%s%%')", first, val);
      else
        sprintf(UDM_STREND(urlstr),"url.url LIKE '%s%s%%')", first, val);
      UdmURLFree(&URL);
    }
    
    if(!strcmp(var,"u"))
    {
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      if(db->DBType==UDM_DB_PGSQL)
        sprintf(UDM_STREND(urlstr),"(url.url || '') LIKE '%s')", val);
      else
        sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')", val);
    }
    
    if(!strcmp(var,"lang") || !strcmp(var,"g"))
    {
      langstr=(char*)UdmRealloc(langstr,strlen(langstr)+strlen(val)+50);
      if(langstr[0])strcpy(UDM_STREND(langstr)-1," OR ");
      else  strcat(langstr,"(");
      sprintf(UDM_STREND(langstr),"il.sval LIKE '%s')",val);
      if (fromurlinfo_lang)
      {
        fromurlinfo_lang = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo il");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND il.url_id=url.rec_id AND il.sname='Content-Language'");
      }
    }

    if(!strncmp(var, "sl.", 3))
    {
      urlinfostr=(char*)UdmRealloc(urlinfostr, strlen(urlinfostr)+strlen(var)+strlen(val)+50);
      if(urlinfostr[0])strcpy(UDM_STREND(urlinfostr)-1," AND ");
      else  strcat(urlinfostr,"(");
      sprintf(UDM_STREND(urlinfostr),"isl%d.sname='%s' AND isl%d.sval LIKE '%s')",fromurlinfo,var+3,fromurlinfo,val);

      fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
      sprintf(UDM_STREND(fromstr), ", urlinfo isl%d", fromurlinfo);
      serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
      sprintf(UDM_STREND(serverstr), " AND isl%d.url_id=url.rec_id", fromurlinfo);
      fromurlinfo++;
    }
    
    if(!strcmp(var,"cat"))
    {
      catstr=(char*)UdmRealloc(catstr,strlen(catstr)+strlen(val)+50);
      if(catstr[0])strcpy(UDM_STREND(catstr)-1," OR ");
      else  strcat(catstr,"(");
      sprintf(UDM_STREND(catstr),"c.path LIKE '%s%%')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
        sprintf(UDM_STREND(fromstr), ", server s, categories c");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id AND s.category=c.rec_id");
      }
    }
    if(!strcmp(var,"type") || !strcmp(var, "typ"))
    {
      /* 
         "type" is a reserved word in ASP,
         so "typ" is added as a workaround
      */
      typestr = (char*)UdmRealloc(typestr, strlen(typestr) + strlen(val) + 50);
      if(typestr[0]) strcpy(UDM_STREND(typestr) - 1, " OR ");
      else  strcat(typestr,"(");

      if(db->DBType==UDM_DB_PGSQL)
        sprintf(UDM_STREND(typestr),"(it.sval || '') LIKE '%s')",val);
      else
        sprintf(UDM_STREND(typestr),"it.sval LIKE '%s')",val);
      if (fromurlinfo_type)
      {
        fromurlinfo_type = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo it");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND it.url_id=url.rec_id AND it.sname='Content-Type'");
      }
    }
    if(!strcmp(var,"site") && intval != 0)
    {
      sitestr=(char*)UdmRealloc(sitestr, strlen(sitestr) + strlen(val) + 50);
      
      if(db->DBSQL_IN)
      {
        if (sitestr[0]) sprintf(UDM_STREND(sitestr) - 1, ",%s%i%s)", qu, intval, qu);
        else  sprintf(sitestr, " url.site_id IN (%s%i%s)", qu, intval, qu);
      }else{
        if (sitestr[0]) strcpy(UDM_STREND(sitestr) - 1, " OR ");
        else  strcat(sitestr, "(");
        sprintf(UDM_STREND(sitestr), "url.site_id=%s%d%s)", qu, intval, qu);
      }
    }
    if (!strcmp(var, "dt"))
    {
      if(!strcasecmp(val, "back")) dt = UDM_DT_BACK;
      else if (!strcasecmp(val, "er")) dt = UDM_DT_ER;
      else if (!strcasecmp(val, "range")) dt = UDM_DT_RANGE;
    }
    if (!strcmp(var, "dx"))
    {
      if (intval == 1 || intval == -1) dx = intval;
      else dx = 1;
    }
    if (!strcmp(var, "dm"))
    {
      dm = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dy"))
    {
      dy = (intval) ? intval : 1970;
    }
    if (!strcmp(var, "dd"))
    {
      dd = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dstmp"))
    {
      dstmp= longval ? longval : 0;
    }
    if (!strcmp(var, "dp"))
    {
      dp = Udm_dp2time_t(val);
    }
    if (!strcmp(var, "db"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DB = mktime(&tm);
    }
    if (!strcmp(var, "de"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DE = mktime(&tm);
    }
  }
  
  switch(dt)
  {
    case UDM_DT_BACK:
      timestr = (char*)UdmRealloc(timestr, 128);
      if (dp) sprintf(timestr, "url.last_mod_time >= %li", time(NULL) - dp);
      break;
    case UDM_DT_ER:
      timestr = (char*)UdmRealloc(timestr, 128);
      tm.tm_mday = dd; tm.tm_mon = dm, tm.tm_year = dy - 1900;
      sprintf(timestr, "url.last_mod_time %s %li", (dx == 1) ? ">=" : "<=", dstmp ? dstmp : mktime(&tm));
      break;
    case UDM_DT_RANGE:
      timestr = (char*)UdmRealloc(timestr, 128);
      sprintf(timestr, "url.last_mod_time >= %li AND url.last_mod_time <= %li", DB, DE);
      break;
    case UDM_DT_UNKNOWN:
    default:
      break;
  }


  if(!urlstr[0] && !tagstr[0] && !statusstr[0] && !catstr[0] && !langstr[0] &&
     !typestr[0] && !serverstr[0] && !fromstr[0] && !sitestr[0] && !timestr[0] &&
     !urlinfostr[0])
  {
    db->where = (char*)UdmStrdup("");
    db->from = (char*)UdmStrdup("");
    goto ret;
  }
  i= strlen(urlstr) + strlen(tagstr) + strlen(statusstr) + strlen(catstr) + 
     strlen(langstr) + strlen(typestr) + strlen(serverstr) + strlen(sitestr) +
     strlen(timestr) + strlen(urlinfostr);
  db->where=(char*)UdmMalloc(i+100);
  db->where[0] = '\0';
  UDM_FREE(db->from);
  db->from = (char*)UdmStrdup(fromstr);
  
  if(urlstr[0])
  {
    strcat(db->where,urlstr);
  }
  if(tagstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,tagstr);
  }
  if(statusstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,statusstr);
  }
  if(catstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,catstr);
  }
  if(langstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,langstr);
  }
  if(typestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, typestr);
  }
  if(sitestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, sitestr);
  }
  if(serverstr[0])
  {
    if(!db->where[0]) strcat(db->where, " 1=1 ");
    strcat(db->where, serverstr);
  }
  if(timestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, timestr);
  }
  if (urlinfostr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, urlinfostr);
  }
ret:
  UDM_FREE(urlstr);
  UDM_FREE(tagstr);
  UDM_FREE(statusstr);
  UDM_FREE(catstr);
  UDM_FREE(langstr);
  UDM_FREE(typestr);
  UDM_FREE(fromstr);
  UDM_FREE(serverstr);
  UDM_FREE(sitestr);
  UDM_FREE(timestr);
  UDM_FREE(urlinfostr);
  return db->where;
}


/********************************************************************/
/* --------------------------------------------------------------- */
/* Almost Unified code for all supported SQL backends              */
/* --------------------------------------------------------------- */
/*******************************************************************/




/************* Servers ******************************************/

static char * strdupnull(const char * src)
{
  if(src)
  {
    if(*src)
    {
      return((char*)UdmStrdup(src));
    }
    else
    {
      return (char*)UdmStrdup("");
    }
  }
  else
  {
    return (char*)UdmStrdup("");
  }
}

static int UdmLoadServerTable(UDM_AGENT * Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  size_t    rows, i, j, jrows;
  UDM_SQLRES  SQLRes, SRes;
  UDM_HREF  Href;
  char    qbuf[1024];
  const char  *filename= UdmVarListFindStr(&db->Vars, "filename", NULL);
  const char  *name = (filename && filename[0]) ? filename : "server";
  const char      *infoname = UdmVarListFindStr(&db->Vars, "srvinfo", "srvinfo");
  int    res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT rec_id,url,tag,category,command,weight,ordre \
FROM %s WHERE enabled=1 AND parent=%s0%s ORDER BY ordre", name, qu, qu);
  
  if(UDM_OK!=(res=UdmSQLQuery(db,&SQLRes,qbuf)))
    return res;
  
  bzero((void*)&Href, sizeof(Href));
  
  rows=UdmSQLNumRows(&SQLRes);
  for(i=0;i<rows;i++)
  {
    const char  *val;
    UDM_SERVER  *Server = Indexer->Conf->Cfg_Srv;
    
    Server->site_id    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    Server->Match.pattern  = strdupnull(UdmSQLValue(&SQLRes,i,1));
    Server->ordre    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 6));
    Server->command    = *UdmSQLValue(&SQLRes, i, 4);
    Server->weight    = UDM_ATOF(UdmSQLValue(&SQLRes, i, 5));
    
    if((val=UdmSQLValue(&SQLRes,i,2)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Tag", val);
    
    if((val=UdmSQLValue(&SQLRes,i,3)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Category", val);
    

    sprintf(qbuf,"SELECT sname,sval FROM %s WHERE srv_id=%s%i%s", infoname, qu, Server->site_id, qu);
    if(UDM_OK != (res = UdmSQLQuery(db, &SRes, qbuf)))
      return res;
    jrows = UdmSQLNumRows(&SRes);
    for(j = 0; j < jrows; j++)
    {
      const char *sname = UdmSQLValue(&SRes, j, 0);
      const char *sval = UdmSQLValue(&SRes, j, 1);
      UdmVarListReplaceStr(&Server->Vars, sname, sval);
    }
    UdmSQLFree(&SRes);

    Server->Match.match_type  = UdmVarListFindInt(&Server->Vars, "match_type", UDM_MATCH_BEGIN);
    Server->Match.case_sense  = UdmVarListFindInt(&Server->Vars, "case_sense", 1);
    Server->Match.nomatch  = UdmVarListFindInt(&Server->Vars, "nomatch", 0);
    Server->Match.arg  = (char *)UdmStrdup(UdmVarListFindStr(&Server->Vars, "Arg", "Disallow"));

    if (Server->command == 'S')
    {
      UdmServerAdd(Indexer, Server);
      if((Server->Match.match_type==UDM_MATCH_BEGIN) &&
         (Indexer->flags & UDM_FLAG_ADD_SERVURL))
      {
        Href.url = Server->Match.pattern;
        Href.method=UDM_METHOD_GET;
        Href.site_id = Server->site_id;
        Href.server_id = Server->site_id;
        Href.hops= (uint4) UdmVarListFindInt(&Server->Vars, "StartHops", 0);
        UdmHrefListAdd(&Indexer->Conf->Hrefs, &Href);
      }
    }
    else
    {
      char errstr[128];
      UdmMatchListAdd(Indexer, &Indexer->Conf->Filters, &Server->Match, errstr, sizeof(errstr), Server->ordre);
    }
    UDM_FREE(Server->Match.pattern);
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}

static int UdmServerTableFlush(UDM_DB *db)
{
  int rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char str[128];
  
  udm_snprintf(str, sizeof(str),  "UPDATE server SET enabled=0 WHERE parent=%s0%s", qu, qu);
  rc = UdmSQLQuery(db, NULL, str);
  return rc;
}

static int UdmServerTableAdd(UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  int    res = UDM_OK, done = 1;
  const char  *alias=UdmVarListFindStr(&S->Server->Vars,"Alias",NULL);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t    i, len= 0;
  char    *buf, *arg;
  udmhash32_t     rec_id = UdmStrHash32(S->Server->Match.pattern);
  UDM_VARLIST  *Vars= &S->Server->Vars;
  
  for (i=0; i < Vars->nvars; i++)
    len= udm_max(len, Vars->Var[i].curlen);
  
  len+= S->Server->Match.pattern ? strlen(S->Server->Match.pattern) : 0;
  len+= alias ? strlen(alias) : 0;
  len+= 2048;
  
  buf = (char*)UdmMalloc(len);
  arg = (char*)UdmMalloc(len);
  if (buf == NULL || arg == NULL)
  {
    UDM_FREE(buf);
    UDM_FREE(arg);
    strcpy(db->errstr, "Out of memory");
    db->errcode = 1;
    return UDM_ERROR;
  }
  
  while(done)
  {
    udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%d%s", qu, rec_id, qu);
    if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
      goto ex;
  
    if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
    {
      rec_id++;
    } else done = 0;
    UdmSQLFree(&SQLRes);
  }

  UdmVarListReplaceInt(&S->Server->Vars, "match_type",  S->Server->Match.match_type);
  UdmVarListReplaceInt(&S->Server->Vars, "case_sense",  S->Server->Match.case_sense);
  UdmVarListReplaceInt(&S->Server->Vars, "nomatch",  S->Server->Match.nomatch);
  if (S->Server->command == 'F' && S->Server->Match.arg != NULL)
  {
    UdmVarListReplaceStr(&S->Server->Vars, "Arg", S->Server->Match.arg);
  } 
  
  udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, \
command, parent, ordre, weight, url, pop_weight \
) VALUES (%s%d%s, 1, '%s', %s, '%c', %s%d%s, %d, %f, '%s', 0\
)",
         qu, rec_id, qu,
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)))
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  udm_snprintf(buf, len, "\
UPDATE server SET enabled=1, tag='%s', category=%s, \
command='%c', parent=%s%i%s, ordre=%d, weight=%f \
WHERE rec_id=%s%d%s",
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         qu, rec_id, qu
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  S->Server->site_id = rec_id;

  sprintf(buf, "DELETE FROM srvinfo WHERE srv_id=%s%i%s", qu, S->Server->site_id, qu);
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;

  for(i = 0; i < S->Server->Vars.nvars; i++)
  {
    UDM_VAR *Sec = &S->Server->Vars.Var[i];
    if(Sec->val && Sec->name && 
       strcasecmp(Sec->name, "Category") &&
       strcasecmp(Sec->name, "Tag"))
    {
      arg = UdmSQLEscStr(db, arg, Sec->val,strlen(Sec->val));
      udm_snprintf(buf, len, "INSERT INTO srvinfo (srv_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
        qu, S->Server->site_id, qu, Sec->name, arg);
      if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf)))break;
    }
  }

ex:
  UDM_FREE(buf);
  UDM_FREE(arg);
  return res;
}

static int UdmServerTableGetId(UDM_AGENT *Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  size_t len = ((S->Server->Match.pattern)?strlen(S->Server->Match.pattern):0) + 1024;
  char *buf = (char*)UdmMalloc(len);
  char *arg = (char*)UdmMalloc(len);
  int res, id = 0, i;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  

  if (buf == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }
  if (arg == NULL)
  {
    UDM_FREE(buf);
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }

  for (i = 0; i < UDM_SERVERID_CACHE_SIZE; i++)
  {
    if (Indexer->ServerIdCacheCommand[i] == S->Server->command)
      if (!strcmp(Indexer->ServerIdCache[i], S->Server->Match.pattern))
      {
        S->Server->site_id = id = Indexer->ServerIdCacheId[i];
        break;
      }
  }

  if (id == 0)
  {
    udmhash32_t     rec_id;
    int done = 1;
  
    udm_snprintf(buf, len, "SELECT rec_id FROM server WHERE command='%c' AND url='%s'",
           S->Server->command,
           UDM_NULL2EMPTY(S->Server->Match.pattern)
     );
    res = UdmSQLQuery(db, &SQLRes, buf);
    if ((res == UDM_OK) && UdmSQLNumRows(&SQLRes))
    {
      id = S->Server->site_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
      UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
      Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
      Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
      Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
      Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return UDM_OK;
    }
    UdmSQLFree(&SQLRes);
    rec_id = UdmStrHash32(S->Server->Match.pattern);
    while(done) 
    {
      udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%i%s", qu, rec_id, qu);
      if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
        return res;
      
      if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
      {
        rec_id++;
      } else done = 0;
      UdmSQLFree(&SQLRes);
    }
    udm_snprintf(buf, len, "SELECT enabled,tag,category,ordre FROM server WHERE rec_id=%s%i%s", qu, S->Server->parent, qu);
    res = UdmSQLQuery(db, &SQLRes, buf);
    if (res != UDM_OK)
    {
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return res;
    }

    udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, command, parent, ordre, weight, url) \
VALUES (%s%d%s, %d, '%s', %s, '%c', %s%d%s, %d, %f, '%s')\
",
           qu, rec_id, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0)),
           UdmSQLValue(&SQLRes, 0, 1),
           UdmSQLValue(&SQLRes, 0, 2),
           S->Server->command,
           qu, S->Server->parent, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 3)),
           S->Server->weight,
           UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)) )
     );
    res = UdmSQLQuery(db, NULL, buf);
    UdmSQLFree(&SQLRes);

    S->Server->site_id = id = rec_id;
    UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
    Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
    Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
    Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
    Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
  }
  UDM_FREE(buf); UDM_FREE(arg);
  return UDM_OK;
}

/************************* find url ********************************/

static int UdmFindURL(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  udmhash32_t  id = 0;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  if(use_crc32_url_id)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    id = UdmStrHash32(url);
  }else{
    size_t i, l = 127;
    const char *o;
    char *qbuf = NULL;
    char *e_url = NULL;
    
    /* Escape URL string */
    if ((e_url = (char*)UdmMalloc( l = (8 * strlen(url)) )) == NULL)
    {
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
    }
    if ((qbuf = (char*)UdmMalloc( l + 100 )) == NULL)
    {
      UDM_FREE(e_url);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
    }
    UdmSQLEscStr(db,e_url,url,strlen(url));


    for(i = 0; i < UDM_FINDURL_CACHE_SIZE; i++)
    {
      if (Indexer->UdmFindURLCache[i])
        if (!strcmp(e_url, Indexer->UdmFindURLCache[i]))
        {
          id = Indexer->UdmFindURLCacheId[i];
          break;
        }
    }

    if (id == 0)
    {
      udm_snprintf(qbuf, l + 100, "SELECT rec_id FROM url WHERE url='%s'",e_url);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
      {
        UDM_FREE(e_url);
        UDM_FREE(qbuf);
        return rc;
      }
      for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
      {
        if((o=UdmSQLValue(&SQLRes,i,0)))
        {
          id=atoi(o);
          break;
        }
      }
      UdmSQLFree(&SQLRes);
      UDM_FREE(Indexer->UdmFindURLCache[Indexer->pURLCache]);
      Indexer->UdmFindURLCache[Indexer->pURLCache] = (char*)UdmStrdup(e_url);
      Indexer->UdmFindURLCacheId[Indexer->pURLCache] = id;
      Indexer->pURLCache = (Indexer->pURLCache + 1) % UDM_FINDURL_CACHE_SIZE;
    }
    UDM_FREE(e_url);
    UDM_FREE(qbuf);
  }
  UdmVarListReplaceInt(&Doc->Sections, "ID", id);
  return  rc;
}


static int UdmFindMessage(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  size_t     i, len;
  char     *qbuf;
  char     *eid;
  UDM_SQLRES  SQLRes;
  const char  *message_id=UdmVarListFindStr(&Doc->Sections,"Header.Message-ID",NULL);
  int    rc;
  
  if(!message_id)
    return UDM_OK;
  
  len = strlen(message_id);
  eid = (char*)UdmMalloc(4 * len + 1);
  if (eid == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 128);
  if (qbuf == NULL)
  { 
    UDM_FREE(eid);
    return UDM_ERROR;
  }

  /* Escape URL string */
  UdmSQLEscStr(db, eid, message_id, len);
  
  udm_snprintf(qbuf, 4 * len + 128, 
     "SELECT rec_id FROM url u, urlinfo i WHERE u.rec_id=i.url_id AND i.sname='Message-ID' AND i.sval='%s'", eid);
  rc = UdmSQLQuery(db,&SQLRes,qbuf);
  UDM_FREE(qbuf);
  UDM_FREE(eid);
  if (UDM_OK != rc)
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char * o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
    {
      UdmVarListReplaceInt(&Doc->Sections,"ID", UDM_ATOI(o));
      break;
    }
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


/********************** Words ***********************************/

static int UdmDeleteWordFromURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[512];
  int  i = 0, rc = UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  unsigned char PrevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0) ? 1 : 0;

  if (! PrevStatus) return(UDM_OK);
  
  switch(db->DBMode){
  
  case UDM_DBMODE_MULTI:
    for(i = 0; i <= MULTI_DICTS; i++)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM dict%02X WHERE url_id=%s%i%s", i, qu, url_id, qu);
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, qbuf))) return(rc);
    }
    break;
  default:  /* UDM_DBMODE_SINGLE */
    udm_snprintf(qbuf, sizeof(qbuf)-1, "DELETE FROM dict WHERE url_id=%s%d%s", qu, url_id, qu);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      return(UDM_ERROR);
    break;
  }
  return(UDM_OK);
}

static size_t udm_put_utf8 (size_t, unsigned char *, unsigned char *);
static size_t udm_get_utf8 (size_t *, const unsigned char *, const unsigned char *);

static char *UdmMultiCachePutIntag (UDM_MULTI_CACHE_WORD *cache, char type)
{
  size_t c;
  size_t len = 2;
  size_t i;
  char *_;
  unsigned char buf[3];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * 6 + 3);
  if (! _) return(NULL);

  if (type == 1)
  {
    strcpy(_, "0x");
  } else {
    *_ = 0;
    len = 0;
  }

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_put_utf8(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    for (i = 0; i < nbytes; i++)
    {
      udm_snprintf(_ + len, 3, "%02X", buf[i]);
      len += 2;
    }

    last = cache->intags[c];
  }

  return(_);
}

static char *UdmMultiCachePutIntag1 (UDM_MULTI_CACHE_WORD *cache)
{
  size_t c;
  size_t len = 0;
  char *_;
  unsigned char buf[3];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * 3 + 1);
  if (! _) return(NULL);

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_put_utf8(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    memcpy(_ + len, buf, nbytes);
    len += nbytes;

    last = cache->intags[c];
  }
  _[len] = 0;

  return(_);
}

int UdmWordCacheWrite (UDM_AGENT *Indexer, UDM_DB *db, size_t limit)
{
  size_t i;
  size_t LastLocked = 0;
  int rc;
  UDM_WORD_CACHE *cache = &db->WordCache;
  UDM_DSTR buf, qbuf;
  UDM_MULTI_CACHE_WORD mintag;
  size_t aintags = 0;

  if (cache->nbytes <= limit) return(UDM_OK);
  UdmLog(Indexer, UDM_LOG_ERROR, "Writing words (%d words, %d bytes%s).", cache->nwords, cache->nbytes, limit ? "" : ", final");

  if(UDM_OK!=(rc=UdmSQLBegin(db)))
  {
    UdmWordCacheFree(cache);
    return(rc);
  }

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  mintag.intags = NULL;
  mintag.word = NULL;

  /* Collect expired url ids for deletion. */
  for (i = 0; i < cache->nurls; i++)
  {
    if (buf.size_data) UdmDSTRAppend(&buf, ",", 1);
    UdmDSTRAppendf(&buf, "'%d'", cache->urls[i]);
  }

  /* Remove expired words. */
  if (buf.size_data) for (i = 0; i <= MULTI_DICTS; i++)
  {
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf, "DELETE FROM dict%02X WHERE url_id IN (%s)", i, buf.data);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
      goto unlock_UdmStoreWordsMulti;
  }

  UdmDSTRReset(&buf);
  UdmDSTRReset(&qbuf);

  UdmWordCacheSort(cache);

  for (i = 0; i < cache->nwords;)
  {
    UDM_WORD_CACHE_WORD *cword = &cache->words[i];
    unsigned char seed = cword->seed;
    char *word = cword->word;
    urlid_t url_id = cword->url_id;
    unsigned char secno = cword->secno;
    char *intag;

    mintag.nintags = 0;

    do
    {
      if (mintag.nintags == aintags)
      {
        uint4 *itmp;
        itmp = UdmRealloc(mintag.intags, (aintags + 256) * sizeof(uint4));
	if (! itmp)
	{
	  goto unlock_UdmStoreWordsMulti;
	}
	mintag.intags = itmp;
	aintags += 256;
      }
      mintag.intags[mintag.nintags] = cword->coord;
      mintag.nintags++;

      i++;
      if (i == cache->nwords) break;
      cword = &cache->words[i];
    }
    while (seed == cword->seed &&
	url_id == cword->url_id &&
        secno == cword->secno &&
	! strcmp(word, cword->word));


    if (db->DBType == UDM_DB_MYSQL)
    {
      intag = UdmMultiCachePutIntag(&mintag, 1);
      if (! intag) continue;

      if (buf.size_data)
      {
        UdmDSTRAppendf(&buf, ",(%d, %d, '%s', %s)",
	               url_id, secno, word, intag);
      } else {
        UdmDSTRAppendf(&buf, "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s)",
                       seed, url_id, secno, word, intag);
      }
      UdmFree(intag);
      if (seed != cword->seed || i == cache->nwords)
      {
        if (LastLocked <= seed)
	{
          if (LastLocked) UdmSQLQuery(db, NULL, "UNLOCK TABLES");
	  LastLocked = seed;
	  UdmDSTRAppendf(&qbuf, "LOCK TABLES dict%02X WRITE", LastLocked);
          for (LastLocked++; LastLocked <= MULTI_DICTS; LastLocked++)
          {
	    if (LastLocked - seed == 0x10) break;
            UdmDSTRAppendf(&qbuf, ",dict%02X WRITE", LastLocked);
          }
          UdmSQLQuery(db, NULL, qbuf.data);
	  UdmDSTRReset(&qbuf);
	}

        if (buf.size_data)
        {
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data)))
          {
            goto unlock_UdmStoreWordsMulti;
          }
	  UdmDSTRReset(&buf);
        }
      }
    }
    else
    {
      const char *quot;
      const char *x;
      const char *castb;
      const char *caste;
      char *tmp;

      if (db->DBType == UDM_DB_ORACLE8 || db->DBType == UDM_DB_DB2) 
        intag = UdmMultiCachePutIntag(&mintag, 0);
      else if (db->DBType == UDM_DB_MSSQL || db->DBType == UDM_DB_ACCESS)
        intag = UdmMultiCachePutIntag(&mintag, 1);
      else
        intag = UdmMultiCachePutIntag1(&mintag);
      if (! intag) continue;
      tmp = UdmSQLEscStr(db, NULL, intag, strlen(intag));
      UdmFree(intag);
      intag = tmp;

      if (db->DBType == UDM_DB_MSSQL || db->DBType == UDM_DB_ACCESS)
        quot="";
      else
        quot="'";

      if (db->DBType == UDM_DB_DB2)
      {
        x="X";
        castb="CAST(";
        caste=" AS BLOB)";
      }
      else
      {
        x="";
        castb="";
        caste="";
      }

      UdmDSTRAppendf(&buf,
        "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s%s%s%s%s%s)",
        seed, url_id, secno, word, castb,x,quot,intag,quot,caste);
      UdmFree(intag);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data))) goto unlock_UdmStoreWordsMulti;
      UdmDSTRReset(&buf);
    }
  }

unlock_UdmStoreWordsMulti:
  UDM_FREE(mintag.intags);
  UdmDSTRFree(&buf);
  UdmDSTRFree(&qbuf);
  if (LastLocked && rc == UDM_OK)
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");

  if(rc==UDM_OK)
    rc=UdmSQLCommit(db);

  UdmWordCacheFree(&db->WordCache);
  UdmLog(Indexer, UDM_LOG_ERROR, "The words are written successfully.%s", limit ? "" : " (final)");
  return(rc);
}

static int UdmStoreWordsMulti (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i;
  int rc = UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  unsigned char PrevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0) ? 1 : 0;
  int WordCacheSize = UdmVarListFindInt(&Indexer->Conf->Vars, "WordCacheSize", 0);

  if (WordCacheSize <= 0) WordCacheSize = 0x800000;

  if (PrevStatus) UdmWordCacheAddURL(&db->WordCache, url_id);

  for (i = 0; i < Doc->Words.nwords; i++)
  {
    if (! Doc->Words.Word[i].coord) continue;
    UdmWordCacheAdd(&db->WordCache, url_id, Doc->Words.Word[i].word, Doc->Words.Word[i].coord);
  }

  rc = UdmWordCacheWrite(Indexer, db, WordCacheSize);
  return(rc);
}

static void UdmMultiAddCoords (UDM_RESULT *Res, UDM_SQLRES *SQLRes, size_t wordnum, int *wf)
{
  size_t i;
  size_t numrows;
  urlid_t url_id;
  size_t crd;
  size_t last;
  size_t nbytes;
  unsigned char secno;
  const unsigned char *intag;
  const unsigned char *s, *e;
  size_t lintag;
  uint4 weight;
  size_t acoords = 0;

  numrows = UdmSQLNumRows(SQLRes);

  for (i = 0; i < numrows; i++)
  {
    size_t tmp = UdmSQLLen(SQLRes, i, 2);
    if (tmp) acoords += tmp;
    else acoords += strlen(UdmSQLValue(SQLRes, i, 2));
  }
  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, (Res->CoordList.ncoords + acoords) * sizeof(UDM_URL_CRD));

  for (i = 0; i < numrows; i++)
  {
    url_id = UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    secno = UDM_ATOI(UdmSQLValue(SQLRes, i, 1));
    intag = (const unsigned char *)UdmSQLValue(SQLRes, i, 2);
    lintag = UdmSQLLen(SQLRes, i, 2);
    weight = wf[secno];
    if (! weight) continue;

    last = 0;

    /* FIXME: Check UdmSQLLen */
    if (! lintag) lintag = strlen((const char *)intag);

    for (s = intag, e = intag + lintag; e > s; s += nbytes)
    {
      nbytes = udm_get_utf8(&crd, s, e);
      if (! nbytes) break;
      last += crd;
      Res->CoordList.Coords[Res->CoordList.ncoords].url_id = url_id;
      Res->CoordList.Coords[Res->CoordList.ncoords].coord = UDM_WRDCOORD(last, secno) + UDM_WRDNUM(wordnum);
      Res->CoordList.ncoords++;
      Res->WWList.Word[wordnum].count++;
    }
  }

  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, Res->CoordList.ncoords * sizeof(UDM_URL_CRD));
}

static int udm_get_int4 (const char *data)
{
  int _;
  memcpy(&_, data, sizeof(int));
  return(_);
}

static void udm_put_int4 (int src, char *dst)
{
  memcpy(dst, &src, sizeof(int));
}

typedef struct {
  char empty;
  urlid_t *urls;
  size_t nurls;
} UDM_URL_TMP;

static int cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}

static
void UdmBlobAddCoords(UDM_RESULT *Res, UDM_SQLRES *SQLRes,
                      size_t wordnum, int *wf, UDM_URL_TMP *urls)
{
  size_t i;
  size_t numrows;
  urlid_t url_id;
  size_t crd;
  size_t last;
  size_t nbytes;
  size_t nrecs;
  unsigned char secno;
  const unsigned char *intag;
  const unsigned char *s, *e;
  size_t lintag;
  uint4 weight;
  size_t acoords = 0;

  numrows = UdmSQLNumRows(SQLRes);

  for (i = 0; i < numrows; i++)
  {
    size_t tmp = UdmSQLLen(SQLRes, i, 1);
    if (tmp) acoords += tmp;
    else acoords += strlen(UdmSQLValue(SQLRes, i, 1));
  }
  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, (Res->CoordList.ncoords + acoords) * sizeof(UDM_URL_CRD));

  for (i = 0; i < numrows; i++)
  {
    secno = UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    intag = (const unsigned char *)UdmSQLValue(SQLRes, i, 1);
    lintag = UdmSQLLen(SQLRes, i, 1);
    weight = wf[secno];
    if (! weight) continue;

    last = 0;

    /* FIXME: Check UdmSQLLen */
    if (! lintag) lintag = strlen((const char *)intag);

    for (s = intag, e = intag + lintag; e > s;)
    {
      if (s + sizeof(urlid_t) > e) break;
      url_id = (urlid_t)udm_get_int4((const char *)s);
      s += sizeof(urlid_t);

      nbytes = udm_get_utf8(&nrecs, s, e);
      if (! nbytes) break;
      s += nbytes;
      last = 0;

      while (nrecs > 0)
      {
        nbytes = udm_get_utf8(&crd, s, e);
        if (! nbytes) break;
        s += nbytes;
        last += crd;
        nrecs--;
        if (! urls->nurls || bsearch(&url_id, urls->urls, urls->nurls, sizeof(urlid_t), (qsort_cmp)cmpaurls))
        {
          Res->CoordList.Coords[Res->CoordList.ncoords].url_id = url_id;
          Res->CoordList.Coords[Res->CoordList.ncoords].coord = UDM_WRDCOORD(last, secno) + UDM_WRDNUM(wordnum);
          Res->CoordList.ncoords++;
          Res->WWList.Word[wordnum].count++;
        }
      }
    }
  }

  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, Res->CoordList.ncoords * sizeof(UDM_URL_CRD));
}

static int UdmBlobGetTable (UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  const char *val;

  return(1);

  rc = UdmSQLQuery(db, &SQLRes, "SELECT n FROM bdictsw");
  if (rc != UDM_OK) return(1);

  if (! UdmSQLNumRows(&SQLRes) || ! (val = UdmSQLValue(&SQLRes, 0, 0))) rc = 2;
  else if (*val != '1') rc = 3;
  else rc = 4;

  UdmSQLFree(&SQLRes);
  return(rc);
}

static const char *UdmBlobGetRTable (UDM_DB *db)
{
  if (db->DBType == UDM_DB_MYSQL)
    return "bdict";
  if (UdmBlobGetTable(db) == 3) return("bdict00");
  return("bdict");
}

static int UdmBlobGetWTable (UDM_DB *db, const char **name)
{
  int rc;
  *name= "bdict";
  if (db->DBType == UDM_DB_MYSQL)
  {
    if ((UDM_OK != (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict_tmp"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp MAX_ROWS=300000000 AVG_ROW_LENGTH=512 SELECT * FROM bdict LIMIT 0"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp ADD KEY (word)"))))
      return rc;
    *name= "bdict_tmp";
  }
  if (UdmBlobGetTable(db) == 4)
    *name= "bdict00";
  return UDM_OK;
}

static int UdmBlobSetTable (UDM_DB *db)
{
  char qbuf[64];
  int rc, t, n;

  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK == (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict")))
      rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp RENAME bdict");
    return rc;
  }
  t= UdmBlobGetTable(db);
  if (t == 1) return(UDM_OK);
  else if (t == 4) n = 0;
  else n = 1;

  rc = UdmSQLQuery(db, NULL, "DELETE FROM bdictsw");
  if (rc != UDM_OK) return(UDM_OK);
  udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO bdictsw VALUES(%d)", n);
  rc = UdmSQLQuery(db, NULL, qbuf);
  if (rc != UDM_OK) return(UDM_OK);
  return(UDM_OK);
}

static size_t udm_utf8_len (const char *str)
{
  size_t _ = 0;
  size_t nbytes;
  size_t lintag;
  size_t crd;
  const char *s, *e;

  if (! str) return(0);
  lintag = strlen(str);

  for (s = str, e = str + lintag; e > s; s += nbytes)
  {
    nbytes = udm_get_utf8(&crd, (const unsigned char *)s, (const unsigned char *)e);
    if (! nbytes) break;
    _++;
  }

  return(_);
}

static char udm_hex_digits[]= "0123456789ABCDEF";

static int UdmBlobCacheWrite (UDM_DB *db, UDM_BLOB_CACHE *cache, const char *table)
{
  size_t w, w1;
  int rc= UDM_OK;
  unsigned char ubuf[3];
  unsigned char *ubufend = ubuf + sizeof(ubuf);
  size_t nbytes;
  UDM_DSTR buf;
  UDM_BLOB_CACHE_WORD *word;
  UDM_BLOB_CACHE_WORD *word1;
  char utmp[4];

  if (! cache->nwords) return(UDM_OK);

  UdmDSTRInit(&buf, 8192);
  for (w = 0; w < cache->nwords; w++)
  {
    word = &cache->words[w];
    for (w1 = w; w1 < cache->nwords; w1++)
    {
      word1 = &cache->words[w1];
      if (word->secno != word1->secno || strcmp(word->word, word1->word)) break;
      udm_put_int4((int)word1->url_id, utmp);
      nbytes = udm_put_utf8(word1->nintags, ubuf, ubufend);
      if (! nbytes) continue;
      UdmDSTRAppend(&buf, utmp, sizeof(urlid_t));
      UdmDSTRAppend(&buf, (char *)ubuf, nbytes);
      UdmDSTRAppend(&buf, word1->intags, word1->ntaglen);
    }

    if (db->DBType == UDM_DB_MYSQL || db->DBType == UDM_DB_MSSQL)
    {
      char *qbuf;
      size_t offs, i;
      qbuf= (char*) UdmMalloc(buf.size_data * 2 + 300);
      offs= sprintf(qbuf, "INSERT INTO %s VALUES('%s',%d,0x",
                    table, word->word, word->secno);
      
      for (i=0; i < buf.size_data; i++)
      {
        unsigned int ch= (unsigned int) (unsigned char) buf.data[i];
        qbuf[offs++]= udm_hex_digits[(ch >> 4) & 0x0F];
        qbuf[offs++]= udm_hex_digits[ch & 0x0F];
      }
      strcpy(qbuf+offs, ")");
      if (UDM_OK != (rc = UdmSQLQuery(db, NULL, qbuf)))
      {
        fprintf(stderr, "UdmSQLQuery failed\n");
        goto end;
      }
      UdmFree(qbuf);
    }
#ifdef HAVE_ORACLE8
    else if (db->DBDriver == UDM_DB_ORACLE8 && db->sql->SQLPrepare)
    {
      char qbuf[1024];
      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('%s', %d, :1)",
        table, word->word, word->secno);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, buf.data, buf.size_data, UDM_SQLTYPE_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);
      db->sql->SQLCommit(db);
    }
#endif
#ifdef HAVE_ODBC
    else if (db->DBDriver == UDM_DB_ODBC)
    {
      char qbuf[1024];
      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('%s', %d, ?)",
        table, word->word, word->secno);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, buf.data, buf.size_data, SQL_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);
      db->sql->SQLCommit(db);
    }
#endif
    else
    {
      char *tmp= UdmSQLEscStr(db, NULL, buf.data, buf.size_data);
      if (! tmp)
      {
        fprintf(stderr, "UdmSQLEscStr failed\n");
        continue;
      }
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "INSERT INTO %s VALUES('%s', %d, '%s')",
        table, word->word, word->secno, tmp);
      UdmFree(tmp);
      if (UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data)))
      {
        fprintf(stderr, "UdmSQLQuery failed\n");
        goto end;
      }
    }

    UdmDSTRReset(&buf); 
    w = w1 - 1;
  }
  
end:
  UdmDSTRFree(&buf);
  return rc;
}

static int UdmBlobWriteURL (UDM_DB *db, const char *table)
{
  UDM_DSTR buf;
  UDM_DSTR r, s, l, p;
  UDM_SQLRES SQLRes;
  int rc= UDM_OK;
  UDM_PSTR row[4];
  int rec_id, site_id, last_mod_time;
  double pop_rank;
  int odbcbind= (db->DBDriver == UDM_DB_ODBC) && (db->DBType != UDM_DB_MSSQL);
  const char *fmt= (db->DBDriver == UDM_DB_ORACLE8 || odbcbind) ? "%c" : "%02X";

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&r, 8192);
  UdmDSTRInit(&s, 8192);
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&p, 8192);

  rc = db->sql->SQLExecDirect(db, &SQLRes, "SELECT rec_id, site_id, last_mod_time, pop_rank FROM url ORDER BY rec_id");
  if (rc != UDM_OK) return rc;

  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    rec_id = UDM_ATOI(row[0].val);
    site_id = UDM_ATOI(row[1].val);
    last_mod_time = UDM_ATOI(row[2].val);
    pop_rank = UDM_ATOF(row[3].val);

    UdmDSTRAppendf(&r, fmt, rec_id & 0xFF);
    UdmDSTRAppendf(&r, fmt, rec_id >> 8 & 0xFF);
    UdmDSTRAppendf(&r, fmt, rec_id >> 16 & 0xFF);
    UdmDSTRAppendf(&r, fmt, rec_id >> 24 & 0xFF);
    
    UdmDSTRAppendf(&s, fmt, site_id & 0xFF);
    UdmDSTRAppendf(&s, fmt, site_id >> 8 & 0xFF);
    UdmDSTRAppendf(&s, fmt, site_id >> 16 & 0xFF);
    UdmDSTRAppendf(&s, fmt, site_id >> 24 & 0xFF);
    
    UdmDSTRAppendf(&l, fmt, last_mod_time & 0xFF);
    UdmDSTRAppendf(&l, fmt, last_mod_time >> 8 & 0xFF);
    UdmDSTRAppendf(&l, fmt, last_mod_time >> 16 & 0xFF);
    UdmDSTRAppendf(&l, fmt, last_mod_time >> 24 & 0xFF);

    /* FIXME: little and big endian incompatibility: */
    if (db->DBDriver == UDM_DB_ORACLE8 || odbcbind)
      UdmDSTRAppend(&p, (char *)&pop_rank, sizeof(pop_rank));
    else
      UdmDSTRAppendf(&p, "%016X", pop_rank);
  }
  UdmSQLFree(&SQLRes);

  if (db->DBDriver == UDM_DB_ORACLE8)
  {
#ifdef HAVE_ORACLE8
      char qbuf[64];
      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#rec_id', 0, :1)", table);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, r.data, r.size_data, UDM_SQLTYPE_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);

      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#site_id', 0, :1)", table);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, s.data, s.size_data, UDM_SQLTYPE_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);

      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#last_mod_time', 0, :1)", table);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, l.data, l.size_data, UDM_SQLTYPE_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);

      udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#pop_rank', 0, :1)", table);
      db->sql->SQLPrepare(db, qbuf);
      if (db->errcode) fprintf(stderr, "[1] %s\n", db->errstr);
      db->sql->SQLBind(db, 1, p.data, p.size_data, UDM_SQLTYPE_LONGVARBINARY);
      if (db->errcode) fprintf(stderr, "[2] %s\n", db->errstr);
      db->sql->SQLExec(db);
      if (db->errcode) fprintf(stderr, "[3] %s\n", db->errstr);

      db->sql->SQLCommit(db);
#endif
  }
#ifdef HAVE_ODBC
  else if (odbcbind)
  {
    int bindtype= SQL_LONGVARBINARY;
    char qbuf[64];
    if (UDM_OK != (rc= db->sql->SQLBegin(db)))
      goto ex;

    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#rec_id', 0, ?)", table);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, r.data, r.size_data, bindtype)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ex;

    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#site_id', 0, ?)", table);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, s.data, s.size_data, bindtype)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ex;

    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#last_mod_time', 0, ?)", table);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, l.data, l.size_data, bindtype)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ex;

    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('#pop_tank', 0, ?)", table);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, p.data, p.size_data, bindtype)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ex;
    
    if (UDM_OK != (rc= db->sql->SQLCommit(db)))
      goto ex;
  }
#endif
  else
  {
    UdmDSTRAppendf(&buf, "INSERT INTO %s VALUES('#rec_id', 0, 0x%s)", table, r.data);
    UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRReset(&buf);

    UdmDSTRAppendf(&buf, "INSERT INTO %s VALUES('#site_id', 0, 0x%s)", table, s.data);
    UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRReset(&buf);

    UdmDSTRAppendf(&buf, "INSERT INTO %s VALUES('#last_mod_time', 0, 0x%s)", table, l.data);
    UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRReset(&buf);

    UdmDSTRAppendf(&buf, "INSERT INTO %s VALUES('#pop_rank', 0, 0x%s)", table, p.data);
    UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRReset(&buf);
  }

ex:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&r);
  UdmDSTRFree(&s);
  UdmDSTRFree(&l);
  UdmDSTRFree(&p);
  return rc;
}

#define BLOB_CACHE_SIZE 0xff

int UdmMulti2BlobSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  size_t t, i;
  int rc;
  UDM_SQLRES SQLRes;
  char buf[128];
  size_t srows = 0;
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  urlid_t url_id;
  unsigned char secno;
  char *intag;
  const char *word;
  size_t nintags;
  udmhash32_t word_seed;
  UDM_PSTR row[4];
  const char *wtable;
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  udm_snprintf(buf, sizeof(buf), "DELETE FROM %s", wtable);
  rc = UdmSQLQuery(db, NULL, buf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  for (i = 0; i <= BLOB_CACHE_SIZE; i++)
  {
    UdmBlobCacheInit(&cache[i]);
  }

  for (t = 0; t <= MULTI_DICTS; t++)
  {
    if (db->DBType == UDM_DB_MYSQL)
    {
      udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict%02X WRITE, %s WRITE", t, wtable);
      rc = UdmSQLQuery(db, NULL, buf);
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }

    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading dict%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id, secno, word, intag FROM dict%02X", t);
    rc = db->sql->SQLExecDirect(db, &SQLRes, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }

    UdmLog(Indexer, UDM_LOG_ERROR, "Converting dict%02X", t);
    while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
    {
      url_id = UDM_ATOI(row[0].val);
      secno = (unsigned char)UDM_ATOI(row[1].val);
      word = row[2].val;
      intag = row[3].val;
      nintags = udm_utf8_len(intag);
      word_seed = UdmStrHash32(word ? word : "") >> 8 & MULTI_DICTS;
      UdmBlobCacheAdd(&cache[word_seed],
                      url_id, secno, word, nintags, intag, row[3].len);
    }
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting dict%02X", t);
    for (i = 0; i <= BLOB_CACHE_SIZE; i++)
    {
      srows += cache[i].nwords;
      UdmBlobCacheSort(&cache[i]);
      rc= UdmBlobCacheWrite(db, &cache[i], wtable);
      UdmBlobCacheFree(&cache[i]);
      if (rc != UDM_OK)
        return rc;
    }
    UdmSQLFree(&SQLRes);

    if (db->DBType == UDM_DB_MYSQL)
    {
      rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }
  }

  UdmLog(Indexer, UDM_LOG_ERROR, "Total records converted: %d", srows);

  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(db, wtable)))
    return rc;

  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;
}


int UdmSingle2BlobSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  int rc;
  char buf[128];
  UDM_PSTR row[3];
  UDM_SQLRES SQLRes;
  size_t t, u, s, w;
  UDM_BLOB_CACHE bcache;
  UDM_MULTI_CACHE mcache;
  urlid_t url_id;
  UDM_WORD words;
  const char *wtable;
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  udm_snprintf(buf, sizeof(buf), "DELETE FROM %s", wtable);
  rc = UdmSQLQuery(db, NULL, buf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict WRITE, %s WRITE", wtable);
    rc = UdmSQLQuery(db, NULL, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  udm_snprintf(buf, sizeof(buf), "SELECT url_id, word, intag FROM dict");
  rc= db->sql->SQLExecDirect(db, &SQLRes, buf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  UdmMultiCacheInit(&mcache);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    url_id = UDM_ATOI(row[0].val);
    words.word = UdmStrdup(row[1].val);
    words.coord = UDM_ATOI(row[2].val);
    UdmMultiCacheAdd(&mcache, url_id, 0, &words);
  }
  UdmSQLFree(&SQLRes);
  UdmBlobCacheInit(&bcache);
  for (t = 0; t <= MULTI_DICTS; t++)
  {
    UDM_MULTI_CACHE_TABLE *table = &mcache.tables[t];
    for (u = 0; u < table->nurls; u++)
    {
      UDM_MULTI_CACHE_URL *url = &table->urls[u];
      for (s = 0; s < url->nsections; s++)
      {
        UDM_MULTI_CACHE_SECTION *section = &url->sections[s];
        for (w = 0; w < section->nwords; w++)
	{
	  UDM_MULTI_CACHE_WORD *word = &section->words[w];
	  char *intag = UdmMultiCachePutIntag1(word);
	  UdmBlobCacheAdd(&bcache, url->url_id, section->secno, word->word,
	                  word->nintags, intag, strlen(intag));
	}
      }
    }
  }
  UdmBlobCacheSort(&bcache);
  rc= UdmBlobCacheWrite(db, &bcache, wtable);
  UdmBlobCacheFree(&bcache);
  UdmMultiCacheFree(&mcache);

  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(db, wtable)))
    return rc;
  
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  UdmBlobSetTable(db);
  return(UDM_OK);
}




static int StoreWordsSingle(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  size_t  i;
  char  qbuf[256]="";
  time_t  stmp;
  int  rc=UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  stmp=time(NULL);
  
  /* Start transaction if supported */
  /* This is to make stuff faster   */
  
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLBegin(db)))
      return rc;
  
  /* Delete old words */
  sprintf(qbuf,"DELETE FROM dict WHERE url_id=%s%i%s", qu, url_id, qu);

  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
    goto unlock_StoreWordsSingle;
  
  /* Insert new words */
  if(db->DBType==UDM_DB_MYSQL)
  {
    if(Doc->Words.nwords)
    {
      size_t nstored=0;
      
      while(nstored < Doc->Words.nwords)
      {
        char * qb,*qe;
        size_t step=1024;
        size_t mlen=1024;
        size_t rstored = 0;

        qb=(char*)UdmMalloc(mlen);
        strcpy(qb,"INSERT INTO dict (word,url_id,intag) VALUES ");
        qe=qb+strlen(qb);

        for(i=nstored;i<Doc->Words.nwords;i++)
        {
          size_t len=qe-qb;
          if(!Doc->Words.Word[i].coord) 
          { 
            nstored++; 
            continue;
          }
          rstored++;
        
          /* UDM_MAXWORDSIZE+100 should be enough */
          if((len + Indexer->Conf->WordParam.max_word_len + 100) >= mlen)
          {
            mlen+=step;
            qb=(char*)UdmRealloc(qb,mlen);
            qe=qb+len;
          }
          
          if(i>nstored)*qe++=',';

          if(db->DBMode==UDM_DBMODE_SINGLE)
          {
            *qe++='(';
            *qe++='\'';
            strcpy(qe,Doc->Words.Word[i].word);
            while(*qe)qe++;
            *qe++='\'';
            *qe++=',';
            qe+=sprintf(qe,"%d,%d",url_id,Doc->Words.Word[i].coord);
            *qe++=')';
            *qe='\0';
          }
          if(qe>qb+UDM_MAX_MULTI_INSERT_QSIZE)
            break;
        }
        nstored = i + 1;
        rc = (rstored > 0) ? UdmSQLQuery(db, NULL, qb) : UDM_OK;
        UDM_FREE(qb);
        if(rc!=UDM_OK) goto unlock_StoreWordsSingle;
      }
    }
  }else{
    for(i=0;i<Doc->Words.nwords;i++)
    {
      if(!Doc->Words.Word[i].coord)continue;
        
      if(db->DBMode==UDM_DBMODE_SINGLE)
      {
        sprintf(qbuf,"INSERT INTO dict (url_id,word,intag) VALUES(%s%i%s,'%s',%d)", qu, url_id, qu, 
          Doc->Words.Word[i].word, Doc->Words.Word[i].coord);
      }
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
        goto unlock_StoreWordsSingle;
    }
  }
unlock_StoreWordsSingle:
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLCommit(db)))
      return rc;
  return(UDM_OK);
}

static size_t udm_get_utf8(size_t *pwc, const unsigned char *s, const unsigned char *e)
{
  unsigned char c;
  
  if (s >= e)
    return 0;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR;
  else if (c < 0xe0) 
  {
    if (s+2 > e) /* We need 2 characters */ 
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40))
      return 0;
    
    *pwc = ((size_t) (c & 0x1f) << 6) | (size_t) (s[1] ^ 0x80);
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (s+3 > e) /* We need 3 characters */
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  return 0;
}

static size_t udm_put_utf8(size_t wc, unsigned char *r, unsigned char *e)
{
  int count;
  
  if (r >= e)
    return 0;
  
  if (wc < 0x80) 
    count = 1;
  else if (wc < 0x800) 
    count = 2;
  else if (wc < 0x10000) 
    count = 3;
  else return 0;
  
  /* 
    e is a character after the string r, not the last character of it.
    Because of it (r+count > e), not (r+count-1 >e )
   */
  if ( r+count > e ) 
    return 0;
  
  switch (count)
  { 
    /* Fall through all cases */
    case 3: r[2] = (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
    case 2: r[1] = (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
    case 1: r[0] = (unsigned char) wc;
  }
  return count;
}

/************************** New fly-mode stuff *****************************/
/*
 * Public functions:
 *
 * UdmDBModeFlyClear - clears all fly-mode related tables
 * UdmDBModeFlyDelete - removes words of given document
 * UdmDBModeFlyStore - stores words from given document
 * UdmDBModeFlyMerge - merges indexing and searching tables
 * UdmDBModeFlyFind - search suite
 *
 * Private functions:
 *
 * UdmDBModeFlyWrite - flushes in-memory words cache
 * UdmDBModeFlyWriteDirect - writes data into fdicts
 *
 * UdmDBModeFlyWordsWrite - writes grouped words
 *
 * UdmDBModeFlyPackCoords - packs coords
 * UdmDBModeFlyUnpackCoords - unpacks coords
 */

static void UdmDBModeFlyAddCoords (UDM_RESULT *Res, UDM_SQLRES *SQLRes, size_t wordnum, int *wf, UDM_URL_TMP *urls)
{
  size_t coordslen;
  const unsigned char *s;
  const unsigned char *e;
  size_t nbytes;
  size_t ncoords;
  unsigned char nsections;
  urlid_t url_id;
  unsigned char secno;
  size_t coord;
  unsigned short last;
  size_t numrows;
  size_t i;
  size_t acoords;

  numrows = UdmSQLNumRows(SQLRes);
  if (! numrows) return;

  for (i = 0; i < numrows; i++)
  {
    size_t tmp = UdmSQLLen(SQLRes, i, 0);
    if (! tmp) continue;
    acoords += tmp;
  }
  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, (Res->CoordList.ncoords + acoords) * sizeof(UDM_URL_CRD));

  for (i = 0; i < numrows; i++)
  {
    s = (const unsigned char *)UdmSQLValue(SQLRes, i, 0);
    coordslen = UdmSQLLen(SQLRes, i, 0);
    if (! coordslen) continue;
    e = s + coordslen;

    while (e > s)
    {
      if (e - s < 7) return;
      url_id= (urlid_t)(s[0]+(s[1]<<8)+(s[2]<<16)+(s[3]<<24));
      s+= 4;
      nsections= *s++;

      while (nsections > 0)
      {
        secno= *s++;
        nbytes= udm_get_utf8(&ncoords, s, e);
        if (! nbytes) return;
        s+= nbytes;
        nsections--;
        last= 0;

        while (ncoords > 0)
        {
          nbytes= udm_get_utf8(&coord, s, e);
          if (! nbytes) return;
          s+= nbytes;
          last+= coord;
          ncoords--;

          if (wf[secno] && (! urls->nurls || bsearch(&url_id, urls->urls, urls->nurls, sizeof(urlid_t), (qsort_cmp)cmpaurls)))
          {
            Res->CoordList.Coords[Res->CoordList.ncoords].url_id = url_id;
            Res->CoordList.Coords[Res->CoordList.ncoords].coord = UDM_WRDCOORD(last, secno) + UDM_WRDNUM(wordnum);
            Res->CoordList.ncoords++;
            Res->WWList.Word[wordnum].count++;
          }
        }
      }
    }
  }

  Res->CoordList.Coords = UdmRealloc(Res->CoordList.Coords, Res->CoordList.ncoords * sizeof(UDM_URL_CRD));
}

static int UdmDBModeFlyClear (UDM_AGENT *Indexer, UDM_DB *db)
{
  if (db->flags & UDM_SQL_HAVE_TRUNCATE)
  {
    if (UdmSQLQuery(db, NULL, "TRUNCATE TABLE fdicti") != UDM_OK) return(UDM_ERROR);
    if (UdmSQLQuery(db, NULL, "TRUNCATE TABLE fdictw") != UDM_OK) return(UDM_ERROR);
    if (UdmSQLQuery(db, NULL, "TRUNCATE TABLE fdicts") != UDM_OK) return(UDM_ERROR);
  }
  else
  {
    if (UdmSQLQuery(db, NULL, "DELETE FROM fdicti") != UDM_OK) return(UDM_ERROR);
    if (UdmSQLQuery(db, NULL, "DELETE FROM fdictw") != UDM_OK) return(UDM_ERROR);
    if (UdmSQLQuery(db, NULL, "DELETE FROM fdicts") != UDM_OK) return(UDM_ERROR);
  }

  return(UDM_OK);
}

static int UdmDBModeFlyDelete (UDM_AGENT *Indexer, UDM_DB *db, UDM_DOCUMENT *Doc)
{
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  UDM_SQLRES res;
  int rc= UDM_OK;
  char qbuf[64];
  size_t i;
  size_t nrows;
  UDM_DSTR wbuf;

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc= UdmSQLQuery(db, NULL, "LOCK TABLES fdictw WRITE");
    if (rc != UDM_OK) return(rc);
  }

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT words FROM fdictw WHERE url_id=%d", url_id);
  rc= UdmSQLQuery(db, &res, qbuf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  nrows= UdmSQLNumRows(&res);

  UdmDSTRInit(&wbuf, 256);
  for (i= 0; i < nrows; i++)
  {
    const char *row = UdmSQLValue(&res, i, 0);
    const char *s;
    const char *e;

    for (s= row; *s;)
    {
      for (e = s + 1; *e; e++) if (*e == ' ') break;
      UdmDSTRReset(&wbuf);
      UdmDSTRAppend(&wbuf, s + 1, e - s - 1);
      UdmWordCacheAdd(&db->WordCache, url_id, wbuf.data, 0);
      s = e;
    }
  }

  UdmDSTRFree(&wbuf);
  UdmSQLFree(&res);

  udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM fdictw WHERE url_id=%d", url_id);
  rc= UdmSQLQuery(db, NULL, qbuf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc= UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (rc != UDM_OK) return(rc);
  }

  return(rc);
}

static int UdmDBModeFlyWordsWriteCMP (UDM_WORD *w1, UDM_WORD *w2)
{
  return(strcmp(w1->word, w2->word));
}

static int UdmDBModeFlyWordsWrite (UDM_AGENT *Indexer, UDM_DB *db, UDM_DOCUMENT *Doc)
{
  size_t i;
  UDM_WORD *cword;
  UDM_WORD *pword= NULL;
  UDM_WORD *tmplist;
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  UDM_DSTR words;
  UDM_DSTR buf;
  int rc= UDM_OK;

  if (! Doc->Words.nwords) return(UDM_OK);

  tmplist= UdmMalloc(Doc->Words.nwords * sizeof(UDM_WORD));
  if (! tmplist) return(UDM_ERROR);
  memcpy(tmplist, Doc->Words.Word, Doc->Words.nwords * sizeof(UDM_WORD));
  UdmSort(tmplist, Doc->Words.nwords, sizeof(UDM_WORD), (qsort_cmp)UdmDBModeFlyWordsWriteCMP);

  UdmDSTRInit(&words, 4096);

  for (i = 0; i < Doc->Words.nwords; i++)
  {
    cword = &tmplist[i];

    if (! cword->coord) continue;

    if (! pword)
      UdmDSTRAppendSTR(&words, cword->word);
    else if (strcmp(cword->word, pword->word))
      UdmDSTRAppendf(&words, " %s", cword->word);

    pword= cword;
  }

  UdmFree(tmplist);

  if (words.size_data)
  {
    UdmDSTRInit(&buf, 4096);
    UdmDSTRAppendf(&buf, "INSERT INTO fdictw (url_id, words) VALUES(%d, '%s')", url_id, words.data);
    rc= UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRFree(&buf);
  }
  UdmDSTRFree(&words);

  return(rc);
}

/*
 * Function:
 *   UdmDBModeFlyPackCoords - groups and packs UDM_WORD_CACHE
 * Arguments:
 *   UDM_WORD_CACHE *cache (IN)
 *   size_t *pos (IN/OUT)
 *   UDM_DSTR *buf (OUT)
 * Returns:
 *   UDM_OK/UDM_ERROR
 */
static int UdmDBModeFlyPackCoords (UDM_WORD_CACHE *cache, size_t *pos, UDM_DSTR *buf)
{
  size_t ncoords;
  unsigned char nsections;
  UDM_DSTR coords;
  UDM_DSTR sections;
  UDM_WORD_CACHE_WORD *cword = &cache->words[*pos];
  UDM_WORD_CACHE_WORD *pword = NULL;
  unsigned char pbuf[3];
  unsigned char *pbufend = pbuf + sizeof(pbuf);
  size_t nbytes;
  const char *fmt = "%02X";
  size_t i;

  UdmDSTRInit(&coords, 1024);
  UdmDSTRInit(&sections, 1024);

  do
  {
    nsections= 0;
    UdmDSTRReset(&sections);
    do
    {
      ncoords= 0;
      UdmDSTRReset(&coords);
      do
      {
        nbytes = udm_put_utf8(cword->coord - (ncoords ? pword->coord : 0), pbuf, pbufend);
        ncoords++;
        UdmDSTRAppend(&coords, (char *)pbuf, nbytes);
        pword= cword;
        (*pos)++;
        if (*pos == cache->nwords) break;
        cword= &cache->words[*pos];
      }
      while (pword->secno == cword->secno &&
             pword->url_id == cword->url_id &&
             pword->seed == cword->seed &&
             ! strcmp(pword->word, cword->word));

      nsections++;
      nbytes = udm_put_utf8(ncoords, pbuf, pbufend);
      UdmDSTRAppend(&sections, (char *)&pword->secno, sizeof(pword->secno));
      UdmDSTRAppend(&sections, (char *)pbuf, nbytes);
      UdmDSTRAppend(&sections, coords.data, coords.size_data);
    }
    while (*pos < cache->nwords &&
           pword->url_id == cword->url_id &&
           pword->seed == cword->seed &&
           ! strcmp(pword->word, cword->word));

/*
    UdmDSTRAppend(buf, (char *)&pword->url_id, sizeof(pword->url_id));
    UdmDSTRAppend(buf, (char *)&nsections, 1);
    UdmDSTRAppend(buf, sections.data, sections.size_data);
*/

    UdmDSTRAppendf(buf, fmt, (unsigned char)(pword->url_id & 0xFF));
    UdmDSTRAppendf(buf, fmt, (unsigned char)(pword->url_id >> 8 & 0xFF));
    UdmDSTRAppendf(buf, fmt, (unsigned char)(pword->url_id >> 16 & 0xFF));
    UdmDSTRAppendf(buf, fmt, (unsigned char)(pword->url_id >> 24 & 0xFF));

    UdmDSTRAppendf(buf, fmt, nsections);
    for (i = 0; i < sections.size_data; i++)
    {
      UdmDSTRAppendf(buf, fmt, (unsigned char)sections.data[i]);
    }
  }
  while (*pos < cache->nwords &&
    pword->seed == cword->seed &&
    ! strcmp(pword->word, cword->word));

  UdmDSTRFree(&coords);
  UdmDSTRFree(&sections);

  return(UDM_OK);
}

static int UdmDBModeFlyUnpackCoords (UDM_WORD_CACHE *cache, const char *word, const unsigned char *coords, size_t coordslen)
{
  const unsigned char *s= coords;
  const unsigned char *e= coords + coordslen;
  size_t nbytes;
  size_t ncoords;
  unsigned char nsections;
  urlid_t url_id;
  unsigned char secno;
  size_t coord;
  unsigned short last;

  while (e > s)
  {
    if (e - s < 7)
    {
      fprintf(stderr, "Cannot extract url_id. Not enough bytes left. Word=%s\n", word);
      return(UDM_OK);
    }
    url_id= (urlid_t)(s[0]+(s[1]<<8)+(s[2]<<16)+(s[3]<<24));
    s+= 4;
    nsections= *s++;

    while (nsections > 0)
    {
      secno= *s++;
      nbytes= udm_get_utf8(&ncoords, s, e);
      if (! nbytes)
      {
        fprintf(stderr, "Cannot extract ncoords. Word=%s\n", word);
        return(UDM_OK);
      }
      s+= nbytes;
      nsections--;
      last= 0;

      while (ncoords > 0)
      {
        nbytes= udm_get_utf8(&coord, s, e);
        if (! nbytes)
        {
          fprintf(stderr, "Cannot extract coord. Word=%s\n", word);
          return(UDM_OK);
        }
        s+= nbytes;
        last+= coord;
        ncoords--;
        UdmWordCacheAdd(cache, url_id, word, UDM_WRDCOORD(last, secno));
      }
    }
  }

  return(UDM_OK);
}

static int UdmDBModeFlyWriteDirect (UDM_AGENT *Indexer, UDM_DB *db)
{
  size_t i;
  int rc;
  UDM_DSTR buf;
  UDM_DSTR coords;
  UDM_WORD_CACHE *cache = &db->WordCache;

  if(UDM_OK!=(rc=UdmSQLBegin(db)))
  {
    return(rc);
  }

  UdmWordCacheSort(cache);

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&coords, 8192);

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc= UdmSQLQuery(db, NULL, "LOCK TABLES fdicts WRITE");
    if (rc != UDM_OK) goto unlock_UdmStoreWordsMulti;
  }

  for (i = 0; i < cache->nwords;)
  {
    UDM_WORD_CACHE_WORD *cword = &cache->words[i];
    unsigned char seed = cword->seed;
    char *word = cword->word;

    UdmDSTRReset(&coords);
    rc= UdmDBModeFlyPackCoords(cache, &i, &coords);
    if (rc != UDM_OK) break;

    if (i < cache->nwords) cword= &cache->words[i];
    if (! coords.size_data) continue;

    if (db->DBType == UDM_DB_MYSQL)
    {
      if (buf.size_data)
      {
        UdmDSTRAppendf(&buf, ",('%s',0x%s)",
	               word, coords.data);
      } else {
        UdmDSTRAppendf(&buf, "INSERT INTO fdicts (word,coords) VALUES('%s',0x%s)",
                       word, coords.data);
      }

      if (seed != cword->seed || i == cache->nwords)
      {
        if (buf.size_data)
        {
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data))) break;
	  UdmDSTRReset(&buf);
        }
      }
    }
  }

unlock_UdmStoreWordsMulti:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&coords);
  UdmWordCacheFree(cache);

  if (rc == UDM_OK && db->DBType == UDM_DB_MYSQL)
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");

  if(rc==UDM_OK)
    rc=UdmSQLCommit(db);

  return(rc);
}

int UdmDBModeFlyMerge (UDM_AGENT *Indexer, UDM_DB *db)
{
  int rc= UDM_OK;
  UDM_SQLRES res;
  size_t i;
  char qbuf[128];

  for (i= 0; i <= 0xFF; i++)
  {
    UDM_PSTR buf[2];
    
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT word,coords FROM fdicti WHERE seed=%d", i);
    fprintf(stderr, "Reading...");
    rc= db->sql->SQLExecDirect(db, &res, qbuf);
    if (rc != UDM_OK)
    {
      fprintf(stderr, "Failed to get row");
      break;
    }

    fprintf(stderr, "Unpacking...%02X", i);
    while (db->sql->SQLFetchRow(db, &res, buf) == UDM_OK)
    {
      UdmDBModeFlyUnpackCoords(&db->WordCache, buf[0].val,
                               (unsigned char *)buf[1].val, buf[1].len);
    }
    fprintf(stderr, "Sorting...nrecs=%d, nbytes=%d\n", db->WordCache.nwords, db->WordCache.nbytes);
    UdmDBModeFlyWriteDirect(Indexer, db);
    UdmSQLFree(&res);
  }

  return(rc);
}

int UdmDBModeFlyWrite (UDM_AGENT *Indexer, UDM_DB *db, size_t limit)
{
  size_t i;
  int rc;
  UDM_WORD_CACHE *cache = &db->WordCache;
  UDM_DSTR buf;
  UDM_DSTR coords;

  if (cache->nbytes <= limit) return(UDM_OK);
  UdmLog(Indexer, UDM_LOG_ERROR, "Writing words (%d words, %d bytes%s).", cache->nwords, cache->nbytes, limit ? "" : ", final");

  if(UDM_OK!=(rc=UdmSQLBegin(db)))
  {
    UdmWordCacheFree(cache);
    return(rc);
  }

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&coords, 8192);

  UdmWordCacheSort(cache);

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc= UdmSQLQuery(db, NULL, "LOCK TABLES fdicti WRITE");
    if (rc != UDM_OK) goto unlock_UdmStoreWordsMulti;
  }

  for (i = 0; i < cache->nwords;)
  {
    UDM_WORD_CACHE_WORD *cword = &cache->words[i];
    unsigned char seed = cword->seed;
    char *word = cword->word;

    UdmDSTRReset(&coords);
    rc= UdmDBModeFlyPackCoords(cache, &i, &coords);
    if (rc != UDM_OK) break;

    if (i < cache->nwords) cword= &cache->words[i];
    if (! coords.size_data) continue;

    if (db->DBType == UDM_DB_MYSQL)
    {
      if (buf.size_data)
      {
        UdmDSTRAppendf(&buf, ",(%d,'%s',0x%s)",
	               seed, word, coords.data);
      } else {
        UdmDSTRAppendf(&buf, "INSERT INTO fdicti (seed,word,coords) VALUES(%d,'%s',0x%s)",
                       seed, word, coords.data);
      }

      if (seed != cword->seed || i == cache->nwords)
      {
        if (buf.size_data)
        {
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data))) break;
	  UdmDSTRReset(&buf);
        }
      }
    }
  }

unlock_UdmStoreWordsMulti:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&coords);

  if (rc == UDM_OK && db->DBType == UDM_DB_MYSQL)
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");

  if(rc==UDM_OK)
    rc=UdmSQLCommit(db);

  UdmWordCacheFree(&db->WordCache);
  UdmLog(Indexer, UDM_LOG_ERROR, "The words are written successfully.%s", limit ? "" : " (final)");
  return(rc);
}

static int UdmDBModeFlyStore (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i;
  int rc = UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  unsigned char PrevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0) ? 1 : 0;
  int WordCacheSize = UdmVarListFindInt(&Indexer->Conf->Vars, "WordCacheSize", 0);

  if (WordCacheSize <= 0) WordCacheSize = 0x800000;

  if (PrevStatus) UdmDBModeFlyDelete(Indexer, db, Doc);

  UdmDBModeFlyWordsWrite(Indexer, db, Doc);

  for (i = 0; i < Doc->Words.nwords; i++)
  {
    if (! Doc->Words.Word[i].coord) continue;
    UdmWordCacheAdd(&db->WordCache, url_id, Doc->Words.Word[i].word, Doc->Words.Word[i].coord);
  }

  rc= UdmDBModeFlyWrite(Indexer, db, WordCacheSize);
  return(rc);
}

/************************* /New fly-mode stuff *****************************/

static int UdmStoreWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int  res;
  
  switch(db->DBMode)
  {
    case UDM_DBMODE_MULTI:
      res=UdmStoreWordsMulti(Indexer,Doc,db);
      break;
    case UDM_DBMODE_FLY:
      res=UdmDBModeFlyStore(Indexer,Doc,db);
      break;
    case UDM_DBMODE_SINGLE:
    default:
      res=StoreWordsSingle(Indexer,Doc,db);
      break;
  }
  return(res);
}


static int UdmDeleteAllFromDict(UDM_AGENT *Indexer,UDM_DB *db)
{
  char  qbuf[512];
  size_t  i;
  int  rc=UDM_OK;
  
  switch(db->DBMode)
  {
  case UDM_DBMODE_MULTI:
    for(i = 0 ; i <= MULTI_DICTS; i++)
    {
      if (db->flags & UDM_SQL_HAVE_TRUNCATE)
        sprintf(qbuf,"TRUNCATE TABLE dict%02X", i);
      else
        sprintf(qbuf,"DELETE FROM dict%02X", i);

      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
        return rc;
    }
    break;
  case UDM_DBMODE_FLY:
    rc= UdmDBModeFlyClear(Indexer, db);
    break;
  default:
    if (db->flags & UDM_SQL_HAVE_TRUNCATE)
      rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE dict");
    else
      rc=UdmSQLQuery(db,NULL,"DELETE FROM dict");
    break;
  }
  return rc;
}


/***************** CrossWords *******************************/

static int UdmDeleteAllFromCrossDict(UDM_AGENT * Indexer,UDM_DB *db)
{
  char  qbuf[1024];
  strcpy(qbuf, "DELETE FROM crossdict");
  return UdmSQLQuery(db,NULL,qbuf);
}


static int UdmDeleteCrossWordFromURL(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t  referrer_id  =UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
  int  rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(url_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE url_id=%s%i%s", qu, url_id, qu);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      return rc;
  }
  if(referrer_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE ref_id=%s%i%s", qu, referrer_id, qu);
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  return rc;
}


static int UdmStoreCrossWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  UDM_DOCUMENT  U;
  size_t    i;
  char    qbuf[1024];
  const char  *lasturl="scrap";
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  urlid_t    referrer = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t    childid = 0;
  int    rc=UDM_OK;
  UDM_HREF        Href;
  UDM_URL         docURL;
  
  UdmDocInit(&U);
  bzero((void*)&Href, sizeof(Href));
  UdmVarListReplaceInt(&Doc->Sections, "Referrer-ID", referrer);
  if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,&U,db)))
  {
    UdmDocFree(&U);
    return rc;
  }
  
  if(Doc->CrossWords.ncrosswords==0)
  {
    UdmDocFree(&U);
    return rc;
  }
  
  UdmURLInit(&docURL);
  UdmURLParse(&docURL, UdmVarListFindStr(&Doc->Sections, "URL", ""));
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(!Doc->CrossWords.CrossWord[i].weight)continue;
    if(strcmp(lasturl,Doc->CrossWords.CrossWord[i].url))
    {
      Href.url = (char*)UdmStrdup(Doc->CrossWords.CrossWord[i].url);
      UdmConvertHref(Indexer, &docURL, &Doc->Spider, &Href);
      UdmVarListReplaceStr(&U.Sections, "URL", Href.url);
      UdmVarListReplaceInt(&U.Sections, "URL_ID", UdmStrHash32(Href.url));
      if(UDM_OK!=(rc=UdmFindURL(Indexer,&U,db)))
      {
        UdmDocFree(&U);
        UdmURLFree(&docURL);
        return rc;
      }
      childid = UdmVarListFindInt(&U.Sections,"ID",0);
      lasturl=Doc->CrossWords.CrossWord[i].url;
      UDM_FREE(Href.url);
    }
    Doc->CrossWords.CrossWord[i].referree_id=childid;
  }
  
  /* Begin transacttion/lock */
  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"LOCK TABLES crossdict WRITE");
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  else
  {
    rc= UdmSQLBegin(db);
  }
  
  if(rc!=UDM_OK)
    goto free_ex;
  
  /* Insert new words */
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(Doc->CrossWords.CrossWord[i].weight && Doc->CrossWords.CrossWord[i].referree_id)
    {
      int weight=UDM_WRDCOORD(Doc->CrossWords.CrossWord[i].pos,Doc->CrossWords.CrossWord[i].weight);
      sprintf(qbuf,"INSERT INTO crossdict (ref_id,url_id,word,intag) VALUES(%s%i%s,%s%i%s,'%s',%d)",
        qu, referrer, qu, qu, Doc->CrossWords.CrossWord[i].referree_id, qu,
        Doc->CrossWords.CrossWord[i].word, weight);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      {
        UdmDocFree(&U);
        goto unlock_UdmStoreCrossWords;
      }
    }
  }

unlock_UdmStoreCrossWords:
  /* COMMIT/UNLOCK */
  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"UNLOCK TABLES");
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  else
  {
    rc= UdmSQLCommit(db);
  }

free_ex:
  UdmDocFree(&U);
  UdmURLFree(&docURL);
  return rc;
}



static void SQLResToDoc(UDM_ENV *Conf, UDM_DOCUMENT *D, UDM_SQLRES *sqlres, size_t i)
{
  time_t    last_mod_time;
  char    dbuf[128];
  const char  *format = UdmVarListFindStr(&Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");
  double          pr;
  
  UdmVarListReplaceStr(&D->Sections,"URL",UdmSQLValue(sqlres,i,1));
  UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(sqlres,i,1)));
  last_mod_time=atol(UdmSQLValue(sqlres,i,2));
  UdmVarListReplaceInt(&D->Sections, "Last-Modified-Timestamp", (int) last_mod_time);
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Last-Modified",dbuf);
  UdmVarListReplaceStr(&D->Sections,"Content-Length",UdmSQLValue(sqlres,i,3));
  pr= atof(UdmSQLValue(sqlres,i,3)) / 1024;
  sprintf(dbuf, "%.2f", pr);
  UdmVarListReplaceStr(&D->Sections,"Content-Length-K",dbuf);  
  last_mod_time=atol(UdmSQLValue(sqlres,i,4));
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Next-Index-Time",dbuf);
  UdmVarListReplaceInt(&D->Sections, "Referrer-ID", UDM_ATOI(UdmSQLValue(sqlres,i,5)));
  UdmVarListReplaceInt(&D->Sections,"crc32",atoi(UdmSQLValue(sqlres,i,6)));
  UdmVarListReplaceStr(&D->Sections, "Site_id", UdmSQLValue(sqlres, i, 7));

#if BAR_COMMA_PERIOD_ORACLE_PROBLEM
  {
	char *num= UdmSQLValue(sqlres, i, 8);
	char *comma= strchr(num, ',');
	if (comma)
	  *comma= '.';
  }
#endif

  pr = atof(UdmSQLValue(sqlres, i, 8));
  snprintf(dbuf, 128, "%.5f", pr);
  UdmVarListReplaceStr(&D->Sections, "Pop_Rank", dbuf);
}

/************************ URLs ***********************************/


static int UdmAddURL(UDM_AGENT *Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    url_seed;
  int    use_crc32_url_id;
  int    usehtdburlid;
  int    rc=UDM_OK;
  size_t          len;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  urlid_t rec_id = 0;

  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");
  usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL)
  { 
    UDM_FREE(e_url);
    return UDM_ERROR;
  }
  
  url_seed = UdmStrHash32(url) & 0xFF;
  
  /* Escape URL string */
  UdmSQLEscStr(db, e_url, url, len);
  
  if(use_crc32_url_id || usehtdburlid)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    if (use_crc32_url_id) rec_id = UdmStrHash32(url);
    else rec_id = UdmVarListFindInt(&Doc->Sections, "HTDB_URL_ID", 0);
    
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (rec_id,url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES (%s%i%s,'%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
           qu, rec_id, qu,
           e_url,
           qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
           UdmVarListFindInt(&Doc->Sections,"Hops",0),
           (int)time(NULL),
           url_seed, (int)time(NULL),
           qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified", UdmVarListFindStr(&Doc->Sections, "Date", "")))
       );
  }else{
    /* Use dabatase generated rec_id */
    /* It depends on used DBType     */
    switch(db->DBType)
    {
    case UDM_DB_SOLID:
    case UDM_DB_ORACLE8:
    case UDM_DB_SAPDB:
      /* FIXME: Dirty hack for stupid too smart databases 
       Change this for config parameter checking */
/*      if (strlen(e_url)>UDM_URLSIZE)e_url[UDM_URLSIZE]=0;*/
      /* Use sequence next_url_id.nextval */
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,next_url_id.nextval,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MIMER:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,NEXT_VALUE OF rec_id_GEN,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;    
    case UDM_DB_IBASE:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,GEN_ID(rec_id_GEN,1),0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MYSQL:
      /* MySQL generates itself */
    default:  
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES ('%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
             e_url,
             qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
             UdmVarListFindInt(&Doc->Sections,"Hops",0),
             (int)time(NULL),
             url_seed, (int)time(NULL),
             qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified", UdmVarListFindStr(&Doc->Sections, "Date", "")))
         );
    }
  }

  /* Exec INSERT now */
  if(UDM_OK!=(rc=UdmSQLQuery(db, NULL, qbuf)))
  {
    UDM_FREE(qbuf); UDM_FREE(e_url); return rc;
  }

  if (! use_crc32_url_id && ! usehtdburlid)
  {
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
    {
      UDM_FREE(qbuf);
      UDM_FREE(e_url);
      return rc;
    }
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO links (ot,k,weight) VALUES (%s%i%s,%s%d%s,0.0)",
                 qu, UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0), qu,
                 qu, rec_id, qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, NULL, qbuf)))
    {
      UDM_FREE(qbuf); UDM_FREE(e_url); return rc;
    }
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

  UDM_FREE(qbuf); UDM_FREE(e_url);
  return UDM_OK;
}


static int UdmAddLink(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  size_t          len;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  urlid_t rec_id = 0;

  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL) 
  { 
    UDM_FREE(e_url); 
    return UDM_ERROR;
  }
  
  if (use_crc32_url_id)
  {
    rec_id = UdmStrHash32(url);
  }
  else
  {
    /* Escape URL string */
    UdmSQLEscStr(db, e_url, url, len);
  
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
    {
      UDM_FREE(qbuf);
      UDM_FREE(e_url);
      return rc;
    }
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO links (ot,k,weight) VALUES (%s%i%s,%s%d%s,0.0)",
                 qu, UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0), qu,
                 qu, rec_id, qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, NULL, qbuf)))
    {
      UDM_FREE(qbuf); UDM_FREE(e_url); return rc;
    }
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

  UDM_FREE(qbuf); UDM_FREE(e_url);
  return UDM_OK;
}



static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db);

static int UdmDeleteBadHrefs(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  UDM_DOCUMENT  rDoc;
  UDM_SQLRES  SQLRes;
  char    q[256];
  size_t    i;
  size_t    nrows;
  int    rc=UDM_OK;
  int    hold_period=UdmVarListFindInt(&Doc->Sections,"HoldBadHrefs",0);
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(q, sizeof(q), "SELECT rec_id FROM url WHERE status > 300 AND status<>304 AND referrer=%s%i%s AND bad_since_time<%d",
    qu, url_id, qu, qu, qu, (int)time(NULL) - hold_period);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,q)))return rc;
  
  nrows = UdmSQLNumRows(&SQLRes);
  
  UdmDocInit(&rDoc);
  for(i = 0; i < nrows ; i++)
  {
    UdmVarListReplaceStr(&rDoc.Sections,"ID", UdmSQLValue(&SQLRes,i,0));
    if(UDM_OK!=(rc=UdmDeleteURL(Indexer, &rDoc, db)))
      break;
  }
  UdmDocFree(&rDoc);
  UdmSQLFree(&SQLRes);
  return rc;
}

static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[128];
  int  rc;
  urlid_t  url_id  =UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  use_crosswords;
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if(use_crosswords)
    if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,Doc,db)))return(rc);
  
  if(UDM_OK!=(rc=UdmDeleteWordFromURL(Indexer,Doc,db)))return(rc);
  
  sprintf(qbuf,"DELETE FROM url WHERE rec_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE k=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if (UDM_OK != (rc=UdmDeleteBadHrefs(Indexer,Doc, db))) return rc;

  sprintf(qbuf,"UPDATE url SET referrer=%s0%s WHERE referrer=%s%i%s", qu, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}

static int UdmDeleteLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char  qbuf[128];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}

static int UdmGetCachedCopy (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
#ifdef HAVE_ZLIB
  UDM_SQLRES SQLRes;
  z_stream zstream;
  char buf[1024];
  const char *sname;
  const char *sval;
  int _;
  int i;

/*  UdmURLAction(Indexer, Doc, UDM_URL_ACTION_FINDBYURL); */
  UdmFindURL(Indexer, Doc, db);
  udm_snprintf(buf, sizeof(buf), "SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);
  if (! UdmSQLNumRows(&SQLRes)) return(UDM_ERROR);
  SQLResToDoc(Indexer->Conf, Doc, &SQLRes, 0);
  UdmSQLFree(&SQLRes);
  udm_snprintf(buf, sizeof(buf), "SELECT sname, sval FROM urlinfo WHERE url_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);

  for (i = 0; i < UdmSQLNumRows(&SQLRes); i++)
  {
    sname = UdmSQLValue(&SQLRes, i, 0);
    sval = UdmSQLValue(&SQLRes, i, 1);
    if (! sname) continue;
    if (! sval) sval = "";

    if (! strcmp(sname, "CachedCopy")) 
    {
      size_t l;
      char *in_buf;

      if (Doc->Buf.content) continue;

      l = strlen(sval);
      Doc->Buf.buf = UdmMalloc(UDM_MAXDOCSIZE);

      in_buf = UdmMalloc(l);
      zstream.next_in = (Byte *)in_buf;
      zstream.avail_in = udm_base64_decode((char *)zstream.next_in, sval, l);
      zstream.next_out = (Byte *)Doc->Buf.buf;
      zstream.avail_out = UDM_MAXDOCSIZE-1;
      zstream.zalloc = Z_NULL;
      zstream.zfree = Z_NULL;
      zstream.opaque = Z_NULL;

      if (inflateInit2(&zstream, 15) != Z_OK) 
      {
        UdmFree(Doc->Buf.buf);
        UdmFree(in_buf);
        Doc->Buf.buf = NULL;
        return(UDM_ERROR);
      }

      inflate(&zstream, Z_FINISH);
      inflateEnd(&zstream);
      Doc->Buf.size = zstream.total_out;
      Doc->Buf.content = Doc->Buf.buf;
      Doc->Buf.buf[Doc->Buf.size]= '\0';
      UdmFree(in_buf);
    } else {
      UdmVarListReplaceStr(&Doc->Sections, sname, sval);
    }
  }
  
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
#else
  return(UDM_ERROR);
#endif
}

static int UdmMarkForReindex(UDM_AGENT *Indexer,UDM_DB *db)
{
  char    qbuf[1024];
  const char  *where;
  UDM_SQLRES   SQLRes;
  size_t          i, j;
  int             rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR buf;

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  if (db->flags & UDM_SQL_HAVE_SUBSELECT) 
  {
    udm_snprintf(qbuf,sizeof(qbuf),"UPDATE url SET next_index_time=%d WHERE rec_id IN (SELECT url.rec_id FROM url%s %s %s)",
       (int)time(NULL), db->from, (where[0]) ? "WHERE" : "", where);
    return UdmSQLQuery(db,NULL,qbuf);
  }

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s %s %s", db->from, (where[0]) ? "WHERE" : "", where);
  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) return rc;

  UdmDSTRInit(&buf, 4096);
  if (db->DBSQL_IN) 
  {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i += 512) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id IN (", (int)time(NULL));
      for (j = 0; (j < 512) && (i + j < UdmSQLNumRows(&SQLRes)); j++) 
      {
        UdmDSTRAppendf(&buf, "%s%s%s%s", (j) ? "," : "", qu, UdmSQLValue(&SQLRes, i + j, 0), qu);
      }
      UdmDSTRAppendf(&buf, ")");
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  } else {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i++) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id=%s", (int)time(NULL),  UdmSQLValue(&SQLRes, i, 0));
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  }
  UdmDSTRFree(&buf);
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int UdmRegisterChild(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  urlid_t  parent_id = UdmVarListFindInt(&Doc->Sections,"Parent-ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf),"insert into links (ot,k,weight) values(%s%i%s,%s%i%s,0.0)", qu, parent_id, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}


static int
UdmUpdateUrl(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char qbuf[256];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  status=UdmVarListFindInt(&Doc->Sections,"Status",0);
  int  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  int  next_index_time=UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time",""));
  int  res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (prevStatus != status && status > 300 && status != 304)
    sprintf(qbuf, "UPDATE url SET status=%d,next_index_time=%d,bad_since_time=%d,site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, (int)time(NULL), qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);
  else
    sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d, site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
            qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);

  if(UDM_OK!=(res=UdmSQLQuery(db,NULL,qbuf)))return res;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  return UdmDeleteBadHrefs(Indexer,Doc,db);
}

static int
UdmUpdateUrlWithLangAndCharset(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  *qbuf;
  int  rc;
  const char  *charset;
  UDM_VAR    *var;
  int    status, prevStatus;
  urlid_t         url_id;
  size_t    i, len = 0;
  char    qsmall[64];
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int IndexTime= UdmVarListFindInt(&Indexer->Conf->Vars, "IndexTime", 0);
  
  status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  
  if((var=UdmVarListFind(&Doc->Sections,"Content-Language")))
  {
    if (var->val == NULL)
    {
      var->val = (char*)UdmStrdup(UdmVarListFindStr(&Doc->Sections, "DefaultLang", "en"));
    }
    len=strlen(var->val);
    for(i = 0; i < len; i++)
    {
      var->val[i] = tolower(var->val[i]);
    }
  }
  
  charset = UdmVarListFindStr(&Doc->Sections, "Charset", 
            UdmVarListFindStr(&Doc->Sections, "RemoteCharset", "iso-8859-1"));
  charset = UdmCharsetCanonicalName(charset);
  UdmVarListReplaceStr(&Doc->Sections, "Charset", charset);
  
  if (prevStatus != status && status > 300 && status != 304)
    udm_snprintf(qsmall, 64, ", bad_since_time=%d", (int)time(NULL));
  else qsmall[0] = '\0';

  if (IndexTime)
  {
    if (! prevStatus) udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", time(NULL));
  }
  else
  {
    udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified", UdmVarListFindStr(&Doc->Sections, "Date", ""))));
  }
  qbuf=(char*)UdmMalloc(1024);
  
  
  udm_snprintf(qbuf, 1023, "\
UPDATE url SET \
status=%d,\
next_index_time=%li,\
docsize=%d,\
crc32=%d%s, site_id=%s%i%s, server_id=%s%i%s \
WHERE rec_id=%s%i%s",
  status,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
  UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
  UdmVarListFindInt(&Doc->Sections,"crc32",0),
  qsmall,
  qu, UdmVarListFindInt(&Doc->Sections,"Site_id",0), qu,
  qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu,
  qu, url_id, qu);
  
  rc=UdmSQLQuery(db,NULL,qbuf);
  UDM_FREE(qbuf);
  return rc;
}


static int UdmLongUpdateURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int    rc=UDM_OK;
  size_t    i;
  char    *qbuf;
  char    *arg;
  char    qsmall[128];
  size_t    len=0;
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char  *c, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int odbcbind= (db->DBDriver == UDM_DB_ODBC) ? 1 : 0;
  int use_crosswords;

#ifdef WIN32
  if (db->DBType != UDM_DB_ORACLE8)
    odbcbind= 0;
#endif

  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  /* Now store words and crosswords */
  if(UDM_OK != (rc = UdmStoreWords(Indexer, Doc, db)))
    return rc;
  
  if(use_crosswords)
    if(UDM_OK != (rc = UdmStoreCrossWords(Indexer, Doc, db)))
      return rc;
  

  /* Copy default languages, if not given by server and not guessed */
  if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL)))
  {
    if((c=UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
      UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
  }
  

  if(UDM_OK != (rc = UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db)))
    return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if(UDM_OK!=(rc=UdmDeleteBadHrefs(Indexer,Doc,db)))
    return rc;
  
  sprintf(qsmall,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qsmall)))return rc;

/* No need delete from links here, it has been done before */
  
  if(!Doc->Sections.nvars) return UDM_OK;
  
  len=0;
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    size_t l = Doc->Sections.Var[i].curlen + (Doc->Sections.Var[i].name ? strlen(Doc->Sections.Var[i].name) : 0);
    if(len < l)
      len = l;
  }
  if(!len)return UDM_OK;
  
  qbuf=(char*)UdmMalloc(2 * len + 128);
  arg=(char*)UdmMalloc(2 * len + 128);
  
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    UDM_VAR *Sec=&Doc->Sections.Var[i];
    if(Sec->val && Sec->name && ((Sec->curlen && Sec->maxlen) || (!strcmp(Sec->name, "Z"))) )
    {
#ifdef HAVE_ORACLE8
      if (db->DBDriver == UDM_DB_ORACLE8 && db->sql->SQLPrepare)
      {
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', :1)",
          url_id, Sec->name);
        db->sql->SQLPrepare(db, qbuf);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot prepare query: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLBind(db, 1, Sec->val, strlen(Sec->val), UDM_SQLTYPE_LONGVARCHAR);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot bind value: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLExec(db);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot execute query: errcode=%d, %s\n", db->errcode, db->errstr);
        continue;
      }
#endif
#ifdef HAVE_ODBC
      if (odbcbind && db->sql->SQLPrepare)
      {
        int bindtype= db->DBType == UDM_DB_MSSQL ? SQL_VARCHAR : SQL_LONGVARCHAR;
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', ?)",
                url_id, Sec->name);
        if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
            UDM_OK != (rc= db->sql->SQLBind(db, 1, Sec->val, strlen(Sec->val), bindtype)) ||
            UDM_OK != (rc= db->sql->SQLExec(db)))
          break;
        continue;
      }
#endif
      arg=UdmSQLEscStr(db,arg,Sec->val,strlen(Sec->val));
      sprintf(qbuf,"INSERT INTO urlinfo (url_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
              qu, url_id, qu, Sec->name, arg);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))break;

    }
  }
  UDM_FREE(qbuf);
  UDM_FREE(arg);
  return rc;
}


static int
UdmUpdateClone(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int rc;
  int use_crosswords;
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer, Doc, db)))
    return rc;
  if(use_crosswords)
  {
    if (UDM_OK != (rc= UdmDeleteCrossWordFromURL(Indexer, Doc, db)))
      return rc;
  }
  rc= UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db);
  return rc;
}


static int
UdmDeleteWordsAndLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char qbuf[128];
  int  rc;
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int use_crosswords;
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (use_crosswords)
    if (UDM_OK!= (rc= UdmDeleteCrossWordFromURL(Indexer,Doc,db)))
      return rc;

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer,Doc,db)))
    return rc;

  if (UDM_OK != (rc= UdmDeleteLinks(Indexer, Doc, db)))
    return rc;

  /* Set status, bad_since_time, etc */
  if (UDM_OK != (rc= UdmUpdateUrl(Indexer, Doc, db)))
    return rc;

  return rc;
}


static int UdmDeleteAllFromUrl(UDM_AGENT *Indexer,UDM_DB *db)
{
  int  rc;
  
  if(db->flags & UDM_SQL_HAVE_TRUNCATE)
    rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE url");
  else
    rc=UdmSQLQuery(db,NULL,"DELETE FROM url");
  
  if(rc!=UDM_OK)return rc;
  
  if(db->flags & UDM_SQL_HAVE_TRUNCATE)
    rc = UdmSQLQuery(db, NULL, "TRUNCATE TABLE links");
  else
    rc = UdmSQLQuery(db, NULL, "DELETE FROM links");
  
  if(rc != UDM_OK) return rc;
  
  if(db->flags & UDM_SQL_HAVE_TRUNCATE)
    rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE urlinfo");
    
  else
    rc=UdmSQLQuery(db,NULL,"DELETE FROM urlinfo");
  
  return rc;
}



/************************ Clones stuff ***************************/
static int UdmFindOrigin(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  size_t    i=0;
  char    qbuf[256]="";
  UDM_SQLRES  SQLRes;
  urlid_t    origin_id = 0;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  int    rc;
  
  if (scrc32==0)return UDM_OK;
  
  if (db->DBSQL_IN)
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status IN (200,304,206)",scrc32);
  else
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206)",scrc32);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char *o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
      if((!origin_id) || (origin_id > UDM_ATOI(o)))
        origin_id = UDM_ATOI(o);
  }
  UdmSQLFree(&SQLRes);
  UdmVarListReplaceInt(&Doc->Sections, "Origin-ID", origin_id);
  return(UDM_OK);
}



/************** Get Target to be indexed ***********************/

int UdmTargetsSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  char    sortstr[128] = "";
  char    updstr[64]="";
  char    lmtstr[64]="";
  size_t    i = 0, j, start, nrows, qbuflen;
  size_t    url_num;
  UDM_SQLRES   SQLRes;
  char    smallbuf[128];
  int    rc=UDM_OK;
  const char  *where;
  char    *qbuf=NULL;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char  *select_seed = (db->DBType == UDM_DB_MIMER) ? ",url.seed" : "";

  url_num = UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_SELECT_CACHE);
  if (Indexer->Conf->url_number < url_num)
    url_num= Indexer->Conf->url_number;
  where = BuildWhere(Indexer->Conf, db);

  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        rc=UdmSQLQuery(db,NULL,"LOCK TABLES url WRITE, server AS s WRITE, categories AS c WRITE");
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
        sprintf(updstr, " FOR UPDATE ");
/*        rc=UdmSQLQuery(db,NULL,"LOCK url");*/
        break;
      case UDM_DB_ORACLE8:
        sprintf(updstr, " FOR UPDATE ");
#if HAVE_ORACLE8
        if(db->DBDriver==UDM_DB_ORACLE8)
        {
          sprintf(lmtstr, " AND ROWNUM <=%d",url_num); 
        }
#endif
        if(!lmtstr[0])
          sprintf(lmtstr, " AND ROWNUM <=%d", url_num); 
        break;
      case UDM_DB_SAPDB:
        sprintf(updstr, " WITH LOCK ");
        strcpy(lmtstr, "");
        break;
      default:
        break;
    }
    if(rc!=UDM_OK)return rc;
  }
  
  qbuflen = 1024 + 4 * strlen(where);
    
  if ((qbuf = (char*)UdmMalloc(qbuflen + 2)) == NULL)
  {
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc= UDM_ERROR;
      goto commit;
  }
  qbuf[0]='\0';

  if ( (Indexer->flags & (UDM_FLAG_SORT_HOPS | UDM_FLAG_SORT_EXPIRED)) || !(Indexer->flags & UDM_FLAG_DONTSORT_SEED)  )
  {
    
    sprintf(sortstr, " ORDER BY %s%s%s", 
      (Indexer->flags & UDM_FLAG_SORT_HOPS) ? "hops" : "",
      (Indexer->flags & UDM_FLAG_DONTSORT_SEED) ? "" : ((Indexer->flags & UDM_FLAG_SORT_HOPS) ? ",seed" : "seed"),
      (Indexer->flags & UDM_FLAG_SORT_EXPIRED) ? 
      ( ((Indexer->flags & UDM_FLAG_SORT_HOPS) || !(Indexer->flags & UDM_FLAG_DONTSORT_SEED)  ) ? 
        ",next_index_time" : "next_index_time") : "");
  } else {
    sortstr[0] = '\0';
  }
    
      
  if(db->flags & UDM_SQL_HAVE_LIMIT)
  {
    udm_snprintf(qbuf, qbuflen, "SELECT url.url,url.rec_id,docsize,status,hops,crc32,last_mod_time,seed FROM url%s WHERE next_index_time<=%d %s %s %s LIMIT %d%s",
      db->from, (int)time(NULL), where[0] ? "AND" : "", where, sortstr, url_num, updstr);
  }
  else
  {
    char top[64]="";
    if (db->DBType == UDM_DB_MSSQL)
      sprintf(top, " TOP %d", url_num);
    db->res_limit=url_num;
    if(!qbuf[0])
      udm_snprintf(qbuf, qbuflen, "SELECT %surl.url,url.rec_id,docsize,status,hops,crc32,last_mod_time%s FROM url%s WHERE next_index_time<=%d %s %s%s%s%s",
        top,select_seed,db->from, (int)time(NULL), where[0] ? "AND" : "", where, lmtstr, sortstr, updstr);
  }
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes, qbuf)))
    goto commit;
  
  if(!(nrows = UdmSQLNumRows(&SQLRes)))
  {
    UdmSQLFree(&SQLRes);
    goto commit;
  }

  start = Indexer->Conf->Targets.num_rows;
  Indexer->Conf->Targets.num_rows += nrows;
  
  Indexer->Conf->Targets.Doc = 
    (UDM_DOCUMENT*)UdmRealloc(Indexer->Conf->Targets.Doc, sizeof(UDM_DOCUMENT)*(Indexer->Conf->Targets.num_rows + 1));
  if (Indexer->Conf->Targets.Doc == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory at realloc %s[%d]", __FILE__, __LINE__);
    rc= UDM_ERROR;
    goto commit;
  }
  
  for(i = 0; i < nrows; i++)
  {
    char    buf[64]="";
    time_t    last_mod_time;
    UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
    
    UdmDocInit(Doc);
    UdmVarListAddStr(&Doc->Sections,"URL",UdmSQLValue(&SQLRes,i,0));
    UdmVarListReplaceInt(&Doc->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLRes,i,0)));
    UdmVarListAddInt(&Doc->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLRes,i,1)));
    UdmVarListAddInt(&Doc->Sections,"Content-Length",atoi(UdmSQLValue(&SQLRes,i,2)));
    UdmVarListAddInt(&Doc->Sections,"Status",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"PrevStatus",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"Hops",atoi(UdmSQLValue(&SQLRes,i,4)));
    UdmVarListAddInt(&Doc->Sections,"crc32",atoi(UdmSQLValue(&SQLRes,i,5)));
    last_mod_time = (time_t) atol(UdmSQLValue(&SQLRes,i,6));
    UdmTime_t2HttpStr(last_mod_time, buf);
    if (last_mod_time != 0 && strlen(buf) > 0)
    {
      UdmVarListReplaceStr(&Doc->Sections,"Last-Modified",buf);
    }
  }
  UdmSQLFree(&SQLRes);
  
  
  if (db->DBSQL_IN)
  {
    char  *urlin=NULL;
    
    if ( (qbuf = (char*)UdmRealloc(qbuf, qbuflen = qbuflen + 35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc= UDM_ERROR;
      goto commit;
    }
    
    if ( (urlin = (char*)UdmMalloc(35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc = UDM_ERROR;
      goto commit;
    }
    urlin[0]=0;
    
    for(i = 0; i < nrows; i+= URL_SELECT_CACHE)
    {

      urlin[0] = 0;

      for (j = 0; (j < URL_SELECT_CACHE) && (i + j < nrows) ; j++)
      {

      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i + j];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      if(urlin[0])strcat(urlin,",");
      sprintf(urlin+strlen(urlin), "%s%i%s", qu, url_id, qu);
      }
      udm_snprintf(qbuf, qbuflen, "UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",
             (int)(time(NULL) + URL_LOCK_TIME), urlin);
      if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
        goto commit;
    }
    UDM_FREE(urlin);
  }
  else
  {
    for(i = 0; i < nrows; i++)
    {
      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      udm_snprintf(smallbuf, 128, "UPDATE url SET next_index_time=%d WHERE rec_id=%i",
             (int)(time(NULL) + URL_LOCK_TIME), url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,smallbuf)))
        goto commit;
    }
  }


commit:

  if (rc != UDM_OK)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "UdmTargetsSQL: DB error: %s", db->errstr);
  }
  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        rc=UdmSQLQuery(db,NULL,"UNLOCK TABLES");
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"END WORK");
        break;
      default:
        break;
    }
  }
  UDM_FREE(qbuf);
  return rc;
}


/************************************************************/
/* Misc functions                                           */
/************************************************************/

int UdmGetDocCount(UDM_AGENT * Indexer,UDM_DB *db)
{
  char    qbuf[200]="";
  UDM_SQLRES  SQLres;
  int    rc;
  
  sprintf(qbuf,NDOCS_QUERY);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  if(UdmSQLNumRows(&SQLres))
  {
    const char * s;
    s=UdmSQLValue(&SQLres,0,0);
    if(s)Indexer->doccount += atoi(s);
  }
  UdmSQLFree(&SQLres);
  return(UDM_OK);
}


int UdmStatActionSQL(UDM_AGENT *Indexer,UDM_STATLIST *Stats,UDM_DB *db)
{
  size_t    i,j,n;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  int    have_group= (db->flags & UDM_SQL_HAVE_GROUPBY);
  const char  *where;
  int    rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(db->DBType==UDM_DB_IBASE)
    have_group=0;

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);

  if(have_group)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(next_index_time<=%d),count(*) FROM url%s WHERE url.rec_id<>0 %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, where[0] ? "AND" : "", where);
        break;
    
      case UDM_DB_PGSQL:
      case UDM_DB_MSSQL:
      case UDM_DB_DB2:
      case UDM_DB_SQLITE:
      default:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(case when next_index_time<=%d then 1 else 0 end),count(*) FROM url%s WHERE url.rec_id<>%s0%s %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, qu, qu, where[0] ? "AND" :"", where);
        break;

      case UDM_DB_ACCESS:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(IIF(next_index_time<=%d, 1, 0)),count(*) FROM url%s WHERE url.rec_id<>%s0%s %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, qu, qu, where[0] ? "AND" :"", where);
        break;

      case UDM_DB_ORACLE8:
      case UDM_DB_SAPDB:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status, SUM(DECODE(SIGN(%d-next_index_time),-1,0,1,1)), count(*) FROM url%s WHERE url.rec_id<>0 %s %s GROUP BY status ORDER BY status",
           Stats->time, db->from, where[0] ? "AND" : "", where);
        break;
    }

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    if((n = UdmSQLNumRows(&SQLres)))
    {
      for(i = 0; i < n; i++)
      {
        for(j=0;j<Stats->nstats;j++)
        {
          if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
          {
            Stats->Stat[j].expired += atoi(UdmSQLValue(&SQLres,i,1));
            Stats->Stat[j].total += atoi(UdmSQLValue(&SQLres,i,2));
            break;
          }
        }
        if(j==Stats->nstats)
        {
          UDM_STAT  *S;
        
          Stats->Stat=(UDM_STAT*)UdmRealloc(Stats->Stat,(Stats->nstats+1)*sizeof(Stats->Stat[0]));
          S=&Stats->Stat[Stats->nstats];
          S->status=atoi(UdmSQLValue(&SQLres,i,0));
          S->expired=atoi(UdmSQLValue(&SQLres,i,1));
          S->total=atoi(UdmSQLValue(&SQLres,i,2));
          Stats->nstats++;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
  }else{
/*  
  FIXME: learn how to get it from SOLID and IBASE
  (HAVE_IBASE || HAVE_SOLID || HAVE_VIRT )
*/
    
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,next_index_time FROM url%s WHERE url.rec_id>0 %s %s ORDER BY status",
      db->from, where[0] ? "AND" : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    for(i=0;i<UdmSQLNumRows(&SQLres);i++)
    {
      for(j=0;j<Stats->nstats;j++)
      {
        if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
        {
          if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
            Stats->Stat[j].expired++;
          Stats->Stat[j].total++;
          break;
        }
      }
      if(j==Stats->nstats)
      {
        Stats->Stat=(UDM_STAT *)UdmRealloc(Stats->Stat,sizeof(UDM_STAT)*(Stats->nstats+1));
        Stats->Stat[j].status = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
        Stats->Stat[j].expired=0;
        if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
          Stats->Stat[j].expired++;
        Stats->Stat[j].total=1;
        Stats->nstats++;
      }
    }
    UdmSQLFree(&SQLres);
  }
  return rc;
}


static int UdmGetReferers(UDM_AGENT *Indexer,UDM_DB *db)
{
  size_t    i,j;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  const char  *where;
  int    rc;

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
    
  udm_snprintf(qbuf,sizeof(qbuf),"SELECT url.status,url2.url,url.url FROM url,url url2%s WHERE url.referrer=url2.rec_id %s %s",
    db->from, where[0] ? "AND" : "", where);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  j=UdmSQLNumRows(&SQLres);
  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  for(i=0;i<j;i++)
  {
    if(Indexer->Conf->RefInfo)Indexer->Conf->RefInfo(
      atoi(UdmSQLValue(&SQLres,i,0)),
      UdmSQLValue(&SQLres,i,2),
      UdmSQLValue(&SQLres,i,1)
    );
  }
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  UdmSQLFree(&SQLres);
  return rc;
}


int UdmClearDBSQL(UDM_AGENT *Indexer,UDM_DB *db)
{
  size_t    i,j;
  int    rc;
  const char*  where;
  int    use_crosswords;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  if(!where[0])
  {
    if(use_crosswords)
    {
      if((UDM_OK!=(rc=UdmDeleteAllFromCrossDict(Indexer,db))))return(rc);
    }
    if((UDM_OK!=(rc=UdmDeleteAllFromDict(Indexer,db))))return(rc);
    if((UDM_OK!=(rc=UdmDeleteAllFromUrl(Indexer,db))))return(rc);
  }else{
    UDM_DSTR qbuf, urlin;
    UdmDSTRInit(&qbuf, 4096);
    UdmDSTRInit(&urlin, 4096);
    j=0;
    while(1)
    {
      char    limit[100]="";
      UDM_SQLRES  SQLres;
      size_t    url_num;
      
      url_num = UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_DELETE_CACHE);

      if(db->flags & UDM_SQL_HAVE_LIMIT)
      {
        sprintf(limit," LIMIT %d", url_num);
      }
      UdmDSTRReset(&qbuf);
      UdmDSTRAppendf(&qbuf,"SELECT url.rec_id, url.url FROM url%s WHERE url.rec_id<>%s0%s AND %s %s", 
        db->from, qu, qu,  where, limit);
      
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf.data)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        UDM_DOCUMENT Doc;
        
        bzero((void*)&Doc, sizeof(Doc));
        if(db->DBSQL_IN)
        {
          UdmDSTRReset(&urlin);
          for(i=0;i<UdmSQLNumRows(&SQLres);i++)
          {
            if(i) UdmDSTRAppend(&urlin,",", 1);
            UdmDSTRAppendf(&urlin, "%s%s%s", qu, UdmSQLValue(&SQLres,i,0), qu);
          }
          j+=i;
          switch(db->DBMode)
          {
          case UDM_DBMODE_MULTI:
            for(i = 0; i <= MULTI_DICTS; i++)
            {
              UdmDSTRReset(&qbuf);
              UdmDSTRAppendf(&qbuf,"DELETE FROM dict%02X WHERE url_id in (%s)", i, urlin.data);
              if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
              {
                UdmSQLFree(&SQLres);
                return rc;
              }
            }
            break;
          default:
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin.data);
            if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            {
              UdmSQLFree(&SQLres);
              return rc;
            }
            break;
          }
          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;
          
          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM urlinfo WHERE url_id in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE ot in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE k in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmSQLFree(&SQLres);
        }
        else
        {
          for(i=0;i<UdmSQLNumRows(&SQLres);i++)
          {
            UdmVarListReplaceInt(&Doc.Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
            if(UDM_OK != UdmDeleteURL(Indexer, &Doc, db))
            {
              UdmSQLFree(&SQLres);
              return(UDM_ERROR);
            }
          }
          j+=i;
          UdmSQLFree(&SQLres);
        }
      }
      else
      {
        UdmSQLFree(&SQLres);
        break;
      }
    }
    UdmDSTRFree(&qbuf);
    UdmDSTRFree(&urlin);
  }
  return(UDM_OK);
}

/********************* Categories ************************************/
static int UdmCatList(UDM_AGENT * Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i, rows;
  char    qbuf[1024];
  UDM_SQLRES  SQLres;
  int    rc;
  
  if(db->DBType==UDM_DB_SAPDB)
  {
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }else{
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
  
  if( (rows = UdmSQLNumRows(&SQLres)) )
  {
    size_t nbytes;
    
    nbytes = sizeof(UDM_CATITEM) * (rows + Cat->ncategories);
    Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,nbytes);
    for(i=0;i<rows;i++)
    {
      UDM_CATITEM *r = &Cat->Category[Cat->ncategories];
      r[i].rec_id=atoi(UdmSQLValue(&SQLres,i,0));
      strcpy(r[i].path,UdmSQLValue(&SQLres,i,1));
      strcpy(r[i].link,UdmSQLValue(&SQLres,i,2));
      strcpy(r[i].name,UdmSQLValue(&SQLres,i,3));
    }
    Cat->ncategories+=rows;
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

static int UdmCatPath(UDM_AGENT *Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i,l;
  char    qbuf[1024];
  int    rc;
  char            *head = NULL;
  
  l=(strlen(Cat->addr)/2)+1;
  Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,sizeof(UDM_CATITEM)*(l+Cat->ncategories));
  head = (char *)UdmMalloc(2 * l + 1);

  if (head != NULL)
  {
    UDM_CATITEM  *r = &Cat->Category[Cat->ncategories];

    for(i = 0; i < l; i++)
    {
      UDM_SQLRES  SQLres;

      strncpy(head,Cat->addr,i*2);head[i*2]=0;

      if(db->DBType==UDM_DB_SAPDB)
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path='%s'",head);
      }
      else
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
      }
    
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
    
      if(UdmSQLNumRows(&SQLres))
      {
        r[i].rec_id=atoi(UdmSQLValue(&SQLres,0,0));
        strcpy(r[i].path,UdmSQLValue(&SQLres,0,1));
        strcpy(r[i].link,UdmSQLValue(&SQLres,0,2));
        strcpy(r[i].name,UdmSQLValue(&SQLres,0,3));
        Cat->ncategories++;
      }
      UdmSQLFree(&SQLres);
    }
    UDM_FREE(head);
  }
  return UDM_OK;
}


/******************* Search stuff ************************************/

int UdmCloneListSQL(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_RESULT *Res, UDM_DB *db)
{
  size_t    i, nr, nadd;
  char    qbuf[256];
  UDM_SQLRES  SQLres;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  urlid_t    origin_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int    rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char  *format = UdmVarListFindStr(&Indexer->Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");

  if (Res->num_rows > 4) return UDM_OK;
  
  sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206) AND rec_id<>%s%i%s", scrc32, qu, origin_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return UDM_OK;
  
  nr = UdmSQLNumRows(&SQLres);
  if( nr == 0)
  {
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  nadd = 5 - Res->num_rows;
  if(nr < nadd) nadd = nr;
  
  Res->Doc = (UDM_DOCUMENT*)UdmRealloc(Res->Doc, (Res->num_rows + nadd) * sizeof(UDM_DOCUMENT));
  
  for(i = 0; i < nadd; i++)
  {
    time_t    last_mod_time;
    char    buf[128];
    UDM_DOCUMENT  *D = &Res->Doc[Res->num_rows + i];
    
    UdmDocInit(D);
    UdmVarListAddInt(&D->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
    UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(&SQLres,i,1));
    UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLres,i,1)));
    last_mod_time=atol(UdmSQLValue(&SQLres,i,2));
    if (strftime(buf, 128, format, localtime(&last_mod_time)) == 0)
    {
      UdmTime_t2HttpStr(last_mod_time, buf);
    }
    UdmVarListAddStr(&D->Sections,"Last-Modified",buf);
    UdmVarListAddInt(&D->Sections,"Content-Length",atoi(UdmSQLValue(&SQLres,i,3)));
    UdmVarListAddInt(&D->Sections,"crc32",scrc32);
    UdmVarListAddInt(&D->Sections, "Origin-ID", origin_id);
  }
  Res->num_rows += nadd;
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

/********************************************************/

int UdmLoadURLDataSQL(UDM_AGENT *A, UDM_RESULT *R, UDM_DB *db)
{
  UDM_SQLRES  SQLres;
  size_t i,j;
  size_t s;
  int rc;
  char qbuf[4*1024];
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t nrows, nbytes;
  UDM_PSTR row[5];
  const char *pattern = UdmVarListFindStr(&A->Conf->Vars, "s", "RP");
  const char *su = UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  char *esu = su ? UdmSQLEscStr(db, NULL, su, strlen(su)) : NULL;
  int need_pop_rank = 0;
  int need_last_mod_time = 0;
  int need_site_id = 1;
  int need_url = 0;
  const char *p;

  for (p = pattern; *p; p++)
  {
    if (*p == 'P' || *p == 'p') need_pop_rank = 1;
    if (*p == 'D' || *p == 'd') need_last_mod_time = 1;
    if (*p == 'U' || *p == 'u') need_url = 1;
  }

  if (R->CoordList.ncoords == 0) return UDM_OK;
  nbytes= R->CoordList.ncoords * sizeof(UDM_URLDATA);
  R->CoordList.Data = (UDM_URLDATA*)UdmRealloc(R->CoordList.Data, nbytes);
  bzero((void*)R->CoordList.Data, nbytes);

#ifdef NO_URL_DATA
        {
    for (i = 0; i < R->CoordList.ncoords; i++)
    {
        R->CoordList.Data[i].url_id = R->CoordList.Coords[i].url_id;
        R->CoordList.Data[i].site_id = 0;
        R->CoordList.Data[i].pop_rank = 0;
        R->CoordList.Data[i].last_mod_time = 0;
	R->CoordList.Data[i].url = NULL;
    }
    return UDM_OK;
        }
#endif

  if (db->DBMode == UDM_DBMODE_BLOB && ! need_url && ! su)
  {
    const urlid_t *rec_id = NULL;
    const int *site_id = NULL;
    const int *last_mod_time = NULL;
    const double *pop_rank = NULL;
    const char *table = UdmBlobGetRTable(db);
    UDM_DSTR buf;
    size_t nrecs = 1;

    UdmDSTRInit(&buf, 64);
    UdmDSTRAppendSTR(&buf, "'#rec_id'");
    if (need_pop_rank)
    {
      UdmDSTRAppendSTR(&buf, ",'#pop_rank'");
      nrecs++;
    }
    if (need_last_mod_time)
    {
      UdmDSTRAppendSTR(&buf, ",'#last_mod_time'");
      nrecs++;
    }
    if (need_site_id)
    {
      UdmDSTRAppendSTR(&buf, ",'#site_id'");
      nrecs++;
    }

    UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from bdict");
    nrows = 0;
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT word, intag FROM %s WHERE word IN (%s)", table, buf.data);
    UdmDSTRFree(&buf);
    rc = UdmSQLQuery(db, &SQLres, qbuf);
    if (rc != UDM_OK)
    {
      UdmLog(A,UDM_LOG_DEBUG,"Couldn't run a query on bdict");
      return(rc);
    }
    if (UdmSQLNumRows(&SQLres) == nrecs)
    {
      for (i = 0; i < nrecs; i++)
      {
        const char *word = UdmSQLValue(&SQLres, i, 0);
        const char *intag = UdmSQLValue(&SQLres, i, 1);
        if (! strcmp(word, "#rec_id"))
        {
          nrows = UdmSQLLen(&SQLres, i, 1) / sizeof(urlid_t);
          rec_id = (const urlid_t *)intag;
        }
	else if (! strcmp(word, "#site_id")) site_id = (const int *)intag;
        else if (! strcmp(word, "#last_mod_time")) last_mod_time = (const int *)intag;
        else if (! strcmp(word, "#pop_rank")) pop_rank = (const double *)intag;
      }
    }

    if (rec_id && 
        (site_id || ! need_site_id) &&
	(last_mod_time || ! need_last_mod_time) &&
	(pop_rank || ! need_pop_rank) &&
	nrows)
    {
      for (j = 0, i = 0; i < nrows; i++)
      {
        if (rec_id[i] == R->CoordList.Coords[j].url_id)
        {
          R->CoordList.Data[j].url_id = R->CoordList.Coords[j].url_id;
          R->CoordList.Data[j].site_id = need_site_id ? site_id[i] : 0;
          R->CoordList.Data[j].pop_rank = need_pop_rank ? pop_rank[i] : 0;
          R->CoordList.Data[j].last_mod_time = need_last_mod_time ? last_mod_time[i] : 0;
          R->CoordList.Data[j].url = NULL;
          R->CoordList.Data[j].section = NULL;
          j++;
          if (j == R->CoordList.ncoords) break;
        }
      }
      UdmSQLFree(&SQLres);
      if (j == R->CoordList.ncoords) return(UDM_OK);
      UdmLog(A,UDM_LOG_DEBUG,"Couldn't load URL data from bdict");
    }
    else
    {
      UdmSQLFree(&SQLres);
      UdmLog(A,UDM_LOG_DEBUG,"There is no URL data in bdict");
    }
  }

  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from url");

  if (db->DBSQL_IN)
  {
    for (j = 0; j < R->CoordList.ncoords; j += 256)
    {
      int notfirst = 0;
      sprintf(qbuf, "SELECT rec_id, site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id IN (", need_url ? ",url" : "");
      for (i = 0; (i < 256) && (j + i < R->CoordList.ncoords); i++)
      {
        sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, R->CoordList.Coords[j + i].url_id, qu);
        notfirst = 1;
      }
      sprintf(UDM_STREND(qbuf), ") ORDER BY rec_id");
      if (UDM_OK != (rc = db->sql->SQLExecDirect(db, &SQLres, qbuf)))
        return rc;
      for (i = 0; db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK; i++)
      {
        s = i + j;
        R->CoordList.Data[s].url_id = (urlid_t)UDM_ATOI(row[0].val);
        
        if (R->CoordList.Data[s].url_id != R->CoordList.Coords[s].url_id)
          UdmLog(A, UDM_LOG_ERROR, "Crd url_id (%d) != Dat url_id (%d)", 
                 R->CoordList.Coords[s].url_id, R->CoordList.Data[s].url_id);
          R->CoordList.Data[s].site_id = UDM_ATOI(row[1].val);
          R->CoordList.Data[s].pop_rank = UDM_ATOF(row[2].val);
          R->CoordList.Data[s].last_mod_time = UDM_ATOI(row[3].val);
          R->CoordList.Data[s].url = need_url ? UdmStrdup(row[4].val) : NULL;
          R->CoordList.Data[s].section = NULL;
      }
      UdmSQLFree(&SQLres);

      if (su)
      {
        notfirst = 0;
        sprintf(qbuf, "SELECT url_id, sval FROM urlinfo WHERE sname='%s' AND url_id IN (", esu);
        for (i = 0; (i < 256) && (j + i < R->CoordList.ncoords); i++)
        {
          sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, R->CoordList.Coords[j + i].url_id, qu);
          notfirst = 1;
        }
        sprintf(UDM_STREND(qbuf), ") ORDER BY url_id");
        if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
          return rc;
        nrows = UdmSQLNumRows(&SQLres);

        i = 0;
        for(s = i + j; i < nrows; s++)
        {
          if (s == R->CoordList.ncoords) break;
          if (R->CoordList.Data[s].url_id != (urlid_t)UDM_ATOI(UdmSQLValue(&SQLres, i, 0)))
          {
            R->CoordList.Data[s].section = UdmStrdup("");
          }
          else
          {
            R->CoordList.Data[s].section = UdmStrdup(UdmSQLValue(&SQLres, i, 1));
            i++;
          }
        }
        UdmSQLFree(&SQLres);
      }
    }
  }
  else
  {
    for (i = 0; i < R->CoordList.ncoords; i++)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id=%i", need_url ? ",url" : "", R->CoordList.Coords[i].url_id);
      if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
        return rc;
      if(UdmSQLNumRows(&SQLres))
      {
        R->CoordList.Data[i].url_id = R->CoordList.Coords[i].url_id;
        R->CoordList.Data[i].site_id = UDM_ATOI(UdmSQLValue(&SQLres, 0, 0));
        R->CoordList.Data[i].pop_rank = UDM_ATOF(UdmSQLValue(&SQLres, 0, 1));
        R->CoordList.Data[i].last_mod_time = UDM_ATOI(UdmSQLValue(&SQLres, 0, 2));
        R->CoordList.Data[i].url = need_url ? UdmStrdup(UdmSQLValue(&SQLres, 0, 3)) : NULL;
        R->CoordList.Data[i].section = NULL;
      }
      UdmSQLFree(&SQLres);
    }
  }
  UDM_FREE(esu);
  return UDM_OK;
}

static int UdmSortAndGroupByURL(UDM_AGENT *A,UDM_RESULT *R, UDM_DB *db)
{
  unsigned long  ticks=UdmStartTimer();
  int    use_site_id = ((!strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "GroupBySite", "no"), "yes")) 
    && (UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0));
  
  UdmLog(A,UDM_LOG_DEBUG,"Start sort by url_id %d coords",R->CoordList.ncoords);
  UdmSortSearchWordsByURL(R->CoordList.Coords, R->CoordList.ncoords);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop sort by url_id:\t\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start group by url_id %d coords",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupByURL(A,R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by url_id:\t%.2f",(float)ticks/1000);


  UdmLog(A,UDM_LOG_DEBUG,"Start load url data %d docs",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmLoadURLDataSQL(A, R, db);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop load url data:\t\t%.2f",(float)ticks/1000);

  if (use_site_id)
  {
    UdmLog(A,UDM_LOG_DEBUG,"Start sort by site_id %d words",R->CoordList.ncoords);
    UdmSortSearchWordsBySite(&R->CoordList, R->CoordList.ncoords);
    ticks=UdmStartTimer()-ticks;
    UdmLog(A,UDM_LOG_DEBUG,"Stop sort by site_id:\t%.2f",(float)ticks/1000);
  
    UdmLog(A,UDM_LOG_DEBUG,"Start group by site_id %d docs", R->CoordList.ncoords);
    ticks=UdmStartTimer();
    UdmGroupBySite(A, R);
    ticks=UdmStartTimer() - ticks;
    UdmLog(A,UDM_LOG_DEBUG,"Stop group by site_id:\t%.2f", (float)ticks/1000);
  } 

  UdmLog(A,UDM_LOG_DEBUG,"Start SORT by PATTERN %d docs", R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmSortSearchWordsByPattern(R, &R->CoordList, R->CoordList.ncoords, UdmVarListFindStr(&A->Conf->Vars, "s", "RP"));
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by PATTERN:\t%.2f",(float)ticks/1000);

  return UDM_OK;
}

static int LoadURL (UDM_DB *db, const char *where, UDM_URL_TMP *buf)
{
  int rc;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  urlid_t *tmp;
  size_t i;

  if (! buf) return(UDM_ERROR);
  buf->urls = NULL;
  buf->nurls = 0;
  buf->empty= 0;
  if (! *where) return(UDM_OK);

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s", db->from, where);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if (rc != UDM_OK) return(UDM_ERROR);
  nrows = UdmSQLNumRows(&SQLRes);
  if (! nrows)
  {
    buf->empty= 1;
    UdmSQLFree(&SQLRes);
    return(UDM_OK);
  }

  tmp = UdmMalloc(sizeof(urlid_t) * nrows);
  buf->urls = UdmMalloc(sizeof(urlid_t) * nrows);
  if (tmp && buf->urls)
  {
    for (i = 0; i < nrows; i++)
    {
      tmp[i] = (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    }
    UdmSort(tmp, nrows, sizeof(urlid_t), (qsort_cmp)cmpaurls);
    i = 0;
    while (i < nrows)
    {
      while (++i < nrows && tmp[i] == tmp[i - 1]);
      buf->urls[buf->nurls++] = tmp[i - 1];
    }
    UDM_FREE(tmp);
    tmp = UdmRealloc(buf->urls, sizeof(urlid_t) * buf->nurls);
    if (tmp) buf->urls = tmp;
  } else {
    UDM_FREE(buf->urls);
    UDM_FREE(tmp);
  }

  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


static inline int udm_int4_get(const char *s)
{
  int res;
  res=  (unsigned char) s[3]; res<<= 8;
  res+= (unsigned char) s[2]; res<<= 8;
  res+= (unsigned char) s[1]; res<<= 8;
  res+= (unsigned char) s[0];
  return res;
}

static int
UdmLoadCachedQueryWords(UDM_AGENT *query, UDM_URLCRDLIST *Coords,
                        UDM_DB *db, const char *pqid)
{
  char qbuf[128], bqid[32];
  char *tm, *end;
  int iid, itm, rc;
  UDM_SQLRES SQLRes;
  
  bzero(Coords, sizeof(*Coords));
  if (!pqid)
    return UDM_OK;
  udm_snprintf(bqid, sizeof(bqid), pqid);
  if (!(tm= strchr(bqid, '-')))
    return UDM_OK;
  *tm++= '\0';
  iid= (int) strtoul(bqid, &end, 16);
  itm= (int) strtol(tm, &end, 16);
  
  sprintf(qbuf, "SELECT doclist FROM qcache WHERE id=%d AND tm=%d", iid, itm);
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes,qbuf)))
      return rc;
  if (UdmSQLNumRows(&SQLRes) == 1)
  {
    size_t i;
    const char *p;
    Coords->ncoords= UdmSQLLen(&SQLRes, 0, 0) / 8;
    Coords->Coords= (UDM_URL_CRD*) UdmMalloc(Coords->ncoords * sizeof(UDM_URL_CRD));
    for (p= UdmSQLValue(&SQLRes, 0, 0), i= 0; i < Coords->ncoords; i++)
    {
      Coords->Coords[i].url_id= udm_int4_get(p); p+= 4;
      Coords->Coords[i].coord= udm_int4_get(p);  p+= 4;
    }
  }
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int cmpurlid (UDM_URL_CRD *s1, UDM_URL_CRD *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  return(UDM_WRDPOS(s1->coord) - UDM_WRDPOS(s2->coord));
}
      
int UdmFindWordsSQL(UDM_AGENT * query,UDM_RESULT *Res,UDM_DB *db)
{
  char    qbuf[1024*4];
  size_t    wordnum;
  int    has_crosswrd=0;
  UDM_SQLRES  SQLres;
  int    word_match;
  int    wf[256];
  const char  *where;
  int    use_crosswords;
  unsigned long   ticks=UdmStartTimer();
  int    rc;
  UDM_URL_TMP     urls;
  const char *btable = NULL;

  UDM_GETLOCK(query, UDM_LOCK_CONF);
  word_match = UdmMatchMode(UdmVarListFindStr(&query->Conf->Vars, "wm", "wrd"));
  where = BuildWhere(query->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "CrossWords", "no"), "yes");
  UdmWeightFactorsInit(UdmVarListFindStr(&query->Conf->Vars,"wf",""),wf);
  UDM_RELEASELOCK(query, UDM_LOCK_CONF);

  if (db->DBMode == UDM_DBMODE_BLOB || db->DBMode == UDM_DBMODE_FLY)
  {
    UdmLog(query,UDM_LOG_DEBUG, "Start loading limits");
    ticks=UdmStartTimer();
    LoadURL(db, where, &urls);
    ticks=UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop loading limits\t%.2f",(float)ticks/1000);
    if (urls.empty) return(UDM_OK);
    btable = UdmBlobGetRTable(db);
  }

  /* Now find each word */
  for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++)
  {
    size_t  numrows,firstnum,curnum,tnum,tmin,tmax = (size_t)-1;
    size_t  i;
    char  tablename[32] = "dict";
    char  escwrd[1000];
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];
    
    if (W->origin == UDM_WORD_ORIGIN_STOP) continue;

    UdmSQLEscStr(db,escwrd,W->word,strlen(W->word));

    if((db->DBMode==UDM_DBMODE_MULTI)&&(word_match!=UDM_MATCH_FULL))
    {
      /* This is for substring search!  */
      /* In Multi mode: we have to scan */
      /* almost all tables except those */
      /* with to short words            */
      
      tmin=0;
      tmax=MULTI_DICTS;
    }else{
      tmin=tmax=UdmStrHash32(W->word) & MULTI_DICTS;
    }

    for(tnum=tmin;tnum<=tmax;tnum++)
    {
        char cmparg[256];
        ticks=UdmStartTimer();
        UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",Res->WWList.Word[wordnum].word);

        if (db->DBMode == UDM_DBMODE_MULTI) sprintf(tablename,"dict%02X",tnum);

        switch(word_match)
        {  
          case UDM_MATCH_BEGIN:
            sprintf(cmparg," LIKE '%s%%'",escwrd);
            break;
          case UDM_MATCH_END:
            sprintf(cmparg," LIKE '%%%s'",escwrd);
            break;
          case UDM_MATCH_SUBSTR:
            sprintf(cmparg," LIKE '%%%s%%'",escwrd);
            break;
          case UDM_MATCH_FULL:
          default:
            sprintf(cmparg,"='%s'",escwrd);
            break;
        }
        if (db->DBMode == UDM_DBMODE_BLOB)
        {
          udm_snprintf(qbuf, sizeof(qbuf), "SELECT secno,intag FROM %s WHERE word%s", btable, cmparg);
        }
        else if (db->DBMode == UDM_DBMODE_FLY)
        {
          udm_snprintf(qbuf, sizeof(qbuf), "SELECT coords FROM fdicts WHERE word%s", cmparg);
        }
        else if(where[0])
        {
          udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s%s%s.intag \
FROM %s, url%s \
WHERE %s.word%s \
AND url.rec_id=%s.url_id AND %s",
          tablename,
          db->DBMode == UDM_DBMODE_MULTI ? tablename : "",
          db->DBMode == UDM_DBMODE_MULTI ? ".secno," : "",
          tablename, tablename, db->from, tablename,
          cmparg,tablename,where);
        } else {
          udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT url_id,%sintag FROM %s WHERE word%s",
             db->DBMode == UDM_DBMODE_MULTI ? "secno," : "",
             tablename, cmparg);
        }

        if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
          return rc;
          
        ticks=UdmStartTimer()-ticks;
        numrows=UdmSQLNumRows(&SQLres);
        UdmLog(query,UDM_LOG_DEBUG,"Stop search for '%s'\t%.2f  %d rows found",Res->WWList.Word[wordnum].word,(float)ticks/1000,numrows);
        
        if (db->DBMode == UDM_DBMODE_MULTI)
        {
          UdmLog(query,UDM_LOG_DEBUG,"Start UdmMultiAddCoords");
          ticks=UdmStartTimer();
          UdmMultiAddCoords(Res, &SQLres, wordnum, wf);
          ticks=UdmStartTimer()-ticks;
          UdmLog(query,UDM_LOG_DEBUG,"Stop UdmMultiAddCoords\t%.2f",(float)ticks/1000);
        }
        else if (db->DBMode == UDM_DBMODE_BLOB)
        {
          UdmLog(query,UDM_LOG_DEBUG,"Start UdmBlobAddCoords");
          ticks=UdmStartTimer();
          UdmBlobAddCoords(Res, &SQLres, wordnum, wf, &urls);
          ticks=UdmStartTimer()-ticks;
          UdmLog(query,UDM_LOG_DEBUG,"Stop UdmBlobAddCoords\t%.2f",(float)ticks/1000);
        }
        else if (db->DBMode == UDM_DBMODE_FLY)
        {
          UdmLog(query,UDM_LOG_DEBUG,"Start UdmDBModeFlyAddCoords");
          ticks=UdmStartTimer();
          UdmDBModeFlyAddCoords(Res, &SQLres, wordnum, wf, &urls);
          ticks=UdmStartTimer()-ticks;
          UdmLog(query,UDM_LOG_DEBUG,"Stop UdmDBModeFlyAddCoords\t%.2f",(float)ticks/1000);
        }
        else {
          /* Add new found word to the list */
          Res->CoordList.Coords = (Res->CoordList.ncoords+numrows) ? 
            (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,(Res->CoordList.ncoords+numrows)*sizeof(UDM_URL_CRD)) 
            : NULL;
        
          firstnum=curnum=Res->CoordList.ncoords;
          for(i=0;i<numrows;i++)
          {
            urlid_t url_id = 0;
            uint4 coord=0;
            uint4 weight;
            uint4 section;
          
            url_id = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
            coord=atoi(UdmSQLValue(&SQLres,i,1));
          
            section=UDM_WRDSEC(coord);
            weight=wf[section];
            if(weight)
            {
              coord = coord & 0xFFFFFF00;
              Res->CoordList.Coords[curnum].url_id=url_id;
              Res->CoordList.Coords[curnum].coord = coord + (wordnum /* Res->WWList.Word[wordnum].order */ & 0xFF);
              curnum++;
            }
          }
          Res->WWList.Word[wordnum].count+=curnum-firstnum;
          Res->CoordList.ncoords=curnum;
          Res->CoordList.Coords = (curnum) ? 
            (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords, curnum * sizeof(UDM_URL_CRD))
            : NULL;
        }
        UdmSQLFree(&SQLres);
      }
  }
  
  /* Now find each word in crosstable */
  has_crosswrd = use_crosswords;
  for(wordnum=0;((has_crosswrd)&&(wordnum<Res->WWList.nwords));wordnum++)
  {
    size_t    numrows,firstnum,curnum;
    char    tablename[32]="";
    char    escwrd[1000];
    size_t    i;
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];
    char cmparg[256];

    if(W->origin == UDM_WORD_ORIGIN_STOP) continue;

    UdmSQLEscStr(db,escwrd,W->word,strlen(W->word));
    
    ticks=UdmStartTimer();
    UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",W->word);
    
    strcpy(tablename,"crossdict");

    switch(word_match)
    {  
      case UDM_MATCH_BEGIN:
        sprintf(cmparg," LIKE '%s%%'",escwrd);
        break;
      case UDM_MATCH_END:
        sprintf(cmparg," LIKE '%%%s'",escwrd);
        break;
      case UDM_MATCH_SUBSTR:
        sprintf(cmparg," LIKE '%%%s%%'",escwrd);
        break;
      case UDM_MATCH_FULL:
      default:
        sprintf(cmparg,"='%s'",escwrd);
        break;
    }
    if(where[0])
    {
      udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url%s \
WHERE %s.word%s \
AND url.rec_id=%s.url_id AND %s",
      tablename,tablename,
      tablename, db->from, tablename,
      cmparg,tablename,where);
    }else{
      udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT url_id,intag FROM %s,url WHERE %s.word%s AND url.rec_id=%s.url_id",
         tablename, tablename, cmparg, tablename);
    }

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    numrows=UdmSQLNumRows(&SQLres);
    ticks=UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop search for '%s'\t%.2f %d found",Res->WWList.Word[wordnum].word,(float)ticks/1000,numrows);
    
    /* Add new found word to the list */
    Res->CoordList.Coords= Res->CoordList.ncoords+numrows ?
          (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,(Res->CoordList.ncoords+numrows)*sizeof(UDM_URL_CRD)):
          NULL;
    
    firstnum=curnum=Res->CoordList.ncoords;
    for(i=0;i<numrows;i++)
    {
      urlid_t url_id = 0;
      int coord=0;
      int weight;
      int section;
      
      url_id = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
      coord=atoi(UdmSQLValue(&SQLres,i,1));
      
      section=UDM_WRDSEC(coord);
      weight=wf[section];
      if(weight)
      {
        coord = coord & 0xFFFFFF00;
        Res->CoordList.Coords[curnum].url_id=url_id;
        Res->CoordList.Coords[curnum].coord = coord + (wordnum /* Res->WWList.Word[wordnum].order */ & 0xFF);
        curnum++;
      }
    }
    UdmSQLFree(&SQLres);
    Res->CoordList.ncoords=curnum;
    Res->WWList.Word[wordnum].count+=curnum-firstnum;
    Res->CoordList.Coords= curnum ? 
          (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,curnum*sizeof(UDM_URL_CRD)) :
          NULL;
  }

  UdmSortAndGroupByURL(query, Res, db);
  {
    UDM_URLCRDLIST Coords;
    const char *pqid= UdmVarListFindStr(&query->Conf->Vars, "pqid", NULL);
    if (pqid && UDM_OK == UdmLoadCachedQueryWords(query, &Coords, db, pqid))
    {
      UdmLog(query,UDM_LOG_DEBUG, "Start applying pqid limit: %d docs", Coords.ncoords);
      if (Coords.ncoords)
      {
        size_t i, to;
        UdmSortSearchWordsByURL(Coords.Coords, Coords.ncoords);
        for (to= 0, i= 0; i < Res->CoordList.ncoords; i++)
        {
          if (bsearch(&Res->CoordList.Coords[i], Coords.Coords,
                      Coords.ncoords, sizeof(UDM_URL_CRD), (qsort_cmp) cmpurlid))
          {
            if (to != i)
              Res->CoordList.Coords[to]= Res->CoordList.Coords[i];
            to++;
          }
        }
        UDM_FREE(Coords.Coords);
        Res->CoordList.ncoords= to;
      }
      else
      {
        Res->CoordList.ncoords= 0;
      }
      UdmLog(query,UDM_LOG_DEBUG, "Stop applying pqid limit: %d docs", Res->CoordList.ncoords);
    }
  }
  Res->total_found = Res->CoordList.ncoords;
  
  return UDM_OK;
}


int UdmTrackSQL(UDM_AGENT * query, UDM_RESULT *Res, UDM_DB *db)
{
  UDM_SQLRES      sqlRes;
  char    *qbuf;
  char    *text_escaped;
  const char  *words=UdmVarListFindStr(&query->Conf->Vars,"q",""); /* "q-lc" was here */
  const char      *IP = UdmVarListFindStr(&query->Conf->Vars, "IP", "");
  size_t          i, escaped_len, qbuf_len;
  int             res, qtime, rec_id;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (*words == '\0') return UDM_OK; /* do not store empty queries */

  escaped_len = 4 * strlen(words);
  qbuf_len = escaped_len + 4096;

  if ((qbuf = (char*)UdmMalloc(qbuf_len)) == NULL) return UDM_ERROR;
  if ((text_escaped = (char*)UdmMalloc(escaped_len)) == NULL)
  { 
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  
  /* Escape text to track it  */
  UdmSQLEscStr(db, text_escaped, words, strlen(words));
  
  udm_snprintf(qbuf, qbuf_len - 1, "INSERT INTO qtrack (ip,qwords,qtime,found) VALUES ('%s','%s',%d,%d)",
     IP, text_escaped, qtime = (int)time(NULL), Res->total_found
     );
  
  res = UdmSQLQuery(db, NULL, qbuf);
  if (res != UDM_OK) goto UdmTrack_exit;
       
  udm_snprintf(qbuf, qbuf_len - 1, "SELECT rec_id FROM qtrack WHERE ip='%s' AND qtime=%d", IP, qtime);
  res = UdmSQLQuery(db, &sqlRes, qbuf);
  if (res != UDM_OK) goto UdmTrack_exit;
  if (UdmSQLNumRows(&sqlRes) == 0) 
  { 
    UdmSQLFree(&sqlRes);
    res = UDM_ERROR;
    goto UdmTrack_exit;
  }
  rec_id = UDM_ATOI(UdmSQLValue(&sqlRes, 0, 0));
  UdmSQLFree(&sqlRes);

  for (i = 0; i < query->Conf->Vars.nvars; i++)
  {
    UDM_VAR *Var = &query->Conf->Vars.Var[i];
    if (strncasecmp(Var->name, "query.",6)==0 && strcasecmp(Var->name, "query.q") && strcasecmp(Var->name, "query.BrowserCharset")
       && strcasecmp(Var->name, "query.IP")  && Var->val != NULL && *Var->val != '\0') 
    {
      udm_snprintf(qbuf, qbuf_len, "INSERT INTO qinfo (q_id,name,value) VALUES (%s%i%s,'%s','%s')", 
       qu, rec_id, qu, &Var->name[6], Var->val);
      res = UdmSQLQuery(db, NULL, qbuf);
      if (res != UDM_OK) goto UdmTrack_exit;
    }
  }
UdmTrack_exit:
  UDM_FREE(text_escaped);
  UDM_FREE(qbuf);
  return res;
}

static int UpdateShows(UDM_DB *db, urlid_t url_id)
{
  char qbuf[64];
  udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET shows = shows + 1 WHERE rec_id = %s%i%s",
               (db->DBType == UDM_DB_PGSQL) ? "'" : "",
               url_id,
               (db->DBType == UDM_DB_PGSQL) ? "'" : "");
  return UdmSQLQuery(db, NULL, qbuf);
}

static void SQLResToSection(UDM_SQLRES *R, UDM_VARLIST *S, size_t row)
{
  const char *sname=UdmSQLValue(R,row,1);
  const char *sval=UdmSQLValue(R,row,2);
  UdmVarListAddStr(S, sname, sval);
}

int UdmResAddDocInfoSQL(UDM_AGENT *query, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  size_t      i;
  char        instr[1024*4]="";
  char        qbuf[1024*4];
  UDM_SQLRES  SQLres;
  int         rc;
  int         use_showcnt = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  double      pr, ratio = 0.0;
  
  if(!Res->num_rows)return UDM_OK;
  if (use_showcnt) ratio = UdmVarListFindDouble(&query->Conf->Vars, "PopRankShowCntRatio", 25.0);
  UdmLog(query, UDM_LOG_DEBUG, "use_showcnt: %d  ratio: %f", use_showcnt, ratio);
  
  if(db->DBSQL_IN)
  {
    size_t  j, sqlrows;
    
    /* Compose IN string and set to zero url_id field */
    for(i=0; i < Res->num_rows; i++)
    {
      const char *comma= instr[0] ? "," : "";
      const char *squot= db->DBType == UDM_DB_PGSQL ? "'" : "";
      if (UdmVarListFindInt(&Res->Doc[i].Sections, "dbnum", 0) == dbnum)
          sprintf(UDM_STREND(instr),"%s%s%i%s", comma, squot,
                  UdmVarListFindInt(&Res->Doc[i].Sections, "ID", 0), squot);
    }
    
    if (!instr[0])
      return UDM_OK;
    
    
    udm_snprintf(qbuf,sizeof(qbuf),"SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id IN (%s)", instr);
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
    {
      UDM_DOCUMENT *D= &Res->Doc[j];
      urlid_t      url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      size_t       url_dbnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)) && url_dbnum == dbnum)
        {
          SQLResToDoc(query->Conf, D, &SQLres, i);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
          break;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
    
    udm_snprintf(qbuf, sizeof(qbuf),"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE u.rec_id IN (%s) AND u.server_id=s.rec_id AND s.category=c.rec_id", instr); 
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;

    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j < Res->num_rows; j++)
    {
      UDM_DOCUMENT  *D = &Res->Doc[j];
      urlid_t    url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
        {
          UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
          break;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
    
    udm_snprintf(qbuf,sizeof(qbuf),"SELECT url_id,sname,sval FROM urlinfo WHERE url_id IN (%s)",instr);
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
    {
      UDM_DOCUMENT *D=&Res->Doc[j];
      urlid_t      url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      size_t       url_dbnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)) && url_dbnum == dbnum)
          SQLResToSection(&SQLres, &D->Sections, i);
      }
    }
    UdmSQLFree(&SQLres);
    
  }
  else
  {
    for(i=0;i<Res->num_rows;i++)
    {
      UDM_DOCUMENT  *D=&Res->Doc[i];
      urlid_t  url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      int      sdnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      size_t   row;
      
      if (sdnum != dbnum)
        continue;
      
      sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        SQLResToDoc(query->Conf, D, &SQLres, 0);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
      }
      UdmSQLFree(&SQLres);
      
      sprintf(qbuf,"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE rec_id=%i AND u.server_id=s.rec_id AND s.category=c.rec_id", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      if(UdmSQLNumRows(&SQLres))
      {
        UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
      }
      UdmSQLFree(&SQLres);
      
      
      sprintf(qbuf,"SELECT url_id,sname,sval FROM urlinfo WHERE url_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      for(row=0;row<UdmSQLNumRows(&SQLres);row++)
        SQLResToSection(&SQLres, &D->Sections, row);
      UdmSQLFree(&SQLres);
    }
    
  }
  return(UDM_OK);
}



int UdmURLDataSQL(UDM_ENV *Conf, UDM_URLDATALIST *L, UDM_DB *db)
{
  UDM_SQLRES  SQLres;
  size_t    i;
  int    rc=UDM_OK;
  
  L->nitems=0;
  L->Item=NULL;
  
  if (UDM_OK != (rc=UdmSQLQuery(db, &SQLres, "SELECT rec_id,site_id,pop_rank,last_mod_time FROM url ORDER by rec_id")))
    return rc;
  
  L->nitems = UdmSQLNumRows(&SQLres);
  L->Item = (UDM_URLDATA*)UdmMalloc(L->nitems*sizeof(UDM_URLDATA));
  if (L->Item == NULL)
  {
    L->nitems = 0;
    rc=UDM_ERROR;
    goto freeex;
  }
  for (i = 0; i < L->nitems; i++)
  {
    L->Item[i].url_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 0));
    L->Item[i].site_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 1));
    L->Item[i].pop_rank = UDM_ATOF(UdmSQLValue(&SQLres, i, 2));
    L->Item[i].last_mod_time = UDM_ATOU(UdmSQLValue(&SQLres, i, 3));
  }
freeex:
  UdmSQLFree(&SQLres);
  return rc;
}



/***********************************************************/
/*  HTDB stuff:  Indexing of database content              */
/***********************************************************/

#define MAXNPART 32

static void include_params(UDM_DB *db,const char *src,char *path,char *dst,
                           size_t start, int limit)
{
  size_t nparts= 0;
  char *part[MAXNPART];
  char *lt;
  
  part[0]= udm_strtok_r(path, "/", &lt);
  while (part[nparts] && nparts < MAXNPART)
  {
    part[++nparts]= udm_strtok_r(NULL, "/", &lt);
  }
  
  for (*dst='\0'; *src; )
  {
    if(*src == '\\')
    {
      *dst++= src[1];
      *dst= '\0';
      src+= 2;
      continue;
    }
    if (*src == '$')
    {
      int i= atoi(++src)-1;
      
      while((*src>='0')&&(*src<='9'))src++;
      if ((i >= 0) && (i < nparts))
      {
        UdmUnescapeCGIQuery(dst,part[i]);
        while(*dst)dst++;
      }
      continue;
    }
    *dst++=*src++;
    *dst='\0';
    continue;
  }
  
  if (limit)
  {
    switch (db->DBType)
    {
      case UDM_DB_MYSQL:
        sprintf(dst, " LIMIT %d,%d", start, limit);
        break;
      case UDM_DB_PGSQL:
      default:
        sprintf(dst, " LIMIT %d OFFSET %d", limit, start);
        break;
    }
  }
}


int UdmHTDBGet(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc)
{
  char    *qbuf;
  char    *end=Doc->Buf.buf;
  UDM_SQLRES  SQLres;
  UDM_URL    realURL;
  UDM_DB      dbnew, *db= &dbnew;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  const char  *htdblist=UdmVarListFindStr(&Doc->Sections,"HTDBList","");
  const char  *htdbdoc=UdmVarListFindStr(&Doc->Sections,"HTDBDoc","");
  const char  *htdbaddr = UdmVarListFindStr(&Doc->Sections, "HTDBAddr", NULL);
  int    usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);
  int    rc = UDM_OK;
  
  Doc->Buf.buf[0]=0;
  UdmURLInit(&realURL);
  UdmURLParse(&realURL,url);
  if ((qbuf = (char*)UdmMalloc(4 * 1024 + strlen(htdblist) + strlen(htdbdoc))) == NULL) return UDM_ERROR;
  *qbuf = '\0';
  
  
  if (htdbaddr)
  {
    UdmDBInit(&dbnew);
    if (UDM_OK != (rc= UdmDBSetAddr(db, htdbaddr, UDM_OPEN_MODE_READ)))
    {
      UdmLog(Indexer,UDM_LOG_ERROR, "Wrong HTDB address");
      return rc;
    }
  }
  else
  {
    if (Indexer->Conf->dbl.nitems != 1)
    {
      UdmLog(Indexer, UDM_LOG_ERROR, "HTDB cannot work with several DBAddr without HTDBAddr");
      return UDM_ERROR;
    }
    db= &Indexer->Conf->dbl.db[0];
  }

  if(realURL.filename != NULL)
  {
    char real_path[1024]="";
    
    udm_snprintf(real_path,sizeof(real_path)-1,"%s%s",realURL.path,realURL.filename);
    real_path[sizeof(real_path)-1]='\0';
    include_params(db,htdbdoc, real_path, qbuf, 0, 0);
    UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBDoc: %s\n",qbuf);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
    {
      goto HTDBexit;
    }
    if(UdmSQLNumRows(&SQLres)==1)
    {
      char *to;
      size_t i; 
      for (to= Doc->Buf.buf, i= 0; i < UdmSQLNumCols(&SQLres); i++)
      {
        size_t len;
        if (i > 0)
        {
          memcpy(to, "\r\n", 2);
          to+= 2;
        }
        len= UdmSQLLen(&SQLres, 0, i);
        memcpy(to, UdmSQLValue(&SQLres, 0, i), len);
        to+= len;
      }
      *to= '\0';
    }
    else
    {
      sprintf(Doc->Buf.buf,"HTTP/1.0 404 Not Found\r\n\r\n");
    }
    UdmSQLFree(&SQLres);
  }
  else
  {
    size_t  i, start = 0;
    urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
    const size_t  htdblimit = UdmVarListFindUnsigned(&Doc->Sections, "HTDBLimit", 0);
    int  done = 0, hops=UdmVarListFindInt(&Doc->Sections,"Hops",0);


    sprintf(Doc->Buf.buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
    end=UDM_STREND(Doc->Buf.buf);
    strcpy(end,"</BODY></HTML>\n");

    while (!done)
    {
      size_t nrows;
      include_params(db,htdblist, realURL.path, qbuf, start, htdblimit);
      UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBList: %s\n",qbuf);
      if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
      goto HTDBexit;

      nrows= UdmSQLNumRows(&SQLres);
      done= htdblimit ? (htdblimit != nrows) : 1;
      start+= nrows;

      for(i = 0; i < nrows; i++)
      {
      UDM_HREF Href;
      UdmHrefInit(&Href);
      Href.referrer=url_id;
      Href.hops=hops+1;
      Href.url = (char*)UdmStrdup(UdmSQLValue(&SQLres,i,0));
      Href.method=UDM_METHOD_GET;
      Href.rec_id = usehtdburlid ? atoi(Href.url) : 0;
      UdmHrefListAdd(&Doc->Hrefs,&Href);
      UDM_FREE(Href.url);
      }
      UdmSQLFree(&SQLres);
      UdmDocStoreHrefs(Indexer, Doc);
      UdmHrefListFree(&Doc->Hrefs);
      UdmStoreHrefs(Indexer);
    }
  }
  end = UDM_STREND(Doc->Buf.buf);
  Doc->Buf.size=end-Doc->Buf.buf;
HTDBexit:
  if (db == &dbnew)
    UdmDBFree(db);
  UdmURLFree(&realURL);
  UDM_FREE(qbuf);
  return rc;
}

/************************** make index stuff *******************************/

static char *BuildLimitQuery(const char * field)
{
  char qbuf[2048];
  char smallbuf[128];

  udm_snprintf(smallbuf, 128, ":%s:", field);
  if (strstr(":status:docsize:next_index_time:last_mod_time:crc32:referrer:hops:seed:bad_since_time:site_id:pop_rank:url:", 
       smallbuf) != NULL)
  {
    udm_snprintf(qbuf, 2048, "SELECT %s,rec_id FROM url", field);
  }
  else if(strstr(":tag:", smallbuf) != NULL)
  {
    udm_snprintf(qbuf, 2048, "SELECT s.%s,u.rec_id FROM server s, url u WHERE s.rec_id=u.server_id", field);
  }
  else if(strstr(":category:", smallbuf) != NULL)
  {
    udm_snprintf(qbuf,2048, "SELECT c.path,u.rec_id FROM server s, url u, categories c WHERE s.rec_id=u.server_id AND s.category=c.rec_id");
  }
  else
  {
    udm_snprintf(qbuf, 2048, "SELECT i.sval,u.rec_id FROM url u,urlinfo i WHERE u.rec_id=i.url_id AND i.sname='%s'", field);
  }
  return (char*)UdmStrdup(qbuf);
}

int UdmLimit8SQL(UDM_ENV *Conf, UDM_UINT8URLIDLIST *L,const char *field, int type, UDM_DB *db)
{
  char    *qbuf = BuildLimitQuery(field);
  size_t    i;
  UDM_SQLRES  SQLres;
  int    rc=UDM_OK;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
  {
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  UDM_FREE(qbuf);
  L->nitems=UdmSQLNumRows(&SQLres);
  
  L->Item=(UDM_UINT8URLID*)UdmMalloc((L->nitems + 1) * sizeof(UDM_UINT8URLID));
  if(!L->Item)
  {
    sprintf(db->errstr,"Error: %s",strerror(errno));
    db->errcode=1;
    UdmSQLFree(&SQLres);
    return UDM_ERROR;
  }
  for(i=0;i<L->nitems;i++)
  {
    const char *val0=UdmSQLValue(&SQLres,i,0);
    const char *val1=UdmSQLValue(&SQLres,i,1);

    switch(type)
    {
      case UDM_IFIELD_TYPE_HEX8STR: UdmDecodeHex8Str(val0,&L->Item[i].hi, &L->Item[i].lo, NULL, NULL); break;
      case UDM_IFIELD_TYPE_INT: L->Item[i].hi = atoi(val0); L->Item[i].lo = 0; break;
    }
    L->Item[i].url_id = UDM_ATOI(val1);
  }
  UdmSQLFree(&SQLres);
  return rc;
}

int UdmLimit4SQL(UDM_ENV *Conf, UDM_UINT4URLIDLIST *L,const char *field, int type, UDM_DB *db)
{
  char    *qbuf = BuildLimitQuery(field);
  size_t    i;
  UDM_SQLRES  SQLres;
  int    rc=UDM_OK;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
  {
    UDM_FREE(qbuf);
    return rc;
  }
  UDM_FREE(qbuf);
  
  L->nitems=UdmSQLNumRows(&SQLres);
  
  L->Item=(UDM_UINT4URLID*)UdmMalloc((L->nitems + 1) * sizeof(UDM_UINT4URLID));
  if(!L->Item)
  {
    sprintf(db->errstr,"Error: %s",strerror(errno));
    db->errcode=0;
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  for(i=0;i<L->nitems;i++)
  {
    const char *val0=UdmSQLValue(&SQLres,i,0);
    const char *val1=UdmSQLValue(&SQLres,i,1);

    switch(type)
    {
      case UDM_IFIELD_TYPE_HOUR: L->Item[i].val = atoi(val0) / 3600; break;
      case UDM_IFIELD_TYPE_MIN: L->Item[i].val = atoi(val0) / 60; break;
      case UDM_IFIELD_TYPE_HOSTNAME:{
        UDM_URL url;
        UdmURLInit(&url);
        if(!UdmURLParse(&url,val0))
        {
          if(url.hostname) L->Item[i].val = UdmStrHash32(url.hostname);
          else L->Item[i].val=0;
        }else
          L->Item[i].val=0;
        UdmURLFree(&url);
      }
        break;
      case UDM_IFIELD_TYPE_STRCRC32: L->Item[i].val = UdmStrHash32(val0); break;
      case UDM_IFIELD_TYPE_INT: L->Item[i].val = atoi(val0); break;
      
    }
    L->Item[i].url_id = UDM_ATOI(val1);
  }
  UdmSQLFree(&SQLres);
  return rc;
}


/***************************************************************************/

static int UdmPopRankCalculate(UDM_AGENT *A, UDM_DB *db)
{
  UDM_SQLRES  SQLres, Res, POPres;
  char    qbuf[1024];
  int             rc = UDM_ERROR, u;
  size_t    i, nrows;
  int    skip_same_site = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankSkipSameSite","no"),"yes");
  int    feed_back = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankFeedBack", "no"), "yes");
  int    use_tracking = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseTracking", "no"), "yes");
  int    use_showcnt = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  double          ratio = UdmVarListFindDouble(&A->Conf->Vars, "PopRankShowCntWeight", 0.01);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (feed_back || use_tracking)
  {
    if (use_tracking) UdmLog(A, UDM_LOG_EXTRA, "Will calculate servers weights by tracking");
    if (feed_back) UdmLog(A, UDM_LOG_EXTRA, "Will calculate feed back servers weights");

    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, "SELECT rec_id FROM server WHERE command='S'")))
      goto Calc_unlock;

    nrows = UdmSQLNumRows(&Res);
    for (i = 0; i < nrows; i++)
    {

      if (use_tracking)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM qinfo WHERE name='site' AND value='%s'", UdmSQLValue(&Res, i, 0) );
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
        u = (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)) == 0);
      }
      if (feed_back && (u || !use_tracking))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(pop_rank) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
      }
      if (*UdmSQLValue(&SQLres, 0, 0))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET weight=%s WHERE rec_id=%s%s%s", UdmSQLValue(&SQLres, 0, 0), 
         qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLQuery(db, NULL, qbuf);
      }
      UdmSQLFree(&SQLres);
    }
    UdmSQLFree(&Res);
    UdmSQLQuery(db, NULL, "UPDATE server SET weight=1 WHERE weight=0 AND command='S'");
  }


  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, "SELECT rec_id, url, weight FROM server WHERE command='S'")))
    goto Calc_unlock;
  
  nrows = UdmSQLNumRows(&SQLres);

  for (i = 0; i < nrows; i++)
  {
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&SQLres, i, 0), qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf)))
    goto Calc_unlock;
    UdmLog(A, UDM_LOG_EXTRA, "Site_id: %s URL: %s Weight: %s URL count: %s",
           UdmSQLValue(&SQLres, i, 0),
           UdmSQLValue(&SQLres, i, 1),
           UdmSQLValue(&SQLres, i, 2),
           UdmSQLValue(&Res, 0, 0)); 
    if (atoi(UdmSQLValue(&Res, 0, 0)) > 0)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET pop_weight=(%s/%s.0) WHERE rec_id=%s%s%s",
      UdmSQLValue(&SQLres, i, 2), UdmSQLValue(&Res, 0, 0), qu, UdmSQLValue(&SQLres, i, 0), qu);
      UdmSQLQuery(db, NULL, qbuf);
    }
    UdmSQLFree(&Res);

  }
  UdmSQLFree(&SQLres);


  UdmLog(A, UDM_LOG_EXTRA, "update links and pages weights");
  if (skip_same_site)  UdmLog(A, UDM_LOG_EXTRA, "Will skip links from same site");
  if (use_showcnt)  UdmLog(A, UDM_LOG_EXTRA, "Will add show count");

        udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id, site_id  FROM url ORDER BY rec_id");

  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) goto Calc_unlock;
  nrows = UdmSQLNumRows(&Res);
  for (i = 0; i < nrows; i++)
  {

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links WHERE ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (*UdmSQLValue(&SQLres, 0, 0)) 
    {
      if (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT pop_weight FROM server WHERE rec_id=%s%s%s", qu, UdmSQLValue(&Res, i, 1), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &POPres, qbuf))) goto Calc_unlock;
        if (UdmSQLNumRows(&POPres) != 1)
        { 
          UdmSQLFree(&POPres);
          UdmSQLFree(&SQLres);
          continue;
        }

        udm_snprintf(qbuf, sizeof(qbuf),
                     "UPDATE links SET weight = (%s/%d.0) WHERE ot=%s%s%s",
                     UdmSQLValue(&POPres, 0, 0),
                     atoi(UdmSQLValue(&SQLres, 0, 0)),
                     qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLFree(&POPres);
        UdmSQLQuery(db, NULL, qbuf);
      }
    }
    UdmSQLFree(&SQLres);

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links WHERE k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (UdmSQLValue(&SQLres,0,0) && *UdmSQLValue(&SQLres, 0, 0))
    {
      if (use_showcnt)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s + (shows * %f) WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), ratio, qu, UdmSQLValue(&Res, i, 0), qu );
      }
      else
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), qu, UdmSQLValue(&Res, i, 0), qu );
      }
    } else {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=(shows * %f) WHERE rec_id=%s%s%s", 
                   ratio, qu, UdmSQLValue(&Res, i, 0), qu );
    }
    UdmSQLQuery(db, NULL, qbuf);
    UdmSQLFree(&SQLres);
  }
  UdmSQLFree(&Res);

  rc = UDM_OK;

Calc_unlock:
  UdmLog(A, UDM_LOG_EXTRA, "Popularity rank done.");
  return rc;
}


static int
UdmWordStatQuery(UDM_AGENT *A, UDM_DB *db, const char *src)
{
  int rc;
  UDM_SQLRES SQLRes;
  size_t row, rows;
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, src)))
    return rc;
  
  rows=UdmSQLNumRows(&SQLRes);
  for(row=0 ; row < rows ; row++)
  {
    const char *word;
    int count;
    size_t wordlen;
    char snd[32];
    char insert[128];
    word= UdmSQLValue(&SQLRes, row, 0);
    wordlen= UdmSQLLen(&SQLRes, row, 0);
    count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
    UdmSoundex(A->Conf->lcs, snd, word, wordlen);
    sprintf(insert,
            "INSERT INTO wrdstat (word, snd, cnt) VALUES ('%s','%s',%d)",
            word, snd, count);
    if (UDM_OK!= (rc= UdmSQLQuery(db, NULL, insert)))
      return rc;
  }
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int
UdmWordStatCreateMulti(UDM_AGENT *A, UDM_DB *db)
{
  int rc;
  size_t i;
  
  for (i=0; i <= 0xFF; i++)
  {
    char qbuf[128];
    UdmLog(A, UDM_LOG_EXTRA, "Processing table %02X", i);

    sprintf(qbuf, "SELECT word, count(*) FROM dict%02X GROUP BY word", i);
    if (UDM_OK != (rc= UdmWordStatQuery(A, db, qbuf)))
      return rc;
  }
  return UDM_OK;
}


static int
UdmWordStatCreateSingle(UDM_AGENT *A, UDM_DB *db)
{
  size_t row, rows;
  UDM_SQLRES SQLRes;
  char qbuf[128];
  sprintf(qbuf, "SELECT word, count(*) FROM dict GROUP BY word");
  return UdmWordStatQuery(A, db, qbuf);
}

static int
UdmWordStatCreate(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  int rc;

  UdmLog(A, UDM_LOG_ERROR, "Calculating word statistics");

  if (db->flags & UDM_SQL_HAVE_TRUNCATE)
  {
    rc= UdmSQLQuery(db, NULL, "TRUNCATE TABLE wrdstat");
  }
  else
  {
    rc= UdmSQLQuery(db, NULL, "DELETE FROM wrdstat");
  }
  
  if (rc != UDM_OK)
    return rc;
  
  if (db->DBMode == UDM_DBMODE_SINGLE)
  {
    rc= UdmWordStatCreateSingle(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_MULTI)
  {
    rc= UdmWordStatCreateMulti(A, db);
  }
  
  UdmLog(A, UDM_LOG_ERROR, "Word statistics done");
  return rc;
}


static int
UdmDocPerSite(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  char qbuf[1024];
  const char *s, *hostinfo= UdmVarListFindStr(&D->Sections, "Hostinfo", NULL);
  int rc, num, prevnum= UdmVarListFindInt(&D->Sections, "DocPerSite", 0);
  UDM_SQLRES SQLRes;
  
  if (!hostinfo)
    return UDM_OK;
  
  for (s= hostinfo; s[0]; s++)
  {
    /*
      Host name good characters: digits, letters, hyphen (-).
      Just check the worst characters.
    */
    if (*s == '\'' || *s == '\"')
    {
      num= 1000000;
      goto ret;
    }
  }
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT COUNT(*) FROM url WHERE url LIKE '%s%%'", hostinfo);
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;
  num= prevnum + atoi(UdmSQLValue(&SQLRes, 0, 0));
  UdmSQLFree(&SQLRes);
ret:
  UdmVarListReplaceInt(&D->Sections, "DocPerSite", num);
  return UDM_OK;
}

 
int UdmURLActionSQL(UDM_AGENT * A, UDM_DOCUMENT * D, int cmd,UDM_DB *db)
{
  int res;

  switch(cmd)
  {
    case UDM_URL_ACTION_DELETE:
      res=UdmDeleteURL(A,D,db);
      break;
      
    case UDM_URL_ACTION_ADD:
      res=UdmAddURL(A,D,db);
      break;
      
    case UDM_URL_ACTION_ADD_LINK:
      res=UdmAddLink(A,D,db);
      break;
      
    case UDM_URL_ACTION_SUPDATE:
      res=UdmUpdateUrl(A,D,db);
      break;
      
    case UDM_URL_ACTION_LUPDATE:
      res=UdmLongUpdateURL(A,D,db);
      break;
      
    case UDM_URL_ACTION_DUPDATE:
      res=UdmDeleteWordsAndLinks(A,D,db);
      break;
      
    case UDM_URL_ACTION_UPDCLONE:
      res=UdmUpdateClone(A,D,db);
      break;
      
    case UDM_URL_ACTION_REGCHILD:
      res=UdmRegisterChild(A,D,db);
      break;
      
    case UDM_URL_ACTION_FINDBYURL:
      res=UdmFindURL(A,D,db);
      break;
      
    case UDM_URL_ACTION_FINDBYMSG:
      res=UdmFindMessage(A,D,db);
      break;
      
    case UDM_URL_ACTION_FINDORIG:
      res=UdmFindOrigin(A,D,db);
      break;
      
    case UDM_URL_ACTION_EXPIRE:
      res=UdmMarkForReindex(A,db);
      break;
      
    case UDM_URL_ACTION_REFERERS:
      res=UdmGetReferers(A,db);
      break;
    
    case UDM_URL_ACTION_DOCCOUNT:
      res=UdmGetDocCount(A,db);
      break;
    
    case UDM_URL_ACTION_LINKS_DELETE:
      res = UdmDeleteLinks(A, D, db);
      break;

    case UDM_URL_ACTION_GET_CACHED_COPY:
      res = UdmGetCachedCopy(A, D, db);
      break;

    case UDM_URL_ACTION_WRDSTAT:
      res= UdmWordStatCreate(A, D, db);
      break;

    case UDM_URL_ACTION_DOCPERSITE:
      res= UdmDocPerSite(A, D, db);
      break;
      
    default:
      UdmLog(A, UDM_LOG_ERROR, "Unsupported URL Action SQL");
      res=UDM_ERROR;
  }
  return res;
}


static size_t
WordProximity(UDM_CHARSET *cs,
              const char *s1, size_t len1,
              const char *s2, size_t len2)
{
  size_t max= len1 > len2 ? len1 : len2;
  size_t min= len1 < len2 ? len1 : len2;
  if ((max - min)*3 > max)  /* Not more than 1/3 of length difference */
    return 0;
  return (1000 * min) / (max ? max : 1);
}


static int
UdmResSuggest(UDM_AGENT *A, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  int rc= UDM_OK;
  size_t i, nwords;
  UDM_WIDEWORDLIST *WWList= &Res->WWList;
  UDM_CONV lcs_uni;
  
  UdmLog(A, UDM_LOG_DEBUG, "Generating suggestion list");
  UdmConvInit(&lcs_uni, A->Conf->lcs, &udm_charset_sys_int, UDM_RECODE_HTML);

  for (i=0, nwords= WWList->nwords; i < nwords; i++)
  {
    UDM_WIDEWORD *W= &WWList->Word[i];
    if (W->origin == UDM_WORD_ORIGIN_QUERY && !W->count)
    {
      char snd[16];
      char qbuf[128];
      UDM_SQLRES SQLRes;
      UDM_WIDEWORD sg;
      size_t row, rows, max_count= 0;
      size_t Wlen= W->len; /* W is not valid after UdmWideWordListAdd() */
      size_t Word= W->order;
      const char *Wword= W->word;
      
      UdmSoundex(A->Conf->lcs, snd, W->word, W->len);
      UdmLog(A, UDM_LOG_DEBUG, "Suggest for '%s': '%s'", W->word, snd);
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT word, cnt FROM wrdstat WHERE snd='%s' ORDER by cnt DESC",
                    snd);
      
      if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
        return rc;
      
      rows=UdmSQLNumRows(&SQLRes);
      UdmLog(A, UDM_LOG_DEBUG, "%d suggestions found", rows);
      bzero((void*)&sg, sizeof(sg));

      for(row=0 ; row < rows ; row++)
      {
        size_t nbytes, proximity_weight, count_weight, final_weight;
        int tmp[128];
        
        sg.word= (char*) UdmSQLValue(&SQLRes, row, 0);
        sg.count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
        sg.len= UdmSQLLen(&SQLRes, row, 0);
        if (max_count <= sg.count)
          max_count= sg.count;
        count_weight= (100 * sg.count) / (max_count ? max_count : 1);
        proximity_weight= WordProximity(A->Conf->lcs, Wword, Wlen, sg.word, sg.len); 
        final_weight= count_weight * proximity_weight;
        UdmLog(A, UDM_LOG_DEBUG, "'%s': %d/%d/%d/%d", 
               sg.word, sg.count, count_weight, proximity_weight, final_weight);
        sg.count= final_weight;

        nbytes= (sg.len + 1) * sizeof(int);
        if (nbytes < sizeof(tmp))
        {
          sg.order= Word;
          sg.origin= UDM_WORD_ORIGIN_SUGGEST;
          sg.uword= (int*) &tmp;
          sg.ulen= UdmConv(&lcs_uni, (char*)&tmp, nbytes, sg.word, sg.len + 1);
          UdmWideWordListAdd(WWList, &sg); /* W is not valid after this */
        }
      }
      UdmSQLFree(&SQLRes);
    }
  }
  return rc;
}


int UdmResActionSQL(UDM_AGENT *Agent, UDM_RESULT *Res, int cmd, UDM_DB *db, size_t dbnum)
{
  switch(cmd)
  {
    case UDM_RES_ACTION_DOCINFO:
      return UdmResAddDocInfoSQL(Agent, db, Res, dbnum);
    case UDM_RES_ACTION_SUGGEST:
      return UdmResSuggest(Agent, db, Res, dbnum);
    default:
      UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Res Action SQL");
      return UDM_ERROR;
  }
}

int UdmCatActionSQL(UDM_AGENT *Agent, UDM_CATEGORY *Cat, int cmd,UDM_DB *db)
{
  switch(cmd)
  {
    case UDM_CAT_ACTION_LIST:
      return UdmCatList(Agent,Cat,db);
    case UDM_CAT_ACTION_PATH:
      return UdmCatPath(Agent,Cat,db);
    default:
              UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Cat Action SQL");
      return UDM_ERROR;
  }
}

int UdmSrvActionSQL(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,UDM_DB *db){
  switch(cmd){
    case UDM_SRV_ACTION_TABLE:
      return UdmLoadServerTable(A,S,db);
    case UDM_SRV_ACTION_FLUSH:
      return UdmServerTableFlush(db);
    case UDM_SRV_ACTION_ADD:
      return UdmServerTableAdd(S, db);
    case UDM_SRV_ACTION_ID:
      return UdmServerTableGetId(A, S, db);
    case UDM_SRV_ACTION_POPRANK:
      return UdmPopRankCalculate(A, db);
    default:
      UdmLog(A, UDM_LOG_ERROR, "Unsupported Srv Action SQL");
      return UDM_ERROR;
  }
}

unsigned int   UdmGetCategoryIdSQL(UDM_ENV *Conf, char *category, UDM_DB *db)
{
  UDM_SQLRES Res;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, 128, "SELECT rec_id FROM categories WHERE path LIKE '%s'", category);
  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) return rc;
  if ( UdmSQLNumRows(&Res) > 0)
  {
    sscanf(UdmSQLValue(&Res, 0, 0), "%u", &rc);
  }
  UdmSQLFree(&Res);
  return rc;
}


int UdmCheckUrlidSQL(UDM_AGENT *Agent, UDM_DB *db, urlid_t id)
{
  UDM_SQLRES SQLRes;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id FROM url WHERE rec_id=%d", id);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if(UDM_OK != rc)
  {
    rc = 1;
  }
  else
  {
    if (UdmSQLNumRows(&SQLRes) != 0) rc = 1;
    else rc = 0;
  }
  UdmSQLFree(&SQLRes);
  return rc;
}

int UdmExportSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  UDM_PSTR row[24];

  printf("<database>\n");
  printf("<urlList>\n");
  rc= db->sql->SQLExecDirect(db, &SQLRes, "SELECT * FROM url");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<url "
      "rec_id=\"%s\" "
      "status=\"%s\" "
      "docsize=\"%s\" "
      "next_index_time=\"%s\" "
      "last_mod_time=\"%s\" "
      "referrer=\"%s\" "
      "hops=\"%s\" "
      "crc32=\"%s\" "
      "seed=\"%s\" "
      "bad_since_time=\"%s\" "
      "site_id=\"%s\" "
      "server_id=\"%s\" "
      "shows=\"%s\" "
      "pop_rank=\"%s\" "
      "url=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val, row[3].val,
      row[4].val, row[5].val, row[6].val, row[7].val,
      row[8].val, row[9].val, row[10].val, row[11].val,
      row[12].val, row[13].val, row[14].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</urlList>\n");

  printf("<linkList>\n");
  rc= db->sql->SQLExecDirect(db, &SQLRes, "SELECT * FROM links");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<link "
      "ot=\"%s\" "
      "k=\"%s\" "
      "weight=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</linkList>\n");

  printf("</database>\n");
  return(0);
}

int UdmUserCacheStoreSQL(UDM_AGENT *Indexer, UDM_RESULT *Res, UDM_DB *db)
{
  const char *usercache = UdmVarListFindStr(&db->Vars, "usercache", NULL);
  const char *prevcache = UdmVarListFindStr(&db->Vars, "prevcache", NULL);
  int rc= UDM_OK;
  char qbuf[64];

  if (usercache)
  {
    size_t i;
    for (i = 0; i < Res->CoordList.ncoords; i++)
    {
      sprintf(qbuf, "INSERT INTO %s VALUES(%d, %d)",
                    usercache,
                    Res->CoordList.Coords[i].url_id,
                    Res->CoordList.Coords[i].coord);
      rc= UdmSQLQuery(db, NULL, qbuf);
      if (rc != UDM_OK)
        return rc;
    }
  }
  
  if (prevcache)
  {
    const char  *words=UdmVarListFindStr(&Indexer->Conf->Vars,"q","");
    const char  *IP = UdmVarListFindStr(&Indexer->Conf->Vars, "IP", "");
    UDM_DSTR buf;
    size_t i, nbytes= Res->CoordList.ncoords * 16;
    udmhash32_t  id;
    int tm= time(0);
    char *s;
    
    udm_snprintf(qbuf, sizeof(qbuf), "%s %s", IP, words);
    id= UdmStrHash32(qbuf);
    sprintf(qbuf, "%08X-%08X", id, tm);

    UdmDSTRInit(&buf, 256);
    UdmDSTRRealloc(&buf, nbytes + 128);
    UdmDSTRAppendf(&buf,
                   "INSERT INTO qcache (id, tm, doclist) VALUES (%d, %d, 0x",
                   id, tm);
    for (s= buf.data + buf.size_data, i= 0; i < Res->CoordList.ncoords; i++)
    {
      uint4 coord;
      id= Res->CoordList.Coords[i].url_id;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2;
      coord= Res->CoordList.Coords[i].coord;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2;
    }
    buf.size_data+= nbytes;
    buf.data[buf.size_data]= '\0';
    UdmDSTRAppend(&buf, ")", 1);
    rc= UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRFree(&buf);
    if (rc == UDM_OK)
      UdmVarListAddStr(&Indexer->Conf->Vars, "qid", qbuf);
  }
  return rc;
}

#endif /* HAVE_SQL */
