#ifndef XMLELEMENTS_H
#define XMLELEMENTS_H

#include <string>
#include <list>

#include "XMLException.h"


//----------------------------------------------------------------------------
class XMLVisitor;
class XMLConstVisitor;


//----------------------------------------------------------------------------
#define DECLARE_XML_VISITOR_API() \
    void do_accept(XMLVisitor &v); \
    void do_accept(XMLConstVisitor &v) const

//----------------------------------------------------------------------------
#define DECLARE_XML_VISITOR_API_BODY(C) \
void C::do_accept(XMLVisitor &v) { v.visit(this); } \
void C::do_accept(XMLConstVisitor &v) const { v.visit(this); }


//----------------------------------------------------------------------------
/**
 * This is a base class for all XML elements.
 */
class XMLElement
{
  protected:
    //------------------------------------------------------------------------
    XMLElement() {}
    XMLElement(const std::string &name) : m_name(name) {}

  public:
    //------------------------------------------------------------------------
    virtual ~XMLElement() {}

    //------------------------------------------------------------------------
    inline const std::string &getName() const { return m_name; }

    //------------------------------------------------------------------------
    inline void accept(XMLVisitor &v) { do_accept(v); }
    inline void accept(XMLConstVisitor &v) const  { do_accept(v); }

  private:

    //------------------------------------------------------------------------
    virtual void do_accept(XMLVisitor &v) = 0;
    virtual void do_accept(XMLConstVisitor &v) const = 0;

    //------------------------------------------------------------------------
    std::string m_name;
};


//----------------------------------------------------------------------------
/**
 * This class represents a property of an XML node.
 */
class XMLProperty : public XMLElement
{
  public:
    //------------------------------------------------------------------------
    XMLProperty(const std::string &name, const bool value);
    XMLProperty(const std::string &name, const int value);
    XMLProperty(const std::string &name, const unsigned value);
    XMLProperty(const std::string &name, const char *value);
    XMLProperty(const std::string &name, const std::string &value);
    ~XMLProperty();

    //------------------------------------------------------------------------
    void setValue(const bool value);
    void setValue(const int value);
    void setValue(const unsigned value);
    void setValue(const char *value);
    void setValue(const std::string &value);

    //------------------------------------------------------------------------
    inline const std::string &getValue() const { return m_value; }

    //------------------------------------------------------------------------
    /**
     * @return The value of this property as bool.
     * @throw XMLException if the value cannot be converted to a bool.
     */
    bool getAsBool() const throw (XMLException);

    /**
     * @return The value of this property as int.
     * @throw XMLException if the value cannot be converted to an int.
     */
    int getAsInt() const throw (XMLException);

    /**
     * @return The value of this property as unsigned.
     * @throw XMLException if the value cannot be converted to an unsigned.
     */
    unsigned getAsUnsigned() const throw (XMLException);

  private:

    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    std::string m_value;
};


//----------------------------------------------------------------------------
/**
 * This class represents a text within an XML node.
 */
class XMLText : public XMLElement
{
  public:
    //------------------------------------------------------------------------
    XMLText(const std::string &text);
    ~XMLText();

    //------------------------------------------------------------------------
    inline const std::string &getText() const { return m_text; }

  private:

    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    std::string m_text;
};


//----------------------------------------------------------------------------
/**
 * This class represents an XML node.
 * It can contain XML properties and further XML nodes.
 */
class XMLNode : public XMLElement
{
    //------------------------------------------------------------------------
    typedef std::list<XMLProperty*> XMLProperties;
    typedef XMLProperties::iterator XMLPropertyIter;
    typedef XMLProperties::const_iterator XMLPropertyCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLText*> XMLTexts;
    typedef XMLTexts::iterator XMLTextIter;
    typedef XMLTexts::const_iterator XMLTextCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLNode*> XMLNodes;
    typedef XMLNodes::iterator XMLNodeIter;
    typedef XMLNodes::const_iterator XMLNodeCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLElement*> XMLChilds;
    typedef XMLChilds::iterator XMLChildIter;
    typedef XMLChilds::const_iterator XMLChildCIter;

  public:
    //------------------------------------------------------------------------
    XMLNode();
    XMLNode(const std::string &name);
    ~XMLNode();

    //------------------------------------------------------------------------
    void clear();

    //------------------------------------------------------------------------
    /**
     * @param p The property to add to this node.
     */
    void addProperty(XMLProperty *p);

    /**
     * @param name The name of the property to get.
     * @return A pointer to the property with the given name.
     * @throw XMLException if no such property exists.
     */
    XMLProperty *getMandatoryProperty(const std::string &name)
        throw (XMLException);

    /**
     * @param name The name of the property to get.
     * @return A const pointer to the property with the given name.
     * @throw XMLException if no such property exists.
     */
    const XMLProperty *getMandatoryProperty(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the property to get.
     * @return A pointer to the property with the given name,
     *         or NULL, if no such property exists.
     */
    XMLProperty *getProperty(const std::string &name);

    /**
     * @param name The name of the property to get.
     * @return A const pointer to the property with the given name,
     *         or NULL, if no such property exists.
     */
    const XMLProperty *getProperty(const std::string &name) const;

    /**
     * @param name The name of the property to check for.
     * @return true, if the property exists, else false.
     */
    bool hasProperty(const std::string &name) const;

    /**
     * @return The number of properties in this node.
     */
    inline size_t getNumberOfProperties() const { return m_properties.size(); }

    //------------------------------------------------------------------------
    /**
     * @param t The text node to add to this node.
     */
    void addText(XMLText *t);

    /**
     * @return true, if a text node exists, else false.
     */
    bool hasText() const;

    /**
     * @return A pointer to the text node.
     * @throw XMLException if no text node exists.
     */
    const std::string &getText() const
        throw (XMLException);


    //------------------------------------------------------------------------
    /**
     * @param n The node to add to this node.
     */
    void addNode(XMLNode *n);

    /**
     * @param name The name of the node to get.
     * @return A pointer to the node with the given name.
     * @throw XMLException if no such node exists.
     */
    XMLNode *getMandatoryNode(const std::string &name)
        throw (XMLException);

    /**
     * @param name The name of the node to get.
     * @return A const pointer to the node with the given name.
     * @throw XMLException if no such node exists.
     */
    const XMLNode *getMandatoryNode(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the node to get.
     * @return A pointer to the node with the given name,
     *         or NULL, if no such node exists.
     */
    XMLNode *getNode(const std::string &name);

    /**
     * @param name The name of the node to get.
     * @return A const pointer to the node with the given name,
     *         or NULL, if no such node exists.
     */
    const XMLNode *getNode(const std::string &name) const;

    /**
     * @param name The name of the node to check for.
     * @return true, if the node exists, else false.
     */
    bool hasNode(const std::string &name) const;

    /**
     * @return The number of nodes in this node.
     */
    inline size_t getNumberOfNodes() const { return m_nodes.size(); }

    //------------------------------------------------------------------------
    /**
     * @param name The name of the property to search for.
     * @return The string value of the found property.
     * @throw XMLException if no such property exists.
     */
    const std::string &getStringProperty(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @return The string value of the found property,
     *         or the given default value, if the property doesn't exist.
     */
    const std::string &getStringProperty(const std::string &name,
                                         const std::string &defaultValue) const;

    /**
     * @param name The name of the property to search for.
     * @return The bool value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to a bool.
     */
    bool getBoolProperty(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The bool value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to a bool.
     */
    bool getBoolProperty(const std::string &name, bool defaultValue) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @return The int value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to an int.
     */
    int getIntProperty(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The int value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to an int.
     */
    int getIntProperty(const std::string &name, int defaultValue) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @return The unsigned value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to an unsigned.
     */
    unsigned getUnsignedProperty(const std::string &name) const
        throw (XMLException);

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The unsigned value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to an unsigned.
     */
    unsigned getUnsignedProperty(const std::string &name,
                                 unsigned defaultValue) const
        throw (XMLException);

    //------------------------------------------------------------------------
    void acceptAllProperties(XMLVisitor &v);
    void acceptAllProperties(XMLConstVisitor &v) const;

    //------------------------------------------------------------------------
    void acceptAllChilds(XMLVisitor &v);
    void acceptAllChilds(XMLConstVisitor &v) const;

  private:

    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    XMLProperties m_properties;
    XMLTexts m_texts;
    XMLNodes m_nodes;
    XMLChilds m_childs;
};

#endif //XMLELEMENTS_H
