#include <unistd.h>
#include <dirent.h>

#include <cerrno>
#include <sstream>
#include <list>

#include "Tools.h"
#include "SyscallException.h"
#include "SystemCalls.h"
#include "Directory.h"
#include "File.h"


//============================================================================
// Implementation of SyscallException.h
//============================================================================

//----------------------------------------------------------------------------
SyscallException::SyscallException(const std::string &message)
        : Exception(message)
{
    m_errno = errno;
}

//----------------------------------------------------------------------------
SyscallException::SyscallException(const std::string &message, int error)
        : Exception(message)
{
    m_errno = error;
}

//----------------------------------------------------------------------------
SyscallException::~SyscallException()
{
}

//----------------------------------------------------------------------------
std::string SyscallException::toString() const
{
    return std::string(m_message).append(": ").append(strerror(m_errno));
}



//============================================================================
// Implementation of SystemCalls.h
//============================================================================

//----------------------------------------------------------------------------
void SYSTEM_CALL::stat(const char *file, struct stat *buf)
    throw (SyscallException)
{
    if (::stat(file, buf) != 0)
    {
        throw SyscallException(
            std::string("Cannot stat() file '").append(file).append("'"));
    }
}

//----------------------------------------------------------------------------
void SYSTEM_CALL::lstat(const char *file, struct stat *buf)
    throw (SyscallException)
{
    if (::lstat(file, buf) != 0)
    {
        throw SyscallException(
            std::string("Cannot lstat() file '").append(file).append("'"));
    }
}



//============================================================================
// Implementation of File.h
//============================================================================

//----------------------------------------------------------------------------
class FilePImpl
{
public:
    //------------------------------------------------------------------------
    FilePImpl(const char *name, const char *mode)
        throw (SyscallException);
    ~FilePImpl();

    //------------------------------------------------------------------------
    const char *getName() const;

    bool isEOF() const;

    void seek(long offset)
        throw (SyscallException);

    size_t read(void *buffer, size_t len)
        throw (SyscallException);
    size_t write(const void *buffer, size_t len)
        throw (SyscallException);

    int puts(const char *s)
        throw (SyscallException);

    void readline(char *buffer, size_t len)
        throw (SyscallException);

private:
    //------------------------------------------------------------------------
    const char *m_name;
    FILE *m_file;
};


//----------------------------------------------------------------------------
FilePImpl::FilePImpl(const char *name, const char *mode)
    throw (SyscallException)
{
    m_name = name;
    m_file = fopen(name, mode);
    if (m_file == NULL)
    {
        throw SyscallException(
            std::string("Cannot open file '").append(name)
            .append("' in mode '").append(mode).append("'"));
    }
}

//----------------------------------------------------------------------------
FilePImpl::~FilePImpl()
{
    fclose(m_file);
    m_file = NULL;
}

//----------------------------------------------------------------------------
const char *FilePImpl::getName() const
{
    return m_name;
}

//----------------------------------------------------------------------------
bool FilePImpl::isEOF() const
{
    return feof(m_file);
}

//----------------------------------------------------------------------------
void FilePImpl::seek(long offset)
    throw (SyscallException)
{
    int rc = fseek(m_file, offset, SEEK_SET);
    if (rc == -1)
    {
        throw SyscallException("Cannot seek in file");
    }
}

//----------------------------------------------------------------------------
size_t FilePImpl::read(void *buffer, size_t len) throw (SyscallException)
{
    size_t rc = fread(buffer, 1, len, m_file);
    if ((rc != len) && !isEOF())
    {
        int e = ferror(m_file);
        if (e != 0)
        {
            clearerr(m_file);
            throw SyscallException("Cannot read from file", e);
        }
    }
    return rc;
}

//----------------------------------------------------------------------------
size_t FilePImpl::write(const void *buffer, size_t len)
    throw (SyscallException)
{
    size_t rc = fwrite(buffer, 1, len, m_file);
    if (rc != len)
    {
        int e = ferror(m_file);
        if (e != 0)
        {
            clearerr(m_file);
            throw SyscallException("Cannot write to file", e);
        }
    }
    return rc;
}

//----------------------------------------------------------------------------
int FilePImpl::puts(const char *s)
    throw (SyscallException)
{
    int rc = fputs(s, m_file);
    if (rc == EOF)
    {
        int e = ferror(m_file);
        if (e != 0)
        {
            clearerr(m_file);
            throw SyscallException("Cannot write to file", e);
        }
    }
    return rc;
}

//----------------------------------------------------------------------------
void FilePImpl::readline(char *buffer, size_t len) throw (SyscallException)
{
    fgets(buffer, len, m_file);
    char *newline = strchr(buffer, '\n');
    if (newline)
    {
        *newline = '\0';
    }
}


//----------------------------------------------------------------------------
File::File(const char *name, const char *mode) throw (SyscallException)
{
    m_pImpl = new FilePImpl(name, mode);
}

//----------------------------------------------------------------------------
File::~File()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
const char *File::getName() const
{
    return m_pImpl->getName();
}

//----------------------------------------------------------------------------
bool File::isEOF() const
{
    return m_pImpl->isEOF();
}


//----------------------------------------------------------------------------
void File::seek(long offset)
    throw (SyscallException)
{
    m_pImpl->seek(offset);
}


//----------------------------------------------------------------------------
uint8_t File::readUint8() throw (SyscallException)
{
    uint8_t value;
    read(&value, 1);
    return value;
}

//----------------------------------------------------------------------------
void File::writeUint8(uint8_t value) throw (SyscallException)
{
    write(&value, 1);
}

//----------------------------------------------------------------------------
uint16_t File::readUint16() throw (SyscallException)
{
    char buffer[2];
    read(buffer, 2);
    uint16_t value = (uint16_t)buffer[0] << 8 | (uint16_t)buffer[1];
    return value;
}

//----------------------------------------------------------------------------
void File::writeUint16(uint16_t value) throw (SyscallException)
{
    char buffer[2];
    buffer[0] = (value & 0xff00) >> 8;
    buffer[1] = value & 0x00ff;
    write(buffer, 2);
}


//----------------------------------------------------------------------------
size_t File::read(void *buffer, size_t len) throw (SyscallException)
{
    return m_pImpl->read(buffer, len);
}

//----------------------------------------------------------------------------
size_t File::write(const void *buffer, size_t len) throw (SyscallException)
{
    return m_pImpl->write(buffer, len);
}

//----------------------------------------------------------------------------
int File::puts(const char *s) throw (SyscallException)
{
    return m_pImpl->puts(s);
}

//----------------------------------------------------------------------------
int File::puts(const std::string &s) throw (SyscallException)
{
    return m_pImpl->puts(s.c_str());
}

//----------------------------------------------------------------------------
void File::readline(char *buffer, size_t len) throw (SyscallException)
{
    m_pImpl->readline(buffer, len);
}


//----------------------------------------------------------------------------
void File::unlink(const char *name) throw (SyscallException)
{
    if ((::unlink(name) != 0) && (errno != ENOENT))
    {
        throw SyscallException(
            std::string("Error removing file '")
            .append(name).append("'"));
    }
}

//----------------------------------------------------------------------------
void File::copy(const char *src, const char *dst) throw (SyscallException)
{
    File srcFile(src, "r");
    File dstFile(dst, "w");

    char buffer[4096];
    size_t len;

    do
    {
        len = srcFile.read(buffer, sizeof(buffer));
        if (len > 0)
        {
            dstFile.write(buffer, len);
        }
    }
    while (len == sizeof(buffer));
}



//============================================================================
// Implementation of Directory.h
//============================================================================

//----------------------------------------------------------------------------
class DirectoryPImpl
{
public:
    //------------------------------------------------------------------------
    class Handle
    {
    public:
        Handle(const char *name)
            throw (SyscallException);
        ~Handle();

        void rewind();
        struct dirent *read();

    private:
        DIR *m_dir;
    };

    //------------------------------------------------------------------------
    typedef std::list<std::string> Entries;
    typedef Entries::iterator iterator;
    typedef Entries::const_iterator const_iterator;
    
    //------------------------------------------------------------------------
    DirectoryPImpl();
    ~DirectoryPImpl();

    //------------------------------------------------------------------------
    inline void clear() { m_entries.clear(); }
    inline size_t size() const { return m_entries.size(); }
    inline bool empty() const { return m_entries.empty(); }

    inline iterator begin() { return m_entries.begin(); }
    inline iterator end() { return m_entries.end(); }

    inline const_iterator begin() const { return m_entries.begin(); }
    inline const_iterator end() const { return m_entries.end(); }

    //------------------------------------------------------------------------
    void read(const char *name)
        throw (SyscallException);

    const char *getName() const;

    bool hasDirectory(const char *name) const;
    bool hasFile(const char *name) const;

    void getFilesWithSuffix(const char* suffix,
                            Directory::FileList &list) const;

private:
    //------------------------------------------------------------------------
    const char *m_name;
    Entries m_entries;
};


//----------------------------------------------------------------------------
DirectoryPImpl::DirectoryPImpl()
{
}

//----------------------------------------------------------------------------
DirectoryPImpl::~DirectoryPImpl()
{
}

//----------------------------------------------------------------------------
void DirectoryPImpl::read(const char *name) throw (SyscallException)
{
    clear();

    m_name = name;
    Handle h(m_name);

    for (;;)
    {
        struct dirent *dirEntry = h.read();
        if (dirEntry == NULL)  break;

        if (strcmp(dirEntry->d_name, ".") &&
            strcmp(dirEntry->d_name, ".."))
        {
            m_entries.push_back(dirEntry->d_name);
        }
    }
}

//----------------------------------------------------------------------------
const char *DirectoryPImpl::getName() const
{
    return m_name;
}

//----------------------------------------------------------------------------
bool DirectoryPImpl::hasDirectory(const char *name) const
{
    try
    {
        std::string fullName(getName());
        fullName.append("/").append(name);

        struct stat statEntry;
        SYSTEM_CALL::lstat(fullName.c_str(), &statEntry);

        return S_ISDIR(statEntry.st_mode);
    }
    catch (SyscallException &e)
    {
        if (e.getErrno() == ENOENT)
        {
            return false;
        }
        else
        {
            throw;
        }
    }
}

//----------------------------------------------------------------------------
bool DirectoryPImpl::hasFile(const char *name) const
{
    try
    {
        std::string fullName(getName());
        fullName.append("/").append(name);

        struct stat statEntry;
        SYSTEM_CALL::lstat(fullName.c_str(), &statEntry);

        return S_ISREG(statEntry.st_mode);
    }
    catch (SyscallException &e)
    {
        if (e.getErrno() == ENOENT)
        {
            return false;
        }
        else
        {
            throw;
        }
    }
}

//----------------------------------------------------------------------------
void DirectoryPImpl::getFilesWithSuffix(const char* suffix,
                                        Directory::FileList &list) const
{
    list.clear();
    for (const_iterator iter = begin(); iter != end(); ++iter)
    {
        if (iter->find(suffix) == iter->length() - strlen(suffix) &&
            hasFile(iter->c_str()))
        {
            list.push_back(
                std::string(m_name).append("/").append(*iter));
        }
    }
}

//----------------------------------------------------------------------------
DirectoryPImpl::Handle::Handle(const char *name)
    throw (SyscallException)
{
    m_dir = opendir(name);
    if (m_dir == NULL)
    {
        throw SyscallException(
            std::string("Cannot open '").append(name).append("'"));
    }
}

//----------------------------------------------------------------------------
DirectoryPImpl::Handle::~Handle()
{
    if (m_dir)
    {
        closedir(m_dir);
        m_dir = NULL;
    }
}

//----------------------------------------------------------------------------
void DirectoryPImpl::Handle::rewind()
{
    rewinddir(m_dir);
}

//----------------------------------------------------------------------------
struct dirent *DirectoryPImpl::Handle::read()
{
    return readdir(m_dir);
}



//----------------------------------------------------------------------------
Directory::Directory()
{
    m_pImpl = new DirectoryPImpl();
}

//----------------------------------------------------------------------------
Directory::Directory(const char *name)
    throw (SyscallException)
{
    m_pImpl = new DirectoryPImpl();
    read(name);
}

//----------------------------------------------------------------------------
Directory::~Directory()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
void Directory::read(const char *name)
    throw (SyscallException)
{
    m_pImpl->read(name);
}

//----------------------------------------------------------------------------
const char *Directory::getName() const
{
    return m_pImpl->getName();
}

//----------------------------------------------------------------------------
bool Directory::hasDirectory(const char *name) const
{
    return m_pImpl->hasDirectory(name);
}

//----------------------------------------------------------------------------
bool Directory::hasFile(const char *name) const
{
    return m_pImpl->hasFile(name);
}


//----------------------------------------------------------------------------
bool Directory::hasDirectoryInCwd(const char *name)
    throw (SyscallException)
{
    Directory d(".");
    return d.hasDirectory(name);
}

//----------------------------------------------------------------------------
bool Directory::hasFileInCwd(const char *name)
    throw (SyscallException)
{
    Directory d(".");
    return d.hasFile(name);
}

//----------------------------------------------------------------------------
void Directory::getFilesWithSuffix(const char *suffix, FileList &list) const
{
    m_pImpl->getFilesWithSuffix(suffix, list);
}

//----------------------------------------------------------------------------
void Directory::chdir(const char *name)
    throw (SyscallException)
{
    if (::chdir(name) != 0)
    {
        throw SyscallException(
            std::string("Error changing to directory '")
            .append(name).append("'"));
    }
}

//----------------------------------------------------------------------------
void Directory::mkdir(const char *name, mode_t mode, bool recursive)
    throw (SyscallException)
{
    while (::mkdir(name, mode) != 0)
    {
        if (recursive && (errno == ENOENT))
        {
            std::string oldDir = name;
            std::string newDir = oldDir.substr(0, oldDir.rfind("/"));
            mkdir(newDir.c_str(), mode, recursive);
        }
        else if (errno != EEXIST)
        {
            throw SyscallException(
                std::string("Error creating directory '")
                .append(name).append("'"));
        }
        else
        {
            break;
        }
    }
}

//----------------------------------------------------------------------------
void Directory::rmdir(const char *name, bool recursive)
    throw (SyscallException)
{
    Directory d;
    while (::rmdir(name) != 0)
    {
        if (recursive && (errno == ENOTEMPTY))
        {
            d.read(name);

            for (DirectoryPImpl::const_iterator iter = d.m_pImpl->begin();
                 iter != d.m_pImpl->end(); ++iter)
            {
                std::string newName(name);
                newName.append("/").append(*iter);

                if (d.hasDirectory(iter->c_str()))
                {
                    rmdir(newName.c_str(), recursive);
                }
                else
                {
                    File::unlink(newName.c_str());
                }
            }
        }
        else if (errno != ENOENT)
        {
            throw SyscallException(
                std::string("Error removing directory '")
                .append(name).append("'"));
        }
        else
        {
            break;
        }
    }
}

//----------------------------------------------------------------------------
void Directory::copy(const char *src, const char *dst)
    throw (SyscallException)
{
    mkdir(dst);

    Directory srcDir(src);

    for (DirectoryPImpl::const_iterator iter = srcDir.m_pImpl->begin();
         iter != srcDir.m_pImpl->end(); ++iter)
    {
        std::string srcName(src);
        srcName.append("/").append(*iter);

        std::string dstName(dst);
        dstName.append("/").append(*iter);

        if (srcDir.hasDirectory(iter->c_str()))
        {
            copy(srcName.c_str(), dstName.c_str());
        }
        else if (srcDir.hasFile(iter->c_str()))
        {
            File::copy(srcName.c_str(), dstName.c_str());
        }
        else
        {
            std::ostringstream msg;
            msg << "Cannot copy '" << *iter
                << "' which is of invalid type" << std::ends;
            throw SyscallException(msg.str());
        }
    }
}
