#ifndef PLAYGROUNDVISITORS_H
#define PLAYGROUNDVISITORS_H

#include "SDLTools.h"
#include "XMLVisitors.h"
#include "ObjectVisitors.h"


//----------------------------------------------------------------------------
class CreateGravityVisitor : public XMLConstVisitor
{
  public:
    //------------------------------------------------------------------------
    CreateGravityVisitor() {}
    ~CreateGravityVisitor() {}

  private:
    //------------------------------------------------------------------------
    void do_visit(const XMLNode *n);
};

//----------------------------------------------------------------------------
class CreateDecorationsVisitor : public XMLConstVisitor
{
  public:
    //------------------------------------------------------------------------
    CreateDecorationsVisitor() {}
    ~CreateDecorationsVisitor() {}

  private:
    //------------------------------------------------------------------------
    void do_visit(const XMLNode *n);
};


//----------------------------------------------------------------------------
/**
 * This const visitor walks through an ObjectList,
 * will take the PlayGround's shadow surface
 * and will overdraw the bounding box of each object
 * with the content of the map surface.
 */
class OverdrawObjectsConstVisitor : public ObjectConstVisitor
{
  public:
    //------------------------------------------------------------------------
    OverdrawObjectsConstVisitor(const SDL_Surface *mapSurface,
                                SDL_Surface *shadowSurface)
        : m_mapSurface(mapSurface), m_shadowSurface(shadowSurface) {}

    ~OverdrawObjectsConstVisitor()
    {
        m_mapSurface = NULL;
        m_shadowSurface = NULL;
    }

  private:
    //------------------------------------------------------------------------
    void do_visit(const Barrier *b) { do_do_visit(b); }
    void do_visit(const BlackHole *b) { do_do_visit(b); }
    void do_visit(const Crate *c) { do_do_visit(c); }
    void do_visit(const GrenadeBase *g) { do_do_visit(g); }
    void do_visit(const Grinder *g) { do_do_visit(g); }
    void do_visit(const MagnetBase *m) { do_do_visit(m); }
    void do_visit(const Missile *m) { do_do_visit(m); }
    void do_visit(const MortarBase *m) { do_do_visit(m); }
    void do_visit(const ParticleBase *p);
    void do_visit(const ParticleFountainBase *f) { do_do_visit(f); }
    void do_visit(const Platform *p) { do_do_visit(p); }
    void do_visit(const ProjectileBase *p) { do_do_visit(p); }
    void do_visit(const SAMBatteryBase *s) { do_do_visit(s); }
    void do_visit(const Ship *s) { do_do_visit(s); }
    void do_visit(const Tank *t) { do_do_visit(t); }
    void do_visit(const TurretBase *t) { do_do_visit(t); }

    //------------------------------------------------------------------------
    /// A common implementation is used for all type of objects.
    void do_do_visit(const ObjectBase *o);

    //------------------------------------------------------------------------
    const SDL_Surface *m_mapSurface;
    SDL_Surface *m_shadowSurface;
};


//----------------------------------------------------------------------------
/**
 * This visitor walks through an ObjectList,
 * and will call every object's update() method.
 */
class UpdateObjectsVisitor : public ObjectVisitor
{
  public:
    //------------------------------------------------------------------------
    UpdateObjectsVisitor() {}
    ~UpdateObjectsVisitor() {}

  private:
    //------------------------------------------------------------------------
    void do_visit(Barrier *b) { do_do_visit(b); }
    void do_visit(BlackHole *b) { do_do_visit(b); }
    void do_visit(Crate *c) { do_do_visit(c); }
    void do_visit(GrenadeBase *g);
    void do_visit(Grinder *g) { do_do_visit(g); }
    void do_visit(MagnetBase *m) { do_do_visit(m); }
    void do_visit(Missile *m);
    void do_visit(MortarBase *m) { do_do_visit(m); }
    void do_visit(ParticleBase *p);
    void do_visit(ParticleFountainBase *f) { do_do_visit(f); }
    void do_visit(Platform *p) { do_do_visit(p); }
    void do_visit(ProjectileBase *p) { do_do_visit(p); }
    void do_visit(SAMBatteryBase *s) { do_do_visit(s); }
    void do_visit(Ship *s);
    void do_visit(Tank *t) { do_do_visit(t); }
    void do_visit(TurretBase *t) { do_do_visit(t); }

    //------------------------------------------------------------------------
    /// A common implementation is used for most type of objects.
    void do_do_visit(ObjectBase *o);
};


//----------------------------------------------------------------------------
class CollisionDetectorVisitor : public ObjectVisitor
{
    //------------------------------------------------------------------------
    class ObjectCollisionDetectorVisitorBase : public ObjectVisitor
    {
      public:
        //--------------------------------------------------------------------
        ObjectCollisionDetectorVisitorBase() : m_collision(false) {}
        ~ObjectCollisionDetectorVisitorBase() {}

        inline void setCollision() { m_collision = true; }
        inline bool isCollision() const { return m_collision; }

      private:
        //--------------------------------------------------------------------
        bool continueForLoopInVisitObjectList() const
        {
            return !isCollision();
        }

        //--------------------------------------------------------------------
        void do_visit(Barrier *b) { do_onCollision(b); }
        void do_visit(BlackHole *b) { do_onCollision(b); }
        void do_visit(Grinder *g) { do_onCollision(g); }
        void do_visit(MagnetBase *m) { do_onCollision(m); }
        void do_visit(ParticleFountainBase *f) { do_onCollision(f); }
        void do_visit(Platform *p) { do_onCollision(p); }

        virtual void do_onCollision(ObjectBase *o) = 0;

        //--------------------------------------------------------------------
        bool m_collision;
    };

    //------------------------------------------------------------------------
    class ParticleCollisionDetectorVisitor
        : public ObjectCollisionDetectorVisitorBase
    {
      public:
        //--------------------------------------------------------------------
        ParticleCollisionDetectorVisitor(ParticleBase *particle)
            : m_particle(particle) {}
        ~ParticleCollisionDetectorVisitor() { m_particle = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(GrenadeBase *g) { do_onCollision(g); }
        void do_visit(Missile *m) { do_onCollision(m); }
        void do_visit(MortarBase *m) { do_onCollision(m); }
        void do_visit(SAMBatteryBase *s) { do_onCollision(s); }
        void do_visit(Ship *s) { do_onCollision(s); }
        void do_visit(Tank *t) { do_onCollision(t); }
        void do_visit(TurretBase *t) { do_onCollision(t); }

        //--------------------------------------------------------------------
        void do_onCollision(ObjectBase *o);

        //--------------------------------------------------------------------
        ParticleBase *m_particle;
    };

    //------------------------------------------------------------------------
    class ShipCollisionDetectorVisitor
        : public ObjectCollisionDetectorVisitorBase
    {
      public:
        //--------------------------------------------------------------------
        ShipCollisionDetectorVisitor(Ship *ship)
            : m_ship(ship) {}
        ~ShipCollisionDetectorVisitor() { m_ship = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(GrenadeBase *g);
        void do_visit(Missile *m);
        void do_visit(MortarBase *m);
        void do_visit(ProjectileBase *p);
        void do_visit(SAMBatteryBase *s);
        void do_visit(Tank *t);
        void do_visit(TurretBase *t);

        //--------------------------------------------------------------------
        void do_onCollision(ObjectBase *o);

        //--------------------------------------------------------------------
        Ship *m_ship;
    };

    //------------------------------------------------------------------------
    class MissileCollisionDetectorVisitor
        : public ObjectCollisionDetectorVisitorBase
    {
      public:
        //--------------------------------------------------------------------
        MissileCollisionDetectorVisitor(Missile *missile)
            : m_missile(missile) {}
        ~MissileCollisionDetectorVisitor() { m_missile = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(GrenadeBase *g);
        void do_visit(MortarBase *m);
        void do_visit(ProjectileBase *p);
        void do_visit(SAMBatteryBase *s);
        void do_visit(Tank *t);
        void do_visit(TurretBase *t);

        //--------------------------------------------------------------------
        void do_onCollision(ObjectBase *o);

        //--------------------------------------------------------------------
        Missile *m_missile;
    };

    //------------------------------------------------------------------------
    class GrenadeCollisionDetectorVisitor
        : public ObjectCollisionDetectorVisitorBase
    {
      public:
        //--------------------------------------------------------------------
        GrenadeCollisionDetectorVisitor(GrenadeBase *grenade)
            : m_grenade(grenade) {}
        ~GrenadeCollisionDetectorVisitor() { m_grenade = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(MortarBase *m);
        void do_visit(SAMBatteryBase *s) { do_onCollision(s); }
        void do_visit(Tank *t);
        void do_visit(TurretBase *t);

        //--------------------------------------------------------------------
        void do_onCollision(ObjectBase *o);
        
        //--------------------------------------------------------------------
        GrenadeBase *m_grenade;
    };

    //------------------------------------------------------------------------
    class ProjectileCollisionDetectorVisitor
        : public ObjectCollisionDetectorVisitorBase
    {
      public:
        //--------------------------------------------------------------------
        ProjectileCollisionDetectorVisitor(ProjectileBase *projectile)
            : m_projectile(projectile) {}
        ~ProjectileCollisionDetectorVisitor() { m_projectile = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(MortarBase *m);
        void do_visit(SAMBatteryBase *s) { do_onCollision(s); }
        void do_visit(Tank *t);
        void do_visit(TurretBase *t);

        //--------------------------------------------------------------------
        void do_onCollision(ObjectBase *o);
        
        //--------------------------------------------------------------------
        ProjectileBase *m_projectile;
    };

  public:
    //------------------------------------------------------------------------
    CollisionDetectorVisitor(ObjectList &objects)
        : m_objects(objects) {}
    ~CollisionDetectorVisitor() {}

  private:
    //------------------------------------------------------------------------
    // Objects handled in the inner CollisionDetectorVisitors.
    void do_visit(ParticleBase *p);
    void do_visit(Ship *s);
    void do_visit(Missile *m);
    void do_visit(GrenadeBase *g);
    void do_visit(ProjectileBase *p);

    //------------------------------------------------------------------------
    ObjectList &m_objects;
};


//----------------------------------------------------------------------------
/**
 * This const visitor walks through an ObjectList,
 * and will blit each object to the given shadow surface of the PlayGround.
 */
class BlitObjectsConstVisitor : public ObjectConstVisitor
{
  public:
    //------------------------------------------------------------------------
    BlitObjectsConstVisitor(SDL_Surface *shadowSurface)
        : m_shadowSurface(shadowSurface) {}
    ~BlitObjectsConstVisitor() { m_shadowSurface = NULL; }

  private:
    //------------------------------------------------------------------------
    void do_visit(const Barrier *b) { do_do_visit(b); }
    void do_visit(const BlackHole *b) { do_do_visit(b); }
    void do_visit(const Crate *c) { do_do_visit(c); }
    void do_visit(const GrenadeBase *g) { do_do_visit(g); }
    void do_visit(const Grinder *g) { do_do_visit(g); }
    void do_visit(const MagnetBase *m) { do_do_visit(m); }
    void do_visit(const Missile *m) { do_do_visit(m); }
    void do_visit(const MortarBase *m) { do_do_visit(m); }
    void do_visit(const ParticleBase *p);
    void do_visit(const ParticleFountainBase *f) { do_do_visit(f); }
    void do_visit(const Platform *p) { do_do_visit(p); }
    void do_visit(const ProjectileBase *p) { do_do_visit(p); }
    void do_visit(const SAMBatteryBase *s) { do_do_visit(s); }
    void do_visit(const Ship *s) { do_do_visit(s); }
    void do_visit(const Tank *t) { do_do_visit(t); }
    void do_visit(const TurretBase *t) { do_do_visit(t); }

    //------------------------------------------------------------------------
    /// A common implementation is used for all type of objects.
    void do_do_visit(const ObjectBase *o);

    //------------------------------------------------------------------------
    SDL_Surface *m_shadowSurface;
};

#endif //PLAYGROUNDVISITORS_H
