#include "Tools.h"
#include "XMLParser.h"
#include "SDLFrameRate.h"
#include "Configuration.h"
#include "File.h"
#include "PlayerStatus.h"

#include "SoundInterface.h"

#include "PlayGround.h"
#include "Ship.h"
#include "Missile.h"
#include "Projectile.h"
#include "Grenade.h"
#include "SAMBattery.h"
#include "Turrets.h"
#include "Mortars.h"
#include "Tank.h"

#include "RaceGame.h"
#include "RescueGame.h"
#include "TestGame.h"

#include "ScoreTable.h"
#include "PlayerConfiguration.h"
#include "GameControlBase.h"


//----------------------------------------------------------------------------
GameControlBase::OutOfFuelDelay GameControlBase::OutOfFuelDelay::sm_instance;
GameControlBase::ExplosionDelay GameControlBase::ExplosionDelay::sm_instance;
GameControlBase::AddFuelToScore GameControlBase::AddFuelToScore::sm_instance;
GameControlBase::AddBonusToScore GameControlBase::AddBonusToScore::sm_instance;
GameControlBase::Final GameControlBase::Final::sm_instance;


//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::updateShipControl(ShipControl control,
                                                       GameControlBase *game)
{
    game->do_updateShipControl(control);
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::updateBonus(GameControlBase *game)
{
    game->do_updateBonus();
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::onShipOutOfFuel(const Ship *ship,
                                                     GameControlBase *game)
{
    if (ship == game->getPlayerShip())
    {
        SoundInterface::getInstance()->onShipOutOfFuel();

        game->initOutOfFuel();
        game->setState(OutOfFuelDelay::getInstance());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::OutOfFuelDelay::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        game->getPlayerShip()->setExplode();
    }
}


//----------------------------------------------------------------------------
void GameControlBase::ExplosionDelay::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        PlayerStatus::getInstance()->decLifes();
        if (PlayerStatus::getInstance()->getLifes() > 0
            && game->continueAfterExplosion())
        {
            game->resetPlayerShip();
            game->setState(game->getRunningState());
        }
        else
        {
            game->setState(Final::getInstance());
        }
    }
}


//----------------------------------------------------------------------------
void GameControlBase::AddFuelToScore::onPreUpdate(GameControlBase *game)
{
   if (!game->getPlayerShip()->decFuel(game->getPlayerFuelDecrement()))
   {
       if (game->hasFrameCounterReachedLimit())
       {
           game->resetFrameCounter();

           PlayerStatus::getInstance()->addToScore(
               ScoreTable::getInstance()->getSpareFuelStep());
       }
   }
   else
   {
       if (PlayerStatus::getInstance()->getBonus() > 0)
       {
           game->initAddBonusToScore();
           game->setState(AddBonusToScore::getInstance());
       }
       else
       {
           game->setState(Final::getInstance());
       }
   }
}


//----------------------------------------------------------------------------
void GameControlBase::AddBonusToScore::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        game->resetFrameCounter();

        PlayerStatus *ps = PlayerStatus::getInstance();
        if (ps->getBonus() > 0)
        {
            ps->addToScore(1);
            ps->decBonus(1);
        }
        else
        {
            game->setState(Final::getInstance());
        }
    }
}


//----------------------------------------------------------------------------
GameControlBase *GameControlBase::sm_instance = NULL;


//----------------------------------------------------------------------------
GameControlBase::GameControlBase()
{
    m_player = NULL;
    m_playerFuel = 0;
    m_playerFuelDecrement = 0;

    m_homePlatform = NULL;

    m_initialXPosition = 0;
    m_initialYPosition = 0;
    m_initialPositionIsPlatform = false;

    m_bonusFrameCounter = 0;
    m_bonusFrameCounterLimit = 0;
    m_bonusDecrement = 0;

    m_frameCounter = 0;
    m_frameCounterLimit = 0;

    m_state = NULL;
}

//----------------------------------------------------------------------------
GameControlBase::~GameControlBase()
{
    m_player = NULL;
    m_homePlatform = NULL;

    m_state = NULL;
}


//----------------------------------------------------------------------------
void GameControlBase::init(const char *mission, const char *level)
    throw (Exception)
{
    destroy();

    std::string full;
    full.append(Configuration::getInstance()->getDataDir())
        .append("/levels/").append(mission).append("/").append(level);

    XMLParser p;
    XMLNode root;
    p.parse(full.c_str(), root);

    XMLNode *levelNode = root.getMandatoryNode("level");

    const std::string &type = levelNode->getStringProperty("type");
    if (type == "race")
    {
        sm_instance = new RaceGame();
    }
    else if (type == "rescue")
    {
        sm_instance = new RescueGame();
    }
    else if (type == "test")
    {
        sm_instance = new TestGame();
    }
    else
    {
        throw XMLException(
            std::string("The level type '")
            .append(type).append("' is invalid"));
    }

    PlayGround::init(mission, levelNode->getMandatoryNode("playground"));
    sm_instance->initBonus(levelNode->getNode("bonus"));
    sm_instance->initGameControl(levelNode->getMandatoryNode("gamecontrol"));
}

//----------------------------------------------------------------------------
void GameControlBase::destroy()
{
    PlayGround::destroy();
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
void GameControlBase::initBonus(const XMLNode *bonusNode)
{
    if (bonusNode)
    {
        PlayerStatus::getInstance()->setInitialBonus(
            bonusNode->getUnsignedProperty("initial"));
        m_bonusDecrement = bonusNode->getUnsignedProperty("decrement");
        m_bonusFrameCounterLimit =
            bonusNode->getUnsignedProperty("delay", 10) *
            SDLFrameRate::getFrameRate() / 10;
    }
    else
    {
        PlayerStatus::getInstance()->setInitialBonus(0);
        m_bonusDecrement = 0;
        m_bonusFrameCounterLimit = 0;
    }
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship)
{
    do_explode(ship);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, GrenadeBase *grenade)
{
    do_explode(ship);
    do_explode(grenade);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, Missile *missile)
{
    do_explode(ship);
    do_explode(missile);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, MortarBase *mortar)
{
    do_explode(ship);
    do_explode(mortar);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getMortarShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, ProjectileBase *projectile)
{
    do_explode(ship);
    do_explode(projectile);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, SAMBatteryBase *sam)
{
    do_explode(ship);
    do_explode(sam);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getSAMBatteryShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, Tank *tank)
{
    do_explode(ship);
    do_explode(tank);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getTankShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, TurretBase *turret)
{
    do_explode(ship);
    do_explode(turret);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getTurretShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile)
{
    do_explode(missile);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, GrenadeBase *grenade)
{
    do_explode(missile);
    do_explode(grenade);

    if (grenade->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMissileShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, MortarBase *mortar)
{
    do_explode(missile);
    do_explode(mortar);

    if (missile->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMortarShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, ProjectileBase *projectile)
{
    do_explode(missile);
    do_explode(projectile);

    if (projectile->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMissileShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, SAMBatteryBase *sam)
{
    do_explode(missile);
    do_explode(sam);

    if (missile->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getSAMBatteryShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, Tank *tank)
{
    do_explode(missile);
    do_explode(tank);

    if (missile->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTankShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, TurretBase *turret)
{
    do_explode(missile);
    do_explode(turret);

    if (missile->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTurretShot());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(GrenadeBase *grenade)
{
    do_explode(grenade);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(GrenadeBase *grenade, MortarBase *mortar)
{
    do_explode(grenade);
    do_explode(mortar);

    if (grenade->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMortarShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(GrenadeBase *grenade, Tank *tank)
{
    do_explode(grenade);
    do_explode(tank);

    if (grenade->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTankShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(GrenadeBase *grenade, TurretBase *turret)
{
    do_explode(grenade);
    do_explode(turret);

    if (grenade->getCreator() == getPlayerShip())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTurretShot());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile)
{
    do_explode(projectile);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile,
                                  MortarBase *mortar)
{
    do_explode(projectile);
    if (mortar->decAndTestHitPoints())
    {
        do_explode(mortar);

        if (projectile->getCreator() == getPlayerShip())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getMortarShot());
        }
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile, Tank *tank)
{
    do_explode(projectile);
    if (tank->decAndTestHitPoints())
    {
        do_explode(tank);

        if (projectile->getCreator() == getPlayerShip())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getTankShot());
        }
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile,
                                  TurretBase *turret)
{
    do_explode(projectile);
    if (turret->decAndTestHitPoints())
    {
        do_explode(turret);

        if (projectile->getCreator() == getPlayerShip())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getTurretShot());
        }
    }
}


//----------------------------------------------------------------------------
void GameControlBase::do_explode(GrenadeBase *grenade)
{
    grenade->createExplosionParticles();
    SoundInterface::getInstance()->onGrenadeExplosion();

    PlayGround::getInstance()->markObjectToRemove(grenade);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Missile *missile)
{
    missile->createExplosionParticles();
    SoundInterface::getInstance()->onMissileExplosion();

    PlayGround::getInstance()->markObjectToRemove(missile);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(MortarBase *mortar)
{
    mortar->createExplosionParticles();
    SoundInterface::getInstance()->onMortarExplosion();

    PlayGround::getInstance()->markObjectToRemove(mortar);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(ProjectileBase *projectile)
{
    projectile->createExplosionParticles();
    SoundInterface::getInstance()->onProjectileExplosion();

    PlayGround::getInstance()->markObjectToRemove(projectile);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(SAMBatteryBase *sam)
{
    sam->createExplosionParticles();
    SoundInterface::getInstance()->onSAMBatteryExplosion();

    PlayGround::getInstance()->markObjectToRemove(sam);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Ship *ship)
{
    ship->removeAllCrates();
    ship->createExplosionParticles();
    SoundInterface::getInstance()->onShipThrustOff();
    SoundInterface::getInstance()->onShipExplosion();

    PlayGround::getInstance()->markObjectToRemove(ship);

    if (ship == m_player)
    {
        m_player = NULL;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Tank *tank)
{
    tank->createExplosionParticles();
    SoundInterface::getInstance()->onTankExplosion();

    PlayGround::getInstance()->markObjectToRemove(tank);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(TurretBase *turret)
{
    turret->createExplosionParticles();
    SoundInterface::getInstance()->onTurretExplosion();

    PlayGround::getInstance()->markObjectToRemove(turret);
}


//----------------------------------------------------------------------------
Ship *GameControlBase::createPlayerShip() const
{
    return new Ship(
        (ShipSurfaces::Ship)
        PlayerConfiguration::getInstance()->getPlayer()->getShipType());
}

//----------------------------------------------------------------------------
void GameControlBase::resetPlayerShip()
{
    if (!m_player)
    {
        m_player = createPlayerShip();
    }

    m_player->resetPosition(
        m_initialXPosition, m_initialYPosition, m_initialPositionIsPlatform);
    m_player->setFuel(getPlayerFuel());

    PlayGround::getInstance()->addObject(m_player);
}

//----------------------------------------------------------------------------
void GameControlBase::initOutOfFuel()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate());
}

//----------------------------------------------------------------------------
void GameControlBase::initExplosion()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate());
}

//----------------------------------------------------------------------------
void GameControlBase::initAddFuelToScore()
{
    // A fuel of 100 seconds means, that in every frame
    // 50 fuel units are decremented from the ship
    // and the score is incremented once.
    //
    // A fuel of 50 seconds thus has to decrement 25 fuel units in every frame
    // and the score shall only be incremented every second frame.
    //
    // This way, the speed of the fuel-to-score adding
    // is not depending on the value of the ship's initial fuel.

    unsigned playerFuelReference = 100 * SDLFrameRate::getFrameRate();
    unsigned ratio = playerFuelReference / getPlayerFuel();
    if (ratio == 0)
    {
        ratio = 1;
    }

    // @todo: Was 50 @96fps and should be 37.5 @72fps.
    //        But since the friction effect is lower, we use 40 here
    //        to keep the fuel bonus nearly constant.
    //        A clean solution depends on a framerate-independend
    //        implementation of the friction model.
    m_playerFuelDecrement = 40 / ratio;
    resetFrameCounter();
    setFrameCounterLimit(ratio);

    SoundInterface::getInstance()->onAddFuelToScore();
}

//----------------------------------------------------------------------------
void GameControlBase::initAddBonusToScore()
{
    resetFrameCounter();
    setFrameCounterLimit(2);

    SoundInterface::getInstance()->onAddBonusToScore();
}

//----------------------------------------------------------------------------
void GameControlBase::update()
    throw (SDLException)
{
    updateBonus();

    PlayGround::getInstance()->clearUpdateRects();

    onPreUpdate();
    PlayGround::getInstance()->update();
    onPostUpdate();
}


//----------------------------------------------------------------------------
void GameControlBase::do_updateShipControl(ShipControl control)
{
    Ship *ship = getPlayerShip();

    switch (control)
    {
    case ROT_LEFT:
        ship->setRotation(Ship::ROT_LEFT);
        break;
    case ROT_RIGHT:
        ship->setRotation(Ship::ROT_RIGHT);
        break;
    case ROT_OFF:
        ship->setRotation(Ship::ROT_NONE);
        break;
    case THRUST_ON:
        ship->setThrust(true);
        break;
    case THRUST_OFF:
        ship->setThrust(false);
        break;
    case ALIGN_ON:
        ship->setAlign(true);
        break;
    case ALIGN_OFF:
        ship->setAlign(false);
        break;
    case FIRE:
        ship->setFire();
        break;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::do_updateBonus()
{
    if (m_bonusFrameCounter++ == m_bonusFrameCounterLimit)
    {
        m_bonusFrameCounter = 0;
        PlayerStatus::getInstance()->decBonus(m_bonusDecrement);
    }
}
