#include <sstream>

#include "SDLFrameRate.h"

#include "SoundInterface.h"

#include "PlayerStatus.h"
#include "PlayGround.h"
#include "Ship.h"
#include "Platform.h"
#include "Projectile.h"
#include "Crates.h"
#include "ScoreTable.h"
#include "RescueGame.h"


//----------------------------------------------------------------------------
RescueGame::Running RescueGame::Running::sm_instance;
RescueGame::UnloadingCrate RescueGame::UnloadingCrate::sm_instance;
RescueGame::UnloadingDelay RescueGame::UnloadingDelay::sm_instance;


//----------------------------------------------------------------------------
void RescueGame::Running::onShipLanded(
    Ship *ship, const Platform *platform, GameControlBase *game)
{
    RescueGame *rescueGame = static_cast<RescueGame*>(game);

    // Keep sure, that the ship's thrusters are turned off,
    // before changing to a game state with disabled ship steering.
    ship->setThrust(false);

    if (platform == rescueGame->m_homePlatform)
    {
        rescueGame->unloadCrate();
    }
    else
    {
        Crate *crate = PlayGround::getInstance()->hasCrateFor(ship);
        if (crate)
        {
            // Pick up the crate.

            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getCratePickUp(crate));
            SoundInterface::getInstance()->onCratePickUp();

            switch (crate->getType())
            {
            case CrateSurfaces::T_SMALL:
            case CrateSurfaces::T_MEDIUM:
            case CrateSurfaces::T_BIG:
                ship->pushCrate(crate);
                break;

            case CrateSurfaces::T_BONUS:
                break;

            case CrateSurfaces::T_FUEL:
                ship->setFuel(game->getPlayerFuel());
                break;

            default:
                break;
            }

            PlayGround::getInstance()->markObjectToRemove(crate, false);
        }
    }
}


//----------------------------------------------------------------------------
void RescueGame::UnloadingCrate::onPreUpdate(GameControlBase *game)
{
    RescueGame *rescueGame = static_cast<RescueGame*>(game);

    if (rescueGame->hasFrameCounterReachedLimit())
    {
        PlayGround::getInstance()->markObjectToRemove(
            rescueGame->m_crateToUnload);
        rescueGame->m_crateToUnload = NULL;

        if (game->getPlayerShip()->hasCrates())
        {
            rescueGame->initUnloadingDelay();
            game->setState(UnloadingDelay::getInstance());
        }
        else
        {
            // @todo Merge this code with unloadCrate()?
            if (rescueGame->countCratesToRescue() > 0)
            {
                game->setState(Running::getInstance());
            }
            else
            {
                game->initAddFuelToScore();
                game->setState(AddFuelToScore::getInstance());
            }
        }
    }
}


//----------------------------------------------------------------------------
void RescueGame::UnloadingDelay::onPreUpdate(GameControlBase *game)
{
    RescueGame *rescueGame = static_cast<RescueGame*>(game);

    if (rescueGame->hasFrameCounterReachedLimit())
    {
        rescueGame->unloadCrate();
    }
}


//----------------------------------------------------------------------------
RescueGame::RescueGame()
{
    m_crateToUnload = NULL;

    setState(Running::getInstance());
}

//----------------------------------------------------------------------------
RescueGame::~RescueGame()
{
    m_crateToUnload = NULL;
}


//----------------------------------------------------------------------------
void RescueGame::initGameControl(const XMLNode *gameControlNode)
    throw (Exception)
{
    PlayGround *playGround = PlayGround::getInstance();

    const XMLNode *child = gameControlNode->getMandatoryNode("startposition");

    m_playerFuel = SDLFrameRate::getFrameRate() *
        child->getUnsignedProperty("fuel", 100);

    m_player = createPlayerShip();

    unsigned platform = child->getUnsignedProperty("platform");

    m_homePlatform = playGround->getPlatform(platform);
    if (m_homePlatform == NULL)
    {
        std::ostringstream s;
        s << "The platform with the number " << platform
          << " doesn't exist" << std::ends;
        throw XMLException(s.str());
    }

    const SDL_Rect &landingZone = m_homePlatform->getLandingZone();

    m_initialXPosition = landingZone.x + landingZone.w/2
        - m_player->getPosition().w/2;
    m_initialYPosition = landingZone.y - m_player->getBottomPixels().y;
    m_initialPositionIsPlatform = true;

    resetPlayerShip();
}


//----------------------------------------------------------------------------
void RescueGame::initUnloadingDelay()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate() / 2);
}

//----------------------------------------------------------------------------
unsigned RescueGame::countCratesToRescue() const
{
    return PlayGround::getInstance()->countCratesToRescue();
}


//----------------------------------------------------------------------------
void RescueGame::unloadCrate()
{
    PlayGround *playGround = PlayGround::getInstance();

    if (m_player->hasCrates())
    {
        assert(m_crateToUnload == NULL);
        m_crateToUnload = m_player->popCrate();

        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getCrateRescued(m_crateToUnload));
        SoundInterface::getInstance()->onCrateUnload();

        // Move the crate to a position beside the ship
        // and add it back to the playground.

        const SDL_Rect &landingZone = m_homePlatform->getLandingZone();
        Sint16 x = m_player->getPosition().x + m_player->getBottomPixels().x
            + m_player->getBottomPixels().w/2;
        if (x - landingZone.x < landingZone.x + landingZone.w - x)
        {
            // Draw the crate at the ship's right side.
            x += m_player->getBottomPixels().w/2;
            x += m_crateToUnload->getPosition().w/4;
        }
        else
        {
            // Draw the crate at the ship's left side.
            x -= m_player->getBottomPixels().w/2;
            x -= m_crateToUnload->getPosition().w/4;
        }
        x -= m_crateToUnload->getPosition().w / 2;

        Sint16 y = m_player->getPosition().y + m_player->getBottomPixels().y
            - m_crateToUnload->getPosition().h + 1;
        // "+ 1", because m_player->getBottomPixels().y
        // lies one pixel above landingZone.y

        m_crateToUnload->setPosition(x, y);

        playGround->addObject(m_crateToUnload);

        initUnloadingDelay();
        setState(UnloadingCrate::getInstance());
    }
    else if (countCratesToRescue() > 0)
    {
        setState(Running::getInstance());
    }
    else
    {
        initAddFuelToScore();
        setState(AddFuelToScore::getInstance());
    }
}
