#ifndef CONTROLLABLEOBJECTBASE_H
#define CONTROLLABLEOBJECTBASE_H

#include <cassert>

#include "SDLFrameRate.h"

#include "StaticDecorationBase.h"  // @todo: needed for SURFACE_ROTATION_ANGLE
#include "MovingObjectBase.h"


//----------------------------------------------------------------------------
/**
 * Base class for objects, that are controllable
 * in terms of rotating and thrusting.
 */
class ControllableObjectBase : public MovingObjectBase
{
  public:
    enum Rotation
    {
        ROT_NONE,
        ROT_LEFT,
        ROT_RIGHT,
        ROT_ALIGN
    };

    virtual ~ControllableObjectBase();

    inline void setRotation(Rotation rot)
    {
        m_rotation = rot;
    }

    inline Rotation getRotation() const
    {
        return m_rotation;
    }

    inline void setAngle(Sint16 angle)
    {
        m_angle = angle / SURFACES_ROTATION_ANGLE;
        m_fineAngle = 1.0 * angle;
    }

    inline Uint16 getAngle() const
    {
        return m_angle * SURFACES_ROTATION_ANGLE;
    }

    inline void setAlign(bool align)
    {
        setRotation(align ? ROT_ALIGN : ROT_NONE);
    }

    inline bool getAlign() const
    {
        return getRotation() == ROT_ALIGN;
    }

    inline void setFuel(unsigned fuel)
    {
        m_fuel = fuel;
    }

    inline unsigned getFuel() const
    {
        return m_fuel;
    }

    inline bool hasFuel() const
    {
        return m_fuel > 0;
    }

    void setThrust(bool enable);

    inline bool isThrust() const
    {
        return m_thrust;
    }

    inline bool isExplode() const
    {
        return m_explode;
    }

    inline void setExplode(bool explode = true)
    {
        m_explode = explode;
    }

    inline const SDL_Rect &getBottomPixels() const
    {
        return m_bottomPixels;
    }

    inline unsigned getMass() const
    {
        return m_mass;
    }

    /**
     * Creates some explosion particles on the object's position.
     */
    virtual void createExplosionParticles();


    //------------------------------------------------------------------------
    /**
     * The number of degrees, this object can rotate in one second.
     */
    virtual unsigned getRotationDegreesPerSecond() const = 0;

    /**
     * The force of the object's thruster.
     */
    virtual unsigned getThrustForce() const = 0;


  protected:

    ControllableObjectBase(const ObjectBase *creator);

    /// Called by the constructor to initialize m_bottomPixels.
    void initBottomPixels();

    /**
     * Updates m_angle and m_surface out of m_rotation.
     */
    void updateRotation();

    /**
     * Updates the velocity and position of this object.
     */
    void updateVelocityAndPosition();


  private:

    /**
     * @return The object's surface for the given rotation frame.
     */
    virtual const SDL_Surface *getRotationSurface(unsigned frame) const = 0;


  protected:

    /// The total mass of the object (must not be 0).
    unsigned m_mass;

    /// The current angle of rotation in steps of SHIP_ROTATION_ANGLE degrees.
    Uint16 m_angle;

    /// The ship's fine scale angle.
    double m_fineAngle;

    /// The current rotation of this object.
    Rotation m_rotation;

    /// The current fuel of this object.
    unsigned m_fuel;

    /// True, if this object's thruster is activated.
    bool m_thrust;

    bool m_explode;

    /**
     * The bottom pixel row used by ThrustParticle
     * and Ship::isInLandingZone().
     */
    SDL_Rect m_bottomPixels;
};

#endif //CONTROLLABLEOBJECTBASE_H
