#ifndef MAGNETS_H
#define MAGNETS_H

#include <map>

#include "Tools.h"
#include "StaticDecorationBase.h"

//----------------------------------------------------------------------------
class MagnetSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(MagnetSurfaces);

    //------------------------------------------------------------------------
    class SurfaceKey
    {
      public:
        SurfaceKey(OrientatingDecorationBase::Orientation o,
                   unsigned frame, unsigned size)
            : m_o(o), m_frame(frame), m_size(size) {}
        ~SurfaceKey() {}

        friend bool operator<(const SurfaceKey &k1, const SurfaceKey &k2)
        {
            COMPARE_SURFACE_KEY_MEMBER(m_o);
            COMPARE_SURFACE_KEY_MEMBER(m_frame);
            COMPARE_SURFACE_KEY_MEMBER(m_size);
            return false;
        }

      private:
        OrientatingDecorationBase::Orientation m_o;
        unsigned m_frame;
        unsigned m_size;
    };

    typedef std::map<SurfaceKey, SDL_Surface*> SurfaceMap;
    typedef SurfaceMap::iterator SurfaceIter;
    typedef SurfaceMap::const_iterator SurfaceCIter;

  public:

    //------------------------------------------------------------------------
    const SDL_Surface *getSurface(OrientatingDecorationBase::Orientation o,
                                  unsigned frame, unsigned size);

  private:

    //------------------------------------------------------------------------
    SDL_Surface *createSurface(OrientatingDecorationBase::Orientation o,
                               unsigned frame, unsigned size) const;


    //------------------------------------------------------------------------
    /// All magnet frames for the left bound.
    SDL_Surface *m_left[4][4];

    /// All magnet frames for the inner parts.
    SDL_Surface *m_center[4][4];

    /// All magnet frames for the right bound.
    SDL_Surface *m_right[4][4];


    /// A map for all dynamically created magnet surfaces via getSurface().
    SurfaceMap m_surfaceMap;
};



//----------------------------------------------------------------------------
#undef DECLARE_UPDATE_STATE
#define DECLARE_UPDATE_STATE(c) \
    class c; \
    friend class c; \
    class c : public UpdateState \
    { \
        STATE_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void update(MagnetBase *magnet);  \
    }


//----------------------------------------------------------------------------
class MagnetBase : public StaticDecorationBase,
                   public AnimatedDecorationBase,
                   public OrientatingDecorationBase
{
    //------------------------------------------------------------------------
    class UpdateState
    {
      protected:
        UpdateState() {}
      public:
        virtual ~UpdateState() {}

        virtual void update(MagnetBase *magnet) = 0;
    };

    //------------------------------------------------------------------------
    DECLARE_UPDATE_STATE(Initial);
    DECLARE_UPDATE_STATE(InitialDelay);
    DECLARE_UPDATE_STATE(PermanentActivation);
    DECLARE_UPDATE_STATE(Activated);
    DECLARE_UPDATE_STATE(Deactivated);


  public:

    //------------------------------------------------------------------------
    class InitializationData
        : public StaticDecorationBase::InitializationData,
          public AnimatedDecorationBase::InitializationData,
          public OrientatingDecorationBase::InitializationData
    {
      public:
        InitializationData(const XMLNode *magnetNode)
            throw (XMLException);
        
        ~InitializationData();

        unsigned wh;

        unsigned strength;
        unsigned distance;
    };


    //------------------------------------------------------------------------
    virtual ~MagnetBase();

    static MagnetBase *create(const XMLNode *magnetNode)
        throw (Exception);


    //------------------------------------------------------------------------
    inline unsigned getSize() const
    {
        return m_size;
    }

    inline unsigned getStrength() const
    {
        return m_strength;
    }

    inline unsigned getDistance() const
    {
        return m_distance;
    }

    inline void setUpdateState(UpdateState *state)
    {
        m_updateState = state;
    }

    //------------------------------------------------------------------------
    void update();

  protected:
    //------------------------------------------------------------------------
    MagnetBase(const InitializationData &init);

    static MagnetBase *create(const InitializationData &init);

    //------------------------------------------------------------------------
    virtual void addGravityToPlayGround() const = 0;
    virtual void subGravityFromPlayGround() const = 0;

    //------------------------------------------------------------------------
    unsigned m_frameDelay;
    unsigned m_currentFrame;
    unsigned m_frameCounter;

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    unsigned m_size;
    unsigned m_strength;
    unsigned m_distance;

    UpdateState *m_updateState;
};

//----------------------------------------------------------------------------
class TopMagnet : public MagnetBase
{
  public:
    TopMagnet(const InitializationData &init);
    ~TopMagnet();

  protected:
    void addGravityToPlayGround() const;
    void subGravityFromPlayGround() const;
};

//----------------------------------------------------------------------------
class BottomMagnet : public MagnetBase
{
  public:
    BottomMagnet(const InitializationData &init);
    ~BottomMagnet();

  protected:
    void addGravityToPlayGround() const;
    void subGravityFromPlayGround() const;
};

//----------------------------------------------------------------------------
class LeftMagnet : public MagnetBase
{
  public:
    LeftMagnet(const InitializationData &init);
    ~LeftMagnet();

  protected:
    void addGravityToPlayGround() const;
    void subGravityFromPlayGround() const;
};

//----------------------------------------------------------------------------
class RightMagnet : public MagnetBase
{
  public:
    RightMagnet(const InitializationData &init);
    ~RightMagnet();

  protected:
    void addGravityToPlayGround() const;
    void subGravityFromPlayGround() const;
};

#endif //MAGNETS_H
