#ifndef MISSILE_H
#define MISSILE_H

#include "Tools.h"
#include "SDLCalls.h"
#include "ControllableObjectBase.h"


//----------------------------------------------------------------------------
class MissileSurfaces : public RotatingSurfacesBase
{
    SURFACES_SINGLETON_OBJECT(MissileSurfaces);
};


//----------------------------------------------------------------------------
#undef DECLARE_HEADING_STRATEGY
#define DECLARE_HEADING_STRATEGY(c) \
    class c; \
    friend class c; \
    class c : public HeadingStrategy \
    { \
        STRATEGY_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        Sint16 calculateHeadingAngle( \
           Sint16 a, Sint16 angle, Sint16 gy, Sint16 gy) const; \
    }

//----------------------------------------------------------------------------
#undef DECLARE_WARHEAD_STRATEGY
#define DECLARE_WARHEAD_STRATEGY(c) \
    class c; \
    friend class c; \
    class c : public WarheadStrategy \
    { \
        STRATEGY_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void updateWarhead(Missile *missile); \
        void onCreateExplosionParticles(Missile *missile); \
    }


//----------------------------------------------------------------------------
class Missile : public ControllableObjectBase
{
    friend class MissileTest;

  public:
    //------------------------------------------------------------------------
    enum EnumHeadingStrategy
    {
        HS_DIRECT,
        HS_SMART
    };

    //------------------------------------------------------------------------
    enum EnumWarheadStrategy
    {
        WH_NORMAL,
        WH_CONEBURST,
        WH_STARBURST
    };


  private:

    //------------------------------------------------------------------------
    /**
     * A base class for all heading strategies.
     */
    class HeadingStrategy
    {
      public:
        virtual ~HeadingStrategy() {}

        /**
         * @param a The acceleration of the missile.
         * @param angle The angle to the player's ship in [0..360) degrees.
         * @param gx The x gravity acceleration.
         * @param gy The y gravity acceleration.
         */
        virtual Sint16 calculateHeadingAngle(
            Sint16 a, Sint16 angle, Sint16 gy, Sint16 gy) const = 0;

        static HeadingStrategy *getStrategy(EnumHeadingStrategy s);

      protected:
        HeadingStrategy() {}
    };

    DECLARE_HEADING_STRATEGY(DirectHeadingStrategy);
    DECLARE_HEADING_STRATEGY(SmartHeadingStrategy);


    //------------------------------------------------------------------------
    /**
     * A base class for all warhead strategies.
     */
    class WarheadStrategy
    {
      public:
        virtual ~WarheadStrategy() {}

        virtual void updateWarhead(Missile *missile) = 0;
        virtual void onCreateExplosionParticles(Missile *missile) = 0;

        static WarheadStrategy *getStrategy(EnumWarheadStrategy s);

      protected:
        WarheadStrategy() {}

        void do_updateWarhead(Missile *missile);
    };

    DECLARE_WARHEAD_STRATEGY(NormalWarheadStrategy);
    DECLARE_WARHEAD_STRATEGY(ConeBurstWarheadStrategy);
    DECLARE_WARHEAD_STRATEGY(StarBurstWarheadStrategy);


  public:

    //------------------------------------------------------------------------
    Missile(EnumHeadingStrategy heading,
            EnumWarheadStrategy warhead,
            const ObjectBase *creator);
    virtual ~Missile();

    void createExplosionParticles();

    //------------------------------------------------------------------------
    inline unsigned getRotationDegreesPerSecond() const { return 432; }
    inline unsigned getThrustForce() const { return 6000; }

    //------------------------------------------------------------------------
    void update();

  protected:

    //------------------------------------------------------------------------
    /// Called by do_update() to update m_rotation via m_headingStrategy.
    void updateHeading();

    /// Called by do_update() to make some updates via m_warheadStrategy.
    void updateWarhead();

  private:
    //------------------------------------------------------------------------
    const SDL_Surface *getRotationSurface(unsigned frame) const;

    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    HeadingStrategy *m_headingStrategy;
    WarheadStrategy *m_warheadStrategy;
};

#endif //MISSILE_H
