#ifndef OBJECTBASE_H
#define OBJECTBASE_H

#include <SDLTools.h>

class ObjectVisitor;
class ObjectConstVisitor;

//----------------------------------------------------------------------------
/**
 * Declares some common stuff for all surface factories.
 */
#define SURFACES_SINGLETON_OBJECT(O) \
    REFERENCE_OBJECT(O); \
  public: \
    ~O(); \
    static void init() throw (Exception); \
    static void destroy(); \
    static inline O *getInstance() { return sm_instance; } \
  protected: \
    O(); \
  private: \
    static O *sm_instance


//----------------------------------------------------------------------------
#define DECLARE_OBJECT_VISITOR_API() \
    void do_accept(ObjectVisitor *v); \
    void do_accept(ObjectConstVisitor *v) const

//----------------------------------------------------------------------------
#define DECLARE_OBJECT_VISITOR_API_BODY(C) \
void C::do_accept(ObjectVisitor *v) { v->visit(this); } \
void C::do_accept(ObjectConstVisitor *v) const { v->visit(this); }


//----------------------------------------------------------------------------
/**
 * Base class for all surface factories.
 */
class SurfacesBase
{
  public:
    //------------------------------------------------------------------------
    virtual ~SurfacesBase() {}

    //------------------------------------------------------------------------
    static void init()
        throw (Exception);

    static void destroy();
    
  protected:
    //------------------------------------------------------------------------
    SurfacesBase() {}
};


//----------------------------------------------------------------------------
/**
 * This is a base class for all objects,
 * such as decorations and ships.
 */
class ObjectBase
{
  public:
    //------------------------------------------------------------------------
    virtual ~ObjectBase();

    //------------------------------------------------------------------------
    inline const SDL_Surface *getSurface() const
    {
        return m_surface;
    }

    inline void setSurface(const SDL_Surface *surface)
    {
        m_surface = surface;
        m_position.w = surface->w;
        m_position.h = surface->h;
    }

    //------------------------------------------------------------------------
    inline const SDL_Rect &getPosition() const
    {
        return m_position;
    }

    inline const SDL_Rect *getPositionPointer() const
    {
        return &m_position;
    }

    //------------------------------------------------------------------------
    inline void setPosition(const SDL_Rect &position)
    {
        m_position = position;
    }

    inline void setPosition(Sint16 x, Sint16 y)
    {
        m_position.x = x;
        m_position.y = y;
    }

    inline void setWidth(Uint16 w)
    {
        m_position.w = w;
    }

    inline void setHeight(Uint16 h)
    {
        m_position.h = h;
    }

    //------------------------------------------------------------------------
    inline void setTilePosition(Sint16 x, Sint16 y)
    {
        setPosition(x*16, y*16);
    }

    inline void setTileWidth(Uint16 w)
    {
        setWidth(w*16);
    }

    inline void setTileHeight(Uint16 h)
    {
        setHeight(h*16);
    }


    //------------------------------------------------------------------------
    /**
     * Called by the PlayGround to update this object.
     */
    virtual void update() = 0;

    //------------------------------------------------------------------------
    /**
     * Called by the PlayGround to blit this object to the given surface.
     *
     * @param s The target surface.
     *
     * @throw SDLException if the blitting failed.
     */
    inline void blitTo(SDL_Surface *s) const
        throw (SDLException)
    {
        SDL_CALLS::BlitSurface(getSurface(), 0, s, getPositionPointer());
    }


    //------------------------------------------------------------------------
    inline const ObjectBase *getCreator() const
    {
        return m_creator;
    }


    //------------------------------------------------------------------------
    inline void accept(ObjectVisitor *v)
    {
        do_accept(v);
    }

    inline void accept(ObjectConstVisitor *v) const
    {
        do_accept(v);
    }


    //------------------------------------------------------------------------
    /**
     * @return true, if there's a collision between o1 and o2.
     */
    static inline bool isCollision(const ObjectBase *o1,
                                   const ObjectBase *o2)
    {
        return SDL_TOOLS::isCollision(
            o1->getSurface(), o1->getPosition(),
            o2->getSurface(), o2->getPosition());
    }

  protected:
    //------------------------------------------------------------------------
    ObjectBase(const ObjectBase *creator);


  private:

    //------------------------------------------------------------------------
    virtual void do_accept(ObjectVisitor *v) = 0;
    virtual void do_accept(ObjectConstVisitor *v) const = 0;


    /// The current surface of the object.
    const SDL_Surface *m_surface;

    /**
     * The current x/y position of the object.
     * The members w and h are set to the surface's width and height.
     */
    SDL_Rect m_position;

    /**
     * The creator of this object.
     */
    const ObjectBase *m_creator;
};

#endif //OBJECTBASE_H
