#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>

#include "Tools.h"
#include "MathTools.h"
#include "SystemCalls.h"
#include "File.h"
#include "Directory.h"
#include "XMLElements.h"
#include "XMLParser.h"
#include "SDLTools.h"

#include <iostream>

//----------------------------------------------------------------------------
class MathToolsTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(MathToolsTest);

    CPPUNIT_TEST(testGetAngle);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testGetAngle()
    {
        CPPUNIT_ASSERT_EQUAL(  0, MATH_TOOLS::getAngle( 0,  5));
        CPPUNIT_ASSERT_EQUAL( 45, MATH_TOOLS::getAngle( 5,  5));
        CPPUNIT_ASSERT_EQUAL( 90, MATH_TOOLS::getAngle( 5,  0));
        CPPUNIT_ASSERT_EQUAL(135, MATH_TOOLS::getAngle( 5, -5));
        CPPUNIT_ASSERT_EQUAL(180, MATH_TOOLS::getAngle( 0, -5));
        CPPUNIT_ASSERT_EQUAL(225, MATH_TOOLS::getAngle(-5, -5));
        CPPUNIT_ASSERT_EQUAL(270, MATH_TOOLS::getAngle(-5,  0));
        CPPUNIT_ASSERT_EQUAL(315, MATH_TOOLS::getAngle(-5,  5));
    }
};



//----------------------------------------------------------------------------
class FileTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(FileTest);

    CPPUNIT_TEST(testFileReadWrite);
    CPPUNIT_TEST(testFileCopy);
    CPPUNIT_TEST(testFileStat);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testFileReadWrite()
    {
        const char *name = "test.dat";
        const char data[] = "1234567890";
        char buffer[sizeof(data)];

        do
        {
            File f(name, "w");
            CPPUNIT_ASSERT_EQUAL(sizeof(data), f.write(data, sizeof(data)));
        }
        while (0);

        do
        {
            File f(name, "r");
            CPPUNIT_ASSERT_EQUAL((size_t)4, f.read(buffer, 4));
            CPPUNIT_ASSERT(!f.isEOF());
            CPPUNIT_ASSERT_EQUAL((size_t)4, f.read(buffer+4, 4));
            CPPUNIT_ASSERT(!f.isEOF());
            CPPUNIT_ASSERT_EQUAL((size_t)3, f.read(buffer+8, 4));
            CPPUNIT_ASSERT(f.isEOF());
            CPPUNIT_ASSERT_EQUAL(0, memcmp(buffer, data, sizeof(buffer)));

            CPPUNIT_ASSERT_EQUAL((size_t)0, f.read(buffer, sizeof(buffer)));
            CPPUNIT_ASSERT(f.isEOF());
            CPPUNIT_ASSERT_EQUAL(0, memcmp(buffer, data, sizeof(buffer)));
        }
        while (0);

        File::unlink(name);
    }

    //------------------------------------------------------------------------
    void testFileCopy()
    {
        const char *src = "test1.dat";
        const char *dst = "test2.dat";
        const char data[] = "1234567890";
        char buffer[sizeof(data)];

        do
        {
            File f1(src, "w");
            f1.write(data, sizeof(data));
        }
        while (0);

        File::copy(src, dst);

        File f2(dst, "r");
        f2.read(buffer, sizeof(buffer));
        CPPUNIT_ASSERT_EQUAL(0, memcmp(buffer, data, sizeof(buffer)));

        File::unlink(src);
        File::unlink(dst);
    }

    //------------------------------------------------------------------------
    void testFileStat()
    {
        const char *name1 = "test1.dat";
        const char *name2 = "test2.dat";

        File f(name1, "w");

        struct stat statEntry;
        SYSTEM_CALL::stat(name1, &statEntry);
        CPPUNIT_ASSERT(S_ISREG(statEntry.st_mode));

        try
        {
            SYSTEM_CALL::stat(name2, &statEntry);
            CPPUNIT_ASSERT(false);
        }
        catch (SyscallException &e) 
        {
        }

        File::unlink(name1);
    }
};



//----------------------------------------------------------------------------
class DirectoryTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(DirectoryTest);

    CPPUNIT_TEST(testDirectoryMkdirRmdir);
    CPPUNIT_TEST(testDirectoryMkdirRmdirRecursive);
    CPPUNIT_TEST(testDirectoryCopy);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testDirectoryMkdirRmdir()
    {
        const char *name = "test";

        Directory::mkdir(name);
        CPPUNIT_ASSERT(Directory::hasDirectoryInCwd(name));
        Directory::mkdir(name);
        CPPUNIT_ASSERT(Directory::hasDirectoryInCwd(name));

        Directory::rmdir(name);
        CPPUNIT_ASSERT(!Directory::hasDirectoryInCwd(name));
        Directory::rmdir(name);
        CPPUNIT_ASSERT(!Directory::hasDirectoryInCwd(name));
    }

    //------------------------------------------------------------------------
    void testDirectoryMkdirRmdirRecursive()
    {
        setUpTestDirectory();

        Directory::mkdir("test/test2/test3", 0755, true);
        CPPUNIT_ASSERT(Directory::hasDirectoryInCwd("test"));

        tearDownTestDirectory();

        Directory::rmdir("test", true);
        CPPUNIT_ASSERT(!Directory::hasDirectoryInCwd("test"));
    }

    //------------------------------------------------------------------------
    void testDirectoryCopy()
    {
        setUpTestDirectory();
        Directory::copy("test", "test_copy");

        do
        {
            Directory d1("test_copy");
            CPPUNIT_ASSERT(d1.hasFile("file1"));
            Directory d3("test_copy/test2/test3");
            CPPUNIT_ASSERT(d3.hasFile("file3"));
        }
        while (0);

        Directory::rmdir("test_copy", true);
        tearDownTestDirectory();
    }

private:

    //------------------------------------------------------------------------
    void setUpTestDirectory()
    {
        Directory::mkdir("test/test2/test3", 0755, true);
        File f1("test/file1", "w");
        File f3("test/test2/test3/file3", "w");
    }

    //------------------------------------------------------------------------
    void tearDownTestDirectory()
    {
        Directory::rmdir("test", true);
        CPPUNIT_ASSERT(!Directory::hasDirectoryInCwd("test"));
    }
};



//----------------------------------------------------------------------------
class XMLTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(XMLTest);

    CPPUNIT_TEST(testXMLPropertyConstructors);
    CPPUNIT_TEST(testXMLParser);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testXMLPropertyConstructors()
    {
        // Use the constructors with matching value types.

        XMLProperty p01("prop", true);
        CPPUNIT_ASSERT_EQUAL(std::string("1"), p01.getValue());

        XMLProperty p02("prop", (int)-42);
        CPPUNIT_ASSERT_EQUAL(std::string("-42"), p02.getValue());

        XMLProperty p03("prop", (unsigned)42);
        CPPUNIT_ASSERT_EQUAL(std::string("42"), p03.getValue());

        XMLProperty p04("prop", "foo");
        CPPUNIT_ASSERT_EQUAL(std::string("foo"), p04.getValue());

        XMLProperty p05("prop", std::string("bar"));
        CPPUNIT_ASSERT_EQUAL(std::string("bar"), p05.getValue());

        // Test, that the implicit type conversions are working correctly.

        XMLProperty p11("prop", (Sint8)-42);
        CPPUNIT_ASSERT_EQUAL(std::string("-42"), p11.getValue());

        XMLProperty p12("prop", (Uint8)42);
        CPPUNIT_ASSERT_EQUAL(std::string("42"), p12.getValue());

        XMLProperty p13("prop", (Sint16)-42);
        CPPUNIT_ASSERT_EQUAL(std::string("-42"), p13.getValue());

        XMLProperty p14("prop", (Uint16)42);
        CPPUNIT_ASSERT_EQUAL(std::string("42"), p14.getValue());
    }

    //------------------------------------------------------------------------
    void testXMLParser()
    {
        XMLNode root;
        XMLParser p;
        p.parse("test.xml", root);

        const XMLNode *node = root.getMandatoryNode("node");
        CPPUNIT_ASSERT(!node->hasText());

        const XMLNode *subnode1 = node->getMandatoryNode("subnode1");
        CPPUNIT_ASSERT_EQUAL(
            std::string("value1"), subnode1->getStringProperty("prop1"));
        CPPUNIT_ASSERT_EQUAL(
            std::string("Some text\n    to parse."), subnode1->getText());

        const XMLNode *subnode2 = node->getMandatoryNode("subnode2");
        CPPUNIT_ASSERT_EQUAL(
            std::string("value2"), subnode2->getStringProperty("prop2"));
        CPPUNIT_ASSERT(!subnode2->hasText());
    }
};



//----------------------------------------------------------------------------
class SDLToolsTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(SDLToolsTest);

    CPPUNIT_TEST(testInside);
    CPPUNIT_TEST(testNoIntersection);
    CPPUNIT_TEST(testEdgeIntersection);
    CPPUNIT_TEST(testPartialIntersection);
    CPPUNIT_TEST(testFullIntersection);
    CPPUNIT_TEST(testIntersectionNegativeCoordinates);
    CPPUNIT_TEST(testIntersectionReuseParameter);

    CPPUNIT_TEST(testUnion);
    CPPUNIT_TEST(testUnionEmptyRectangle);
    CPPUNIT_TEST(testUnionNegativeCoordinates);
    CPPUNIT_TEST(testUnionReuseParameter);

    CPPUNIT_TEST(testRotate);

    CPPUNIT_TEST(testIsCollision);
    CPPUNIT_TEST(testGetAngle);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testInside()
    {
        SDL_Rect r; r.x = 5; r.y = 5; r.w = 5; r.h = 5;

        // Outside points.
        CPPUNIT_ASSERT(!SDL_TOOLS::inside( 4,  4, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside( 7,  4, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside(10,  4, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside(10,  7, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside(10, 10, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside( 7, 10, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside( 4, 10, r));
        CPPUNIT_ASSERT(!SDL_TOOLS::inside( 4,  7, r));

        // Points on the corners and edges.
        CPPUNIT_ASSERT(SDL_TOOLS::inside(5, 5, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(7, 5, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(9, 5, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(9, 7, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(9, 9, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(7, 9, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(5, 9, r));
        CPPUNIT_ASSERT(SDL_TOOLS::inside(5, 7, r));

        // Inner point.
        CPPUNIT_ASSERT(SDL_TOOLS::inside(7, 7, r));

        // 1x1 rectangle.
        r.w = r.h = 1;
        CPPUNIT_ASSERT(SDL_TOOLS::inside(5, 5, r));

        // 0x0 rectangle.
        r.w = r.h = 0;
        CPPUNIT_ASSERT(!SDL_TOOLS::inside(5, 5, r));
    }

    //------------------------------------------------------------------------
    void testNoIntersection()
    {
        SDL_Rect r1; r1.x = 5; r1.y = 5; r1.w = 5; r1.h = 5;
        SDL_Rect r2; r2.w = 5; r2.h = 5;
        SDL_Rect i;

        r2.y = 0;
        for (r2.x = 0; r2.x < 10; r2.x++)
        {
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r1, r2, i));
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r2, r1, i));
        }

        r2.x = 10;
        for (r2.y = 0; r2.y < 10; r2.y++)
        {
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r1, r2, i));
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r2, r1, i));
        }

        r2.y = 10;
        for (r2.x = 10; r2.x > 0; r2.x--)
        {
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r1, r2, i));
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r2, r1, i));
        }

        r2.x = 0;
        for (r2.y = 10; r2.y > 0; r2.y--)
        {
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r1, r2, i));
            CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r2, r1, i));
        }
    }

    //------------------------------------------------------------------------
    void testEdgeIntersection()
    {
        SDL_Rect r1; r1.x = 5; r1.y = 5; r1.w = 5; r1.h = 5;
        SDL_Rect r2; r2.w = 2; r2.h = 2;

        r2.x = 4; r2.y = 4;
        helperAssertIntersect(r1, r2, 5, 5, 1, 1);
        for (r2.x = 5; r2.x < 9; r2.x++)
        {
            helperAssertIntersect(r1, r2, r2.x, 5, 2, 1);
        }

        r2.x = 9;
        helperAssertIntersect(r1, r2, 9, 5, 1, 1);
        for (r2.y = 5; r2.y < 9; r2.y++)
        {
            helperAssertIntersect(r1, r2, 9, r2.y, 1, 2);
        }
        
        r2.y = 9;
        helperAssertIntersect(r1, r2, 9, 9, 1, 1);
        for (r2.x = 8; r2.x > 4; r2.x--)
        {
            helperAssertIntersect(r1, r2, r2.x, 9, 2, 1);
        }

        r2.x = 4;
        helperAssertIntersect(r1, r2, 5, 9, 1, 1);
        for (r2.y = 8; r2.y > 4; r2.y--)
        {
            helperAssertIntersect(r1, r2, 5, r2.y, 1, 2);
        }
    }

    //------------------------------------------------------------------------
    void testPartialIntersection()
    {
        SDL_Rect r1; r1.x = 5; r1.y = 5; r1.w = 5; r1.h = 5;
        SDL_Rect r2; r2.w = 4; r2.h = 4;

        r2.y = 3;
        r2.x = 3;
        helperAssertIntersect(r1, r2, 5, 5, 2, 2);
        r2.x = 4;
        helperAssertIntersect(r1, r2, 5, 5, 3, 2);
        r2.x = 5;
        helperAssertIntersect(r1, r2, 5, 5, 4, 2);
        r2.x = 6;
        helperAssertIntersect(r1, r2, 6, 5, 4, 2);
        r2.x = 7;
        helperAssertIntersect(r1, r2, 7, 5, 3, 2);

        r2.x = 8;
        r2.y = 3;
        helperAssertIntersect(r1, r2, 8, 5, 2, 2);
        r2.y = 4;
        helperAssertIntersect(r1, r2, 8, 5, 2, 3);
        r2.y = 5;
        helperAssertIntersect(r1, r2, 8, 5, 2, 4);
        r2.y = 6;
        helperAssertIntersect(r1, r2, 8, 6, 2, 4);
        r2.y = 7;
        helperAssertIntersect(r1, r2, 8, 7, 2, 3);

        r2.y = 8;
        r2.x = 8;
        helperAssertIntersect(r1, r2, 8, 8, 2, 2);
        r2.x = 7;
        helperAssertIntersect(r1, r2, 7, 8, 3, 2);
        r2.x = 6;
        helperAssertIntersect(r1, r2, 6, 8, 4, 2);
        r2.x = 5;
        helperAssertIntersect(r1, r2, 5, 8, 4, 2);
        r2.x = 4;
        helperAssertIntersect(r1, r2, 5, 8, 3, 2);
        
        r2.x = 3;
        r2.y = 8;
        helperAssertIntersect(r1, r2, 5, 8, 2, 2);
        r2.y = 7;
        helperAssertIntersect(r1, r2, 5, 7, 2, 3);
        r2.y = 6;
        helperAssertIntersect(r1, r2, 5, 6, 2, 4);
        r2.y = 5;
        helperAssertIntersect(r1, r2, 5, 5, 2, 4);
        r2.y = 4;
        helperAssertIntersect(r1, r2, 5, 5, 2, 3);
    }
    
    //------------------------------------------------------------------------
    void testFullIntersection()
    {
        SDL_Rect r1; r1.x = 5; r1.y = 5; r1.w = 5; r1.h = 5;
        SDL_Rect r2; r2.w = 3; r2.h = 3;

        for (r2.y = 5; r2.y <= 7; r2.y++)
        {
            for (r2.x = 5; r2.x <= 7; r2.x++)
            {
                helperAssertIntersect(r1, r2, r2.x, r2.y, r2.w, r2.h);
            }
        }
    }

    //------------------------------------------------------------------------
    void testIntersectionNegativeCoordinates()
    {
        SDL_Rect r1; r1.x = 0; r1.y = 0; r1.w = 10; r1.h = 10;
        SDL_Rect r2; r2.w = 3; r2.h = 3;
        SDL_Rect i;

        r2.x = -5; r2.y = -5;
        CPPUNIT_ASSERT(!SDL_TOOLS::intersect(r1, r2, i));

        r2.x = -1; r2.y = -1;
        helperAssertIntersect(r1, r2, 0, 0, 2, 2);
    }

    //------------------------------------------------------------------------
    void testIntersectionReuseParameter()
    {
        // Assure, that the destination rectangle
        // can be one of the source rectangles.

        SDL_Rect r1;
        SDL_Rect r2;

        r1.x = 5; r1.y = 5; r1.w = 10; r1.h = 10;
        r2.x = 3; r2.y = 3; r2.w =  5; r2.h =  5;

        SDL_Rect i; // This is the reference value.
        SDL_TOOLS::intersect(r1, r2, i);

        SDL_TOOLS::intersect(r1, r2, r1);
        CPPUNIT_ASSERT_EQUAL(i.x, r1.x);
        CPPUNIT_ASSERT_EQUAL(i.y, r1.y);
        CPPUNIT_ASSERT_EQUAL(i.w, r1.w);
        CPPUNIT_ASSERT_EQUAL(i.h, r1.h);

        r1.x = 5; r1.y = 5; r1.w = 10; r1.h = 10;  // Reinitialize r1.
        SDL_TOOLS::intersect(r1, r2, r2);
        CPPUNIT_ASSERT_EQUAL(i.x, r2.x);
        CPPUNIT_ASSERT_EQUAL(i.y, r2.y);
        CPPUNIT_ASSERT_EQUAL(i.w, r2.w);
        CPPUNIT_ASSERT_EQUAL(i.h, r2.h);
    }


    //------------------------------------------------------------------------
    void testUnion()
    {
        SDL_Rect r1; r1.x = 5; r1.y = 5; r1.w = 3; r1.h = 1;
        SDL_Rect r2; r2.w = 1; r2.h = 3;

        // No and partial intersection.
        for (r2.y = 1; r2.y <= 7; r2.y++)
        {
            for (r2.x = 3; r2.x <= 9; r2.x++)
            {
                helperAssertUnite(r1, r2);
            }
        }

        // Full intersection.
        r1.x = 5; r1.y = 5; r1.w = 5; r1.h = 5;
        r2.x = 6; r2.y = 6; r2.w = 3; r2.h = 3;
        helperAssertUnite(r1, r2);
    }

    //------------------------------------------------------------------------
    void testUnionEmptyRectangle()
    {
        SDL_Rect r1;
        SDL_Rect r2;
        SDL_Rect u;

        r1.x = 5; r1.y = 5; r1.w = 3; r1.h = 3;
        r2.x = 2; r2.y = 2; r2.w = 0; r2.h = 0;

        SDL_TOOLS::unite(r1, r2, u);
        CPPUNIT_ASSERT_EQUAL(r1.x, u.x);
        CPPUNIT_ASSERT_EQUAL(r1.y, u.y);
        CPPUNIT_ASSERT_EQUAL(r1.w, u.w);
        CPPUNIT_ASSERT_EQUAL(r1.h, u.h);
        
        SDL_TOOLS::unite(r2, r1, u);
        CPPUNIT_ASSERT_EQUAL(r1.x, u.x);
        CPPUNIT_ASSERT_EQUAL(r1.y, u.y);
        CPPUNIT_ASSERT_EQUAL(r1.w, u.w);
        CPPUNIT_ASSERT_EQUAL(r1.h, u.h);
    }

    //------------------------------------------------------------------------
    void testUnionNegativeCoordinates()
    {
        SDL_Rect r1; r1.x = 0; r1.y = 0; r1.w = 10; r1.h = 10;
        SDL_Rect r2; r2.w = 3; r2.h = 3;

        r2.x = -5; r2.y = -5;
        helperAssertUnite(r1, r2);

        r2.x = -1; r2.y = -1;
        helperAssertUnite(r1, r2);
    }

    //------------------------------------------------------------------------
    void testUnionReuseParameter()
    {
        // Assure, that the destination rectangle
        // can be one of the source rectangles.

        SDL_Rect r1;
        SDL_Rect r2;

        r1.x = 5; r1.y = 5; r1.w = 10; r1.h = 10;
        r2.x = 0; r2.y = 0; r2.w =  3; r2.h =  3;

        SDL_Rect u;  // This is the reference value.
        SDL_TOOLS::unite(r1, r2, u);

        SDL_TOOLS::unite(r1, r2, r1);
        CPPUNIT_ASSERT_EQUAL(u.x, r1.x);
        CPPUNIT_ASSERT_EQUAL(u.y, r1.y);
        CPPUNIT_ASSERT_EQUAL(u.w, r1.w);
        CPPUNIT_ASSERT_EQUAL(u.h, r1.h);
        
        r1.x = 5; r1.y = 5; r1.w = 10; r1.h = 10;  // Reinitialize r1.
        SDL_TOOLS::unite(r1, r2, r2);
        CPPUNIT_ASSERT_EQUAL(u.x, r2.x);
        CPPUNIT_ASSERT_EQUAL(u.y, r2.y);
        CPPUNIT_ASSERT_EQUAL(u.w, r2.w);
        CPPUNIT_ASSERT_EQUAL(u.h, r2.h);
    }


    //------------------------------------------------------------------------
    void testRotate()
    {
        SDL_Surface *i1 = SDL_CALLS::LoadBMP("gfx/rotate1.bmp");
        SDL_Surface *i2 = SDL_CALLS::LoadBMP("gfx/rotate2.bmp");
        SDL_Surface *i3 = SDL_CALLS::LoadBMP("gfx/rotate3.bmp");
        SDL_Surface *i4 = SDL_CALLS::LoadBMP("gfx/rotate4.bmp");
        SDL_Surface *r;

        r = SDL_TOOLS::rotate(i1, 0);
        helperCheckEqual(r, i1);
        SDL_CALLS::FreeSurface(r);

        r = SDL_TOOLS::rotate(i1, 90);
        helperCheckEqual(r, i2);
        SDL_CALLS::FreeSurface(r);

        r = SDL_TOOLS::rotate(i1, 180);
        helperCheckEqual(r, i3);
        SDL_CALLS::FreeSurface(r);

        r = SDL_TOOLS::rotate(i1, 270);
        helperCheckEqual(r, i4);
        SDL_CALLS::FreeSurface(r);

        SDL_CALLS::FreeSurface(i1);
        SDL_CALLS::FreeSurface(i2);
        SDL_CALLS::FreeSurface(i3);
        SDL_CALLS::FreeSurface(i4);
    }


    //------------------------------------------------------------------------
    void testIsCollision()
    {
        SDL_Surface *ship1 = SDL_CALLS::LoadBMP("gfx/ship.bmp");
        SDL_Surface *ship2 = SDL_CALLS::LoadBMP("gfx/ship.bmp");
        SDL_Rect pos1; pos1.w = ship1->w; pos1.h = ship1->h;
        SDL_Rect pos2; pos2.w = ship2->w; pos2.h = ship2->h;


        pos1.x = 32; pos1.y = 0;

        // Sweep the x coordinate of ship2 over ship1,
        // while incrementing the y coordinate of ship2 for each sweep.

        // The first 6 sweeps will result in identical collision detections,
        // since the ship's graphic contains a vertical edge of 6 pixels
        // on each side.
        for (pos2.y=0; pos2.y<6; pos2.y++)
        {
            for (pos2.x=0; pos2.x<64; pos2.x++)
            {
                bool c = SDL_TOOLS::isCollision(ship1, pos1, ship2, pos2);
                CPPUNIT_ASSERT_EQUAL(
                    c, SDL_TOOLS::isCollision(ship2, pos2, ship1, pos1));

                if ((pos2.x > 4) && (pos2.x < 60))
                {
                    CPPUNIT_ASSERT(c);
                }
                else
                {
                    CPPUNIT_ASSERT(!c);
                }
            }
        }

        // The next sweeps will reduce the collision zone
        // by one pixel on the right and one pixel to the left.
        // Now, the diagonal edges of the ships will collide.
        for (pos2.y=6; pos2.y<28; pos2.y++)
        {
            for (pos2.x=0; pos2.x<64; pos2.x++)
            {
                bool c = SDL_TOOLS::isCollision(ship1, pos1, ship2, pos2);
                CPPUNIT_ASSERT_EQUAL(
                    c, SDL_TOOLS::isCollision(ship2, pos2, ship1, pos1));
                if ((pos2.x > 4  + (pos2.y-6+1)) &&
                    (pos2.x < 60 - (pos2.y-6+1)))
                {
                    CPPUNIT_ASSERT(c);
                }
                else
                {
                    CPPUNIT_ASSERT(!c);
                }
            }
        }

        // The last sweep must not result in a collision.
        // The upper/lower edge of the ship will touch but won't overlap.
        for (pos2.x=0; pos2.x<64; pos2.x++)
        {
            pos2.y = 28;
            CPPUNIT_ASSERT(!SDL_TOOLS::isCollision(ship1, pos1, ship2, pos2));
            CPPUNIT_ASSERT(!SDL_TOOLS::isCollision(ship2, pos2, ship1, pos1));
        }


        SDL_CALLS::FreeSurface(ship1);
        SDL_CALLS::FreeSurface(ship2);
    }

    //------------------------------------------------------------------------
    void testGetAngle()
    {
        SDL_Rect pos1 = { 0, 0, 8, 8 };
        SDL_Rect pos2 = { 0, 0, 8, 8 };

        pos2.x = 0; pos2.y = 8;
        CPPUNIT_ASSERT_EQUAL(0, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = 8; pos2.y = 8;
        CPPUNIT_ASSERT_EQUAL(45, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = 8; pos2.y = 0;
        CPPUNIT_ASSERT_EQUAL(90, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = 8; pos2.y = -8;
        CPPUNIT_ASSERT_EQUAL(135, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = 0; pos2.y = -8;
        CPPUNIT_ASSERT_EQUAL(180, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = -8; pos2.y = -8;
        CPPUNIT_ASSERT_EQUAL(225, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = -8; pos2.y = 0;
        CPPUNIT_ASSERT_EQUAL(270, SDL_TOOLS::getAngle(pos1, pos2));

        pos2.x = -8; pos2.y = 8;
        CPPUNIT_ASSERT_EQUAL(315, SDL_TOOLS::getAngle(pos1, pos2));
    }


protected:

    //------------------------------------------------------------------------
    void helperAssertIntersect(const SDL_Rect &r1, const SDL_Rect &r2,
                               Sint16 x, Sint16 y, Uint16 w, Uint16 h)
    {
        SDL_Rect i;

        CPPUNIT_ASSERT(SDL_TOOLS::intersect(r1, r2, i));
        CPPUNIT_ASSERT_EQUAL(x, i.x);
        CPPUNIT_ASSERT_EQUAL(y, i.y);
        CPPUNIT_ASSERT_EQUAL(w, i.w);
        CPPUNIT_ASSERT_EQUAL(h, i.h);

        CPPUNIT_ASSERT(SDL_TOOLS::intersect(r2, r1, i));
        CPPUNIT_ASSERT_EQUAL(x, i.x);
        CPPUNIT_ASSERT_EQUAL(y, i.y);
        CPPUNIT_ASSERT_EQUAL(w, i.w);
        CPPUNIT_ASSERT_EQUAL(h, i.h);
    }

    //------------------------------------------------------------------------
    void helperAssertUnite(const SDL_Rect &r1, const SDL_Rect &r2)
    {
        SDL_Rect u;

        SDL_TOOLS::unite(r1, r2, u);
        CPPUNIT_ASSERT_EQUAL((Sint16)MIN(r2.x, r1.x), u.x);
        CPPUNIT_ASSERT_EQUAL((Sint16)MIN(r2.y, r1.y), u.y);
        CPPUNIT_ASSERT_EQUAL((Uint16)(MAX(r1.x+r1.w, r2.x+r2.w)-u.x), u.w);
        CPPUNIT_ASSERT_EQUAL((Uint16)(MAX(r1.y+r1.h, r2.y+r2.h)-u.y), u.h);

        SDL_TOOLS::unite(r2, r1, u);
        CPPUNIT_ASSERT_EQUAL((Sint16)MIN(r2.x, r1.x), u.x);
        CPPUNIT_ASSERT_EQUAL((Sint16)MIN(r2.y, r1.y), u.y);
        CPPUNIT_ASSERT_EQUAL((Uint16)(MAX(r1.x+r1.w, r2.x+r2.w)-u.x), u.w);
        CPPUNIT_ASSERT_EQUAL((Uint16)(MAX(r1.y+r1.h, r2.y+r2.h)-u.y), u.h);
    }

    //------------------------------------------------------------------------
    void helperCheckEqual(SDL_Surface *s1, SDL_Surface *s2_)
    {
        SDL_Surface *s2 =
            SDL_CALLS::ConvertSurface(s2_, s1->format, s1->flags);

        CPPUNIT_ASSERT_EQUAL(s1->h, s2->h);
        CPPUNIT_ASSERT_EQUAL(s1->w, s2->w);
        CPPUNIT_ASSERT_EQUAL(s1->pitch, s2->pitch);
        CPPUNIT_ASSERT_EQUAL((Uint8)32, s1->format->BitsPerPixel);
        CPPUNIT_ASSERT_EQUAL((Uint8)32, s2->format->BitsPerPixel);

        for (Sint32 y=0; y<s1->h; y++)
        {
            for (Sint32 x=0; x<s1->w; x++)
            {
                CPPUNIT_ASSERT_EQUAL(
                    *(Uint32*)((Uint8*)s1->pixels + y*s1->pitch + x*4),
                    *(Uint32*)((Uint8*)s2->pixels + y*s2->pitch + x*4));
            }
        }
        
        SDL_CALLS::FreeSurface(s2);
    }
};


//----------------------------------------------------------------------------
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(MathToolsTest, "MathTools");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(FileTest, "File");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(DirectoryTest, "Directory");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(XMLTest, "XML");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(SDLToolsTest, "SDLTools");
