#ifndef STRINGTOOLS_H
#define STRINGTOOLS_H

#include <string>
#include <vector>

//----------------------------------------------------------------------------
namespace STRING_TOOLS
{
    //------------------------------------------------------------------------
    /**
     * @param s A string to convert to a boolean.
     * @return The bool value of s.
     * @throw Exception if s doesn't contain a boolean value.
     */
    bool toBool(const std::string &s);

    //------------------------------------------------------------------------
    /**
     * @param s A string to convert to an int.
     * @return The int value of s.
     * @throw Exception if s doesn't contain an integer value.
     */
    int toInt(const std::string &s);

    //------------------------------------------------------------------------
    /**
     * @param s A string to convert to an unsigned.
     * @return The unsigned value of s.
     * @throw Exception if s doesn't contain an unsigned value.
     * @todo Negative numbers in s shall lead to an exception.
     */
    unsigned toUnsigned(const std::string &s);

    //------------------------------------------------------------------------
    /**
     * @param s A string containing comma-seperated unsigned values.
     * @param values An unsigned vector where the values will be pushed.
     * @throw Exception if s contains a token that is not an unsigned value.
     */
    void toUnsignedVector(const std::string &s, std::vector<unsigned> &values);

    //------------------------------------------------------------------------
    /**
     * Checks, if the given unsigned value is contained in the token string t.
     *
     * @param t The token string, using ',' as seperation character.
     * @param value The unsigned value to check for.
     *
     * @return true, if value is contained in t, else false.
     */
    bool hasUnsignedInTokenString(const std::string &t, const unsigned value);

    //------------------------------------------------------------------------
    /**
     * Tries to pop the first token of the token string t,
     * which value is equal to the given unsigned value.
     *
     * @param t The token string, using ',' as seperation character.
     * @param value The unsigned value to pop.
     *
     * @return true, if value was popped from t, else false.
     */
    bool popUnsignedFromTokenString(std::string &t, const unsigned value);

    //------------------------------------------------------------------------
    /**
     * Same as popUnsignedFromTokenString(),
     * but only the first token of t is checked.
     *
     * @param t The token string, using ',' as seperation character.
     * @param value The unsigned value to pop.
     *
     * @return true, if value is the first token and was popped from t,
     *         else false.
     */
    bool popFirstUnsignedFromTokenString(std::string &t, const unsigned value);

    //------------------------------------------------------------------------
    /**
     * This method converts unsigned number ranges in the given token string
     * into a comma-seperated list of unsigned values.
     *
     * For example, the string "1,3-5,10" will be converted to "1,3,4,5,10".
     *
     * @param t The token string to convert.
     *
     * @return The converted token string.
     */
    std::string &convertNumberRanges(std::string &t);

} // STRING_TOOLS

#endif //STRINGTOOLS_H
