#ifndef PLATFORM_H
#define PLATFORM_H

#include <map>

#include "Tools.h"
#include "StaticDecorationBase.h"

//----------------------------------------------------------------------------
class PlatformSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(PlatformSurfaces);

  public:

    //------------------------------------------------------------------------
    /**
     * An enum, which values define the available type of edges.
     */
    enum EdgeType
    {
        ET_NO_EDGE = 0,

        ET_NUMBER_1 = 1,
        ET_NUMBER_2 = 2,
        ET_NUMBER_3 = 3,
        ET_NUMBER_4 = 4,
        ET_NUMBER_5 = 5,
        ET_NUMBER_6 = 6,
        ET_NUMBER_7 = 7,
        ET_NUMBER_8 = 8,
        ET_NUMBER_9 = 9,

        ET_RED = 10,
        ET_YELLOW = 11,
        ET_GREEN = 12,
        ET_CYAN = 13,
        ET_BLUE = 14,
        ET_MAGENTA = 15,

        ET_TOTAL_NUMBER = 16
    };

  private:

    //------------------------------------------------------------------------
    class SurfaceKey
    {
      public:
        SurfaceKey(EdgeType left, EdgeType right,
                   unsigned frame, unsigned size)
            : m_left(left), m_right(right), m_frame(frame), m_size(size) {}
        ~SurfaceKey() {}

        friend bool operator<(const SurfaceKey &k1, const SurfaceKey &k2)
        {
            COMPARE_SURFACE_KEY_MEMBER(m_left);
            COMPARE_SURFACE_KEY_MEMBER(m_right);
            COMPARE_SURFACE_KEY_MEMBER(m_frame);
            COMPARE_SURFACE_KEY_MEMBER(m_size);
            return false;
        }

      private:
        EdgeType m_left;
        EdgeType m_right;
        unsigned m_frame;
        unsigned m_size;
    };

    //------------------------------------------------------------------------
    typedef std::map<SurfaceKey, SDL_Surface*> SurfaceMap;
    typedef SurfaceMap::iterator SurfaceIter;
    typedef SurfaceMap::const_iterator SurfaceCIter;

  public:

    //------------------------------------------------------------------------
    const SDL_Surface *getSurface(EdgeType left, EdgeType right,
                                  unsigned frame, unsigned size);

  private:

    //------------------------------------------------------------------------
    SDL_Surface *createSurface(EdgeType left, EdgeType right,
                               unsigned frame, unsigned size) const;


    /// All frames for the left edge.
    SDL_Surface *m_left[ET_TOTAL_NUMBER][2];

    /// The platform frame for the inner parts.
    SDL_Surface *m_center;

    /// All frames for the right edge.
    SDL_Surface *m_right[ET_TOTAL_NUMBER][2];


    /// A map for all dynamically created platform surfaces via getSurface().
    SurfaceMap m_surfaceMap;
};


//----------------------------------------------------------------------------
class Platform : public StaticDecorationBase
{
  public:
    //------------------------------------------------------------------------
    class InitializationData : public StaticDecorationBase::InitializationData
    {
      public:
        InitializationData(const XMLNode *platformNode);
        ~InitializationData();

        unsigned w;

        PlatformSurfaces::EdgeType left;
        PlatformSurfaces::EdgeType right;

      private:
        PlatformSurfaces::EdgeType getEdgeType(const XMLNode *node,
                                               const char *prop) const;
    };


    //------------------------------------------------------------------------
    virtual ~Platform();

    static Platform *create(const XMLNode *platformNode);

    //------------------------------------------------------------------------
    inline unsigned getWidth() const
    {
        return m_width;
    }

    inline const SDL_Rect &getLandingZone() const
    {
        return m_landingZone;
    }

    //------------------------------------------------------------------------
    void update();

  protected:
    //------------------------------------------------------------------------
    Platform(const InitializationData &init);

    /// Called by the constructor to initialize m_landingZone.
    void initLandingZone();


    //------------------------------------------------------------------------
    unsigned m_frameCounter;
    unsigned m_currentFrame;

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    unsigned m_width;
    PlatformSurfaces::EdgeType m_left;
    PlatformSurfaces::EdgeType m_right;

    /// A one pixel high rectangle on top of the center platform tiles.
    SDL_Rect m_landingZone;
};

#endif //PLATFORM_H
