#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>
#include "MyAssertionTraits.h"

#include "Configuration.h"
#include "XMLVisitors.h"

#include "Tiles.h"
#include "PlayerConfiguration.h"
#include "PlayerStatus.h"

#include "AllObjects.h"
#include "ObjectRepository.h"
#include "GameBridge.h"
#include "GameControlBase.h"


//----------------------------------------------------------------------------
class GameControlTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(GameControlTest);

    CPPUNIT_TEST(testLandObjectives);
    CPPUNIT_TEST(testDestroyObjectives);
    CPPUNIT_TEST(testRescueObjectives);
    CPPUNIT_TEST(testRescueCrashAfterPickup);
    CPPUNIT_TEST(testRescueCrashAfterPickupLastCrate);
    CPPUNIT_TEST(testRescueCrashAfterCrateUnloading);
    CPPUNIT_TEST(testRescueCrashAfterCrateUnloadingDelay);
    CPPUNIT_TEST(testLandCrashBeforeLandOnHomePlatform);
    CPPUNIT_TEST(testEventHandlerOnPickup);
    CPPUNIT_TEST(testOutOfFuel);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();

        SDLFrameRate::setFrameRate(72);

        Tiles::init();
        SurfacesBase::init();

        PlayerConfiguration::create();

        GameInterface::setInstance(GameBridge::getInstance());
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        GameControlBase::destroy();
        PlayerConfiguration::destroy();
        PlayerStatus::destroy();

        SurfacesBase::destroy();
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testLandObjectives()
    {
        GameControlBase::init("test", "land.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        gc->onShipLanded(ship, getPlatform(1));
        assertObjectives(
            "<land platform=\"2,3\"/>\n"
            "<land platform=\"4,5\"/>\n");

        gc->onShipLanded(ship, getPlatform(2));
        assertObjectives(
            "<land platform=\"3\"/>\n"
            "<land platform=\"4,5\"/>\n");

        gc->onShipLanded(ship, getPlatform(3));
        assertObjectives(
            "<land platform=\"4,5\"/>\n");

        gc->onShipLanded(ship, getPlatform(4));
        assertObjectives(
            "<land platform=\"5\"/>\n");

        gc->onShipLanded(ship, getPlatform(5));
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"AddFuelToScore", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testDestroyObjectives()
    {
        GameControlBase::init("test", "destroy.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        gc->onShipLanded(ship, getPlatform(2));
        assertObjectives(
            "<destroy id=\"10,11\"/>\n"
            "<destroy id=\"12,13\"/>\n"
            "<land platform=\"2\"/>\n");

        gc->onObjectDestroyed(getObject(13));
        simulateUpdateLoop();
        assertObjectives(
            "<destroy id=\"10,11\"/>\n"
            "<destroy id=\"12\"/>\n"
            "<land platform=\"2\"/>\n");

        gc->onObjectDestroyed(getObject(10));
        simulateUpdateLoop();
        assertObjectives(
            "<destroy id=\"11\"/>\n"
            "<destroy id=\"12\"/>\n"
            "<land platform=\"2\"/>\n");

        gc->onObjectDestroyed(getObject(12));
        simulateUpdateLoop();
        assertObjectives(
            "<destroy id=\"11\"/>\n"
            "<land platform=\"2\"/>\n");

        gc->onObjectDestroyed(getObject(11));
        simulateUpdateLoop();
        assertObjectives(
            "<land platform=\"2\"/>\n");

        gc->onShipLanded(ship, getPlatform(2));
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"AddFuelToScore", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testRescueObjectives()
    {
        GameControlBase::init("test", "rescue.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        gc->onShipLanded(ship, getPlatform(1)); // Land on the wrong platform.
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 10.
        unloadCrates(1);
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        movePlayerShipToCrate(12);
        gc->onShipLanded(ship, getPlatform(4)); // 12 can't be picked up yet.
        CPPUNIT_ASSERT(!ship->hasCrates());
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        movePlayerShipToCrate(11);
        gc->onShipLanded(ship, getPlatform(3)); // Pick up crate 11.
        simulateUpdateLoop();
        assertObjectives(
            "<pickup id=\"\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 11.
        unloadCrates(1);
        assertObjectives(
            "<pickup id=\"12\" platform=\"1\"/>\n");
        
        movePlayerShipToCrate(12);
        gc->onShipLanded(ship, getPlatform(4)); // Pick up crate 12.
        simulateUpdateLoop();
        assertObjectives(
            "<pickup id=\"\" platform=\"1\"/>\n");

        gc->onShipLanded(ship, getPlatform(1)); // Unload crate 12.
        unloadCrates(1);
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"AddFuelToScore", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testRescueCrashAfterPickup()
    {
        GameControlBase::init("test", "rescue.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        gc->onCollision(ship);  // Crash before reaching platform 5.
        handleExplosionDelay();
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Running", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testRescueCrashAfterPickupLastCrate()
    {
        GameControlBase::init("test", "rescue.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        // This is the same control flow as in testRescueObjectives()
        // except for the last crate.

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 10.
        unloadCrates(1);
        movePlayerShipToCrate(11);
        gc->onShipLanded(ship, getPlatform(3)); // Pick up crate 11.
        simulateUpdateLoop();
        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 11.
        unloadCrates(1);
        movePlayerShipToCrate(12);
        gc->onShipLanded(ship, getPlatform(4)); // Pick up crate 12.
        simulateUpdateLoop();

        gc->onCollision(ship); // Crash before reaching platform 1.
        handleExplosionDelay();
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Final", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testRescueCrashAfterCrateUnloading()
    {
        GameControlBase::init("test", "rescue.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 10.

        gc->onCollision(ship);
        handleExplosionDelay();

        CPPUNIT_ASSERT(!ship->hasCrates());
        CPPUNIT_ASSERT(getObject(10) == NULL);
        assertObjectives(
            "<pickup id=\"11\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Running", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testRescueCrashAfterCrateUnloadingDelay()
    {
        GameControlBase::init("test", "rescue.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        movePlayerShipToCrate(11);
        gc->onShipLanded(ship, getPlatform(3)); // Pick up crate 11.
        simulateUpdateLoop();
        assertObjectives(
            "<pickup id=\"\" platform=\"5\"/>\n"
            "<pickup id=\"12\" platform=\"1\"/>\n");
        
        gc->onShipLanded(ship, getPlatform(5)); // Unload crate 10.
        CPPUNIT_ASSERT_EQUAL(
            (const char*)"UnloadingCrate", gc->getNameOfCurrentState());
        while (!strcmp(gc->getNameOfCurrentState(), "UnloadingCrate"))
        {
            simulateUpdateLoop();
        }
        CPPUNIT_ASSERT_EQUAL(
            (const char*)"UnloadingDelay", gc->getNameOfCurrentState());

        gc->onCollision(ship);
        handleExplosionDelay();

        CPPUNIT_ASSERT(!ship->hasCrates());
        CPPUNIT_ASSERT(getObject(10) == NULL);
        // Crate 11 wasn't deleted yet, but it must still be hidden.
        CPPUNIT_ASSERT(getObject(11)->isHidden());
        assertObjectives(
            "<pickup id=\"12\" platform=\"1\"/>\n");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Running", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testLandCrashBeforeLandOnHomePlatform()
    {
        GameControlBase::init("test", "landhome.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();
        
        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();

        gc->onCollision(ship); // Crash before reaching platform 3.
        handleExplosionDelay();

        // The <pickup> objective is deleted, because it can't be finished.
        // If the remaining <land> objective refers to the home platform,
        // it must be deleted too.
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Final", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testEventHandlerOnPickup()
    {
        GameControlBase::init("test", "onpickup.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        // Only crate 10 is visible at first.
        CPPUNIT_ASSERT(!getObject(10)->isHidden());
        CPPUNIT_ASSERT(getObject(11)->isHidden());
        CPPUNIT_ASSERT(getObject(12)->isHidden());

        movePlayerShipToCrate(10);
        gc->onShipLanded(ship, getPlatform(2)); // Pick up crate 10.
        simulateUpdateLoop();
        CPPUNIT_ASSERT(!getObject(11)->isHidden());
        assertObjectives(
            "<pickup id=\"11,12\" platform=\"5\">\n"
            "  <onpickup show=\"12\"/>\n"
            "</pickup>\n");

        movePlayerShipToCrate(11);
        gc->onShipLanded(ship, getPlatform(3)); // Pick up crate 11.
        simulateUpdateLoop();
        CPPUNIT_ASSERT(!getObject(12)->isHidden());
        assertObjectives(
            "<pickup id=\"12\" platform=\"5\"/>\n");

        movePlayerShipToCrate(12);
        gc->onShipLanded(ship, getPlatform(3)); // Pick up crate 12.
        simulateUpdateLoop();
        assertObjectives(
            "<pickup id=\"\" platform=\"5\"/>\n");

        gc->onShipLanded(ship, getPlatform(5)); // Unload all crates.
        unloadCrates(3);
        assertObjectives(
            "");

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"AddFuelToScore", gc->getNameOfCurrentState());
    }

    //------------------------------------------------------------------------
    void testOutOfFuel()
    {
        GameControlBase::init("test", "land.xml");
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        gc->onShipOutOfFuel(ship);
        
        handleOutOfFuelDelay();
        gc->onCollision(ship);  // UpdateObjectsVisitor::do_visit(Ship *s)
        handleExplosionDelay();

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"Running", gc->getNameOfCurrentState());
        CPPUNIT_ASSERT(!ship->isExplode());
    }


private:

    //------------------------------------------------------------------------
    ObjectBase *getObject(unsigned id)
    {
        return ObjectRepository::getInstance()->getObject(id);
    }

    //------------------------------------------------------------------------
    Platform *getPlatform(unsigned id)
    {
        return dynamic_cast<Platform*>(
            ObjectRepository::getInstance()->getObject(id));
    }

    //------------------------------------------------------------------------
    void movePlayerShipToCrate(unsigned crateId)
    {
        Ship *ship = GameControlBase::getInstance()->getPlayerShip();
        const SDL_Rect &shipPos = ship->getPosition();

        Crate *crate = dynamic_cast<Crate*>(
            ObjectRepository::getInstance()->getObject(crateId));
        const SDL_Rect &cratePos = crate->getPosition();

        ship->setPosition(cratePos.x + cratePos.w/2 - shipPos.w/2,
                          cratePos.y + cratePos.h - shipPos.h);
    }

    //------------------------------------------------------------------------
    void unloadCrates(unsigned num)
    {
        GameControlBase *gc = GameControlBase::getInstance();

        for (unsigned i=num; i>0; i--)
        {
            CPPUNIT_ASSERT_EQUAL(
                (const char*)"UnloadingCrate",
                gc->getNameOfCurrentState());
            while (!strcmp(gc->getNameOfCurrentState(), "UnloadingCrate"))
            {
                simulateUpdateLoop();
            }

            if (i > 1)
            {
                CPPUNIT_ASSERT_EQUAL(
                    (const char*)"UnloadingDelay",
                    gc->getNameOfCurrentState());
                while (!strcmp(gc->getNameOfCurrentState(), "UnloadingDelay"))
                {
                    simulateUpdateLoop();
                }
            }
        }
    }

    //------------------------------------------------------------------------
    void handleOutOfFuelDelay()
    {
        GameControlBase *gc = GameControlBase::getInstance();
        Ship *ship = gc->getPlayerShip();

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"OutOfFuelDelay", gc->getNameOfCurrentState());

        while (!ship->isExplode())
        {
            simulateUpdateLoop();
        }
    }

    //------------------------------------------------------------------------
    void handleExplosionDelay()
    {
        GameControlBase *gc = GameControlBase::getInstance();

        CPPUNIT_ASSERT_EQUAL(
            (const char*)"ExplosionDelay", gc->getNameOfCurrentState());
        while (!strcmp(gc->getNameOfCurrentState(), "ExplosionDelay"))
        {
            simulateUpdateLoop();
        }
    }

    //------------------------------------------------------------------------
    void simulateUpdateLoop()
    {
        GameControlBase *gc = GameControlBase::getInstance();

        gc->onPreUpdate();

        // The only relevant action of PlayGround::update() needed here.
        ObjectRepository::getInstance()->removeObjectsToRemove();

        gc->onPostUpdate();
    }
    
    //------------------------------------------------------------------------
    void assertObjectives(const char *desired)
    {
        std::ostringstream s;
        XMLWriteToStreamVisitor v(s);
        GameControlBase::getInstance()->getGameControlNode()->accept(v);

        CPPUNIT_ASSERT_EQUAL(std::string(desired), s.str());
    }
};

CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(GameControlTest, "GameControl");
