#include <ctype.h>
#include <iostream>

#include "Directory.h"
#include "Configuration.h"
#include "SoundInterface.h"
#include "LevelReader.h"
#include "LevelSelectorMenu.h"
#include "HighscoreMenu.h"
#include "PlayGroundMenu.h"
#include "PlayerConfiguration.h"
#include "GuiException.h"

#include <pgdraw.h>
#include <pgdropdown.h>

#define LEVEL_STRING "Start at level %d"


MENU_SINGLETON_INIT(LevelSelector);

//----------------------------------------------------------------------
LevelSelectorMenu::LevelSelectorMenu()
{
  getWidget(ID_LEVELSEL_GO)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onGo);
  getWidget(ID_LEVELSEL_HIGHSCORE)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onHighscore);
  getWidget(ID_LEVELSEL_BACK)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onBack);
  getWidget(ID_LEVELSEL_PLEV)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onPreviousLevel);
  getWidget(ID_LEVELSEL_NLEV)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onNextLevel);
  getWidget(ID_LEVELSEL_FLEV)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onFirstLevel);
  getWidget(ID_LEVELSEL_LLEV)->SetEventObject(
    MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onLastLevel);

  // deleted by paragui on shutdown

  PG_Widget *p = getWidget(ID_LEVELSEL_PREVIEW);
  m_previewWidget = new ReplaceableSurfaceWidget(p, PG_Rect(0, 0, 640, 480));

  getWidget(ID_LEVELSEL_LABEL)->Show();
  getWidget(ID_LEVELSEL_BACK)->Show();
  getWidget(ID_LEVELSEL_RESUME)->Show();
  getWidget(ID_LEVELSEL_GO)->Show();
  getWidget(ID_LEVELSEL_HIGHSCORE)->Show();
  getWidget(ID_LEVELSEL_PREVIEW)->Show();
}

//----------------------------------------------------------------------
LevelSelectorMenu::~LevelSelectorMenu()
{
}

//----------------------------------------------------------------------
void LevelSelectorMenu::initMissionSelect()
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    PG_DropDown *widget = (PG_DropDown*)getWidget(ID_LEVELSEL_MISSION);
    widget->SetEditable(false);
    widget->SetIndent(30);
    m_selectedIndex = -1;

    try
    {
        MissionReader mr;
        LevelReader lr;
        unsigned i = 0;
        for (MissionReader::iterator iter = mr.begin();
             iter != mr.end(); ++iter)
        {
            lr.init(iter->first.c_str());

            missionConfig->trimToRealNumberOfLevels(iter->first, lr.size());

            m_levelNames.push_back(iter->first);
            m_fullLevelNames.push_back(iter->second);
            m_numberOfLevels.push_back(lr.size());

            widget->AddItem(iter->second.c_str(), (void*)i++);

            if (iter->first == missionConfig->getCurrentMission())
            {
                m_selectedIndex = std::distance(mr.begin(), iter);
            }
        }

        if (m_selectedIndex == -1)
        {
            m_selectedIndex = 0;
            missionConfig->updateLastPlayed(mr.begin()->first, 0);

            widget->SelectFirstItem();
        }
        else
        {
            widget->SelectItem(m_selectedIndex);
        }
    }
    catch (Exception &e)
    {
        throw GuiException("Error reading levels directory: " + e.toString());
    }

    widget->SetEventObject(
        MSG_SELECTITEM, this,
        (MSG_CALLBACK_OBJ)&LevelSelectorMenu::onMissionSelected);

    widget->Update();
    widget->Show();
}


//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onMissionSelected)
{
    PG_ListBoxItem *item = (PG_ListBoxItem*)data;
    m_selectedIndex = (long)item->GetUserData();

    PlayerConfiguration::getInstance()->getMission()
        ->updateLastPlayed(getCurrentlySelected(), 0);
    updateLevelSelection();

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onGo)
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    PlayGroundMenu *playground = PlayGroundMenu::getInstance();
    playground->setMission(missionConfig->getCurrentMission());
    playground->setLevel(missionConfig->getLastPlayedLevel());

    MenuManager::getInstance()->changeTo(
        PlayGroundMenu::getInstance(), false, true);

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onHighscore)
{
    try
    {
        HighscoreMenu *highscore = HighscoreMenu::getInstance();
        highscore->setInsertEntryOnShow(false);
        MenuManager::getInstance()->changeTo(highscore);
    }
    catch (GuiMenuException &e)
    {
        std::cout << e << std::endl;
    }

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onBack)
{
  MenuManager::getInstance()->changeTo(getPrevious(), true, false);
  return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onPreviousLevel)
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    const std::string &mission = missionConfig->getCurrentMission();
    unsigned lastPlayedLevel = missionConfig->getLastPlayedLevel();

    if (lastPlayedLevel > 1)
    {
        missionConfig->updateLastPlayed(mission, lastPlayedLevel-1);
        updateLevelSelection();
    }

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onNextLevel)
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    const std::string &mission = missionConfig->getCurrentMission();
    unsigned lastPlayedLevel = missionConfig->getLastPlayedLevel();
    unsigned handicap = missionConfig->getHandicap();

    if (lastPlayedLevel < handicap)
    {
        missionConfig->updateLastPlayed(mission, lastPlayedLevel+1);
        updateLevelSelection();
    }

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onFirstLevel)
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    const std::string &mission = missionConfig->getCurrentMission();

    missionConfig->updateLastPlayed(mission, 1);
    updateLevelSelection();

    return true;
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(LevelSelectorMenu::onLastLevel)
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    const std::string &mission = missionConfig->getCurrentMission();
    unsigned handicap = missionConfig->getHandicap();

    missionConfig->updateLastPlayed(mission, handicap);
    updateLevelSelection();

    return true;
}

//----------------------------------------------------------------------
void LevelSelectorMenu::updateLevelSelection()
{
    PlayerConfiguration::Mission *missionConfig =
        PlayerConfiguration::getInstance()->getMission();

    unsigned lastPlayedLevel = missionConfig->getLastPlayedLevel();
    unsigned handicap = missionConfig->getHandicap();

    if (lastPlayedLevel > 1)
    {
        getWidget(ID_LEVELSEL_PLEV)->Show();
        getWidget(ID_LEVELSEL_FLEV)->Show();
    }
    else
    {
        getWidget(ID_LEVELSEL_PLEV)->Hide();
        getWidget(ID_LEVELSEL_FLEV)->Hide();
    }

    if (lastPlayedLevel < handicap)
    {
        getWidget(ID_LEVELSEL_NLEV)->Show();
        getWidget(ID_LEVELSEL_LLEV)->Show();
    }
    else
    {
        getWidget(ID_LEVELSEL_NLEV)->Hide();
        getWidget(ID_LEVELSEL_LLEV)->Hide();
    }

    getWidget(ID_LEVELSEL_RESUME)->SetTextFormat(
        LEVEL_STRING, lastPlayedLevel);

    createPreview();
}

//----------------------------------------------------------------------
void LevelSelectorMenu::eventShow()
{
    static bool initialized = false;
    if (!initialized)
    {
        initMissionSelect();
        initialized = true;
    }

    updateLevelSelection();
    getWidget(ID_LEVELSEL)->Show();

    SoundInterface::getInstance()->playTitleMusic();
}

//----------------------------------------------------------------------
void LevelSelectorMenu::eventHide()
{
    PlayerConfiguration::getInstance()->writeConfiguration();

    getWidget(ID_LEVELSEL)->Hide();
}

//----------------------------------------------------------------------
void LevelSelectorMenu::createPreview()
{
    try
    {
        PlayerConfiguration::Mission *missionConfig =
            PlayerConfiguration::getInstance()->getMission();

        const std::string &mission = missionConfig->getCurrentMission();
        unsigned lastPlayedLevel = missionConfig->getLastPlayedLevel();

        const PG_Rect *previewWidgetRect =
            getWidget(ID_LEVELSEL_PREVIEW)->GetClipRect();

        PlayGroundMenu *playGroundMenu = PlayGroundMenu::getInstance();
        playGroundMenu->setMission(mission);
        playGroundMenu->setLevel(lastPlayedLevel); 

        const SDL_Surface *levelSurface = playGroundMenu->getLevelPreview();

        float xScale = 1.0 * (previewWidgetRect->w - 16) / levelSurface->w;
        float yScale = 1.0 * (previewWidgetRect->h - 16) / levelSurface->h;

        SDL_Surface *previewWidgetSurface =
            SDL_CALLS::CreateRGBSurface(
                SDL_SWSURFACE, previewWidgetRect->w, previewWidgetRect->h, 32);

        SDL_Surface *previewSurface =
            SDL_CALLS::rotozoomSurface(
                levelSurface, 0.0f,
                (xScale < yScale ? xScale : yScale), SMOOTHING_OFF);

        SDL_Rect clip;
        clip.x = (previewWidgetSurface->w - previewSurface->w) / 2;
        clip.y = (previewWidgetSurface->h - previewSurface->h) / 2;
        clip.w = previewSurface->w;
        clip.h = previewSurface->h;

        SDL_CALLS::BlitSurface(
            previewSurface, NULL, previewWidgetSurface, &clip);

        ZAP_SURFACE(previewSurface);

        m_previewWidget->SetWidgetSurface(previewWidgetSurface);
        m_previewWidget->Show();
    }
    catch(Exception &e)
    {
        std::cout << "Error creating preview: " << e << std::endl;
    }
}
