#ifndef SHIP_H
#define SHIP_H

#include <vector>
#include <list>
#include <cassert>

#include "Tools.h"
#include "SDLCalls.h"
#include "ControllableObjectBase.h"

class Crate;
class Platform;

//----------------------------------------------------------------------------
class ShipSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(ShipSurfaces);

  public:

    enum Ship
    {
        S_DEFAULT = 0,
        S_ARROW = 1,
        S_GALAXY = 2,
        S_AKIRA = 3,
        S_DEFIANT = 4,
        S_DANUBE = 5,
	S_INTREPID = 6,
        S_PROMETHEUS = 7,
        S_NORWAY = 8,
        S_STEAMRUNNER = 9,
        S_BIRDOFPREY = 10,
        S_KTINGA = 11,
        S_VORCHA = 12,
        S_TOTAL_NUMBER = 13
    };

    inline const SDL_Surface *getSurface(Ship ship, unsigned frame) const
    {
        assert(ship >= 0);
        assert(ship < S_TOTAL_NUMBER);
        assert(frame < SURFACES_ROTATION_STEPS);

        return m_ships[ship][frame];
    }

    Uint16 getTipOffset(Ship ship) const;

  private:

    /// All ship surfaces.
    SDL_Surface *m_ships[S_TOTAL_NUMBER][SURFACES_ROTATION_STEPS];
};



//----------------------------------------------------------------------------
#undef DECLARE_UPDATE_STATE
#define DECLARE_UPDATE_STATE(c) \
    class c; \
    friend class c; \
    class c : public UpdateState \
    { \
        STATE_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void update(Ship *s); \
    }

//----------------------------------------------------------------------------
class Ship : public ControllableObjectBase
{
    friend class ShipTest;

    //------------------------------------------------------------------------
    /**
     * The max. fuel capacity in seconds.
     */
    static const unsigned MAX_FUEL_CAPACITY = 100;

    //------------------------------------------------------------------------
    typedef std::list<Crate*> Crates;
    typedef Crates::iterator CrateIter;
    typedef Crates::const_iterator CrateCIter;

    //------------------------------------------------------------------------
    class UpdateState
    {
      protected:
        UpdateState() {}
      public:
        virtual ~UpdateState() {}

        virtual void update(Ship *s) = 0;
    };

    //------------------------------------------------------------------------
    DECLARE_UPDATE_STATE(Landed);
    DECLARE_UPDATE_STATE(Flying);

  public:
    //------------------------------------------------------------------------
    Ship(ShipSurfaces::Ship shipType);
    Ship(unsigned id, ShipSurfaces::Ship shipType);
    virtual ~Ship();


    //------------------------------------------------------------------------
    inline void setFire()
    {
        m_fire = true;
    }


    //------------------------------------------------------------------------
    /**
     * @return true, if the ship has capacity to hold the given crate.
     */
    bool hasCapacity(const Crate *crate) const;

    /**
     * @return true, if the ship has at least one crate loaded.
     */
    inline bool hasCrates() const
    {
       return !m_crates.empty();
    }

    /**
     * Adds the given crate to m_crates.
     */
    void pushCrate(Crate *crate);

    /**
     * @return The first loaded crate, which will be removed from m_crates.
     */
    Crate *popCrate();


    //------------------------------------------------------------------------
    inline const Platform *getLandedPlatform() const
    {
        return m_platform;
    }

    inline Uint16 getTipOffset() const
    {
        return m_tipOffset;
    }

    /**
     * @param r The landing zone rectangle of a platform.
     * @return true, if the ship's bottom pixel row is inside the landing zone.
     */
    bool isInLandingZone(const SDL_Rect &r) const;

    /**
     * Resets the ship's position to the given values.
     * Further, the angle and the velocity will be set to 0.
     */
    void resetPosition(Sint16 x, Sint16 y, bool landed);

    //------------------------------------------------------------------------
    inline ShipSurfaces::Ship getShipType() const
    {
        return m_shipType;
    }

    //------------------------------------------------------------------------
    inline unsigned getRotationDegreesPerSecond() const { return 216; }
    inline unsigned getThrustForce() const { return 30000; }

    //------------------------------------------------------------------------
    void update();

  protected:
    //------------------------------------------------------------------------
    /// Called by the constructor to initialize m_tipPixelX and m_tipPixelY.
    void initTipPixel();

    /**
     * Creates a Projectile, if m_fire is true.
     */
    void updateFire();

    inline void setUpdateState(UpdateState *s)
    {
        m_updateState = s;
    }

  private:
    //------------------------------------------------------------------------
    const SDL_Surface *getRotationSurface(unsigned frame) const;

    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

  protected:
    //------------------------------------------------------------------------
    /// The type of ship, needed to access the ShipSurfaces factory.
    ShipSurfaces::Ship m_shipType;

    /// True, if the ship shall fire a projectile.
    bool m_fire;

    /// The platform where this ship has landed.
    const Platform *m_platform;

    /// A list of crates loaded in the ship.
    Crates m_crates;

    /// The y offset of the ship's tip.
    Uint16 m_tipOffset;

  private:
    //------------------------------------------------------------------------
    UpdateState *m_updateState;
};

#endif //SHIP_H
