///////////////////////////////////////////////////////////////////////////////
// $Id: mapeditor.cpp,v 1.19 2004/12/14 11:03:29 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    mapeditor.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the editor window for maps
*/

// Qt includes
#include <qaction.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qclipboard.h>
#include <qcombobox.h>
#include <qheader.h>
#include <qinputdialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qtoolbar.h>

// local includes
#include "mainwindow.h"
#include "mapdocument.h"
#include "mapdrag.h"
#include "mapeditor.h"
#include "mapview.h"
#include "resizedialog.h"
#include "tilegroup.h"
#include "tileset.h"

// action icons
#include "line.xpm"
#include "point.xpm"
#include "rectangle.xpm"
#include "select.xpm"

static const char* const cutpix[] = {
"22 22 3 1",
". c None",
"# c #000000",
"a c #000083",
"......................",
".......#.....#........",
".......#.....#........",
".......#.....#........",
".......#....##........",
".......##...#.........",
"........#...#.........",
"........##.##.........",
".........###..........",
".........###..........",
"..........#...........",
".........a#a..........",
"........aa.aaa........",
".......a.a.a..a.......",
"......a..a.a...a......",
".....a...a.a....a.....",
"....a....a.a....a.....",
"....a....a..a...a.....",
"....a....a..a..a......",
"....a...a....aa.......",
".....aaa..............",
"......................"};

static const char* const copypix[] = {
"22 22 6 1",
". c None",
"# c #000000",
"b c #000083",
"c c #393cff",
"d c #8b89ff",
"a c #ffffff",
"......................",
"......................",
"########..............",
"#aaaaaa##.............",
"#a####a#a#............",
"#aaaaaa#aa#...........",
"#a####a#bbbbbbbb......",
"#aaaaaa#baaaaaabb.....",
"#a#####aba####abcb....",
"#aaaaaaabaaaaaabdcb...",
"#a#####aba####abadcb..",
"#aaaaaaabaaaaaabbbbbb.",
"#a#####aba####aaaaaab.",
"#aaaaaaabaaaaaaaaaaab.",
"#a#####aba#########ab.",
"#aaaaaaabaaaaaaaaaaab.",
"########ba#########ab.",
"........baaaaaaaaaaab.",
"........ba#########ab.",
"........baaaaaaaaaaab.",
"........bbbbbbbbbbbbb.",
"......................"};

static const char* const pastepix[] = {
"22 22 8 1",
". c None",
"# c #000000",
"e c #000083",
"c c #838100",
"b c #838183",
"d c #c5c2c5",
"a c #ffff00",
"f c #ffffff",
"......................",
".......#####..........",
"..######aaa######.....",
".######aaaaa######....",
"##bcb##a###a##bcb##...",
"#bcb#ddddddddd#bcb#...",
"#cbc#ddddddddd#cbc#...",
"#bcb###########bcb#...",
"#cbcbcbcbcbcbcbcbc#...",
"#bcbcbcbcbcbcbcbcb#...",
"#cbcbcbceeeeeeeeee#...",
"#bcbcbcbefffffffefe...",
"#cbcbcbcefeeeeefeffe..",
"#bcbcbcbefffffffefffe.",
"#cbcbcbcefeeeeefeffffe",
"#bcbcbcbefffffffeeeeee",
"#cbcbcbcefeeeeeffffffe",
"#bcbcbcbeffffffffffffe",
"#cbcbcbcefeeeeeeeeeefe",
".#######effffffffffffe",
"........eeeeeeeeeeeeee",
"......................"};

///////////////////////////////////////////////////////////////////////////////

MapEditor::MapEditor(MainWindow* mainwindow, QWidget* parent, const char* name)
	: MapEditorBase(parent, name),
	  m_mainWindow(mainwindow)
{
	m_doc = new MapDocument();
	m_view = new MapView(m_doc, m_lblPos, this, "map view");
	layout()->add(m_view);
	
	QObject::connect(m_view, SIGNAL(showing()), this, SLOT(slotViewShowing()));
	QObject::connect(m_view, SIGNAL(hiding()), this, SLOT(slotViewHiding()));
	QObject::connect(m_view, SIGNAL(contextMenu(const QPoint&)),
	                 this, SLOT(slotContextMenu(const QPoint&)));

	QObject::connect(m_lvTiles, SIGNAL(currentChanged(QListViewItem*)),
	                 this, SLOT(slotTileSelected(QListViewItem*)));

	QObject::connect(m_lvLayers, SIGNAL(selectionChanged(QListViewItem*)),
	                 this, SLOT(slotLayerSelected(QListViewItem*)));
	QObject::connect(m_lvLayers, SIGNAL(clicked(QListViewItem*, const QPoint&, int)),
	                 this, SLOT(slotLayerClicked(QListViewItem*, const QPoint&, int)));
	QObject::connect(m_lvLayers->header(), SIGNAL(clicked(int)),
	                 this, SLOT(slotLayerHeaderClicked(int)));
	QObject::connect(m_btnAddLayer,    SIGNAL(clicked()), this, SLOT(slotAddLayer()));
	QObject::connect(m_btnRemoveLayer, SIGNAL(clicked()), this, SLOT(slotRemoveLayer()));
	QObject::connect(m_btnLayerUp,     SIGNAL(clicked()), this, SLOT(slotMoveLayerUp()));
	QObject::connect(m_btnLayerDown,   SIGNAL(clicked()), this, SLOT(slotMoveLayerDown()));
	QObject::connect(m_cbBackground,   SIGNAL(toggled(bool)),
	                 this, SLOT(slotLayerBackgroundToggled(bool)));
	QObject::connect(m_spAlpha, SIGNAL(valueChanged(int)),
	                 this, SLOT(slotLayerPropertiesChanged()));
	QObject::connect(m_spBrightness, SIGNAL(valueChanged(int)),
	                 this, SLOT(slotLayerPropertiesChanged()));

	createTools();
	createContextMenu();
}

///////////////////////////////////////////////////////////////////////////////

MapEditor::~MapEditor()
{
	delete m_doc;
}


///////////////////////////////////////////////////////////////////////////////

bool MapEditor::newDocument()
{
	QString tilesetfile = m_mainWindow->tilesDirectory();
	if (tilesetfile.isEmpty()) return false;
	tilesetfile += "/categories.idx";

	ResizeDialog dlg(this);
	dlg.setCaption(tr("Map size"));
	if (dlg.exec() != QDialog::Accepted) return false;

	bool result = m_doc->init(dlg.numRows(), dlg.numCols(), tilesetfile);

	m_view->slotDocumentChanged();
	updateLayerSelector();
	return result;
}

///////////////////////////////////////////////////////////////////////////////

bool MapEditor::openDocument(const QString& filename)
{
	QString tilesetfile = m_mainWindow->tilesDirectory();
	if (tilesetfile.isEmpty()) return false;
	tilesetfile += "/categories.idx";

	bool result = m_doc->init(tilesetfile);
	result = result && m_doc->openFile(filename);

	m_view->slotDocumentChanged();
	updateLayerSelector();
	return result;
}

///////////////////////////////////////////////////////////////////////////////

bool MapEditor::saveDocument()
{
	return m_doc->saveFile();
}

///////////////////////////////////////////////////////////////////////////////

bool MapEditor::saveDocumentAs(const QString& filename)
{
	return m_doc->saveFileAs(filename);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::resize()
{
	ResizeDialog dlg(this);
	dlg.setCaption(tr("Resize map"));
	dlg.setNumRows(m_doc->numRows());
	dlg.setNumCols(m_doc->numCols());

	if (dlg.exec() != QDialog::Accepted) return;
	if (dlg.numRows() == m_doc->numRows() && dlg.numCols() == m_doc->numCols()) return;

	m_doc->resize(dlg.numRows(), dlg.numCols());

	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::cut()
{
	QRect area = m_doc->selectionArea();
	if (!area.isValid()) return;

	MapDocument* transferDoc = new MapDocument();
	transferDoc->init(*m_doc->tileSet());

	bool result = transferDoc->copyData(area,
	                                    m_doc->currentLayer(),
	                                    *m_doc, QPoint(0, 0));

	if (result)
	{
		MapDrag* drag = new MapDrag(transferDoc);

		QClipboard* clipboard = QApplication::clipboard();
		if (clipboard == 0)
			delete drag;
		else
		{
			clipboard->setData(drag, QClipboard::Clipboard);

			QListViewItem* currentTileItem = m_lvTiles->currentItem();
			m_doc->setCurrentTile(0);
			for (int r = area.y(); r <= area.bottom(); ++r)
			{
				for (int c = area.x(); c <= area.right(); ++c)
				{
					m_doc->drawCurrentTileAt(r, c);
				}
			}
			slotTileSelected(currentTileItem);
			m_view->slotDocumentChanged();
		}
	}
	else
		delete transferDoc;
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::copy()
{
	if (!m_doc->selectionArea().isValid()) return;

	MapDocument* transferDoc = new MapDocument();
	transferDoc->init(*m_doc->tileSet());

	bool result = transferDoc->copyData(m_doc->selectionArea(),
	                                    m_doc->currentLayer(),
	                                    *m_doc, QPoint(0, 0));

	if (result)
	{
		MapDrag* drag = new MapDrag(transferDoc);

		QClipboard* clipboard = QApplication::clipboard();
		if (clipboard == 0)
			delete drag;
		else
		{
			clipboard->setData(drag, QClipboard::Clipboard);
		}
	}
	else
		delete transferDoc;
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::paste()
{
	if (!m_doc->selectionArea().isValid()) return;

	QClipboard* clipboard = QApplication::clipboard();
	if (clipboard == 0) return;

	QMimeSource* source = clipboard->data(QClipboard::Clipboard);

	if (MapDrag::canDecode(source))
	{
		MapDocument* doc = new MapDocument();
		doc->init(*m_doc->tileSet());
		if (MapDrag::decode(source, doc))
		{
			m_doc->copyData(QRect(0, 0, doc->numCols(), doc->numRows()),
			                      0, *doc, m_doc->selectionArea().topLeft(),
			                      false, false);
			m_view->slotDocumentChanged();
		}
		else
			qWarning("MapDrag::decode failed");

		delete doc;
	}
	else
		qWarning("MapDrag cannot decode clipboard");
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::selectAll()
{
	m_doc->setSelectionArea(QRect(0, 0, m_doc->numCols(), m_doc->numRows()));
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

QString MapEditor::fileName() const
{
	return m_doc->fileName();
}

///////////////////////////////////////////////////////////////////////////////

QString MapEditor::fileFilter() const
{
	return m_doc->fileFilter();
}

///////////////////////////////////////////////////////////////////////////////

bool MapEditor::isModified() const
{
	return m_doc->isModified();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::showGrid(bool show)
{
	m_view->showGrid(show);
}

///////////////////////////////////////////////////////////////////////////////

bool MapEditor::showsGrid() const
{
	return m_view->showsGrid();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::updateLayerSelector()
{
	QListViewItem* item = m_lvLayers->selectedItem();
	QString layerNumber = item != 0 ? item->text(1) : QString::null;
	item = 0;

	m_lvLayers->clear();

	TileSet* tileset = m_doc->tileSet();
	if (tileset == 0) return;

	m_lvLayers->setSorting(1);
	for (uint i = 0; i < m_doc->layerCount(); ++i)
	{
		int groupIndex = m_doc->groupForLayer(i);
		if (groupIndex == -1)
		{
			qWarning("MapEditor: getting groupIndex -1 for layer %d of %s",
			         i,
			         (m_doc->fileName().isEmpty() ? "Untitled" : m_doc->fileName().latin1()));
		}
		else
		{
			TileGroup* group = tileset->group(groupIndex);
			if(group != 0)
			{
				QString number;
				number.sprintf("%03d", i);

				//QListViewItem* newItem = new QListViewItem(m_lvLayers, number, group->name());
				QCheckListItem* newItem =
					new QCheckListItem(m_lvLayers, QString::null, QCheckListItem::CheckBox);
				newItem->setText(1, number);
				newItem->setText(2, group->name());
				newItem->setOn(m_doc->isLayerVisible(i));
				if (layerNumber == number) item = newItem;
			}
		}
	}

	m_lvLayers->sort();
	m_lvLayers->setSorting(-1);

	if (item != 0)
		m_lvLayers->setSelected(item, true);
	else if (m_lvLayers->firstChild() != 0)
		m_lvLayers->setSelected(m_lvLayers->firstChild(), true);
	else
		slotLayerSelected(0);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::createTools()
{
	m_toolMenu   = new QPopupMenu(m_mainWindow, "Mapeditor ToolMenu");
	m_toolMenuID = m_mainWindow->menuBar()->insertItem("Tools", m_toolMenu);
	m_toolBar    = new QToolBar(m_mainWindow, "Mapeditor ToolBar");
	m_toolBar->setLabel(tr("Tools"));

	QActionGroup* actions = new QActionGroup(this, "Mapeditor ToolActions");
	QObject::connect(actions, SIGNAL(selected(QAction*)),
	                 this, SLOT(slotToolAction(QAction*)));

	QPixmap selectIcon(select_);
	m_actSelectTool = new QAction(tr("Selecting areas"),
	                              QIconSet(selectIcon),
	                              tr("Select"),
	                              QKeySequence("Ctrl+Shift+S"),
	                              actions, "select tool action", true);
	m_actSelectTool->addTo(m_toolMenu);
	m_actSelectTool->addTo(m_toolBar);
	QPixmap pointIcon(point);
	m_actPointTool = new QAction(tr("Drawing points freely"),
	                             QIconSet(pointIcon),
	                             tr("Freehand"),
	                             QKeySequence("Ctrl+Shift+F"), actions,
	                             "point tool action", true);
	m_actPointTool->addTo(m_toolMenu);
	m_actPointTool->addTo(m_toolBar);
	QPixmap lineIcon(line);
	m_actLineTool = new QAction(tr("Drawing lines"),
	                            QIconSet(lineIcon),
	                            tr("Line"),
	                            QKeySequence("Ctrl+Shift+L"), actions,
	                            "line tool action", true);
	m_actLineTool->addTo(m_toolMenu);
	m_actLineTool->addTo(m_toolBar);
	QPixmap rectangleIcon(rectangle);
	m_actRectangleTool = new QAction(tr("Drawing rectangles"),
	                                 QIconSet(rectangleIcon),
	                                 tr("Rectangle"),
	                                 QKeySequence("Ctrl+Shift+R"),
	                                 actions, "rectangle tool action", true);
	m_actRectangleTool->addTo(m_toolMenu);
	m_actRectangleTool->addTo(m_toolBar);

	m_actPointTool->toggle();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::createContextMenu()
{
	m_contextMenu = new QPopupMenu(this, "mapeditor-context-menu");

	QAction* actCut = new QAction(tr("Cut"), QPixmap(cutpix),
	                              tr("Cut"), 0, this);
	actCut->addTo(m_contextMenu);
	QObject::connect(actCut, SIGNAL(activated()), this, SLOT(slotCut()));

	QAction* actCopy = new QAction(tr("Copy"), QPixmap(copypix),
	                               tr("Copy"), 0, this);
	actCopy->addTo(m_contextMenu);
	QObject::connect(actCopy, SIGNAL(activated()), this, SLOT(slotCopy()));

	QAction* actPaste = new QAction(tr("Paste"), QPixmap(pastepix),
	                                tr("Paste"), 0, this);
	actPaste->addTo(m_contextMenu);
	QObject::connect(actPaste, SIGNAL(activated()), this, SLOT(slotPaste()));
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotTileGroupChanged(int index)
{
	TileSet* tileset = m_doc->tileSet();
	if (tileset == 0) return;

	TileGroup* tilegroup = tileset->group(index);
	if (tilegroup == 0) return;

	m_lvTiles->clear();

	for (uint i = 0; i < tilegroup->count(); ++i)
	{
		QPixmap* pix = tilegroup->pixmap(i);
		if (pix == 0) continue;

		QString number;
		number.sprintf("%03d", i);
		QListViewItem* item = new QListViewItem(m_lvTiles, number);
		item->setPixmap(1, *pix);
	}

	m_doc->setCurrentTile(0);

	m_lvTiles->setSelected(m_lvTiles->firstChild(), true);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotTileSelected(QListViewItem* item)
{
	int index = 0;
	if (item != 0)
	{
		bool ok;
		index = item->text(0).toInt(&ok);
		if (!ok || index < 0) index = 0;
	}

	m_doc->setCurrentTile(index);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotLayerSelected(QListViewItem* item)
{
	TileSet* tileset = m_doc->tileSet();
	if (tileset == 0) return;

	m_btnAddLayer->setEnabled(m_doc->layerCount() < tileset->count());

	if (item == 0)
	{
		m_btnRemoveLayer->setEnabled(false);
		m_btnLayerUp->setEnabled(false);
		m_btnLayerDown->setEnabled(false);
		m_spAlpha->setEnabled(false);
		m_spBrightness->setEnabled(false);
		m_lvTiles->clear();
	}
	else
	{
		m_btnRemoveLayer->setEnabled(true);
		m_btnLayerUp->setEnabled(item != m_lvLayers->firstChild());
		m_btnLayerDown->setEnabled(item->itemBelow() != 0);

		bool ok;
		int layerIndex = item->text(1).toInt(&ok);
		if (!ok) return;

		m_doc->setCurrentLayer(layerIndex);
		slotTileGroupChanged(m_doc->groupForLayer(layerIndex));
		
		m_spAlpha->setEnabled(true);
		m_spBrightness->setEnabled(true);
		int alpha = 0;
		int brightness = 0;
		m_doc->getLayerImageParameters(alpha, brightness);
		m_spAlpha->setValue(alpha);
		m_spBrightness->setValue(brightness);
		m_cbBackground->setChecked(m_doc->isLayerBackground(layerIndex));
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotLayerClicked(QListViewItem* item, const QPoint&, int col)
{
	if (item == 0 || col != 0) return;

	TileSet* tileset = m_doc->tileSet();
	if (tileset == 0) return;

	bool ok;
	int layerIndex = item->text(1).toInt(&ok);
	if (!ok) return;

	QCheckListItem* checkItem = static_cast<QCheckListItem*>(item);
	m_doc->setLayerVisible(layerIndex, checkItem->isOn());
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotLayerHeaderClicked(int col)
{
	if (col != 0) return;

	QListViewItem* item = m_lvLayers->firstChild();
	for (; item != 0; item = item->nextSibling())
	{
		QCheckListItem* checkItem = static_cast<QCheckListItem*>(item);
		checkItem->setOn(true);

		bool ok;
		int layerIndex = item->text(1).toInt(&ok);
		if (!ok) continue;

		m_doc->setLayerVisible(layerIndex, true);
	}
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotAddLayer()
{
	TileSet* tileset = m_doc->tileSet();
	if (tileset == 0) return;

	QStringList groups;
	for (uint i = 0; i < tileset->count(); ++i)
	{
		// allow only one background
		if (i == 0 && m_doc->groupForLayer(0) == 0) continue;

		TileGroup* group = tileset->group(i);
		if (group == 0) continue;

		groups.append(group->name());
	}

	if (groups.count() == 0) return;

	bool ok;
	QString groupName =
		QInputDialog::getItem(tr("Add layer"), tr("Select tile category"),
	                          groups, 0, false, &ok, this);

	if (!ok) return;

	for (uint i = 0; i < tileset->count(); ++i)
	{
		TileGroup* group = tileset->group(i);
		if (group == 0) continue;

		if (group->name() == groupName)
		{
			QListViewItem* item = m_lvLayers->selectedItem();
			if (item == 0)
				m_doc->insertLayer(i);
			else
			{
				bool ok;
				int index = item->text(1).toInt(&ok);
				if (!ok || index <= 0)
					m_doc->insertLayer(i);
				else
					m_doc->insertLayer(i, index);
			}

			updateLayerSelector();
			m_view->slotDocumentChanged();
			return;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotRemoveLayer()
{
	QListViewItem* item = m_lvLayers->selectedItem();
	if (item == 0) return;

	bool ok;
	int index = item->text(1).toInt(&ok);
	if (!ok || index < 0) return;

	m_doc->removeLayer(index);

	updateLayerSelector();
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotMoveLayerUp()
{
	QListViewItem* item = m_lvLayers->selectedItem();
	if (item == 0) return;

	bool ok;
	int index = item->text(1).toInt(&ok);
	if (!ok || index < 1) return;
	QString groupName = item->text(2);

	m_doc->moveLayer(index, index-1);

	updateLayerSelector();

	QString layer;
	layer.sprintf("%03d", (index-1));
	item = m_lvLayers->findItem(layer, 0);
	if (item != 0)
		m_lvLayers->setCurrentItem(item);

	slotTileGroupChanged(m_doc->groupForLayer(index-1));
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotMoveLayerDown()
{
	QListViewItem* item = m_lvLayers->selectedItem();
	if (item == 0) return;

	bool ok;
	int index = item->text(1).toInt(&ok);
	if (!ok || index < 0) return;

	m_doc->moveLayer(index, index+1);

	updateLayerSelector();

	QString layer;
	layer.sprintf("%03d", (index+1));
	item = m_lvLayers->findItem(layer, 0);
	if (item != 0)
		m_lvLayers->setCurrentItem(item);

	slotTileGroupChanged(m_doc->groupForLayer(index+1));
	m_view->slotDocumentChanged();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotLayerPropertiesChanged()
{
	int alpha;
	int brightness;
	m_doc->getLayerImageParameters(alpha, brightness);
	
	// if line step from 255 down, add 1 to be on multiples of 16
	if (alpha == 255 && m_spAlpha->value() == 239)
	{
		alpha = 240;
		m_spAlpha->blockSignals(true);
		m_spAlpha->setValue(240);
		m_spAlpha->blockSignals(false);
	}
	else
		alpha = m_spAlpha->value();
	if (brightness == 255 && m_spBrightness->value() == 239)
	{
		brightness = 240;
		m_spBrightness->blockSignals(true);
		m_spBrightness->setValue(240);
		m_spBrightness->blockSignals(false);
	}
	else
		brightness = m_spBrightness->value();
	
	m_doc->setLayerImageParameters(alpha, brightness);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotLayerBackgroundToggled(bool on)
{
	m_doc->setLayerBackground(m_doc->currentLayer(), on);
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotViewShowing()
{
	m_mainWindow->menuBar()->setItemVisible(m_toolMenuID, true);
	m_toolBar->show();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotViewHiding()
{
	m_mainWindow->menuBar()->setItemVisible(m_toolMenuID, false);
	m_toolBar->hide();
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotToolAction(QAction* action)
{
	if (action == m_actSelectTool)
	{
		m_view->activateTool(MapView::Select);
	}
	else if (action == m_actPointTool)
	{
		m_view->activateTool(MapView::Point);
	}
	else if (action == m_actLineTool)
	{
		m_view->activateTool(MapView::Line);
	}
	else if (action == m_actRectangleTool)
	{
		m_view->activateTool(MapView::Rectangle);
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapEditor::slotContextMenu(const QPoint& pos)
{
	qDebug("Context menu at (%d, %d)", pos.x(), pos.y());
	m_contextMenu->popup(pos);
}


// End of file
