/*
 *  Moblin Background Management Application
 *  Source: capplets/background/wp-item.c
 *  Author: Todd Brandt <todd.e.brandt@intel.com>
 *  Copyright 2007-2008 Intel Corp (moblin.org)
 *
 *  Based on capplets/background/gnome-wp-item.c from
 *  gnome-control-center-2.18.1: Rodney Dawes <dobey@ximian.com>
 *  Copyright 2003-2006 Novell, Inc. (www.novell.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>

#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <gnome.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include "wp-item.h"
#include "wp-utils.h"

extern gchar *master_options;
extern gchar *master_color_value;

MoblinWPItem * moblin_wp_item_new (const gchar * filename,
				 GHashTable * wallpapers,
				 GnomeThumbnailFactory * thumbnails) {
  MoblinWPItem * item = NULL;
  GConfClient * client;

  client = gconf_client_get_default ();

  item = g_new0 (MoblinWPItem, 1);
  
  item->filename = gnome_vfs_unescape_string_for_display (filename);

  item->fileinfo = moblin_wp_info_new (item->filename, thumbnails);

  item->pixbuf = NULL;

  if (item->fileinfo != NULL &&
      !strncmp (item->fileinfo->mime_type, "image/", strlen ("image/"))) {
    if (item->name == NULL) {
      if (g_utf8_validate (item->fileinfo->name, -1, NULL))
	item->name = g_strdup (item->fileinfo->name);
      else
	item->name = g_filename_to_utf8 (item->fileinfo->name, -1, NULL,
					 NULL, NULL);
    }
    moblin_wp_item_update_description (item);
     
    g_hash_table_insert (wallpapers, item->filename, item);
  } else {
    moblin_wp_item_free (item);
    item = NULL;
  }

  g_object_unref (client);

  return item;
}

void moblin_wp_item_free (MoblinWPItem * item) {
  if (item == NULL) {
    return;
  }

  g_free (item->name);
  g_free (item->filename);
  g_free (item->description);

  moblin_wp_info_free (item->fileinfo);

  gtk_tree_row_reference_free (item->rowref);

  item = NULL;
}

static void collect_save_options (GdkPixbuf * pixbuf,
				  gchar *** keys,
				  gchar *** vals,
				  gint width,
				  gint height) {
  gchar ** options;
  gint n, count;

  count = 0;

  options = g_object_get_qdata (G_OBJECT (pixbuf),
				g_quark_from_static_string ("gdk_pixbuf_options"));
  if (options) {
    for (n = 0; options[2 * n]; n++) {
      ++count;
      
      *keys = g_realloc (*keys, sizeof (gchar *) * (count + 1));
      *vals = g_realloc (*vals, sizeof (gchar *) * (count + 1));

      (*keys)[count - 1] = g_strdup (options[2 * n]);
      (*vals)[count - 1] = g_strdup (options[2 * n + 1]);
      
      (*keys)[count] = NULL;
      (*vals)[count] = NULL;
    }
  }
  ++count;

  *keys = g_realloc (*keys, sizeof (gchar *) * (count + 1));
  *vals = g_realloc (*vals, sizeof (gchar *) * (count + 1));

  (*keys)[count - 1] = g_strdup ("tEXt::Thumb::Image::Width");
  (*vals)[count - 1] = g_strdup_printf ("%d", width);

  (*keys)[count] = NULL;
  (*vals)[count] = NULL;

  ++count;

  *keys = g_realloc (*keys, sizeof (gchar *) * (count + 1));
  *vals = g_realloc (*vals, sizeof (gchar *) * (count + 1));

  (*keys)[count - 1] = g_strdup ("tEXt::Thumb::Image::Height");
  (*vals)[count - 1] = g_strdup_printf ("%d", height);

  (*keys)[count] = NULL;
  (*vals)[count] = NULL;
}

#define LIST_IMAGE_WIDTH 64
extern GdkColor *master_color;

GdkPixbuf * moblin_wp_item_get_thumbnail (MoblinWPItem * item,
					 GnomeThumbnailFactory * thumbs) {
  GdkPixbuf *pixbuf = NULL;
  GdkPixbuf *bgpixbuf = NULL;
  GdkPixbuf *scaled = NULL;
  gint sw=0, sh=0, bw=0, bh=0;
  gint pw=0, ph=0, tw=0, th=0;
  gdouble ratio;
  GError *error = NULL;

    if(item->fileinfo == NULL)
	item->fileinfo = moblin_wp_info_new (item->filename, thumbs);

    /*
     Get the size of the screen and calculate our aspect ratio divisor
     We do this, so that images are thumbnailed as they would look on
     the screen in reality
    */
    sw = gdk_screen_get_width (gdk_screen_get_default ());
    sh = gdk_screen_get_height (gdk_screen_get_default ());
    ratio = (gdouble) sw / (gdouble) LIST_IMAGE_WIDTH;
    bw = sw / ratio;
    bh = sh / ratio;
    bgpixbuf = moblin_wp_pixbuf_new_solid (master_color, bw, bh);

    if (!strcmp (item->filename, "(none)")) {
	return bgpixbuf;
    } else {
	gchar *escaped_path, *thumbnail_filename;

	escaped_path = gnome_vfs_escape_path_string (item->filename);
	thumbnail_filename = gnome_thumbnail_factory_lookup (thumbs,
			escaped_path, item->fileinfo->mtime);

	if (thumbnail_filename == NULL)
	{
#if 1
	    pixbuf = gdk_pixbuf_new_from_file_at_size(item->fileinfo->uri, bw, bh, &error);
	    if(error) {
		g_warning ("gdk_pixbuf_new_from_file_at_size error: %s", error->message);
		g_error_free (error);
	    }
#else
	    pixbuf = gnome_thumbnail_factory_generate_thumbnail (thumbs,
			escaped_path, item->fileinfo->mime_type);
#endif
	    if(pixbuf)
	    {
		gnome_thumbnail_factory_save_thumbnail (thumbs, pixbuf,
			escaped_path, item->fileinfo->mtime);
		g_object_unref (pixbuf);
		pixbuf = NULL;
		thumbnail_filename = gnome_thumbnail_factory_lookup (thumbs,
			escaped_path, item->fileinfo->mtime);
	    }
	}

	if (thumbnail_filename != NULL) {
	    pixbuf = gdk_pixbuf_new_from_file (thumbnail_filename, NULL);
	    if (pixbuf != NULL) {
		g_free (item->fileinfo->thumburi);
		item->fileinfo->thumburi = thumbnail_filename;
		thumbnail_filename = NULL;
	    }
	    g_free (thumbnail_filename);
	}
        g_free (escaped_path);
    }

    if (pixbuf != NULL) {
	const gchar * w_val, * h_val;

	w_val = gdk_pixbuf_get_option (pixbuf, "tEXt::Thumb::Image::Width");
	h_val = gdk_pixbuf_get_option (pixbuf, "tEXt::Thumb::Image::Height");
	if (item->width <= 0 || item->height <= 0) {
	    if (w_val && h_val) {
		item->width = atoi (w_val);
		item->height = atoi (h_val);
	    } else {
		gchar ** keys = NULL;
		gchar ** vals = NULL;

		gdk_pixbuf_get_file_info (item->filename, 
                                  &item->width, &item->height); 
		collect_save_options (pixbuf, &keys, &vals, item->width, item->height);
		gdk_pixbuf_savev (pixbuf, item->fileinfo->thumburi, "png",
			  keys, vals, NULL);

		g_strfreev (keys);
		g_strfreev (vals);
	    }
	}

	pw = gdk_pixbuf_get_width (pixbuf);
	ph = gdk_pixbuf_get_height (pixbuf);

	if (item->width <= bw && item->height <= bh)
	    ratio = 1.0;

	tw = item->width / ratio;
	th = item->height / ratio;

	scaled = moblin_wp_pixbuf_tile (pixbuf, bgpixbuf, tw, th);
	g_object_unref (pixbuf);
    }

    g_object_unref (bgpixbuf);

    return scaled;
}

void 
moblin_wp_item_update_description (MoblinWPItem * item)
{
    g_free (item->description);

    if (item->fileinfo==NULL) {
	item->description = g_markup_printf_escaped ("<b>%s</b>\n"
	    "<i>Thumbnail loading...</i>", item->name);
    } else if(!strcmp (item->filename, "(none)")) {
	item->description = g_strdup_printf ("<b>%s</b>", item->name);
    } else {
	gchar * info;

	/*
	Translators: This message is used to render the type and size of the
	background images in gnome-background-properites. The first "%s" will
	be replaced by the image type, and the two "%d %s"s will be replaced
	with the images' dimensions. For example, in US English, this may be
	displayed as "JPEG Image, 1600 pixels x 1200 pixels".

	Do not translate the "background size|" type. Remove it from the
	translation.
	*/
    
	if((item->width > 0)&&(item->height > 0))
	{
	    info = g_strdup_printf (Q_("background size|%s, %d %s x %d %s"),
			    gnome_vfs_mime_get_description (item->fileinfo->mime_type),
			    item->width,
			    ngettext ("pixel", "pixels", item->width),
			    item->height,
			    ngettext ("pixel", "pixels", item->height));

	    item->description = g_markup_printf_escaped ("<b>%s</b>\n"
		"%s",
		item->name,
		info);

	    g_free (info);
	}
	else
	{
	    item->description = g_markup_printf_escaped ("<b>%s</b>\n"
		"%s", item->name,
		gnome_vfs_mime_get_description(item->fileinfo->mime_type));
	}
    }
}
