/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* bt_llist.c: APR-based linked list implementation */

#include <apr.h>
#include <apr_pools.h>
#include <apr_errno.h>

#include <stdint.h>
#include <stdio.h>

#include <libbtutil/const.h>
#include <libbtutil/types/bt_llist.h>

bt_llist* bt_llist_new(apr_pool_t* p, uint32_t max) {
    bt_llist* rv;
    apr_pool_t* subpool;
    apr_status_t ret;
    char err[BT_SHORT_STRING];
    
    if((ret = apr_pool_create(&subpool, p)) != APR_SUCCESS) {
        fprintf(
            stderr, "bt_llist_new: failure creating pool: %s\n",
            apr_strerror(ret, err, sizeof(err))
        );
        return NULL;
    }
    
    rv = apr_pcalloc(subpool, sizeof(bt_llist));
    rv->p = subpool;
    rv->max = max;
    
    return rv;
}

void bt_llist_destroy(bt_llist* list) {
    apr_pool_destroy(list->p);
}

bt_llist_e* bt_llist_e_new(bt_llist* list, void* data, apr_size_t size) {
    /* fail if there's no room for another entry */
    if(list->max && (list->n >= list->max))
        return NULL;
    
    bt_llist_e* rv = apr_pcalloc(list->p, sizeof(bt_llist_e));
    rv->s = size;
    rv->d = data;
    rv->l = list;
    return rv;
}

void bt_llist_e_unlink(bt_llist_e* e) {
    if(e->ln) {
        if(e->l->first == e)
            e->l->first = e->next;
    
        if(e->l->last == e)
            e->l->last = e->prev;
    
        if(e->prev)
            e->prev->next = e->next;
    
        if(e->next)
            e->next->prev = e->prev;
    
        e->prev = NULL;
        e->next = NULL;
        
        e->ln = 0;
        e->l->n--;
    }
    
    return;
}

apr_status_t bt_llist_e_link_first(bt_llist_e* e) {
    if(e->l->max && ((e->l->n - e->ln) >= e->l->max))
        return APR_ENOMEM;
    
    bt_llist_e_unlink(e);
    e->next = e->l->first;
    e->l->first = e;

    if(e->next)
        e->next->prev = e;
    else
        e->l->last = e;

    e->ln = 1;
    e->l->n++;

    return APR_SUCCESS;
}

apr_status_t bt_llist_e_link_last(bt_llist_e* e) {
    if(e->l->max && ((e->l->n - e->ln) >= e->l->max))
        return APR_ENOMEM;
    
    bt_llist_e_unlink(e);
    e->prev = e->l->last;
    e->l->last = e;
    
    if(e->prev)
        e->prev->next = e;
    else
        e->l->first = e;

    e->ln = 1;
    e->l->n++;
    
    return APR_SUCCESS;
}

apr_status_t bt_llist_e_link_before(bt_llist_e* e, bt_llist_e* before) {
    if(!before->ln)
        return APR_NOTFOUND;
    
    if(before->l != e->l)
        return APR_EBADF;
    
    if(e->l->first == before)
        return bt_llist_e_link_first(e);
    
    if(e->l->max && ((e->l->n - e->ln) >= e->l->max))
        return APR_ENOMEM;

    bt_llist_e_unlink(e);
    e->prev = before->prev;
    e->next = before;
    before->prev->next = e;
    before->prev = e;
    e->l->n++;
    
    return APR_SUCCESS;
}

apr_status_t bt_llist_e_link_after(bt_llist_e* e, bt_llist_e* after) {
    if(!after->ln)
        return APR_NOTFOUND;
    
    if(after->l != e->l)
        return APR_EBADF;
    
    if(e->l->last == after)
        return bt_llist_e_link_last(e);
    
    if(e->l->max && ((e->l->n - e->ln) >= e->l->max))
        return APR_ENOMEM;

    bt_llist_e_unlink(e);
    e->next = after->next;
    e->prev = after;
    after->next->prev = e;
    after->next = e;
    e->l->n++;
    
    return APR_SUCCESS;
}

apr_status_t bt_llist_add_last(bt_llist* l, void* data, apr_size_t len) {
    bt_llist_e* e;
    if((e = bt_llist_e_new(l, data, len)))
        return bt_llist_e_link_last(e);
    else
        return APR_ENOMEM;
}

apr_status_t bt_llist_add_first(bt_llist* l, void* data, apr_size_t len) {
    bt_llist_e* e;
    if((e = bt_llist_e_new(l, data, len)))
        return bt_llist_e_link_first(e);
    else
        return APR_ENOMEM;
}

apr_status_t bt_llist_add_before(
    bt_llist* l, bt_llist_e* before, void* data, apr_size_t len
) {
    bt_llist_e* e;
    if((e = bt_llist_e_new(l, data, len)))
        return bt_llist_e_link_before(e, before);
    else
        return APR_ENOMEM;
}

apr_status_t bt_llist_add_after(
    bt_llist* l, bt_llist_e* after, void* data, apr_size_t len
) {
    bt_llist_e* e;
    if((e = bt_llist_e_new(l, data, len)))
        return bt_llist_e_link_after(e, after);
    else
        return APR_ENOMEM;
}
