/*
 * $Id: mod_cband.c,v 1.16 2005/12/07 00:47:39 dembol Exp $
 *
 * mod_cband - A per-user, per-virtualhost and per-destination bandwidth limiter for the Apache HTTP Server Version 2
 *
 * Copyright (c) 2005 Lukasz Dembinski <dembol@cband.linux.pl>
 * All Rights Reserved
 * 
 * Date:	2005/12/06
 * Info:	mod_cband Apache 2 module
 * Contact:	mailto: <dembol@cband.linux.pl>
 * Version:	0.9.6.0
 *
 * Authors:
 * - Lukasz Dembinski <dembol@cband.linux.pl>
 * - Sergey V. Beduev <shaman@interdon.net>
 * - Kyle Poulter <kyle@unixowns.us>
 * - Adam Dawidowski <drake@oomkill.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *	     
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *		     
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *					 
 */

#include "httpd.h"
#include "http_main.h"
#include "http_core.h"
#include "http_config.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_log.h"
#include "apr_strings.h"
#include "apr_file_io.h"
#include "apr_file_info.h"
#include "apr_signal.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_pools.h"
#include "util_filter.h"
#include "util_cfgtree.h"
#include <sys/shm.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <unistd.h>

#include "mod_cband.h"

static mod_cband_config_header *config = NULL;
static const char mod_cband_filter_name[] = "CBAND_FILTER";

module AP_MODULE_DECLARE_DATA cband_module;

ap_filter_rec_t *mod_cband_output_filter_handle;
apr_status_t patricia_cleanup(void *);
const void mod_cband_signal_handler(int);
unsigned long mod_cband_get_period_sec(char *period);
unsigned long mod_cband_get_limit_kb(char *limit, int *mult);

mod_cband_shmem_data *mod_cband_shmem_init(void)
{
    int shm_id;
    mod_cband_shmem_data *data;

    shm_id = shmget(IPC_PRIVATE, sizeof(mod_cband_shmem_data), IPC_CREAT | 0666);
    data = (mod_cband_shmem_data *)shmat(shm_id, 0, 0);
    memset(data, 0, sizeof(mod_cband_shmem_data));
    data->total_last_refresh = (unsigned long)(apr_time_now() / 1000000);
    data->total_last_time    = (unsigned long)(apr_time_now() / 1000000);
    
    return data;
}

void mod_cband_sem_init(int sem_id)
{
    union semun arg;
    unsigned short values[1];
    
    values[0] = 1;
    arg.val   = 1;
    arg.array = values;
    
    semctl(sem_id, 0, SETALL, arg);    
    
    /* 
     * We should also set owner of the semaphore ...
     */
}

void mod_cband_sem_remove(int sem_id)
{
    union semun arg;

    arg.val   = 0;    
    semctl(sem_id, 0, IPC_RMID, arg);    
}

void mod_cband_sem_down(int sem_id)
{
    struct sembuf sops;
    
    sops.sem_num  = 0;
    sops.sem_op   = -1;
    sops.sem_flg  = SEM_UNDO;
    
    semop(sem_id, &sops, 1);
}

void mod_cband_sem_up(int sem_id)
{
    struct sembuf sops;
        
    sops.sem_num  = 0;
    sops.sem_op   = 1;
    sops.sem_flg  = SEM_UNDO;
    
    semop(sem_id, &sops, 1);
}

/**
 * get virtualhost entry or create new one
 */
mod_cband_virtualhost_config_entry *mod_cband_get_virtualhost_entry_(char *virtualhost, apr_port_t port, unsigned line, int create)
{
    mod_cband_virtualhost_config_entry *entry;
    mod_cband_virtualhost_config_entry *new_entry;
    int i;

    if (virtualhost == NULL || config == NULL)
	return NULL;
    
    entry = config->next_virtualhost;
    
    while(entry != NULL) {
	if (!strcmp(entry->virtual_name, virtualhost) && (port == entry->virtual_port) && (line == entry->virtual_defn_line))
	    return entry;
    
	if (entry->next == NULL)
	    break;
	    
	entry = entry->next;
    }

    if (create) {
	new_entry = apr_palloc(config->p, sizeof(mod_cband_virtualhost_config_entry));
	
	memset(new_entry, 0, sizeof(mod_cband_virtualhost_config_entry));
	
	new_entry->virtual_name      = virtualhost;
	new_entry->virtual_defn_line = line;
	new_entry->virtual_port      = port;
	new_entry->shmem_data	     = mod_cband_shmem_init();
	
	for (i = 0; i < DST_CLASS; i++)
	    new_entry->virtual_class_limit_mult[i] = 1024;
	
	if (entry == NULL)
	    config->next_virtualhost = new_entry;
	else
	    entry->next = new_entry;
	
	return new_entry;    
    }
    
    return NULL;    
}
 
mod_cband_virtualhost_config_entry *mod_cband_get_virtualhost_entry(server_rec *s, ap_conf_vector_t *module_config, int create)
{
    char *virtualhost;
    apr_port_t port;
    unsigned line;

    if (s == NULL)
	return NULL;

    virtualhost = s->server_hostname;
    port        = s->port;
    line        = s->defn_line_number;

    return mod_cband_get_virtualhost_entry_(virtualhost, port, line, create);
}

/**
 * get user entry or create new one
 */
mod_cband_user_config_entry *mod_cband_get_user_entry(char *user, ap_conf_vector_t *module_config, int create)
{
    mod_cband_user_config_entry *entry;
    mod_cband_user_config_entry *new_entry;
    int i;

    if (user == NULL || config == NULL)
	return NULL;
    
    entry = config->next_user;
    
    while(entry != NULL) {
	if (!strcmp(entry->user_name, user))
	    return entry;
    
	if (entry->next == NULL)
	    break;
	    
	entry = entry->next;
    }
    
    if (create) {
	new_entry = apr_palloc(config->p, sizeof(mod_cband_user_config_entry));
	
	memset(new_entry, 0, sizeof(mod_cband_user_config_entry));
	
	new_entry->user_name  = user;
	new_entry->shmem_data = mod_cband_shmem_init();
	
	for (i = 0; i < DST_CLASS; i++)
	    new_entry->user_class_limit_mult[i] = 1024;

	if (entry == NULL)
	    config->next_user = new_entry;
	else
	    entry->next = new_entry;
	
	return new_entry;    
    }
	
    return NULL;    
}

/**
 * get class entry or create new one
 */
mod_cband_class_config_entry *mod_cband_get_class_entry(char *dest, ap_conf_vector_t *module_config, int create)
{
    mod_cband_class_config_entry *entry;
    mod_cband_class_config_entry *new_entry;

    if (dest == NULL || config == NULL)
	return NULL;
    
    entry = config->next_class;
    
    while(entry != NULL) {
	if (!strcmp(entry->class_name, dest))
	    return entry;
    
	if (entry->next == NULL)
	    break;
	    
	entry = entry->next;
    }

    if (create) {
	new_entry = apr_palloc(config->p, sizeof(mod_cband_class_config_entry));
	
	memset(new_entry, 0, sizeof(mod_cband_class_config_entry));
	
	new_entry->class_name = dest;
    
	if (entry == NULL)
	    config->next_class = new_entry;
	else
	    entry->next = new_entry;
	
	return new_entry;    
    }
    
    return NULL;    
}

int mod_cband_check_duplicate(void *ptr, const char *command, const char *arg, server_rec *s)
{
    if (ptr != NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, s, "Duplicate command '%s %s' for %s:%d, the first has precedence", (char *)command, (char *)arg, s->server_hostname, s->port);
	return 1;
    }
    
    return 0;
}

static const char *mod_cband_set_default_url(cmd_parms *parms, void *mconfig, const char *arg)
{
    if (!mod_cband_check_duplicate(config->default_limit_exceeded, "CBandDefaultExceededURL", arg, parms->server))
	config->default_limit_exceeded = (char *)arg;
      
    return NULL;
}

static const char *mod_cband_set_score_flush_period(cmd_parms *parms, void *mconfig, const char *arg)
{
    if (!mod_cband_check_duplicate((void *)config->score_flush_period, "CBandScoreFlushPeriod", arg, parms->server))
	config->score_flush_period = atol((char *)arg);
      
    return NULL;
}

static const char *mod_cband_set_limit(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_virtualhost_config_entry *entry;

    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandLimit %s', undefined virtualhost name", (char *)arg);
	return NULL;
    }

    if (!mod_cband_check_duplicate((void *)entry->virtual_limit, "CBandLimit", arg, parms->server))
	entry->virtual_limit = mod_cband_get_limit_kb((char *)arg, &entry->virtual_limit_mult);
    
    return NULL;
}

static const char *mod_cband_set_period(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_virtualhost_config_entry *entry;

    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandPeriod %s', undefined virtualhost name", (char *)arg);
	return NULL;
    }

    if (!mod_cband_check_duplicate((void *)entry->refresh_time, "CBandPeriod", arg, parms->server))
	entry->refresh_time = mod_cband_get_period_sec((char *)arg);
    
    return NULL;
}

static const char *mod_cband_set_url(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_virtualhost_config_entry *entry;

#ifdef DEBUG
    fprintf(stderr, "apache2_mod_cband: %s:%d %s:%d %d\n", parms->server->server_hostname, parms->server->port,
    parms->server->addrs->virthost, parms->server->addrs->host_port, parms->server->defn_line_number);
    fflush(stderr);
#endif
    
    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandExceededURL %s', undefined virtualhost name", (char *)arg);
	return NULL;
    }
    
    if (!mod_cband_check_duplicate(entry->virtual_limit_exceeded, "CBandExceededURL", arg, parms->server))
	entry->virtual_limit_exceeded = (char *)arg;
      
    return NULL;
}

static const char *mod_cband_set_speed(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    mod_cband_virtualhost_config_entry *entry;

    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandSpeed', undefined virtualhost name");
	return NULL;
    }

    if (!mod_cband_check_duplicate((void *)entry->shmem_data->max_kbps, "CBandSpeed", arg1, parms->server)) {
    	entry->shmem_data->max_kbps  = entry->shmem_data->curr_kbps  = atol((char *)arg1);
	entry->shmem_data->max_rps   = entry->shmem_data->curr_rps   = atol((char *)arg2);
	entry->shmem_data->max_delay = entry->shmem_data->curr_delay = atol((char *)arg3);
    }
    
    return NULL;
}

static const char *mod_cband_set_exceeded_speed(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    mod_cband_virtualhost_config_entry *entry;

    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandExceededSpeed', undefined virtualhost name");
	return NULL;
    }

    if (!mod_cband_check_duplicate((void *)entry->shmem_data->over_kbps, "CBandExceededSpeed", arg1, parms->server)) {
    	entry->shmem_data->over_kbps  = atol((char *)arg1);
	entry->shmem_data->over_rps   = atol((char *)arg2);
	entry->shmem_data->over_delay = atol((char *)arg3);
    }
    
    return NULL;
}

static const char *mod_cband_set_scoreboard(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_virtualhost_config_entry *entry;
    
    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandScoreboard %s', undefined virtualhost name", (char *)arg);
	return NULL;
    }
    
    if (!mod_cband_check_duplicate(entry->virtual_scoreboard, "CBandScoreboard", arg, parms->server))
	entry->virtual_scoreboard = (char *)arg;
      
    return NULL;
}

static const char *mod_cband_set_user(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_virtualhost_config_entry *entry;
    
    if ((entry = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandUser %s', undefined virtualhost name", (char *)arg);
	return NULL;
    }

    if (mod_cband_get_user_entry((char *)arg, parms->server->module_config, 0) == NULL)  {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandUser %s', undefined user", (char *)arg);
	return NULL;
    }
    
    if (!mod_cband_check_duplicate(entry->virtual_user, "CBandUser", arg, parms->server))
	entry->virtual_user = (char *)arg;
      
    return NULL;
}

static char *username_arg  = NULL;
static char *classname_arg = NULL;
int class_nr = -1;

static const char *mod_cband_user_section(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_user_config_entry *entry;
    const char *endp = ap_strrchr_c(arg, '>');
    char *username;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;

    if (endp == NULL)
	return apr_pstrcat(parms->pool, parms->cmd->name, "> directive missing closing '>'", NULL);

    username = apr_pstrndup(parms->pool, arg, endp - arg);

#ifdef DEBUG
    fprintf(stderr, "apache2_mod_cband: user %s\n", username);
    fflush(stderr);
#endif

    if ((mod_cband_get_user_entry(username, parms->server->module_config, 0)) != NULL) 
	return apr_pstrcat(parms->pool, parms->cmd->name, " ", username, "> duplicate user definition", NULL);

    if ((entry = mod_cband_get_user_entry(username, parms->server->module_config, 1)) == NULL) 
        return ap_walk_config(parms->directive->first_child, parms, parms->context);

    entry->user_name = username;
    username_arg = username;

    return ap_walk_config(parms->directive->first_child, parms, parms->context);
}

static const char *mod_cband_set_user_limit(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_user_config_entry *entry;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 0)) == NULL)
	return NULL;

    if (!mod_cband_check_duplicate((void *)entry->user_limit, "CBandUserLimit", arg, parms->server))
	entry->user_limit = mod_cband_get_limit_kb((char *)arg, &entry->user_limit_mult);

    return NULL;
}

static const char *mod_cband_set_user_period(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_user_config_entry *entry;
    const char *err;

    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;

    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 1)) == NULL) {
	return NULL;
    }

    if (!mod_cband_check_duplicate((void *)entry->refresh_time, "CBandUserPeriod", arg, parms->server))
	entry->refresh_time = mod_cband_get_period_sec((char *)arg);
    
    return NULL;
}

static const char *mod_cband_set_user_url(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_user_config_entry *entry;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 0)) == NULL)
	return NULL;
    
    if (!mod_cband_check_duplicate(entry->user_limit_exceeded, "CBandUserExceededURL", arg, parms->server))
	entry->user_limit_exceeded = (char *)arg;

    return NULL;
}

static const char *mod_cband_set_user_speed(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    mod_cband_user_config_entry *entry;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 0)) == NULL)
	return NULL;

    if (!mod_cband_check_duplicate((void *)entry->shmem_data->max_kbps, "CBandUserSpeed", arg1, parms->server)) {
    	entry->shmem_data->max_kbps  = entry->shmem_data->curr_kbps  = atol((char *)arg1);
	entry->shmem_data->max_rps   = entry->shmem_data->curr_rps   = atol((char *)arg2);
	entry->shmem_data->max_delay = entry->shmem_data->curr_delay = atol((char *)arg3);
    }

    return NULL;
}

static const char *mod_cband_set_user_exceeded_speed(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    mod_cband_user_config_entry *entry;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 0)) == NULL)
	return NULL;

    if (!mod_cband_check_duplicate((void *)entry->shmem_data->over_kbps, "CBandUserExceededSpeed", arg1, parms->server)) {
    	entry->shmem_data->over_kbps  = atol((char *)arg1);
	entry->shmem_data->over_rps   = atol((char *)arg2);
	entry->shmem_data->over_delay = atol((char *)arg3);
    }

    return NULL;
}

static const char *mod_cband_set_user_scoreboard(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_user_config_entry *entry;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_user_entry(username_arg, parms->server->module_config, 0)) == NULL)
	return NULL;
    
    if (!mod_cband_check_duplicate(entry->user_scoreboard, "CBandUserScoreboard", arg, parms->server))
	entry->user_scoreboard = (char *)arg;
      
    return NULL;
}

int mod_cband_check_IP(char *addr)
{
    int i;
    int dig, dot, mask;
    int len;
    
    len = (strlen(addr) > MAX_DST_LEN)?MAX_DST_LEN:strlen(addr);
    
    dig = 0;
    dot = 0;
    for (i = 0; i < len; i++) {
	if (addr[i] >= '0' && addr[i] <= '9') {
	    if (++dig > 3)
		return 0;
	} else
	if (addr[i] == '.') {
	    if (dig == 0)
		return 0;
	
	    if (++dot > 3)
		return 0;
		
	    dig = 0;
	} else
	if (addr[i] == '/') {
	    if (dig == 0)
		return 0;
	
	    mask = atoi(&addr[i + 1]);
	    if (mask < 0 || mask > 32)
		return 0;
	
	    return 1;
	} else
	    return 0;
    }

    return 1;
}

static const char *mod_cband_class_section(cmd_parms *parms, void *mconfig, const char *arg)
{
    mod_cband_class_config_entry *entry;
    const char *endp = ap_strrchr_c(arg, '>');
    char *classname;
    const char *err;
    
    class_nr++;
    
    if (class_nr >= DST_CLASS)
        return ap_walk_config(parms->directive->first_child, parms, parms->context);
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;

    if (endp == NULL)
	return apr_pstrcat(parms->pool, parms->cmd->name, "> directive missing closing '>'", NULL);

    classname = apr_pstrndup(parms->pool, arg, endp - arg);
    
#ifdef DEBUG
    fprintf(stderr, "apache2_mod_cband: class %s (class %d)\n", classname, class_nr);
    fflush(stderr);
#endif

    if ((mod_cband_get_class_entry(classname, parms->server->module_config, 0)) != NULL) 
	return apr_pstrcat(parms->pool, parms->cmd->name, " ", classname, "> duplicate class definition", NULL);

    if ((entry = mod_cband_get_class_entry(classname, parms->server->module_config, 1)) == NULL) 
        return ap_walk_config(parms->directive->first_child, parms, parms->context);

    entry->class_name = classname;
    entry->class_nr   = class_nr;
    classname_arg     = classname;

    return ap_walk_config(parms->directive->first_child, parms, parms->context);
}

static const char *mod_cband_set_class_dst(cmd_parms *parms, void *mconfig, const char *arg)
{
    patricia_node_t *node;
    char class_nr_str[MAX_CLASS_STR_LEN];

    if (config->tree == NULL)
	config->tree = New_Patricia(32); 

    if ((class_nr < DST_CLASS) && (mod_cband_check_IP((char *)arg))) {
#ifdef DEBUG
	fprintf(stderr, "apache2_mod_cband: class dst %s (class %d)\n", (char *)arg, class_nr);
	fflush(stderr);
#endif
    
        sprintf(class_nr_str, "%d", class_nr);
	node = make_and_lookup(config->tree, (char *)arg);
		
	if (node)
    	    node->user1 = apr_pstrdup(config->p, class_nr_str);
    } else {
	if (class_nr >= DST_CLASS) {
	    ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "You can define only %d destination classes", DST_CLASS);
	} else {
	    ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid IP address %s", arg);
	}
    }

    return NULL;
}

static const char *mod_cband_set_class_limit(cmd_parms *parms, void *mconfig, const char *arg, const char *limit)
{
    mod_cband_class_config_entry *entry;
    mod_cband_virtualhost_config_entry *entry_virtual;
    
    if ((entry = mod_cband_get_class_entry((char *)arg, parms->server->module_config, 0)) == NULL) {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandClassLimit %s %s', undefined class name", (char *)arg, (char *)limit);
	return NULL;
    }
    
    if ((entry_virtual = mod_cband_get_virtualhost_entry(parms->server, parms->server->module_config, 1)) == NULL) {
	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandClassLimit %s %s', undefined virtualhost name", (char *)arg, (char *)limit);
	return NULL;
    }

    entry_virtual->virtual_class_limit[entry->class_nr] = mod_cband_get_limit_kb((char *)limit,
	&entry_virtual->virtual_class_limit_mult[entry->class_nr]);
    
    return NULL;
}

static const char *mod_cband_set_user_class_limit(cmd_parms *parms, void *mconfig, const char *arg, const char *limit)
{
    mod_cband_class_config_entry *entry;
    mod_cband_user_config_entry *entry_user;
    const char *err;
    
    if ((err = ap_check_cmd_context(parms, GLOBAL_ONLY)) != NULL)
	return err;
    
    if ((entry = mod_cband_get_class_entry((char *)arg, parms->server->module_config, 0)) == NULL) {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandUserClassLimit %s %s', undefined class name", (char *)arg, (char *)limit);
	return NULL;
    }

    if ((entry_user = mod_cband_get_user_entry(username_arg, parms->server->module_config, 1)) == NULL) {
    	ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server, "Invalid command 'CBandUserClassLimit %s %s', undefined user name", (char *)arg, (char *)limit);
	return NULL;
    }

    entry_user->user_class_limit[entry->class_nr] = mod_cband_get_limit_kb((char *)limit,
	&entry_user->user_class_limit_mult[entry->class_nr]);
    
    return NULL;
}

unsigned long mod_cband_get_period_sec(char *period)
{
    unsigned long val;
    char unit;

    sscanf(period, "%lu%c", &val, &unit);
    
    if (unit == 's' || unit == 'S')
	return val;
    else
    if (unit == 'm' || unit == 'M')
	return val * 60;
    else
    if (unit == 'h' || unit == 'H')
	return val * 60 * 60;
    else
    if (unit == 'd' || unit == 'D')
	return val * 60 * 60 * 24;
    else
    if (unit == 'w' || unit == 'W')
	return val * 60 * 60 * 24 * 7;

    return atol(period);
}

unsigned long mod_cband_get_limit_kb(char *limit, int *mult)
{
    unsigned long val;
    char unit, unit2 = 0;

    sscanf(limit, "%lu%c%c", &val, &unit, &unit2);
    if (unit2 == 'i' || unit2 == 'I')
	*mult = 1024;
    else
	*mult = 1000;

    if (unit == 'k' || unit == 'K')
	return val;
    else
    if (unit == 'm' || unit == 'M')
	return val * *mult;
    else
    if (unit == 'g' || unit == 'G')
	return val * *mult * *mult;

    return atol(limit);
}

char *mod_cband_create_time(apr_pool_t *p, unsigned long sec)
{
    unsigned h, m, s, d, w;
    char period[MAX_PERIOD_LEN];

    s = sec % 60;
    sec /= 60;
    m = sec % 60;
    sec /= 60;
    h = sec % 24;
    sec /= 24;
    d = sec % 7;
    sec /= 7;
    w = sec;

    sprintf(period, "%uW ", w);
    sprintf(period + strlen(period), "%uD ", d);
    sprintf(period + strlen(period), "%02u:%02u:%02u", h, m, s);
	
    return apr_pstrndup(p, period, strlen(period));
}

char *mod_cband_create_traffic_size(apr_pool_t *p, unsigned long kb, char *unit, int mult)
{
    char traffic[MAX_TRAFFIC_LEN];
    char dest_unit[3] = {0, 0, 0};
    float v;
    unsigned long vi;

    if ((unit != "" && unit[0] == 'G') || (unit == "" && kb >= mult*mult)) {
	dest_unit[0] = 'G';
	v = (float)kb / (mult*mult);
    } else
    if ((unit != "" && unit[0] == 'M') || (unit == "" && kb >= mult)) {
	dest_unit[0] = 'M';
	v = (float)kb / mult;
    } else {
	dest_unit[0] = 'K';
	v = kb;
    }
    if (mult == 1024)
	dest_unit[1] = 'i';

    vi = (unsigned long)truncf(v * 100);
    v  = (float)vi / 100;
    
    if (vi % 100)
	sprintf(traffic, "%0.2f%sB", v, dest_unit);
    else
    	sprintf(traffic, "%0.0f%sB", v, dest_unit);

    return apr_pstrndup(p, traffic, strlen(traffic));
}

char *mod_cband_create_period(apr_pool_t *p, unsigned long start, unsigned long refresh)
{
    unsigned sec;

    if (start == 0 || refresh == 0)
	return "never";

    sec = start + refresh;
    sec -= (unsigned long)(apr_time_now() / 1000000);
    
    return mod_cband_create_time(p, sec);
}

static const command_rec mod_cband_cmds[] =
{
  AP_INIT_TAKE1(
      "CBandDefaultExceededURL",
      mod_cband_set_default_url,
      NULL,
      RSRC_CONF,
      "CBandDefaultExceededURL - The URL to redirect when bandwidth is exceeded."
    ),
  
  AP_INIT_TAKE1(
      "CBandScoreFlushPeriod",
      mod_cband_set_score_flush_period,
      NULL,
      RSRC_CONF,
      "CBandScoreFlushPeriod"
    ),
  
  AP_INIT_TAKE1(
      "CBandLimit",
      mod_cband_set_limit,
      NULL,
      RSRC_CONF,
      "CBandLimit - The limit bandwidth in KB for virtualhost."
    ),

  AP_INIT_TAKE1(
      "CBandPeriod",
      mod_cband_set_period,
      NULL,
      RSRC_CONF,
      "CBandPeriod - The time after the scoreboard will be cleared"
    ),

  AP_INIT_TAKE1(
      "CBandExceededURL",
      mod_cband_set_url,
      NULL,
      RSRC_CONF,
      "CBandExceededURL - The URL to redirect when virtualhost's bandwidth is exceeded."
    ),

  AP_INIT_TAKE3 (
      "CBandSpeed",
      mod_cband_set_speed,
      NULL,
      RSRC_CONF,
      "CBandSpeed - Maximal speed for virtualhost."
    ),

  AP_INIT_TAKE3 (
      "CBandExceededSpeed",
      mod_cband_set_exceeded_speed,
      NULL,
      RSRC_CONF,
      "CBandExceededSpeed - Over limit speed for virtualhost."
    ),

  AP_INIT_TAKE1(
      "CBandScoreboard",
      mod_cband_set_scoreboard,
      NULL,
      RSRC_CONF,
      "CBandScoreboard - The path to the virtualhost's scoreboard file."
    ),

  AP_INIT_TAKE1(
      "CBandUser",
      mod_cband_set_user,
      NULL,
      RSRC_CONF,
      "CBandUser - virtualhost user owner."
    ),

  AP_INIT_RAW_ARGS(
      "<CBandUser", 
      mod_cband_user_section, 
      NULL,
      RSRC_CONF,
      "CBandUser section"
    ),

  AP_INIT_TAKE1(
      "CBandUserLimit",
      mod_cband_set_user_limit,
      NULL,
      RSRC_CONF,
      "CBandUserLimit - The limit bandwidth in KB for user."
    ),

  AP_INIT_TAKE1(
      "CBandUserPeriod",
      mod_cband_set_user_period,
      NULL,
      RSRC_CONF,
      "CBandUserPeriod - The time after the scoreboard will be cleared"
    ),
    
  AP_INIT_TAKE1(
      "CBandUserExceededURL",
      mod_cband_set_user_url,
      NULL,
      RSRC_CONF,
      "CBandUserExceededURL - The URL to redirect when user's bandwidth is exceeded."
    ),

  AP_INIT_TAKE3(
      "CBandUserSpeed",
      mod_cband_set_user_speed,
      NULL,
      RSRC_CONF,
      "CBandUserSpeed - Maximal speed for user."
    ),

  AP_INIT_TAKE3(
      "CBandUserExceededSpeed",
      mod_cband_set_user_exceeded_speed,
      NULL,
      RSRC_CONF,
      "CBandUserExceededSpeed - Over limit speed for user."
    ),

  AP_INIT_TAKE1(
      "CBandUserScoreboard",
      mod_cband_set_user_scoreboard,
      NULL,
      RSRC_CONF,
      "CBandUserScoreboard - The path to the user's scoreboard file."
    ),

  AP_INIT_RAW_ARGS(
      "<CBandClass", 
      mod_cband_class_section, 
      NULL,
      RSRC_CONF,
      "CBandClass section"
    ),

  AP_INIT_TAKE1(
      "CBandClassDst",
      mod_cband_set_class_dst,
      NULL,
      RSRC_CONF,
      "CBandClassDst"
    ),

  AP_INIT_TAKE2(
      "CBandClassLimit",
      mod_cband_set_class_limit,
      NULL,
      RSRC_CONF,
      ""
    ),

  AP_INIT_TAKE2(
      "CBandUserClassLimit",
      mod_cband_set_user_class_limit,
      NULL,
      RSRC_CONF,
      ""
    ),

    {NULL}
};

apr_status_t patricia_cleanup(void *p)
{
    patricia_tree_t *t = (patricia_tree_t *) p;
    Clear_Patricia(t,NULL);
    
    return APR_SUCCESS;
}

int mod_cband_get_dst(request_rec *r) 
{
    patricia_node_t *node;
    prefix_t p;
    char *leaf;
	      
    if (config->tree == NULL)
	return -1;
    
    p.bitlen = 32;
    p.ref_count = 0;
    p.family = AF_INET;
    p.add.sin.s_addr = inet_addr(r->connection->remote_ip);
		      
    node = patricia_search_best(config->tree, &p);
				    
    if (node) {
        leaf = node->user1;
					            
        if (leaf) {
#ifdef DEBUG
            fprintf(stderr,"%s leaf %s\n",r->connection->remote_ip,leaf);
            fflush(stderr);
#endif
	    return atoi(leaf);
        }
    }
    
    return -1;
}

int mod_cband_get_score(server_rec *s, char *path, unsigned long long *val, int dst, mod_cband_shmem_data *shmem_data)
{
    if (path == NULL)
	return -1;

    if (dst < 0)
	*val = shmem_data->total_usage.total_bytes;
    else
	*val = shmem_data->total_usage.class_bytes[dst];
    
    return 0;
}

int mod_cband_get_score_all(server_rec *s, char *path, mod_cband_scoreboard_entry *val)
{
    apr_file_t *fd;
    apr_size_t nbuf;
    
    if (path == NULL)
	return -1;
    
    if ((apr_file_open(&fd, path, APR_READ | APR_BINARY, 0, config->p)) != APR_SUCCESS)
	return -1;
    
    nbuf = sizeof(mod_cband_scoreboard_entry);
    apr_file_read(fd, val, &nbuf);
    apr_file_close(fd);
    
    return 0;
}

int mod_cband_save_score(char *path, mod_cband_scoreboard_entry *scoreboard)
{
    apr_file_t *fd;
    apr_size_t nbuf;
    
    if (scoreboard->was_request == 0)
	return -1;

    if ((apr_file_open(&fd, path,
		APR_CREATE | APR_READ | APR_WRITE | APR_BINARY, APR_UREAD | APR_UWRITE, config->p)) != APR_SUCCESS)
	return -1;
   
    apr_file_lock(fd, APR_FLOCK_EXCLUSIVE);
    nbuf = sizeof(mod_cband_scoreboard_entry);
    apr_file_write(fd, scoreboard, &nbuf);
    apr_file_unlock(fd);
    apr_file_close(fd);
    
    return 0;
}

int mod_cband_flush_score(char *path, mod_cband_scoreboard_entry *scoreboard)
{
    if ((path == NULL) || (scoreboard == NULL))
	return -1;
    
    scoreboard->was_request = 1;
	
    if (--(scoreboard->score_flush_count) <= 0) {
	mod_cband_save_score(path, scoreboard);
	scoreboard->score_flush_count = config->score_flush_period;
    }
        
    return 0;
}

int mod_cband_update_score(char *path, unsigned long long *bytes_served, int dst, mod_cband_scoreboard_entry *scoreboard)
{
    scoreboard->total_bytes	   += (unsigned long long)(*bytes_served);
    if (dst >= 0)
	scoreboard->class_bytes[dst] += (unsigned long long)(*bytes_served);

    return 0;
}

int mod_cband_clear_score(mod_cband_scoreboard_entry *scoreboard)
{
    memset(scoreboard, 0, sizeof(mod_cband_scoreboard_entry));    

    return 0;
}

int mod_cband_update_score_cache(server_rec *s)
{
    mod_cband_virtualhost_config_entry *entry = NULL;
    mod_cband_user_config_entry *entry_user = NULL;

    entry = config->next_virtualhost;
    while(entry != NULL) {
        mod_cband_get_score_all(s, entry->virtual_scoreboard, &(entry->shmem_data->total_usage));
        if ((entry = entry->next) == NULL)
	    break;
    }

    entry_user = config->next_user;
    while(entry_user != NULL) {
        mod_cband_get_score_all(s, entry_user->user_scoreboard, &(entry_user->shmem_data->total_usage));
        if ((entry_user = entry_user->next) == NULL)
	    break;
    }
    
    return OK;
}

int mod_cband_save_score_cache(void)
{
    mod_cband_virtualhost_config_entry *entry = NULL;
    mod_cband_user_config_entry *entry_user = NULL;

    entry = config->next_virtualhost;
    while(entry != NULL) {
        mod_cband_save_score(entry->virtual_scoreboard, &(entry->shmem_data->total_usage));
        if ((entry = entry->next) == NULL)
	    break;
    }

    entry_user = config->next_user;
    while(entry_user != NULL) {
        mod_cband_save_score(entry_user->user_scoreboard, &(entry_user->shmem_data->total_usage));
        if ((entry_user = entry_user->next) == NULL)
	    break;
    }
    
    return OK;
}

unsigned long mod_cband_get_start_time(mod_cband_scoreboard_entry *scoreboard)
{
    if (scoreboard == NULL)
	return 0;

    return scoreboard->start_time;
}

int mod_cband_set_start_time(mod_cband_scoreboard_entry *scoreboard, unsigned long start_time)
{
    if (scoreboard == NULL)
	return 0;

    scoreboard->start_time = start_time;
    
    return 0;
}

int mod_cband_get_speed(mod_cband_shmem_data *shmem_data, float *bps, float *rps)
{
    unsigned long time_now;
    unsigned long time_delta;

    *bps   = (unsigned long)0;
    *rps   = (unsigned long)0;
    time_now          = (unsigned long)(apr_time_now() / 1000000);
    time_delta        = time_now - shmem_data->total_last_refresh;

    if (time_delta > 0) {
        *bps = (float)(shmem_data->total_TX * 8) / time_delta;    
        *rps = (float)(shmem_data->total_conn) / time_delta;    
    }
    
    return 0;
}

int mod_cband_update_speed(mod_cband_shmem_data *shmem_data, unsigned long long bytes_served)
{
    unsigned long time_delta;
    unsigned long time_last_request;
    unsigned long time_now;
    unsigned long divider;
    
    time_now          = (unsigned long)(apr_time_now() / 1000000);
    time_delta        = time_now - shmem_data->total_last_refresh;
    time_last_request = time_now - shmem_data->total_last_time;
    
    if (bytes_served > (unsigned long long)0) {
    	shmem_data->total_last_time = time_now;
	shmem_data->total_TX += bytes_served;
        shmem_data->total_conn++;
    } 
    
    divider = 0;
    if ((time_delta > PERIOD_LEN) && (time_last_request < PERIOD_LEN))
	divider = time_delta;
    else
    if ((time_delta > PERIOD_LEN) && (time_last_request < 2 * PERIOD_LEN))
	divider = time_last_request;
    else
    if ((time_delta > PERIOD_LEN) && (time_last_request >= 2 * PERIOD_LEN)) {
	shmem_data->total_last_refresh = time_now - 1;
	shmem_data->total_TX   = 0;
	shmem_data->total_conn = 0;
    }

    if (divider > 0) {
	shmem_data->total_last_refresh = time_now - 1;
        shmem_data->total_TX   /= divider;
	shmem_data->total_conn /= divider;
    }
        
    return 0;
}

void mod_cband_check_virtualhost_refresh(mod_cband_virtualhost_config_entry *entry_virtual, unsigned long sec)
{
    mod_cband_scoreboard_entry *scoreboard;

    if (entry_virtual->refresh_time == 0 || entry_virtual->virtual_scoreboard == NULL)
	return;

    scoreboard = &(entry_virtual->shmem_data->total_usage);

    if (mod_cband_get_start_time(scoreboard) < 0)
    	mod_cband_set_start_time(scoreboard, sec);
    
    if ((mod_cband_get_start_time(scoreboard) + entry_virtual->refresh_time) < sec) {
    	mod_cband_clear_score(scoreboard);
	mod_cband_set_start_time(scoreboard, sec);
    }
}

void mod_cband_check_user_refresh(mod_cband_user_config_entry *entry_user, unsigned long sec)
{
    mod_cband_scoreboard_entry *scoreboard;

    if (entry_user->refresh_time == 0 || entry_user->user_scoreboard == NULL)
	return;

    scoreboard = &(entry_user->shmem_data->total_usage);

    if (mod_cband_get_start_time(scoreboard) < 0)
    	mod_cband_set_start_time(scoreboard, sec);
    
    if ((mod_cband_get_start_time(scoreboard) + entry_user->refresh_time) < sec) {
    	mod_cband_clear_score(scoreboard);
	mod_cband_set_start_time(scoreboard, sec);
    }
}

int mod_cband_set_overlimit_speed(mod_cband_shmem_data *shmem_data)
{
    if (shmem_data == NULL)
	return -1;

    shmem_data->curr_kbps  = shmem_data->over_kbps;
    shmem_data->curr_rps   = shmem_data->over_rps;
    shmem_data->curr_delay = shmem_data->over_delay;

    return 0;
}

int mod_cband_set_normal_speed(mod_cband_shmem_data *shmem_data)
{
    if (shmem_data == NULL)
	return -1;

    shmem_data->curr_kbps  = shmem_data->max_kbps;
    shmem_data->curr_rps   = shmem_data->max_rps;
    shmem_data->curr_delay = shmem_data->max_delay;

    return 0;
}

int mod_cband_reset_virtualhost(char *name)
{
    mod_cband_virtualhost_config_entry *entry;
    char virtualhost[MAX_VIRTUALHOST_NAME];
    unsigned port, line;

    if (name == NULL)
	return -1;

    if (!strcasecmp(name, "all")) {
        entry = config->next_virtualhost;
    
	while(entry != NULL) {
	    mod_cband_clear_score(&(entry->shmem_data->total_usage));
	    mod_cband_set_start_time(&(entry->shmem_data->total_usage), (unsigned long)(apr_time_now() / 1000000));
	    mod_cband_set_normal_speed(entry->shmem_data);
    
	    if (entry->next == NULL)
		break;
	    
	    entry = entry->next;
        }
    } else {
	sscanf(name, "%[^:]:%u:%u", virtualhost, &port, &line);
	
	if ((entry = mod_cband_get_virtualhost_entry_(virtualhost, (apr_port_t)port, line, 0)) != NULL) {
	    mod_cband_clear_score(&(entry->shmem_data->total_usage));
	    mod_cband_set_start_time(&(entry->shmem_data->total_usage), (unsigned long)(apr_time_now() / 1000000));
	    mod_cband_set_normal_speed(entry->shmem_data);
	}
    }
    
    return 0;
}

int mod_cband_reset_user(char *name)
{
    mod_cband_user_config_entry *entry;

    if (name == NULL)
	return -1;

    if (!strcasecmp(name, "all")) {
        entry = config->next_user;
    
	while(entry != NULL) {
	    mod_cband_clear_score(&(entry->shmem_data->total_usage));
	    mod_cband_set_start_time(&(entry->shmem_data->total_usage), (unsigned long)(apr_time_now() / 1000000));
	    mod_cband_set_normal_speed(entry->shmem_data);
    
	    if (entry->next == NULL)
		break;
	    
	    entry = entry->next;
        }
    } else {
	if ((entry = mod_cband_get_user_entry(name, NULL, 0)) != NULL) {
	    mod_cband_clear_score(&(entry->shmem_data->total_usage));
	    mod_cband_set_start_time(&(entry->shmem_data->total_usage), (unsigned long)(apr_time_now() / 1000000));
	    mod_cband_set_normal_speed(entry->shmem_data);
	}
    }
    
    return 0;
}

void mod_cband_status_print_limit(request_rec *req, unsigned long limit, unsigned long usage, char *unit, unsigned int mult)
{
    unsigned char normal_r, normal_g, normal_b;
    unsigned char exceeded_r, exceeded_g, exceeded_b;
    unsigned char r, g, b;
    const char *color;
    
    exceeded_r = 0xff;
    exceeded_g = 0x30;
    exceeded_b = 0x50;
    normal_r = 0x30;
    normal_g = 0xf0;
    normal_b = 0x30;
        
    if (limit == (unsigned long)0) 
        ap_rprintf(req, "<td style=\"background-color: yellow\">U/%s</td>\n", mod_cband_create_traffic_size(req->pool, usage, unit, mult));
    else {
	if (usage < limit) {
	    r = normal_r;
	    g = normal_g;
	    b = normal_b;
	
	    if (usage > 0)
		r += (unsigned char)((float)(exceeded_r - normal_r) * (float)((float)usage / limit));

	    if (usage > 0)
		g -= (unsigned char)((float)(normal_g - exceeded_g) * (float)((float)usage / limit));

	    if (usage > 0)
		b += (unsigned char)((float)(exceeded_b - normal_b) * (float)((float)usage / limit));
	} else {
	    r = exceeded_r;
	    g = exceeded_g;
	    b = exceeded_b;
	}
    
	if (usage < limit / 2)
	    color = "black";
	else
	    color = "yellow";
	    
	ap_rprintf(req, "<td style=\"color: %s; background-color: #%02X%02X%02X\">%s/%s</td>\n", color, r, g, b, mod_cband_create_traffic_size(req->pool, limit, unit, mult), mod_cband_create_traffic_size(req->pool, usage, unit, mult));
    }
}

void mod_cband_status_print_speed(request_rec *req, unsigned long limit, float usage)
{
    unsigned char normal_r, normal_g, normal_b;
    unsigned char exceeded_r, exceeded_g, exceeded_b;
    unsigned char r, g, b;
    const char *color;
    
    exceeded_r = 0xff;
    exceeded_g = 0x20;
    exceeded_b = 0x20;
    normal_r = 0xf0;
    normal_g = 0xa1;
    normal_b = 0xa1;
        
    if (limit == (unsigned long)0) 
        ap_rprintf(req, "<td class=\"speed\">U/%0.2f</td>\n", usage);
    else {
	if (usage < limit) {
	    r = normal_r;
	    g = normal_g;
	    b = normal_b;
	
	    if (usage > 0)
		g -= (unsigned char)((float)(normal_g - exceeded_g) * (float)((float)usage / limit));

	    if (usage > 0)
	    	b -= (unsigned char)((float)(normal_b - exceeded_b) * (float)((float)usage / limit));
	} else {
	    r = exceeded_r;
	    g = exceeded_g;
	    b = exceeded_b;
	}

	if (usage < limit / 2)
	    color = "black";
	else
	    color = "yellow";
    
	ap_rprintf(req, "<td style=\"color: %s; background-color: #%02X%02X%02X\">%lu/%0.2f</td>\n", color, r, g, b, limit, usage);
    }
}

void mod_cband_status_print_virtualhost_row(request_rec *r, mod_cband_virtualhost_config_entry *entry,
	int handler_type, int refresh, char *unit) {
    
    mod_cband_scoreboard_entry *virtual_usage;
    float bps, rps;
    int i;

    virtual_usage = &entry->shmem_data->total_usage;
	    
    ap_rputs("<tr>\n", r);
    ap_rprintf(r, "<td><a href=\"http://%s\">%s</a>:%d (%d)</td>\n", entry->virtual_name, entry->virtual_name, entry->virtual_port, entry->virtual_defn_line);	
    if (handler_type == CBAND_HANDLER_ALL)
	ap_rprintf(r, "<td><a href=\"?reset=%s:%d:%d&amp;refresh=%d&amp;unit=%s\">reset</a></td>\n", entry->virtual_name, entry->virtual_port, entry->virtual_defn_line, refresh, unit);
    ap_rprintf(r, "<td class=\"refresh\">%s</td>\n", mod_cband_create_period(r->pool, virtual_usage->start_time, entry->refresh_time));	

    mod_cband_status_print_limit(r, entry->virtual_limit, (unsigned long)(virtual_usage->total_bytes / entry->virtual_limit_mult),
	unit, entry->virtual_limit_mult);

    for (i = 0; i < DST_CLASS; i++) {
	mod_cband_status_print_limit(r, entry->virtual_class_limit[i], (unsigned long)(virtual_usage->class_bytes[i] / entry->virtual_class_limit_mult[i]),
	    unit, entry->virtual_class_limit_mult[i]);
    }

    mod_cband_update_speed(entry->shmem_data, 0);
    mod_cband_get_speed(entry->shmem_data, &bps, &rps);
    mod_cband_status_print_speed(r, entry->shmem_data->curr_kbps, bps / 1024);
    mod_cband_status_print_speed(r, entry->shmem_data->curr_rps,  rps);

    if (entry->virtual_user)
        ap_rprintf(r, "<td>%s</td>\n", entry->virtual_user);	
    else
	ap_rprintf(r, "<td>none</td>\n");	

    ap_rputs("</tr>\n", r);
}

void mod_cband_status_print_user_row(request_rec *r, mod_cband_user_config_entry *entry_user,
	int handler_type, int refresh, char *unit) {
    
    mod_cband_scoreboard_entry *user_usage;
    float bps, rps;
    int i;

    user_usage = &entry_user->shmem_data->total_usage;
	    
    ap_rputs("<tr>\n", r);
    ap_rprintf(r, "<td>%s</td>\n", entry_user->user_name);	
    if (handler_type == CBAND_HANDLER_ALL)
	ap_rprintf(r, "<td><a href=\"?reset_user=%s&amp;refresh=%d&amp;unit=%s\">reset</a></td>\n", entry_user->user_name, refresh, unit);
    ap_rprintf(r, "<td class=\"refresh\">%s</td>\n", mod_cband_create_period(r->pool, user_usage->start_time, entry_user->refresh_time));

    mod_cband_status_print_limit(r, entry_user->user_limit, (unsigned long)(user_usage->total_bytes / entry_user->user_limit_mult),
	unit, entry_user->user_limit_mult);

    for (i = 0; i < DST_CLASS; i++) {
	mod_cband_status_print_limit(r, entry_user->user_class_limit[i], (unsigned long)(user_usage->class_bytes[i] / entry_user->user_class_limit_mult[i]),
	    unit, entry_user->user_class_limit_mult[i]);
    }

    mod_cband_update_speed(entry_user->shmem_data, 0);
    mod_cband_get_speed(entry_user->shmem_data, &bps, &rps);
    mod_cband_status_print_speed(r, entry_user->shmem_data->curr_kbps, bps / 1024);
    mod_cband_status_print_speed(r, entry_user->shmem_data->curr_rps,  rps);

    ap_rputs("</tr>\n", r);
}

void mod_cband_status_print_virtualhost_XML_row(request_rec *r, mod_cband_virtualhost_config_entry *entry,
	int handler_type) {

    float bps, rps;
    mod_cband_class_config_entry *entry_class;
    mod_cband_scoreboard_entry *virtual_usage;
    int i;

    virtual_usage = &entry->shmem_data->total_usage;

    mod_cband_update_speed(entry->shmem_data, 0);
    mod_cband_get_speed(entry->shmem_data, &bps, &rps);
	    
    ap_rprintf(r, "\t\t<%s>\n", entry->virtual_name);
    ap_rprintf(r, "\t\t\t<port>%d</port>\n", entry->virtual_port);
    ap_rprintf(r, "\t\t\t<line>%d</line>\n", entry->virtual_defn_line);
    ap_rprintf(r, "\t\t\t<limits>\n");
 	    
    ap_rprintf(r, "\t\t\t\t<total>%lu%s</total>\n", entry->virtual_limit,
	      (entry->virtual_limit_mult == 1024 ? "KiB" : "KB"));

    entry_class = config->next_class;
    i = 0;
    while(entry_class != NULL) {
	ap_rprintf(r, "\t\t\t\t<%s>%lu%s</%s>\n", entry_class->class_name, entry->virtual_class_limit[i],
        (entry->virtual_class_limit_mult[i] == 1024 ? "KiB" : "KB"), entry_class->class_name);
	i++;
        entry_class = entry_class->next;
    }
    ap_rprintf(r, "\t\t\t\t<kbps>%lu</kbps>\n", entry->shmem_data->curr_kbps);
    ap_rprintf(r, "\t\t\t\t<rps>%lu</rps>\n", entry->shmem_data->curr_rps);
    ap_rprintf(r, "\t\t\t</limits>\n");

    ap_rprintf(r, "\t\t\t<usages>\n");
    ap_rprintf(r, "\t\t\t\t<total>%luKiB</total>\n", (unsigned long)(virtual_usage->total_bytes / 1024));
    entry_class = config->next_class;
    i = 0;
    while(entry_class != NULL) {
        ap_rprintf(r, "\t\t\t\t<%s>%lu%s</%s>\n", entry_class->class_name,
	    (unsigned long)(virtual_usage->class_bytes[i] / entry->virtual_class_limit_mult[i]),
	    (entry->virtual_class_limit_mult[i] == 1024 ? "KiB" : "KB"), entry_class->class_name);
        i++;
        entry_class = entry_class->next;
    }
    ap_rprintf(r, "\t\t\t\t<kbps>%0.2f</kbps>\n", bps / 1024);
    ap_rprintf(r, "\t\t\t\t<rps>%0.2f</rps>\n", rps);
    ap_rprintf(r, "\t\t\t</usages>\n");

    ap_rprintf(r, "<time_to_refresh>%s</time_to_refresh>", mod_cband_create_period(r->pool, virtual_usage->start_time, entry->refresh_time));
    
    if (entry->virtual_user)
	ap_rprintf(r, "\t\t\t<user>%s</user>\n", entry->virtual_user);
    else
    	ap_rprintf(r, "\t\t\t<user>none</user>\n");	

    if (entry->virtual_scoreboard)
	ap_rprintf(r, "\t\t\t<scoreboard>%s</scoreboard>\n", entry->virtual_scoreboard);	
    else
    	ap_rprintf(r, "\t\t\t<scoreboard>none</scoreboard>\n");	

    if (entry->virtual_limit_exceeded)
	ap_rprintf(r, "\t\t\t<limit_exceeded_URL>%s</limit_exceeded_URL>\n", entry->virtual_limit_exceeded);	
    else
    	ap_rprintf(r, "\t\t\t<limit_exceeded_URL>none</limit_exceeded_URL>\n");	

    ap_rprintf(r, "\t\t</%s>\n", entry->virtual_name);
}

void mod_cband_status_print_user_XML_row(request_rec *r, mod_cband_user_config_entry *entry_user,
	int handler_type) {
    
    float bps, rps;
    mod_cband_class_config_entry *entry_class;
    mod_cband_scoreboard_entry *user_usage;
    int i;

    user_usage = &entry_user->shmem_data->total_usage;

    mod_cband_update_speed(entry_user->shmem_data, 0);
    mod_cband_get_speed(entry_user->shmem_data, &bps, &rps);
    
    ap_rprintf(r, "\t\t<%s>\n", entry_user->user_name);	
    ap_rprintf(r, "\t\t\t<limits>\n");
    ap_rprintf(r, "\t\t\t\t<total>%lu%s</total>\n", entry_user->user_limit,
    (entry_user->user_limit_mult == 1024 ? "KiB" : "KB"));

    entry_class = config->next_class;
    i = 0;
    while(entry_class != NULL) {
	ap_rprintf(r, "\t\t\t\t<%s>%lu%s</%s>\n", entry_class->class_name, entry_user->user_class_limit[i],
	          (entry_user->user_class_limit_mult[i] == 1024 ? "KiB" : "KB"), entry_class->class_name);
	i++;
        entry_class = entry_class->next;
    }
    
    ap_rprintf(r, "\t\t\t\t<kbps>%lu</kbps>\n", entry_user->shmem_data->curr_kbps);
    ap_rprintf(r, "\t\t\t\t<rps>%lu</rps>\n", entry_user->shmem_data->curr_rps);
    ap_rprintf(r, "\t\t\t</limits>\n");

    ap_rprintf(r, "\t\t\t<usages>\n");
    ap_rprintf(r, "\t\t\t\t<total>%luKiB</total>\n", (unsigned long)(user_usage->total_bytes / 1024));
    entry_class = config->next_class;
    i = 0;
    while(entry_class != NULL) {
        ap_rprintf(r, "\t\t\t\t<%s>%lu%s</%s>\n", entry_class->class_name,
	    (unsigned long)(user_usage->class_bytes[i] / entry_user->user_class_limit_mult[i]),
	    (entry_user->user_class_limit_mult[i] == 1024 ? "KiB" : "KB"), entry_class->class_name);
        i++;
        entry_class = entry_class->next;
    }
    ap_rprintf(r, "\t\t\t\t<kbps>%0.2f</kbps>\n", bps / 1024);
    ap_rprintf(r, "\t\t\t\t<rps>%0.2f</rps>\n", rps);
    ap_rprintf(r, "\t\t\t</usages>\n");

    ap_rprintf(r, "<time_to_refresh>%s</time_to_refresh>", mod_cband_create_period(r->pool, user_usage->start_time, entry_user->refresh_time));

    if (entry_user->user_limit_exceeded)
	ap_rprintf(r, "\t\t\t<limit_exceeded_URL>%s</limit_exceeded_URL>\n", entry_user->user_limit_exceeded);	
    else
    	ap_rprintf(r, "\t\t\t<limit_exceeded_URL>none</limit_exceeded_URL>\n");	
	    
    if (entry_user->user_scoreboard)
	ap_rprintf(r, "\t\t\t<scoreboard>%s</scoreboard>\n", entry_user->user_scoreboard);	
    else
    	ap_rprintf(r, "\t\t\t<scoreboard>none</scoreboard>\n");	

    ap_rprintf(r, "\t\t</%s>\n", entry_user->user_name);
}

static const char mod_cband_status_handler_style[] = 
"\n<style type=\"text/css\">\n"
".small 	{ font-family: sans; font-size: 8pt; }\n"
".normal 	{ font-family: sans; font-size: 10pt; }\n"
"table		{ font-size: 12px; font-family: tachoma, helvetica, verdana, sans; border: 1px solid #d0d0d0 }\n"
"tr		{ font-size: 12px; font-family: tachoma, helvetica, verdana, sans; border: 1px solid #d0d0d0 }\n"
"td		{ font-size: 11px; font-family: sans; padding-left: 2px}\n"
"td.refresh	{ font-size: 11px; font-family: sans; padding-left: 2px; padding-right: 2px; background-color: #0de2cb; text-align: right }\n"
"td.speed	{ font-size: 11px; font-family: sans; padding-left: 2px; padding-right: 2px; background-color: #ffa1a1; text-align: left }\n"
"td.speedc	{ font-size: 11px; font-family: sans; padding-left: 2px; padding-right: 2px; background-color: #ffb1b1; text-align: left }\n"
"a		{ font-size: 11px; font-family: sans; font-weight: normal; color: #606060}\n"
"h1, h2		{ font-family: tachoma, helvetica, verdana, sans; }\n"
"h1		{ font-size: 16px; }\n"
"h2		{ font-size: 12px; }\n"
"</style>\n\n";

static const char mod_cband_status_handler_foot[] = 
"\n<br><center>\n"
"<p class=\"small\">\n"
"<a href=\"http://cband.linux.pl\">mod_cband 0.9.6.0</a><br>\n"
"Copyright 2005 by <a href=\"mailto: dembol _at_ cband.linux.pl\">Lukasz Dembinski</a><br>\n"
"All rights reserved.</p>\n"
"</center>\n";

/**
 * /cband-status handler output in HTML
 */
static int mod_cband_status_handler_HTML (request_rec *r, int handler_type)
{
    mod_cband_virtualhost_config_entry *entry, *entry_me = NULL;
    mod_cband_user_config_entry *entry_user, *entry_user_me = NULL;
    mod_cband_class_config_entry *entry_class;
    unsigned long uptime, sec;
    unsigned i;
    char *arg;
    char *val, *key;
    int refresh = -1;
    char *unit = "";

    if (r->args != NULL) {

	arg = r->args;
	while(*arg != 0) {
	    val = ap_getword_nc(r->pool, &arg, '&');
	    
	    if (val == NULL)
		break;
	
	    key = ap_getword_nc(r->pool, &val, '=');
	
	    if (key == NULL)
		break;
		
	    apr_table_setn(r->notes, key, val);
	}
    }

    if ((arg = (char *)apr_table_get(r->notes, "refresh")) != NULL)
	refresh = atoi(arg);

    if ((arg = (char *)apr_table_get(r->notes, "unit")) != NULL) {
	if (arg[0] == 'G' || arg[0] == 'g' || arg[0] == 'M' || arg[0] == 'm' || arg[0] == 'K' || arg[0] == 'k') {
	    arg[0] = toupper(arg[0]);
	    if (arg[1] == 'I' || arg[1] == 'i') {
		arg[1] = 'i';
		arg[2] = 0;
		unit = arg;
	    } else {
		arg[1] = 0;
		unit = arg;
	    }
	}
    }
    
    if (refresh < 0)
	refresh = DEFAULT_REFRESH;
	
    if (handler_type == CBAND_HANDLER_ALL) {
	if ((arg = (char *)apr_table_get(r->notes, "reset")) != NULL) {
    	    mod_cband_reset_virtualhost(arg);
	
	    apr_table_setn(r->headers_out, "Location", apr_psprintf(r->pool, "%s?refresh=%d&unit=%s", r->uri, refresh, unit));
	    return HTTP_MOVED_PERMANENTLY;
	}

	if ((arg = (char *)apr_table_get(r->notes, "reset_user")) != NULL) {
    	    mod_cband_reset_user(arg);
	
	    apr_table_setn(r->headers_out, "Location", apr_psprintf(r->pool, "%s?refresh=%d&unit=%s", r->uri, refresh, unit));
	    return HTTP_MOVED_PERMANENTLY;
	}
    } else {
    	if ((entry_me = mod_cband_get_virtualhost_entry(r->server, r->server->module_config, 0)) != NULL) {
	    if (entry_me->virtual_user != NULL)  
	       entry_user_me = mod_cband_get_user_entry(entry_me->virtual_user, r->server->module_config, 0);
	}
    }

    sec = (unsigned long)(apr_time_now() / 1000000);
    uptime = (unsigned long)(sec - config->start_time);

    ap_set_content_type(r, "text/html");

    ap_rputs(DOCTYPE_HTML_4_0T "<HTML><HEAD><TITLE>mod_cband status</TITLE>\n", r);
    ap_rputs(mod_cband_status_handler_style, r);
    ap_rputs("</HEAD>\n", r);
    ap_rputs("<BODY>", r);
    ap_rputs("<center><H1>mod_cband status page</h1></center>", r);
    ap_rprintf(r, "<center><h2>Server uptime %s</h2></center><br>\n", mod_cband_create_time(r->pool, uptime));
    ap_rprintf(r, "<h2>Virtualhosts <a href=\"?refresh=%d&amp;unit=%s\">[refresh]</a> &nbsp; <a href=\"?refresh=%d\">[human-readable]</a> <a href=\"?refresh=%d&amp;unit=G\">[GB]</a> <a href=\"?refresh=%d&amp;unit=M\">[MB]</a> <a href=\"?refresh=%d&amp;unit=K\">[KB]</a>\n</h2>\n", refresh, unit, refresh, refresh, refresh, refresh);
    
    if (((handler_type == CBAND_HANDLER_ME) && (entry_me == NULL)) || 
        ((handler_type == CBAND_HANDLER_ALL) && (config->next_virtualhost == NULL)))
	ap_rputs("<p class=\"normal\">no limits for virtualhosts</p>\n", r);
    else {
	ap_rputs("<table cellspacing=\"0\" cellpadding=\"0\">\n", r);
	ap_rputs("<tr>\n<td width=\"150\">Virtualhost name</td>\n", r);

	if (handler_type == CBAND_HANDLER_ALL) {
	    ap_rputs("<td width=\"50\">", r);
	    ap_rprintf(r, "<a href=\"?reset=all&amp;refresh=%d&amp;unit=%s\">reset all</a></td>\n", refresh, unit);
	}
	
	ap_rputs("<td width=\"10%\">time to refresh</td>\n", r);
	ap_rprintf(r,"<td width=\"100\">Total<br>Limit/Used</td>\n");
	entry_class = config->next_class;
	
	i = 0;
	while(entry_class != NULL) {
	    ap_rprintf(r,"<td width=\"100\">%s<br>Limit/Used</td>\n", entry_class->class_name);
	
	    entry_class = entry_class->next;
            i++;
	}
	
	for (; i < DST_CLASS; i++)
	    ap_rprintf(r,"<td width=\"100\">Class %d<br>Limit/Used</td>\n",i);
		
	ap_rprintf(r,"<td width=\"100\">kbps<br>Limit/Current</td>\n");
	ap_rprintf(r,"<td width=\"100\">rps<br>Limit/Current</td>\n");
	
	ap_rputs("<td width=\"10%\">user</td>\n", r);
	ap_rputs("</tr>\n", r);
	
	if (handler_type == CBAND_HANDLER_ALL) {	
	    entry = config->next_virtualhost;
	    while(entry != NULL) {
		mod_cband_check_virtualhost_refresh(entry, sec);
	        mod_cband_status_print_virtualhost_row(r, entry, handler_type, refresh, unit);
	    	    
		if ((entry = entry->next) == NULL)
		    break;
	    }
	} else {
	    if (entry_me != NULL) {
	    	mod_cband_check_virtualhost_refresh(entry_me, sec);
	        mod_cband_status_print_virtualhost_row(r, entry_me, handler_type, refresh, unit);
	    }
	}
	
	ap_rputs("</table>\n", r);
    }

    ap_rprintf(r, "<br><h2>Users <a href=\"?refresh=%d&amp;unit=%s\">[refresh]</a> &nbsp; <a href=\"?refresh=%d\">[human-readable]</a> <a href=\"?refresh=%d&amp;unit=G\">[GB]</a> <a href=\"?refresh=%d&amp;unit=M\">[MB]</a> <a href=\"?refresh=%d&amp;unit=K\">[KB]</a>\n</h2>\n", refresh, unit, refresh, refresh, refresh, refresh);

    if (((handler_type == CBAND_HANDLER_ME) && (entry_user_me == NULL)) || 
        ((handler_type == CBAND_HANDLER_ALL) && (config->next_user == NULL)))
        ap_rputs("<p class=\"normal\">no limits for users</p>\n", r);
    else {
	ap_rputs("<table cellspacing=\"0\" cellpadding=\"0\">\n", r);
	ap_rputs("<tr>\n<td width=\"150\">Username</td>\n", r);
	
	if (handler_type == CBAND_HANDLER_ALL) {
	    ap_rputs("<td width=\"50\">", r);
	    ap_rprintf(r, "<a href=\"?reset_user=all&amp;refresh=%d&amp;unit=%s\">reset all</a></td>\n", refresh, unit);
	}
	
	ap_rputs("<td width=\"10%\">time to refresh</td>\n", r);    
	ap_rprintf(r,"<td width=\"100\">Total<br>Limit/Used</td>\n");
	entry_class = config->next_class;
	
	i = 0;
	while(entry_class != NULL) {
	    ap_rprintf(r,"<td width=\"100\">%s<br>Limit/Used</td>\n", entry_class->class_name);
	
	    entry_class = entry_class->next;
            i++;
	}
	
	for (; i < DST_CLASS; i++)
	    ap_rprintf(r,"<td width=\"100\">Class %d<br>Limit/Used</td>\n",i);

	ap_rprintf(r,"<td width=\"100\">kbps<br>Limit/Current</td>\n");
	ap_rprintf(r,"<td width=\"100\">rps<br>Limit/Current</td>\n");
	
	ap_rputs("</tr>\n", r);
	
	if (handler_type == CBAND_HANDLER_ALL) {	
	    entry_user = config->next_user;
	
	    while(entry_user != NULL) {
		mod_cband_check_user_refresh(entry_user, sec);
		mod_cband_status_print_user_row(r, entry_user, handler_type, refresh, unit);
	    
		if ((entry_user = entry_user->next) == NULL)
		    break;
	    }
	} else {
	    if (entry_user_me != NULL) {
	        mod_cband_check_user_refresh(entry_user_me, sec);
		mod_cband_status_print_user_row(r, entry_user_me, handler_type, refresh, unit);
	    }
	}
	
	ap_rputs("</table>\n", r);
    }

    ap_rputs(mod_cband_status_handler_foot, r);
    ap_rputs("</BODY></HTML>\n", r);
    
    apr_table_setn(r->headers_out, "Refresh", apr_psprintf(r->pool, "%d", refresh));

    return OK;
}

/**
 * /cband-status handler output in XML
 */
static int mod_cband_status_handler_XML (request_rec *r, int handler_type)
{
    mod_cband_virtualhost_config_entry *entry, *entry_me = NULL;
    mod_cband_user_config_entry *entry_user, *entry_user_me = NULL;
    unsigned long sec, uptime;

    if (handler_type == CBAND_HANDLER_ME) {
    	if ((entry_me = mod_cband_get_virtualhost_entry(r->server, r->server->module_config, 0)) != NULL) {
	    if (entry_me->virtual_user != NULL)  
	       entry_user_me = mod_cband_get_user_entry(entry_me->virtual_user, r->server->module_config, 0);
	}
    }

    sec = (unsigned long)(apr_time_now() / 1000000);
    uptime = (unsigned long)(sec - config->start_time);

    ap_set_content_type(r, "text/xml");

    ap_rputs("<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n", r);
    ap_rputs("<mod_cband>\n", r);
    ap_rputs("\t<Server>\n", r);
    ap_rprintf(r, "\t\t<uptime>%s</uptime>\n", mod_cband_create_time(r->pool, uptime));    
    ap_rputs("\t</Server>\n", r);
    
    ap_rputs("\t<Virtualhosts>\n", r);
    if (handler_type == CBAND_HANDLER_ALL) {	
        entry = config->next_virtualhost;
    
	while(entry != NULL) {
    	    mod_cband_check_virtualhost_refresh(entry, sec);
	    mod_cband_status_print_virtualhost_XML_row(r, entry, handler_type);
	    	    
	    if ((entry = entry->next) == NULL)
		break;
	}
    } else {
	if (entry_me != NULL) {
	    mod_cband_check_virtualhost_refresh(entry_me, sec);
	    mod_cband_status_print_virtualhost_XML_row(r, entry_me, handler_type);
	}
    }
	
    ap_rputs("\t</Virtualhosts>\n", r);
    ap_rputs("\t<Users>\n", r);

    if (handler_type == CBAND_HANDLER_ALL) {	
        entry_user = config->next_user;
	
	while(entry_user != NULL) {
	    mod_cband_check_user_refresh(entry_user, sec);
	    mod_cband_status_print_user_XML_row(r, entry_user, handler_type);
	    
	    if ((entry_user = entry_user->next) == NULL)
		break;
	}
    } else {
	if (entry_user_me != NULL) {
	    mod_cband_check_user_refresh(entry_user_me, sec);
	    mod_cband_status_print_user_XML_row(r, entry_user_me, handler_type);
	}
    }
	
    ap_rputs("\t</Users>\n", r);
    ap_rputs("</mod_cband>", r);

    return OK;
}

/*
 * /cband-status handler
 */
static int mod_cband_status_handler (request_rec *r)
{
    int handler_type;

    if (strcmp(r->handler, "cband-status") && strcmp(r->handler, "cband-status-me"))
	return DECLINED;

    if (!strcmp(r->handler, "cband-status"))
	handler_type = CBAND_HANDLER_ALL;
    else
    	handler_type = CBAND_HANDLER_ME;
	
    if ((r->args != NULL) && (!strcasecmp(r->args, "xml")))
	return mod_cband_status_handler_XML(r, handler_type);
    else
	return mod_cband_status_handler_HTML(r, handler_type);
}

int mod_cband_check_limit(request_rec *r, mod_cband_shmem_data *shmem_data, unsigned long limit, unsigned int mult, unsigned long long usage, char *limit_exceeded)
{
    /* Check if the bandwidth limit has been reached */
    if ((limit > 0) && (((unsigned long long)limit * (unsigned long long)mult) < usage)) {
	if (limit_exceeded != NULL) {
	    apr_table_setn(r->headers_out, "Location", limit_exceeded);
	    return HTTP_MOVED_PERMANENTLY;
	}
	else
	if ((shmem_data->over_kbps > (unsigned long)0) || (shmem_data->over_rps > (unsigned long)0))
	    mod_cband_set_overlimit_speed(shmem_data);
	else
	if (config->default_limit_exceeded != NULL) {
	    apr_table_setn(r->headers_out, "Location", config->default_limit_exceeded);
	    return HTTP_MOVED_PERMANENTLY;
	}
    }
    
    return OK;
}

/**
 * cband request handler
 * check bandwidth usage for virtualhosts. If it's exceeded, redirect to the specified URL
 */
static int mod_cband_request_handler (request_rec *r)
{
    mod_cband_virtualhost_config_entry *entry = NULL;
    mod_cband_user_config_entry *entry_user = NULL;
    unsigned long virtual_limit, virtual_class_limit;
    unsigned long user_limit, user_class_limit;
    unsigned long long virtual_usage, virtual_class_usage;
    unsigned long long user_usage, user_dst_usage;
    unsigned long sec;
    unsigned int virtual_limit_mult, virtual_class_limit_mult, user_limit_mult, user_class_limit_mult;
    char *user_scoreboard = NULL;
    char *virtual_limit_exceeded = NULL;
    char *user_limit_exceeded = NULL;
    int dst = -1;
    float virtualhost_bps = 0, virtualhost_rps = 0;
    float user_bps = 0, user_rps = 0;
    int request_overlimit;
    mod_cband_shmem_data *shmem_data = NULL;
    int loops;

#ifdef DEBUG
    fprintf(stderr, "apache2_mod_cband: req=%s:%d %s:%d %d\n", r->server->server_hostname, r->server->port,
    r->server->addrs->virthost, r->server->addrs->host_port, r->server->defn_line_number);
    fflush(stderr);
#endif

    if ((entry = mod_cband_get_virtualhost_entry(r->server, r->server->module_config, 0)) == NULL)
	return DECLINED;

    shmem_data = entry->shmem_data;
    shmem_data->total_usage.was_request = 1;

    sec = (unsigned long)(apr_time_now() / 1000000);
    dst = mod_cband_get_dst(r);
    mod_cband_check_virtualhost_refresh(entry, sec);
    
    virtual_limit = entry->virtual_limit;
    virtual_limit_mult = entry->virtual_limit_mult;
    if (dst >= 0) {
	virtual_class_limit = entry->virtual_class_limit[dst];
	virtual_class_limit_mult = entry->virtual_class_limit_mult[dst];
    } else {
	virtual_class_limit = 0;
	virtual_class_limit_mult = 0;
    }

    virtual_limit_exceeded  = entry->virtual_limit_exceeded;
    user_limit              = 0;
    user_limit_mult         = 0;
    user_class_limit        = 0;
    user_class_limit_mult   = 0;
    user_scoreboard         = NULL;
    user_limit_exceeded     = NULL;
    
    if ((entry->virtual_user != NULL) && ((entry_user = mod_cband_get_user_entry(entry->virtual_user, r->server->module_config, 0)) != NULL)) {
        user_limit          = entry_user->user_limit;
	user_limit_mult     = entry_user->user_limit_mult;
        user_limit_exceeded = entry_user->user_limit_exceeded;
	user_scoreboard     = entry_user->user_scoreboard;
	if (dst >= 0) {
	    user_class_limit       = entry_user->user_class_limit[dst];
	    user_class_limit_mult  = entry_user->user_class_limit_mult[dst];
	}
	
	mod_cband_check_user_refresh(entry_user, sec);
    }

    if ((virtual_limit == 0) && (user_limit == 0) && (user_class_limit == 0) && (virtual_class_limit == 0))
        return DECLINED;

    /* BEGIN CRITICAL SECTION */
    mod_cband_sem_down(config->sem_id);
    loops = 0;
    do {
	mod_cband_get_speed(entry->shmem_data, &virtualhost_bps, &virtualhost_rps);
	if (entry_user != NULL)
	    mod_cband_get_speed(entry_user->shmem_data, &user_bps, &user_rps);
	
	request_overlimit = 0;
	if (((entry->shmem_data->curr_kbps > 0) && ((virtualhost_bps / 1024) > entry->shmem_data->curr_kbps)) || ((entry->shmem_data->curr_rps > 0) && (virtualhost_rps > entry->shmem_data->curr_rps))) {
	    mod_cband_sem_up(config->sem_id);
	    sleep(1);	
	    mod_cband_sem_down(config->sem_id);
	    request_overlimit = 1;
	} else
	if ((entry_user != NULL) && (((entry_user->shmem_data->curr_kbps > 0) && ((user_bps / 1024) > entry_user->shmem_data->curr_kbps)) || ((entry_user->shmem_data->curr_rps > 0) && (user_rps > entry_user->shmem_data->curr_rps)))) {
	    mod_cband_sem_up(config->sem_id);
	    sleep(1);	
	    mod_cband_sem_down(config->sem_id);
	    request_overlimit = 2;
	}
	
	if (request_overlimit == 1) {
	    if (loops > entry->shmem_data->curr_delay)
		break;
	} else
	if (request_overlimit == 2) {
	    if (entry_user != NULL && (loops > entry_user->shmem_data->curr_delay))
		break;
	}
    } while (request_overlimit);

    virtual_usage       = 0;
    virtual_class_usage = 0;
    user_usage          = 0;
    user_dst_usage      = 0;

    mod_cband_get_score(r->server, entry->virtual_scoreboard, &virtual_usage, -1, entry->shmem_data);
    
    if (dst >= 0) 
	mod_cband_get_score(r->server, entry->virtual_scoreboard, &virtual_class_usage, dst, entry->shmem_data);
    
    if (entry->virtual_user != NULL)
	mod_cband_get_score(r->server, user_scoreboard, &user_usage, -1, entry_user->shmem_data);

    if ((entry->virtual_user != NULL) && (dst >= 0))
	mod_cband_get_score(r->server, user_scoreboard, &user_dst_usage, dst, entry_user->shmem_data);

    mod_cband_sem_up(config->sem_id);
    /* END CRITICAL SECTION */

    if ((virtual_usage == 0) && (user_usage == 0) && (virtual_class_usage == 0) && (user_dst_usage == 0))
	return DECLINED;
      
    if (mod_cband_check_limit(r, entry->shmem_data, virtual_limit, virtual_limit_mult, virtual_usage, virtual_limit_exceeded) == HTTP_MOVED_PERMANENTLY)
	return HTTP_MOVED_PERMANENTLY;

    if (mod_cband_check_limit(r, entry->shmem_data, virtual_class_limit, virtual_class_limit_mult, virtual_class_usage, virtual_limit_exceeded) == HTTP_MOVED_PERMANENTLY)
	return HTTP_MOVED_PERMANENTLY;

    if ((entry_user != NULL) && (mod_cband_check_limit(r, entry_user->shmem_data, user_limit, user_limit_mult, user_usage, user_limit_exceeded) == HTTP_MOVED_PERMANENTLY))
	return HTTP_MOVED_PERMANENTLY;

    if ((entry_user != NULL) && (mod_cband_check_limit(r, entry_user->shmem_data, user_class_limit, user_class_limit_mult, user_dst_usage, user_limit_exceeded) == HTTP_MOVED_PERMANENTLY))
	return HTTP_MOVED_PERMANENTLY;
      
    return DECLINED;
}

static apr_status_t mod_cband_logger(request_rec *r)
{
    mod_cband_virtualhost_config_entry *entry = NULL;
    mod_cband_user_config_entry *entry_user = NULL;
    unsigned long long bytes_served;
    char *user_scoreboard = NULL;
    int dst = -1;
    
    if (r->method_number != M_GET)
	return DECLINED;
	
    if ((entry = mod_cband_get_virtualhost_entry(r->server, r->server->module_config, 0)) == NULL)
        return DECLINED; 

#ifdef DEBUG
    fprintf(stderr, "apache2_mod_cband: req=%s:%d %s:%d %d %s\n", r->server->server_hostname, r->server->port,
    r->server->addrs->virthost, r->server->addrs->host_port, r->server->defn_line_number, entry->virtual_scoreboard);
    fflush(stderr);
#endif

    bytes_served = (unsigned long long)r->bytes_sent;

    if ((entry->virtual_user != NULL) && ((entry_user = mod_cband_get_user_entry(entry->virtual_user, r->server->module_config, 0)) != NULL)) {
        user_scoreboard  = entry_user->user_scoreboard;
	mod_cband_flush_score(user_scoreboard, &(entry_user->shmem_data->total_usage));
    }
    mod_cband_flush_score(entry->virtual_scoreboard, &(entry->shmem_data->total_usage));

    dst = mod_cband_get_dst(r);

    /* BEGIN CRITICAL SECTION */
    mod_cband_sem_down(config->sem_id);
    mod_cband_update_speed(entry->shmem_data, bytes_served);            
    if (entry_user != NULL)
	mod_cband_update_speed(entry_user->shmem_data, bytes_served);            
    
    /* Update virtualhost's bandwidth score */
    mod_cband_update_score(entry->virtual_scoreboard, &bytes_served, dst, &(entry->shmem_data->total_usage));

    /* Update user's bandwidth score */
    if ((entry_user != NULL) && (user_scoreboard != NULL))
	mod_cband_update_score(user_scoreboard, &bytes_served, dst, &(entry_user->shmem_data->total_usage));

    mod_cband_sem_up(config->sem_id);
    /* END CRITICAL SECTION */

    return DECLINED;    
}

static apr_status_t mod_cband_cleanup1(void *s)
{
    mod_cband_sem_remove(config->sem_id);
    mod_cband_save_score_cache();

    return APR_SUCCESS;
}

static apr_status_t mod_cband_cleanup2(void *s)
{
    return APR_SUCCESS;
}

static apr_status_t mod_cband_post_config(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptmp, 
					  server_rec *s)
{
    mod_cband_update_score_cache(s);

    return OK;
}

/**
 * register mod_cband hooks 
 */
static void mod_cband_register_hooks (apr_pool_t *p)
{
    ap_hook_handler(mod_cband_status_handler, NULL, NULL, APR_HOOK_FIRST);
    ap_hook_handler(mod_cband_request_handler, NULL, NULL, APR_HOOK_FIRST);
    ap_hook_log_transaction(mod_cband_logger, NULL, NULL, APR_HOOK_MIDDLE);
    apr_pool_cleanup_register(p, NULL, mod_cband_cleanup1, mod_cband_cleanup2);
    ap_hook_post_config(mod_cband_post_config, NULL, NULL, APR_HOOK_MIDDLE);
}

/**
 * allocate config_header for mod_cband - this will store module 
 * settings, as read from config file
 */
static void *mod_cband_create_config(apr_pool_t *p, server_rec *s)
{
    if (config == NULL) {
	config = (mod_cband_config_header *) apr_palloc(p, sizeof(mod_cband_config_header));
	config->next_virtualhost = NULL;
	config->next_user = NULL;
	config->next_class = NULL;
	config->default_limit_exceeded = NULL;
	config->p = p;
	config->tree = NULL;
	config->start_time = (unsigned long)(apr_time_now() / 1000000);
	config->sem_id = semget(IPC_PRIVATE, 1, IPC_CREAT | 0666);
	config->score_flush_period = 0;
	mod_cband_sem_init(config->sem_id);
    } 
    
    return (void *)config;
}

module AP_MODULE_DECLARE_DATA cband_module =
{
	STANDARD20_MODULE_STUFF,
	NULL,
	NULL,
	mod_cband_create_config,
	NULL,
	mod_cband_cmds,
	mod_cband_register_hooks
};
