/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.5 2003/07/08 10:25:50 miham Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

/* init the plugin */
int mplugins_input_ippl_dlinit(mconfig *ext_conf) {
	config_input *conf = NULL;
	const char *errptr;
	int erroffset = 0;

	if (0 != strcmp(ext_conf->version, VERSION)) {
		M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "version string doesn't match: (mla) %s != (plugin) %s\n", ext_conf->version, VERSION);
		return -1;
	}

/* get the neccesary space */
	conf = malloc(sizeof(config_input));
	memset(conf, 0, sizeof(config_input));

/* set some defaults */
	conf->inputfilename = NULL;

	conf->buf = buffer_init();

/* compile the match */
	if ((conf->match_timestamp =
	      pcre_compile(
	           /* source ip */
	                    "^([a-zA-Z]{3}) ([ 0-9]{2}) ([0-9]{2}):([0-9]{2}):([0-9]{2})",
	                    0, &errptr, &erroffset, NULL)) == NULL) {

                fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr );
                return -1;
        }

	if ((conf->match_ip = 
			  /* ipaddress                                                 */
	      pcre_compile( "^([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})$", 
		            0, &errptr, &erroffset, NULL )) == NULL ) {
		fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
		return -1;
	}

	conf->match_linetype = NULL;
	conf->match_linetype_extra = NULL;
	conf->match_tcpline = NULL;
	conf->match_tcp_extra = NULL;
	conf->match_udpline = NULL;
	conf->match_udp_extra = NULL;
	conf->match_icmpline = NULL;
	conf->match_icmp_extra = NULL;
	conf->match_ipmon = NULL;       // Added 06/09/03
	conf->match_ipmon_extra = NULL; // for IPMON lines
	if ((conf->match_repline =
	      pcre_compile( "[a-zA-Z]{3} [0-9 ]{2} [:0-9]{8} "
		      	    "last message repeated ([0-9]+) time\\(s\\)", 0, &errptr, 
			    &erroffset, NULL )) == NULL ) {
		fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
		return -1;
	}
	conf->match_rep_extra = pcre_study(conf->match_repline, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}

	conf->ignored_shosts	= mlist_init();
	conf->ignored_dhosts	= mlist_init();
	conf->ignored_sports	= mlist_init();
	conf->ignored_dports	= mlist_init();
	conf->self_host		= NULL;
	
/* connect the plugin config to the master config */
	ext_conf->plugin_conf = conf;

	return 0;
}

/* destructor */

int mplugins_input_ippl_dlclose(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;

/* clean up everything you have opened, reserved */
	mclose(&(conf->inputfile));

	buffer_free(conf->buf);

	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;

	return 0;
}

int mplugins_input_ippl_parse_config(mconfig *ext_conf, const char *filename, const char *section) {
	config_input *conf = ext_conf->plugin_conf;

/* add your config options here. the format of the struct:
 * {"<name of the option>", <type>, <how to handle the further
 * occurences of the same value>, <address of the pointer where
 * the value should go to> }
 *
 * you only have the change config_values[] the rest is done in the
 * background (-> ./src/mconfig.c)
 */
	const mconfig_values config_values[] = {
		{"inputfile",	M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->inputfilename)},
		{"loglevel",	M_CONFIG_TYPE_INT,	M_CONFIG_VALUE_OVERWRITE, &(conf->loglevel)},
		{"has_identlog",	M_CONFIG_TYPE_INT,	M_CONFIG_VALUE_OVERWRITE, &(conf->has_identlog)},
		{"ignore_shost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND,	&(conf->ignored_shosts)},
		{"ignore_dhost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND,	&(conf->ignored_dhosts)},
		{"ignore_sport",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND,	&(conf->ignored_sports)},
		{"ignore_dport",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND,	&(conf->ignored_dports)},
		{"self_host",	M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, 	&(conf->self_host)},
		/* terminate the array of options */
		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	return mconfig_parse_section(ext_conf, filename, section, config_values);
}


int mplugins_input_ippl_set_defaults(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;
	const char *errptr;
	int erroffset = 0;

	/* For the repeat stuff.. */
	conf->internal_copy 	= mrecord_init();
	conf->repeatnum 	= 0;

	if (conf->inputfilename && strcmp(conf->inputfilename, "-") != 0) {
		if (mopen(&(conf->inputfile), conf->inputfilename)) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "%s: %s\n", conf->inputfilename, strerror(errno));
			return -1;
		}
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(ippl) using %s as inputfile\n", conf->inputfilename);
	} else {
		/* stdin */
		if (mopen(&(conf->inputfile), NULL)) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "%s: %s\n", conf->inputfilename, strerror(errno));
			return -1;
		}
		
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(ippl) using (stdin) as inputfile\n");
	}

	if (conf->has_identlog < 0 || conf->has_identlog > 1 ) {
		fprintf( stderr, "Bad configvalue (has_identlog: %d [range: %d-%d]) in configuration!\n", 
			 conf->loglevel, 0, 1 );
		return -1;
	}
	
	switch ( conf->loglevel ) {
	case 0:
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(ippl) using parser for 'short' logformat\n");
		/* Set the short logregexps .. */
		if ((conf->match_linetype = 
	        	          /* date                             */
		      pcre_compile( "[a-zA-Z]{3} [0-9 ]{2} [:0-9]{8} "
				  /* First word could be: last (ok), ICMP(ok), * (not ok)
				   * Second word could be: message (ok), UDP (ok), connection (ok) */
		                    "(last|ICMP|UDP|TCP)",
				    0, &errptr, &erroffset, NULL )) == NULL ) {
		  fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
		  return -1;
		}
	
		if ((conf->match_tcpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* TCP portnum|portname    closed|- (IP opts)|- "-" */
		                    "TCP (port [0-9]{1,5}|[^ ]+)( closed|)( \\(IP opts\\)|) - "
				  /* [ident@]host */
				    "([@.0-9a-zA-Z-]+)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_udpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* UDP portnum|portname                   (IP opts)|- from */
		                    "UDP (port [0-9]{1,5}|[^ ]+)( \\(IP opts\\)|) from "
				  /* host    */
				    "([.0-9a-zA-Z-]+)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_icmpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
		                    "ICMP ([a-z ]+)( - [_A-Za-z ]|)( \\(IP opts\\)|) - ([.0-9a-zA-Z-]+)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if (conf->self_host == NULL) {
			fprintf(stderr, "Input-ippl: config_parse: No self_host supplied when loglevel is %d!\n", conf->loglevel);
			return -1;
		}
		break;
	
	case 1: 
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(ippl) using parser for 'normal' logformat\n");
		/* Set the normal logregexps .. */
		if ((conf->match_linetype = 
	        	          /* date                             */
		      pcre_compile( "[a-zA-Z]{3} [0-9 ]{2} [:0-9]{8} "
				  /* First word could be: last (ok), ICMP(ok), * (not ok)
				   * Second word could be: message (ok), UDP (ok), connection (ok) */
		                    "(last|ICMP|port [0-9]{1,5}|[a-z-]+) (message|UDP|connection)", 
				    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_tcpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* portnum|portname                   attempt|closed */
		                    "(port [0-9]{1,5}|[^ ]+) connection (attempt|closed)"
				  /* (IP opts)|-       from */
				    "( \\(IP opts\\)|) from "
				  /* [ident@]host */
				    "([@.0-9a-zA-Z-]+)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_udpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* portnum|portname                   (IP opts)|- from */
		                    "(port [0-9]{1,5}|[^ ]+) UDP datagram( \\(IP opts\\)|) from "
				  /* host    */
				    "([^ ]+)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_icmpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
		                    "ICMP message type "
				  /* Main part .. message type .. */
				    "([a-z ]+)( - [_A-Za-z ]+|)( \\(IP opts\\)|) from "
				  /* host           */
				    "([.0-9a-zA-Z-]+)", 0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if (conf->self_host == NULL) {
			fprintf(stderr, "Input-ippl: config_parse: No self_host supplied when loglevel is %d!\n", conf->loglevel);
			return -1;
		}
	
		break;
	
	case 2:
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(ippl) using parser for 'detailed' logformat\n");
		/* Set the detailed logregexps .. */
		if ((conf->match_linetype = 
	        	          /* date                             */
		      pcre_compile( "[a-zA-Z]{3} [0-9 ]{2} [:0-9]{8} "
				  /* First word could be: last (ok), ICMP(ok), * (not ok)
				   * Second word could be: message (ok), UDP (ok), connection (ok) */
		                    //"(last|ICMP|port [0-9]{1,5}|[a-z-]+) (message|UDP|connection)", 
				    "(.+?) (message|UDP|connection|ipmon\\[[0-9]+\\]:)", // Adds support for ipmon lines
	                	    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_tcpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* portnum|portname                   attempt|closed */
		                    "(port [0-9]{1,5}|[^ ]+) connection (attempt|closed)"
				  /* (IP opts)|-       from */
				    "( \\(IP opts\\)|) from "
				  /* [ident@]host (src_ip:srcport    ->dst_ip:dstport) */
				    "([@.0-9a-zA-Z-]+) \\(([.0-9]+):([0-9]{1,5})->([.0-9]+):([0-9]{1,5})\\)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_udpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* portnum|portname                   (IP opts)|- from */
		                    "(port [0-9]{1,5}|[^ ]+) UDP datagram( \\(IP opts\\)|) from "
				  /* host      (src_ip:srcport    ->dst_ip:dstport) */
				    "([^ ]+) \\(([.0-9]+):([0-9]{1,5})->([.0-9]+):([0-9]{1,5})\\)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}
	
		if ((conf->match_icmpline =
				  /* date */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) "
				  /* Fix message .. don't need to get back */
		                    "ICMP message type "
				  /* Main part .. message type .. */
				    "([a-z ]+)( - [_A-Za-z ]+|)( \\(IP opts\\)|) from "
				  /* host      (src_ip:srcport    ->dst_ip:dstport) */
				    "([^ ]+) \\(([.0-9]+)->([.0-9]+)\\)",
		                    0, &errptr, &erroffset, NULL )) == NULL ) {
			fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
			return -1;
		}


		// Added 06/09/03. Matches ipmon lines
		if ((conf->match_ipmon =
				  /* date                              host  "ipmon[106]" */
		      pcre_compile( "([a-zA-Z]{3} [0-9 ]{2} [:0-9]{8}) (.+?) (.+?): "

			  /*              "#x" + fxp#                                        */
          	          /* start_time                       group-rule(@G:R) action(1 ltr) */
			  "(.+?) ([0-9]+x fxp[0-9]+|fxp[0-9]+) (\\@[0-9]+:[0-9]+) ([a-zA-Z]) "

			  /* src_ip src_port)    ->   dest_ip dest_port)   "PR"     protocol */
			  "([.0-9]+),([0-9]+) (-\\>) ([.0-9]+),([0-9]+) ([a-zA-Z]+) ([a-zA-Z]+) "

			  /* "len" hdr_lnth         opts(-...) +
			                  tot_lnth                "IN" */
			  "(len) ([0-9]+) ([0-9]+) (-[a-zA-Z]+ IN |IN )$",

			  0, &errptr, &erroffset, NULL )) == NULL ) {

		  fprintf(stderr, "%s.%d: regexp compilation error at %s\n", __FILE__, __LINE__, errptr );
		  return -1;
		}
		  // End IPMON
	
		break;
		
	default:
		fprintf( stderr, "Bad configvalue (loglevel: %d [range: %d-%d]) in configuration!\n", 
			 conf->loglevel, 0, 2 );
		return -1;
	}
	
	conf->match_linetype_extra = pcre_study(conf->match_linetype, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}
	conf->match_tcp_extra = pcre_study(conf->match_tcpline, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}
	conf->match_udp_extra = pcre_study(conf->match_udpline, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}
	conf->match_icmp_extra = pcre_study(conf->match_icmpline, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}

	// Added 06/09/03 for IPMON lines
	conf->match_ipmon_extra = pcre_study(conf->match_ipmon, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}

	return 0;
}


int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_input_ippl_dlinit;
	func->dlclose = mplugins_input_ippl_dlclose;
	func->parse_config = mplugins_input_ippl_parse_config;
	func->set_defaults = mplugins_input_ippl_set_defaults;
	func->get_next_record = mplugins_input_ippl_get_next_record;
	func->insert_record = NULL;
	func->gen_report = NULL;
        func->gen_history = NULL;

	return 0;
}
