/* 
 *   Creation Date: <1999/11/02 01:17:09 samuel>
 *   Time-stamp: <2002/11/24 15:15:04 samuel>
 *   
 *	<of1275.c>
 *	
 *	OF interface implementation (IEEE 1275)
 *   
 *   Copyright (C) 1999, 2000, 2002 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

/* The draft 1275 standard is available from ftp://playground.sun.com/pub/p1275/coredoc/ */

#include "of.h"
#include "string.h"
#include "ofmem.h"
#include "prom.h"
#include "phandles.h"
#include "processor.h"
#include "osi_calls.h"
#include "fs.h"
#include "adb_to_char.h"

#define LOG(a,...)	do {} while(0)

static char*		my_strncpy( char *dest, const char *src, size_t n );
static int 		of_call_method_( ulong args[], ulong ret[], int nargs, int nret );

typedef struct prom_args {
        const char 	*service;
        int 		nargs;
        int 		nret;
        ulong 		args[10];		/* MAX NUM ARGS! */
} prom_args_t;

static prom_args_t 	*s_pargs;
static char 		*s_service;
static int		met_nret;
static int		met_nargs;


/************************************************************************/
/*	parameter passing						*/
/************************************************************************/

static char *
local_buf( const char *buf, int len )
{
	char *p;
	if( !buf )
		return NULL;
	p = malloc( len );
	memcpy( p, buf, len );
	return p;
}

/* strncpy without 0-pad */
static char*
my_strncpy( char *dest, const char *src, size_t n )
{
	int len = MIN( n, strlen(src)+1 );
	return memcpy( dest, src, len );
}


/************************************************************************/
/*	OF implementation						*/
/************************************************************************/

static int 
of_nop( ulong args[], ulong ret[] ) 
{
	int i;
	printm("*** nop_'%s' [%d,%d]  ", s_service, s_pargs->nargs, s_pargs->nret );

	for(i=0; i<s_pargs->nargs; i++ )
		printm(" %08lX", args[i] );
	printm("\n");

	return 0;
}

static int 
of_interpret( ulong args[], ulong ret[] ) 
{
	char buf[32];
	strncpy( buf, (char*)args[0], sizeof(buf) );
	buf[ sizeof(buf)-1 ] = 0;
	printm("of_interpret '%s'\n", buf );
	return 0;
}


/* -- int msecs */
static int 
of_milliseconds( ulong args[], ulong ret[] ) 
{
	static ulong mticks=0, usecs=0;
	ulong t;
	
	asm volatile("mftb %0" : "=r" (t) : );
	if( mticks )
		usecs += OSI_MticksToUsecs( t-mticks );
	mticks = t;
	ret[0] = usecs/1000;
	return 0;
}


/* phandle -- int sibling_phandle */
static int 
of_peer( ulong args[], ulong ret[] ) 
{
	ret[0] = prom_peer( args[0] );
	return 0;
}

/* int phandle -- int parent_phandle */
static int 
of_parent( ulong args[], ulong ret[] ) 
{
	ret[0] = prom_parent( args[0] );
	/* printm("of_parent '%s' -- '%s'\n", dn->full_name, dn->parent ? dn->parent->full_name : "root" ); */
	return 0;
}

/* int phandle -- int child_phandle */
static int 
of_child( ulong args[], ulong ret[] ) 
{
	ret[0] = prom_child( args[0] );
	/* printm("of_child '%s' -- '%s'\n", dn->full_name, dn->child ? dn->child->full_name : "last" );  */
	return 0;
}

/* char *device_specifier --- int phandle */
static int 
of_finddevice( ulong args[], ulong ret[] ) 
{
	ret[0] = prom_find_device( (char*)args[0] );
	/* printm("of_finddevice %s [ %08lX ]\n", (char*)args[0], ret[0] );  */
	return 0;
}


/* int phandle, char *previous, void *buf -- int flag */
static int 
of_nextprop( ulong args[], ulong ret[] ) 
{
	char *prev = strdup( (char*)args[1] );
	char *next = malloc( 32 );
	*next = 0;
	ret[0] = prom_next_prop( args[0], prev, next );
	my_strncpy( (char*)args[2], next, 32 );
	free( prev );
	free( next );
	return 0;
}

/* int phandle, char *name, void *buf, int buflen --- int size */
static int 
of_getprop( ulong args[], ulong ret[] ) 
{
	char *buf, *prop = strdup( (char*)args[1] );

	/* Sometimes args[0]==0. This might be due to a bug somewhere.
	 * Possibly it is correct to return the root node, but I'm not sure.
	 */
	if( !args[0] ) {
#if 0
		static int warned=0;
		if( !warned++ )
			printm("************ WARNING, phandle == 0 ***********\n");
#endif
		args[0] = prom_find_device("/");
	}
	if( (ret[0]=prom_get_prop_len( args[0], prop )) >= 0 ) {
		buf = malloc( ret[0] );
		if( (ret[0]=prom_get_prop( args[0], prop, buf, ret[0] )) > 0 )
			memcpy( (char*)args[2], buf, MIN(ret[0],args[3]) );
		free( buf );
	}
	free( prop );
	return 0;
}

/* int phandle, char *name --- int size */
static int 
of_getproplen( ulong args[], ulong ret[] ) 
{
	char *prop = strdup( (char*)args[1] );
	ret[0] = prom_get_prop_len( args[0], prop );
	free( prop );
	return 0;
}

/* int phandle, char *name, void *buf, int len -- int size */
static int 
of_setprop( ulong args[], ulong ret[] ) 
{
	char *prop = strdup( (char*)args[1] );
	char *buf = local_buf( (char*)args[2], args[3] );

	ret[0] = prom_set_prop( args[0], prop, buf, args[3] );

	free( buf );
	free( prop );
	return 0;
}

/* int phandle, void *buf, int buflen -- int length */
static int
of_pack_to_path( ulong args[], ulong ret[] ) 
{
	char buf[256];
	
	ret[0] = prom_package_to_path( args[0], buf, sizeof(buf) );
	if( args[2] && (int)ret[0] != -1 )
		my_strncpy( (char*)args[1], buf, args[2] );

	/* printm("of_canon: %08lx -- '%s'\n", args[0], (char*)args[1] ); */
	return 0;
}

/* char *device_specifier, void *buf, int buflen -- int length */
static int 
of_canon( ulong args[], ulong ret[] ) 
{
	char *path = strdup( (char*)args[0] );
	mol_phandle_t ph = prom_find_device( path );
	free( path );

	if( ph == -1 ) {
		ret[0] = -1;
		return 0;
	}
	args[0] = ph;
	return of_pack_to_path( args, ret );
}

/* int ihandle, void *buf, int buflen -- int length */
static int
of_inst_to_path( ulong args[], ulong ret[] ) 
{
	args[0] = ihandle_to_phandle( args[0] );

	if( !args[0] ) {
		ret[0] = -1;
		return 0;
	}
	/* printm("of_inst_to_path... %08lX %08lX %08lX\n", args[0], args[1], args[2] ); */
	return of_pack_to_path( args, ret );
}

/* int ihandle -- int phandle */
static int
of_inst_to_pack( ulong args[], ulong ret[] ) 
{
	ret[0] = ihandle_to_phandle( args[0] );
	if( !ret[0] )
		ret[0] = -1;

	/* printm("of_inst_to_pack %08lx %08lX\n", args[0], ret[0] ); */
	return 0;
}

static int
parse_dev_spec( char *spec, int *partition, char **filename )
{
	char *s, *endp;

	*filename = NULL;
	if( (s=strchr(spec, ':')) ) {
		if( !s[1] )
			return -1;
		*partition = strtol( s+1, &endp, 10 );
		if( !*endp )
			return 0;
		if( *endp != ',' ) {
			*partition = -1;	/* unspecified partition  */
			endp = s+1;
		}
		*filename = s+1;
	}
	return -1;
}

static int
of_open_pseudofs( char *spec ) 
{
	file_desc_t fd;
	char *name;
	int par;
	
	/* fd == 0 is the 'raw' device */

	parse_dev_spec( spec, &par, &name );
	
	/* printm("pseudofs_open %s -- %d %s\n", spec, par, name ); */
	if( !name )
		return 0;
	if( !(fd=pseudo_open(name)) )
		return -1;
	return (int)fd;
}

/* char *device_specifier -- int ihandle */
static int
of_open( ulong args[], ulong ret[] ) 
{
	char *path = strdup( (char*)args[0] );
	mol_phandle_t ph;
	int molih, fd=0;

	ph = prom_find_device( path );
	free( path );

	/* printm("of_open: %s\n", (char*)args[0] ); */

	ret[0] = 0;
	if( ph == -1 ) {
		printm("of_open: node '%s' missing\n", (char*)args[0]);
		/* XXX: check return codes.... */
		return -1;
	}
	molih = phandle_to_molih( ph );
	
	switch( molih ) {
	case 0: /* unknown node */
		printm("of_open <%s>\n", (char*)args[0] );
		break;
	case molih_pseudofs:
		fd = of_open_pseudofs( (char*)args[0] );
		break;
	case molih_blockdev:
	default:
		break;
	}
	if( fd < 0 )
		return -1;
	ret[0] = allocate_ihandle( molih, ph, (void*)fd );
	/* printm("of_open: %s (%ld)\n", (char*)args[0], ret[0] ); */
	return 0;
}

/* int ihandle --  */
static int
of_close( ulong args[], ulong ret[] ) 
{
	int ih = args[0];
	int molih = ihandle_to_molih( ih );
	file_desc_t fd;
	
	switch( molih ) {
	case molih_pseudofs:
		if( !get_ihandle_data(ih, (void**)&fd) && fd )
			pseudo_close( fd );
		break;
	case molih_blockdev:
		break;
	}
	free_ihandle( args[0] );
	return 0;
}

/* int ihandle, void *addr, int len -- int actual */
static int 
of_read( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"read", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 ); 	
	ret[0] = lret[1];
	return err;
}

/* int ihandle, void *addr, int len -- int actual */
static int 
of_write( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"write", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 ); 	
	ret[0] = lret[1];
	return err;
}

/* int ihandle, int pos_hi, int pos_lo -- int status */
static int
of_seek( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"seek", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 );
	ret[0] = lret[1];
	return err;
}

/* void *virt, int size, int align -- void *baseaddr */
static int 
of_claim( ulong args[], ulong ret[] ) 
{
	/* virt should be used if align == 0 */

	ret[0] = ofmem_claim( args[2] ? 0 : args[0], args[1], args[2] );
	return 0;
}


/* void *virt, int size -- */
static int 
of_release( ulong args[], ulong ret[] ) 
{
	ofmem_release( args[0], args[1] );
	return 0;
}

/* -- */
static int 
of_exit( ulong args[], ulong ret[] ) 
{
	printm("********** of_exit *********\n");
	OSI_Exit();
	return 0;
}

/* -- */
static int 
of_quiesce( ulong args[], ulong ret[] ) 
{
#if 0
	ulong msr;
	/* This seems to be the correct thing to do - but I'm not sure */
	asm volatile("mfmsr %0" : "=r" (msr) : );
	msr &= ~(MSR_IR | MSR_DR);
	asm volatile("mtmsr %0" :: "r" (msr) );
#endif
	printm("=============================================================\n\n");
	prom_close();
	return 0;
}

/* char *method, int ihandle, stack_arg_top, stack_arg2, stackarg3 
 * -- catch_result, stack-res1, stack-res2
 */
static int
of_call_method( ulong args[], ulong ret[] ) 
{
	return of_call_method_( args, ret, s_pargs->nargs, s_pargs->nret );
}


/************************************************************************/
/*	node methods							*/
/************************************************************************/

static int 
met_nop( ulong ih, ulong args[], ulong ret[] ) 
{
	printm("* met_nop '%s' [%d/%d]\n", (char*)s_pargs->args[0], met_nargs, met_nret );
	return 0;
}

static int 
met_set_hybernot_flag( ulong ih, ulong args[], ulong ret[] ) 
{
	return 0;
}

/* ( phys, size, align --- base ) */
static int 
met_mem_claim( ulong ih, ulong args[], ulong ret[] ) 
{
	ret[0] = ofmem_claim_phys( args[0], args[1], args[2] );
	return (ret[0] == -1) ? -1 : 0;
}

/* ( virt, size, align ---  base ) */
static int 
met_mmu_claim( ulong ih, ulong args[], ulong ret[] ) 
{
	ret[0] = ofmem_claim_virt( args[0], args[1], args[2] );
	return (ret[0] == -1) ? -1 : 0;
}

/* ( virt --- phys mode true ) */
static int 
met_mmu_translate( ulong ih, ulong args[], ulong ret[] ) 
{
	/*printm("met_mmu_translate: %08lx\n", args[0] ); */
	ret[1] = 0;
	ret[0] = ofmem_translate( args[0], &ret[1] );
	ret[2] = 0;

	return (ret[0] == -1) ? -1 : 0;
}

/* ( ??? virt --- phys mode true ) */
static int
met_mmu_translate23( ulong ih, ulong args[], ulong ret[] ) 
{
	return met_mmu_translate( ih, &args[1], ret );
}


/* ( phys, virt, size, mode -- [ret] ) */
static int 
met_mmu_map( ulong ih, ulong args[], ulong ret[] ) 
{
	ulong r;

	r = ofmem_map( args[0], args[1], args[2], args[3] ) ? -1 : 0;
	if( met_nret == 1 )
		ret[0] = r;	/* ? */
	return r;	/* or ? */
}


/* ( physbase -- rtas_callback ) */
static int
met_rtas_instantiate( ulong ih, ulong args[], ulong ret[] )
{
	int s=0x1000, size = (int)of_rtas_end - (int)of_rtas_start;
	ulong virt;
	
	while( s < size )
		s += 0x1000;
	virt = ofmem_claim_virt( 0, s, 0x1000 );
	ofmem_map( args[0], virt, s, -1 );
	memcpy( (char*)virt, of_rtas_start, size );

	printm("RTAS instantiated at %08lx\n", args[0] );
	flush_icache_range( (char*)virt, (char*)virt + size );
	ret[0] = args[0];
	return 0;
}


/* ( addr, len -- actual ) */
static int 
met_stdout_write( ulong ih, ulong args[], ulong ret[] ) 
{
	char *s = malloc( args[1] + 1);
	my_strncpy( s, (char*)args[0], args[1] );
	s[args[1]]=0;
	
	/* printm( "%s", s ); */
	console_draw_str( s );
	free( s );
	return 0;
}

/************************************************************************/
/*	pseudofs 							*/
/************************************************************************/

/* ( pos_hi, pos_lo -- status ) */
static int 
met_pseudofs_seek( ulong ih, ulong args[], ulong ret[] ) 
{
	file_desc_t fd;
	
	/* printm("pseudofs_seek %08lX %08lX\n", args[0], args[1] );  */
	if( get_ihandle_data(ih, &fd) || !fd ) {
		ret[0] = 0;
	} else {
		pseudo_lseek( fd, args[1], SEEK_SET );
		ret[0] = 0;
	}
	return 0;
}

/* ( addr, len -- actual ) */
static int 
met_pseudofs_read( ulong ih, ulong args[], ulong ret[] ) 
{
	file_desc_t fd;
	
	if( get_ihandle_data(ih, &fd) || !fd ) {
		int len = MIN( 0x1000, args[1] );
		memset( (char*)args[0], 0, len );
		ret[0] = len;
	} else
		ret[0] = pseudo_read( fd, (char*)args[0], args[1] );
	return 0;
}

/* ( -- blocksize ) */
static int
met_pseudofs_blocksize( ulong ih, ulong args[], ulong ret[] )
{
	ret[0] = 512;
	return 0;
}


/************************************************************************/
/*	nvram								*/
/************************************************************************/

static int nvram_mark=0;

/* ( pos_hi, pos_lo -- status ) */
static int 
met_nvram_seek( ulong ih, ulong args[], ulong ret[] ) 
{
	/* printm("NVRAM: seek %08lX %08lX\n", args[0], args[1] );  */
	nvram_mark = args[1];
	return 0;
}

/* ( addr, len -- actual ) */
static int 
met_nvram_read( ulong ih, ulong args[], ulong ret[] ) 
{
	char *p = (char*)args[0];
	int n;

	/* printm("NVRAM: read %08lX %08lX\n", args[0], args[1] ); */

	for( n=args[1]; n>0 ; n-- )
		*p++ = OSI_ReadNVRamByte( nvram_mark++ );
	ret[0] = args[1];
	return 0;
}

/* ( addr, len -- actual ) */
static int 
met_nvram_write( ulong ih, ulong args[], ulong ret[] ) 
{
	char *p = (char*)args[0];
	int n;

	/* printm("NVRAM: write %08lX %08lX\n", args[0], args[1] ); */

	for( n=args[1]; n>0 ; n-- )
		OSI_WriteNVRamByte( nvram_mark++, *p++ );
	ret[0] = args[1];
	return 0;

}


/* ( -- size ) */
static int 
met_nvram_size( ulong ih, ulong args[], ulong ret[] ) 
{
	/* printm("NVRAM: size\n"); */
	ret[0] = OSI_NVRamSize();
	return 0;
}


/************************************************************************/
/*	video								*/
/************************************************************************/

/* ( -- width height ) (?) */
static int
met_video_dimensions( ulong ih, ulong args[], ulong ret[] )
{
	return video_get_res( (int*)&ret[0], (int*)&ret[1] );
}

/* ( table, start, count ) (?) */
static int
met_video_set_colors( ulong ih, ulong args[], ulong ret[] )
{
	uchar *p = (char*)args[0];
	int i;

	for( i=0; i<args[2]; i++, p+=3 ) {
		ulong col = (p[0] << 16) | (p[1] << 8) | p[2];
		set_color( i + args[1], col );
	}
	refresh_palette();
	return 0;
}

/* r g b index -- */
static int
met_video_color_bang( ulong ih, ulong args[], ulong ret[] ) 
{
	ulong col = ((args[0] << 16) & 0xff0000) | ((args[1] << 8) & 0x00ff00) | (args[2] & 0xff);
	/* printm("color!: %08lx %08lx %08lx %08lx\n", args[0], args[1], args[2], args[3] ); */
	set_color( args[3], col );
	refresh_palette();
	return 0;
}

/* ( color_ind x y width height -- ) (?) */
static int
met_video_fill_rect( ulong ih, ulong args[], ulong ret[] )
{
	fill_rect( args[0], args[1], args[2], args[3], args[4] );
	return 0;
}


/************************************************************************/
/*	keyboard							*/
/************************************************************************/

static char keytable[32];

/* ( -- keymap ) (?) */
/* should return a pointer to an array with 32 bytes (256 bits) */
static int
met_kbd_get_key_map( ulong ih, ulong args[], ulong ret[] )
{
	/* printm("met_kbd_get_key_map\n"); */

	ret[0] = (ulong)keytable;
	keytable[5] = 0x40;	/* test key is tested... */
	return 0;
}

/* ( buf len --- actlen ) */
static int
met_kbd_read( ulong ih, ulong args[], ulong ret[] )
{
	char *p = (char*)args[0];
	int key;
	
	if( !p || !args[1] ) {
		ret[0] = -1;
		return 0;
	}
	OSI_USleep(1);	/* be nice */ 
	ret[0] = 0;
	for( ; (key=OSI_GetAdbKey()) >= 0 ; ) {
		if( key & 0x80 )
			continue;
		ret[0] = 1;
		*p = adb_to_char_table[ key & 0x7f ];
		if( !*p )
			*p = 'x';
	}
	return 0;
}


/************************************************************************/
/*	OF command table						*/
/************************************************************************/

static struct {
	char 			*service;
	int			(*proc)( ulong args[], ulong ret[] );
	int			nargs;
	int			nret;
} cmd_table[] = {
/* 6.3.2.1 Client interface */
	{"test", 		of_nop, 	1,1},	/* char *name -- int missing */

/* 6.3.2.2 Device tree */
	{"peer", 		of_peer, 	1,1},	/* int phandle -- int sibling_phandle */
	{"child", 		of_child, 	1,1},	/* int phandle -- int child_phandle */
	{"parent", 		of_parent, 	1,1},	/* int phandle -- int parent_phandle */
	{"getproplen", 		of_getproplen, 	2,1},	/* int phandle, char *name -- int proplen */
	{"getprop", 		of_getprop, 	4,1},	/* int phandle, char *name, void *buf, int buflen -- int size */
	{"nextprop", 		of_nextprop, 	3,1},	/* int phandle, char *previous, void *buf -- int flag */
	{"setprop",		of_setprop,	4,1},	/* int phandle, char *name, void *buf, int len -- int size */
	{"canon",		of_canon,	3,1},	/* char *device_specifier, void *buf, int buflen -- int length */
	{"finddevice", 		of_finddevice, 	1,1},	/* char *device_specifier -- int phandle */
	{"instance-to-package", of_inst_to_pack,1,1},	/* int ihandle -- int phandle */
	{"instance-to-path",	of_inst_to_path,3,1},	/* int ihandle, void *buf, int buflen -- int length */
	{"package-to-path",	of_pack_to_path,3,1},	/* int phandle, void *buf, int buflen -- int length */
	{"call-method",		of_call_method,	-1,-1},	/* char *method, int ihandle -- int args[] */

/* 6.3.2.3 Device I/O */
	{"open",		of_open,	1,1},	/* char *device_specifier -- int ihandle */
	{"close",		of_close,	1,0},	/* int ihandle -- */
	{"read",		of_read,	3,1},	/* int ihandle, void *addr, int len -- int actual */
	{"write",		of_write,	3,1},	/* int ihandle, void *addr, int len -- int actual */
	{"seek",		of_seek,	3,1},	/* int ihandle, int pos_hi, int pos_lo -- int status */

/* 6.3.2.4 Memory */
	{"claim",		of_claim,	3,1},	/* void *virt, int size, int align -- void *baseaddr */
	{"release",		of_release,	2,0},	/* void *virt, int size -- */

/* 6.3.2.5 Control transfer */
	{"boot",		of_nop,		1,0},	/* char *bootspec -- */
	{"enter",		of_nop,		0,0},	/* -- */
	{"exit",		of_exit,	0,0},	/* -- */
	{"chain",		of_nop,		4,0},	/* void *virt, int size, void *entry, void *args, int len -- */

/* 6.3.2.6 User interface */
	{"interpret",		of_interpret,	-1,-1},	/* char *cmd args[] -- args[] */
	{"set-callback",	of_nop,		1,1},	/* void *newfunc -- void *oldfunc */
	{"set-symbol-lookup",	of_nop,		2,0},	/* void *sym_to_value, void *value_to_sym, -- */

/* 6.3.2.7 Time */
	{"milliseconds",	of_milliseconds,0,1},	/* -- int ms */

/* Documented in one of the supplements? */
	{"quiesce",		of_quiesce,	0,0},	/* -- */
	
	{NULL, 0, 0, 0 }
};


/* This is the OF client interface. r3 points to the argument array.
 * On return, r3 should contain 0==true or -1==false. r4-r12,cr0,cr1 may
 * be modified freely.
 */
int
of_client_interface( int *params )
{
	prom_args_t *args = (prom_args_t*)params;
	char *service = (char*)args->service;
	ulong largs[10];	/* passed arguments must be local */
	int i,j;

	/* printm("of_client_interface: %s\n", args->service ); */
	
	if( args->nargs < 0 || args->nret < 0 ) {
		LOG("nargs or nret negative (%d,%d)\n",args->nargs, args->nret );
		return 0;
	}

	for( i=0; cmd_table[i].service; i++ ) {
		if( strcmp( cmd_table[i].service, service ) )
			continue;
		if( (cmd_table[i].nargs == args->nargs || cmd_table[i].nargs == -1 ) &&
		    (cmd_table[i].nret == args->nret || cmd_table[i].nret == -1 )) 
		{
			for( j=0; j<args->nargs; j++ )
				largs[j] = args->args[j];

			s_pargs = args;
			s_service = service;
			return (*cmd_table[i].proc)( largs, &args->args[args->nargs] );
		}
	}
	printm( "Unimplemented service '%s' [%d,%d]\n", service, args->nargs, args->nret );
	return 0;
}


/************************************************************************/
/*	Node methods							*/
/************************************************************************/

typedef struct {
	char 		*method;
	int		molih;
	int		(*proc)( ulong ihandle, ulong args[], ulong ret[] );
	int		nargs;
	int		nret;
} of_method_t;

/* node methods (invoked by call-method) */
static of_method_t method_table[] = {
	{"claim",		molih_memory, 	met_mem_claim,		3,1},	/* ( phys, size, align --- base ) */
	{"release",		molih_memory, 	met_nop,		2,0},	/* ( phys, size --- ) */

	{"claim",		molih_mmu, 	met_mmu_claim,		3,1},	/* ( virt, size, align ---  base ) */
	{"release",		molih_mmu, 	met_nop,		2,0},	/* ( virt, size --- ) */
	{"map",			molih_mmu, 	met_mmu_map,		4,-1},	/* ( phys, virt, size, mode -- [ret] ) */
	{"unmap",		molih_mmu, 	met_nop,		2,0},	/* ( virt, size --- ) */
	{"translate",		molih_mmu,	met_mmu_translate,	1,3},	/* ( virt --- phys mode true ) */
	{"translate",		molih_mmu,	met_mmu_translate23,	2,3},	/* ( virt, ? --- phys mode true ) */

	{"instantiate-rtas",	molih_rtas,	met_rtas_instantiate,	1,1},	/* ( physbase -- rtas_callback ) */

	{"seek",		molih_nvram,	met_nvram_seek,		2,1},	/* ( pos_hi, pos_lo -- status ) */
	{"size",		molih_nvram,	met_nvram_size,		0,1},	/* ( -- size ) */
	{"read",		molih_nvram,	met_nvram_read,		2,1},	/* ( addr, len -- actual ) */
	{"write",		molih_nvram,	met_nvram_write,	2,1},	/* ( addr, len -- actual ) */

	{"write",		molih_stdout,	met_stdout_write,	2,1},	/* ( addr, len -- actual ) */

	{"dimensions",		molih_video,	met_video_dimensions,	0,2},
	{"set-colors",		molih_video,	met_video_set_colors,	3,0},
	{"fill-rectangle",	molih_video,	met_video_fill_rect,	5,0},

	{"color!",		molih_video,	met_video_color_bang,	4,0},

	{"get-key-map",		molih_kbd,	met_kbd_get_key_map,	0,1},
	{"read",		molih_kbd,	met_kbd_read,		2,1},

	{"set-hybernot-flag",	molih_powermgt,	met_set_hybernot_flag,	0,0},

	{"read",		molih_pseudofs,	met_pseudofs_read,	2,1},	/* ( addr, len -- actual ) */
	{"seek",		molih_pseudofs,	met_pseudofs_seek,	2,1},	/* ( pos_hi, pos_lo -- status ) */
	{"block-size",		molih_pseudofs, met_pseudofs_blocksize,	0,1},	/* ( -- blocksize ) */

	{NULL,			0,0,0,0	}
};


static int
of_call_method_( ulong args[], ulong ret[], int tot_nargs, int tot_nret ) 
{
	int molih = ihandle_to_molih( args[1] );
	ulong largs[10], lret[10];
	int i, nargs, nret;
	of_method_t *t;

	nargs = tot_nargs - 2;
	nret = tot_nret - 1;

	for( t=method_table; t->method; t++ ){
		if( strcmp( (char*)args[0], t->method ) || molih != t->molih )
			continue;
		if( (t->nargs != -1 && nargs != t->nargs) || (t->nret != -1 && nret != t->nret ) )
			continue;
		/* stack_argN, ..., arg1  --  retN ... ret1 */
		for( i=0; i<nargs; i++ )
			largs[i] = args[1 + nargs - i];

		/* catch result in r[0] --- normally 0 */
		met_nargs = nargs;
		met_nret = nret;

		ret[0] = (*t->proc)( args[1], largs, lret );

		for(i=0; i<nret; i++ )
			ret[i+1] = lret[nret-i-1];
		return 0;
	}
#if 1
	{
		char buf[80];
		printm("* of_call_method: cmd '%s' ih: %08lX[%d] (%d/%d) Args: ", 
		       (char*)args[0], args[1], molih, nargs, nret );
		for(i=2; i<tot_nargs; i++ ) {
			printm(" %08lX", args[i] );
		}
		printm("\n");
		if( prom_package_to_path( ihandle_to_phandle(args[1]), buf, sizeof(buf) ) != -1 )
			printm("[%s]\n", buf );
	}
#endif
	return 0;
}
