/* libmondo-string.c

- string manipulation


05/06/2003
- cleaned up severity_of_difference() a bit

11/17/2002
- strip_spaces() now accommodates _smaller_ strings auto'y

11/08
- if decimal point in string sent to friendly_sizestr_to_sizelong()
  then fatal error: we expect integers only

10/01 - 10/31
- commented subroutines
- strip_spaces() allows up to MAX_STR_LEN-len input string

08/01 - 08/31
- fixed bug in friendly_sizestr_to_sizelong() which stopped
  it from working with capital G's and K's
- fixed bug in build_partition_name()

07/24
- created
*/



#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo-string.h"
#include "lib-common-externs.h"
#include "libmondo-gui-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-files-EXT.h"



extern int g_current_media_number;
extern long long g_tape_posK;


 /*************************************************************************
 * build_partition_name() -- Philippe De Muyter                        
 *                                                                       
 * Purpose:   
 * Called by: does_partition_exist()                                      *
 * Params:    partition [returned]       string containing partition name *
 *            drive                      e.g. /dev/hda                    *
 *            partno                     e.g. 1                           *
 * Returns:   partition                  ... e.g. /dev/hda1 :-)           *
 * NB:        If drive ends in a digit then 'p' is added before partno,   *
 *            for Compaq and other odd devices                            *
 *************************************************************************/
char *
build_partition_name (char *partition, char *drive, int partno)
{
  char *p;

  p = strcpy(partition, drive);
  p += strlen(p);
  if (isdigit (p[-1]))
    {
      *p++ = 'p';
    }
  sprintf (p, "%d", partno);
  return (partition);
}














/*************************************************************************
 * center_string() -- Hugo Rabson                                        *
 *                                                                       *
 * Purpose:  Center-pads a string                                        *
 * Called by:...                                                         *
 * Params:   in_out          string to be center-padded                  *
 *           width           how wide the output field is (e.g. 80 chars)*
 * Returns:  void                                                        *
 *           [in_out's string's value is changed]                        *
 *************************************************************************/
void center_string (char *in_out, int width)
{
  char scratch[MAX_STR_LEN];
  char *p;
  int i;			/* purpose */
  int len;			/* purpose */
  int mid;			/* purpose */
  int x;			/* purpose */

  assert(in_out!=NULL);
  if (in_out[0] == '\0')
    {
      return;
    }
  for (p = in_out; *p == ' '; p++);
  strcpy (scratch, p);
  len = strlen (scratch);
  mid = width / 2;
  x = mid - len / 2;
  for (i = 0; i < x; i++)
    {
      in_out[i] = ' ';
    }
  in_out[i] = '\0';
  strcat (in_out, scratch);
}






char* commarize(char*input)
{
  char pos_w_commas[MAX_STR_LEN];
  static char output[MAX_STR_LEN];
  char tmp[MAX_STR_LEN];
  int j;

  strcpy(tmp, input);
  if (strlen(tmp) > 6)
    {
      strcpy(pos_w_commas, tmp);
      j = strlen(pos_w_commas);
      tmp[j-6] = ',';
      strcpy(tmp+j-5, pos_w_commas+j-6);
//      tmp[j-2] = ',';
//      strcpy(tmp+j-1, pos_w_commas+j-3);
      strcpy(pos_w_commas, tmp);
    }
  if (strlen(tmp) > 3)
    {
      j = strlen(tmp);
      strcpy(pos_w_commas, tmp);
      pos_w_commas[j-3] = ',';
      strcpy(pos_w_commas+j-2, tmp+j-3);
    }
  else
    {
      strcpy(pos_w_commas, tmp);
    }
  strcpy(output, pos_w_commas);
  return(output);
}










/*************************************************************************
 * disklist_entry_to_string() -- Hugo Rabson	                         *
 *                                                                       *
 * Purpose:  Turn entry from RAID editor's disklist (list of constituent *
 *           partitions) into a GUI-friendly string to be displayed.     *
 * Called by:redraw_disklist                                             *
 * Params:   disklist        pointer to array disklist                   *
 *           lino            element# within array (first element is #0) *
 * Returns:  static string containing reformatted entry disklist[lino]   *
 *************************************************************************/
char *
disklist_entry_to_string (struct list_of_disks *disklist, int lino)
{

	/** buffers ***********************************************************/
  static char output[MAX_STR_LEN];



  sprintf (output, "%-24s %8d", disklist->el[lino].device,
	   disklist->el[lino].index);
  return (output);
}






/*************************************************************************
 * friendly_sizestr_to_sizelong() -- Hugo Rabson                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by: ...                                                        *
 * Params:    incoming          string containing user-friendly size,    *
 *                              e.g. 40m for 40 Megs or 2g for 3 Gigs    *
 * Returns:   value of incoming in Kilobytes, or 0 if error              *
 *************************************************************************/
long
friendly_sizestr_to_sizelong (char *incoming)
{
  long outval;
  int i;
  char tmp[MAX_STR_LEN];
  char ch;

  if (!incoming[0]) { return(0); }
  if (strchr(incoming, '.'))
    { fatal_error("Please use integers only. No decimal points."); }
  strcpy (tmp, incoming);
  i = strlen(tmp);
  if (tmp[i-1]=='B' || tmp[i-1]=='b') { tmp[i-1]='\0'; }
  for (i = 0; i < strlen (tmp) && isdigit (tmp[i]); i++);
  ch = tmp[i];
  tmp[i] = '\0';
  outval = atol (tmp);
  if (ch == 'g' || ch == 'G')
    {
      outval = outval * 1024;
    }
  else if (ch == 'k' || ch == 'K')
    {
      outval = outval / 1024;
    }
  else if (ch != 'm' && ch != 'M')
    {
      sprintf(tmp, "Re: parameter '%s' - bad multiplier ('%c')", incoming, ch);
      fatal_error(tmp);
    }
  return (outval);
}




/*************************************************************************
 * leftpad_string() -- Hugo Rabson                                       *
 *                                                                       *
 * Purpose:   Left-align string in array                                 *
 * Called by: ...                                                        *
 * Params:    incoming                           incoming string         *
 *            width                              width of array          *
 * Returns:   pointer to left-aligned string                             *
 *************************************************************************/
char *
leftpad_string (char *incoming, int width)
{
	/** buffers ******************************************************/
  static char output[MAX_STR_LEN];

	/** ints *********************************************************/
  int i;

	/** end vars *****************************************************/
  strcpy (output, incoming);
  for (i = strlen (output); i < width; i++)
    {
      output[i] = ' ';
    }
  output[i] = '\0';
  return (output);
}




/*************************************************************************
 * marker_to_string () -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:   Return user-readable string describing the meaning of a    *
 *            given marker byte (e.g. BLK_START_OF_BACKUP                *
 * Called by: ...                                                        *
 * Params:    marker                      integer containing marker byte *
 * Returns:   pointer to user-readable string                            *
 *************************************************************************/
char *
marker_to_string (int marker)
{
	/** buffer *******************************************************/
  static char outstr[MAX_STR_LEN];


	/** end vars ****************************************************/

  switch (marker)
    {
    case BLK_START_OF_BACKUP:
      strcpy (outstr, "BLK_START_OF_BACKUP");
      break;
    case BLK_START_OF_TAPE:
      strcpy (outstr, "BLK_START_OF_TAPE");
      break;
    case BLK_START_AN_AFIO_OR_SLICE:
      strcpy (outstr, "BLK_START_AN_AFIO_OR_SLICE");
      break;
    case BLK_STOP_AN_AFIO_OR_SLICE:
      strcpy (outstr, "BLK_STOP_AN_AFIO_OR_SLICE");
      break;
    case BLK_START_AFIOBALLS:
      strcpy (outstr, "BLK_START_AFIOBALLS");
      break;
    case BLK_STOP_AFIOBALLS:
      strcpy (outstr, "BLK_STOP_AFIOBALLS");
      break;
    case BLK_STOP_BIGGIEFILES:
      strcpy (outstr, "BLK_STOP_BIGGIEFILES");
      break;
    case BLK_START_A_BIGGIE:
      strcpy (outstr, "BLK_START_A_BIGGIE");
      break;
    case BLK_START_BIGGIEFILES:
      strcpy (outstr, "BLK_START_BIGGIEFILES");
      break;
    case BLK_STOP_A_BIGGIE:
      strcpy (outstr, "BLK_STOP_A_BIGGIE");
      break;
    case BLK_END_OF_TAPE:
      strcpy (outstr, "BLK_END_OF_TAPE");
      break;
    case BLK_END_OF_BACKUP:
      strcpy (outstr, "BLK_END_OF_BACKUP");
      break;
    case BLK_ABORTED_BACKUP:
      strcpy (outstr, "BLK_ABORTED_BACKUP");
      break;
    case BLK_START_FILE:
      strcpy (outstr, "BLK_START_FILE");
      break;
    case BLK_STOP_FILE:
      strcpy (outstr, "BLK_STOP_FILE");
      break;
    default:
      sprintf (outstr, "BLK_UNKNOWN (%d)", marker);
      break;
    }
  return (outstr);
}





/*************************************************************************
 * mountlist_entry_to_string() -- Hugo Rabson                            *
 *                                                                       *
 * Purpose:  Format line from mountlist as a GUI-friendly string         *
 * Called by:reformat_mountlist                                          *
 * Params:   mountlist       array containing mountlist                  *
 *           lino            element# within array (first element is #0) *
 * Returns:  static string containing reformatted line                   *
 *************************************************************************/
char *
mountlist_entry_to_string (struct mountlist_itself *mountlist, int lino)
{

	/** buffer ************************************************************/
  static char output[MAX_STR_LEN];



  sprintf (output, "%-24s %-24s %-10s %8ld", mountlist->el[lino].device,
	   mountlist->el[lino].mountpoint, mountlist->el[lino].format,
	   mountlist->el[lino].size / 1024);
  return (output);
}







/*************************************************************************
 * number_of_disks_as_string() -- Hugo Rabson      							         *
 *                                                                       *
 * Purpose:  Generate friendly string saying, "Blah blah.. N disks"      *
 * Called by:edit_raidlist_entry                                         *
 * Params:   noof_disks              number of disks                     *
 *           label                   'Blah blah' label at front          *
 * Returns:  static string containing friendly string                    *
 *************************************************************************/
char *
number_of_disks_as_string (int noof_disks, char *label)
{

	/** buffers **********************************************************/
  static char output[MAX_STR_LEN];

	/** char     *********************************************************/
  char p;


  if (noof_disks > 1)
    {
      p = 's';
    }
  else
    {
      p = ' ';
    }
  sprintf (output, "%d %s disk%c", noof_disks, label, p);
  while (strlen (output) < 14)
    {
      strcat (output, " ");
    }
  return (output);
}




/*************************************************************************
 * number_to_text() -- Hugo Rabson                                       *
 *                                                                       *
 * Purpose:   Turn supplied integer into user-readable, user-friendly str*
 * Called by: ...                                                        *
 * Parms:     i               integer to be turned into string           *
 * Returns:   char*           pointer to user-readable string            *
 *************************************************************************/
char *
number_to_text (int i)
{

	/** buffers ******************************************************/
  static char output[MAX_STR_LEN];


	/** end vars ****************************************************/

  switch (i)
    {
    case 0:
      strcpy (output, "zero");
      break;
    case 1:
      strcpy (output, "one");
      break;
    case 2:
      strcpy (output, "two");
      break;
    case 3:
      strcpy (output, "three");
      break;
    case 4:
      strcpy (output, "four");
      break;
    case 5:
      strcpy (output, "five");
      break;
    case 6:
      strcpy (output, "six");
      break;
    case 7:
      strcpy (output, "seven");
      break;
    case 8:
      strcpy (output, "eight");
      break;
    case 9:
      strcpy (output, "nine");
    case 10:
      strcpy (output, "ten");
    default:
      sprintf (output, "%d", i);
    }
  return (output);
}





/*************************************************************************
 * resolve_naff_tokens() -- Hugo Rabson                                  *
 *                                                                       *
 * Purpose:  Finds occurrences of token in string; replaces them with    *
 *           the string which the token represents; writes whole thing   *
 *           to output.                                                  *
 * Called by:...                                                         *
 * Params:   ip          input string                                    *
 *           output      output string                                   *
 *           token       token, to be replaced with...                   *
 *           value       value of token                                  *
 * Returns:  void                                                        *
 *           [output's string's value is changed]                        *
 *************************************************************************/
void
resolve_naff_tokens (char *output, char *ip, char *value, char *token)
{
	/** Buffers ****/
  char input[MAX_STR_LEN*2];

	/** Pointers **/
  char *p;

  strcpy (output, ip);		/* just in case the token doesn't appear in string at all */
  for (strcpy (input, ip); strstr (input, token); strcpy (input, output))
    {
      strcpy (output, input);
      p = strstr (output, token);
      *p = '\0';
      strcat (output, value);
      p = strstr (input, token) + strlen (token);
      strcat (output, p);
    }
}






/*************************************************************************
 * slice_fname() -- Hugo Rabson                                          *
 *                                                                       *
 * Purpose:   Generate filename of slice X of file Y in directory D, so  *
 *            that other subroutines can easily manipulate slices        *
 * Called by: verify_all_slices_*(), slice_up_file_etc()                 *
 * Params:    bigfileno                   big file number                *
 *            sliceno                     slice number (of this file)    *
 *            path                        path to append to returned str.*
 *            s                           suffix; if nonzero then use it *
 * Returns:   $path/slice-$bigfileno-$sliceno[.$s]                       *
 * NB:        If s is not null then add '.$s'; otherwise, don't add it   *
 *************************************************************************/
char *
slice_fname (long bigfileno, long sliceno, char *path, char *s)
{

	/** buffers *****************************************************/
  static char output[MAX_STR_LEN];
  static char suffix[MAX_STR_LEN];

	/** end vars ****************************************************/


  if (s[0] != '\0')
    {
      sprintf (suffix, ".%s", s);
    }
  else
    {
      suffix[0] = '\0';
    }
  sprintf (output, "%s/slice-%07ld.%05ld.dat%s", path, bigfileno, sliceno,
	   suffix);
  return (output);
}



/*************************************************************************
 * special_dot_char() -- Hugo Rabson                                     *
 *                                                                       *
 * Purpose:   Generate spinning symbol, based on incoming integer %4     *
 * Params:    i               integer (%4 --> symbol)                    *
 * Called by: ...                                                        *
 * Returns:   char of spinning globe                                     *
 *************************************************************************/
int
special_dot_char (int i)
{
  switch (i % 4)
    {
    case 0:
      return ('/');
    case 1:
      return ('-');
    case 2:
      return ('\\');
    case 3:
      return ('|');
    default:
      return ('.');
    }
  return ('.');
}





/*************************************************************************
 * spread_flaws_across_three_lines() -- Hugo Rabson                      *
 *                                                                       *
 * Purpose:   Take incoming string. Spread/wrap across 3 other strings.  *
 * Called by: ...                                                        *
 * Params:    flaws_str                   original string---input        *
 *            flaws_str_A [ret'd]         wrapped string [1/3]           *
 *            flaws_str_B [ret'd]         wrapped string [2/3]           *
 *            flaws_str_C [ret'd]         wrapped string [3/3]           *
 *            res                         result of earlier operation    *
 * Returns:   TRUE if res==0; else, FALSE                                *
 * NB:        flaws_str_[A,B,C]'s values are returned,changed.           *
 *            The returned boolean reflects the value of res.            *
 *************************************************************************/
bool
spread_flaws_across_three_lines (char *flaws_str, char *flaws_str_A,
				 char *flaws_str_B, char *flaws_str_C,
				 int res)
{

	/** int **************************************************************/
  int i = 0;

	/** initialize *******************************************************/
  flaws_str_A[0] = flaws_str_B[0] = flaws_str_C[0] = '\0';


  if (!res && !strlen (flaws_str))
    {
      return (TRUE);
    }
  if (strlen (flaws_str) > 0)
    {
      sprintf (flaws_str_A, "%s", flaws_str + 1);
    }
  if (strlen (flaws_str_A) >= 74)
    {
      for (i = 74; flaws_str_A[i] != ' '; i--);
      strcpy (flaws_str_B, flaws_str_A + i + 1);
      flaws_str_A[i] = '\0';
    }
  if (strlen (flaws_str_B) >= 74)
    {
      for (i = 74; flaws_str_B[i] != ' '; i--);
      strcpy (flaws_str_C, flaws_str_B + i + 1);
      flaws_str_B[i] = '\0';
    }
  if (res)
    {
      return (FALSE);
    }
  else
    {
      return (TRUE);
    }
}




/*************************************************************************
 * strcmp_inc_numbers() -- Hugo Rabson                                   *
 *                                                                       *
 * Purpose:   Like strcmp(a,b) but we say hda5 is 'less than' hda11  :-) *
 * Called by: ...                                                        *
 * Params:    stringA, stringB           strings to be compared          *
 * Returns:   <0 if A<B; 0 if A==B; 1 if B>A                             *
 * NB:        This subroutine _might_ malfunction if an incoming string  *
 *            has more than one set of numbers - e.g. /dev/sb2d/cd2      *
 *************************************************************************/
int
strcmp_inc_numbers (char *stringA, char *stringB)
{
	/** int **********************************************************/
  int i;
  int start_of_numbers_in_A;
  int start_of_numbers_in_B;
  int res;

	/** long ********************************************************/
  long numA;
  long numB;

	/** end vars ****************************************************/

  if (strlen (stringA) == strlen (stringB))
    {
      return (strcmp (stringA, stringB));
    }
  for (i = strlen (stringA); i > 0 && isdigit (stringA[i - 1]); i--);
  if (i == strlen (stringA))
    {
      return (strcmp (stringA, stringB));
    }
  start_of_numbers_in_A = i;
  for (i = strlen (stringB); i > 0 && isdigit (stringB[i - 1]); i--);
  if (i == strlen (stringB))
    {
      return (strcmp (stringA, stringB));
    }
  start_of_numbers_in_B = i;
  if (start_of_numbers_in_A != start_of_numbers_in_B)
    {
      return (strcmp (stringA, stringB));
    }
  res = strncmp (stringA, stringB, i);
  if (res)
    {
      return (res);
    }
  numA = atol (stringA + start_of_numbers_in_A);
  numB = atol (stringB + start_of_numbers_in_B);
  /*
     sprintf(tmp,"Comparing %s and %s --> %ld,%ld\n",stringA,stringB,numA,numB);
     log_to_screen(tmp);
   */
  return (numA - numB);
}




/*************************************************************************
 *strip_afio_output_line() -- Hugo Rabson                                *
 *                                                                       *
 * Purpose:   Strip 'afio: ' and other crap from either side of supplied *
 *            string, which probably is a copy of a line output from afio*
 * Called by: ...                                                        *
 * Params:    input                                 incoming string      *
 * Returns:   (static char*)stripped string                              *
 *************************************************************************/
char *
strip_afio_output_line (char *input)
{
	/** buffer *******************************************************/
  static char output[MAX_STR_LEN];

	/** pointers *****************************************************/
  char *p;
  char *q;
	/** end vars ****************************************************/


  strcpy (output, input);
  p = strchr (input, '\"');
  if (p)
    {
      q = strchr (++p, '\"');
      if (q)
	{
	  strcpy (output, p);
	  *(strchr (output, '\"')) = '\0';
	}
    }
  return (output);
}




/*************************************************************************
 * strip_spaces() -- Hugo Rabson                                         *
 *                                                                       *
 * Purpose:   Remove spaces, control chars, etc. from both sides of str. *
 * Called by: ...                                                        *
 * Params:    incoming                           incoming/outgoing str.  *
 * Returns:   value of incoming [changed]                                *
 *************************************************************************/
void
strip_spaces (char *in_out)
{
	/** buffers ******************************************************/
  char tmp[MAX_STR_LEN];

	/** pointers *****************************************************/
  char *p;

	/** int *********************************************************/
  int i;
  int original_incoming_length;

	/** end vars ****************************************************/

  original_incoming_length = strlen(in_out);
  for (i = 0; in_out[i] <= ' ' && i < strlen (in_out); i++);
  strcpy (tmp, in_out + i);
  for (i = strlen (tmp); i>0 && tmp[i - 1] <= 32; i--);
  tmp[i] = '\0';
  for (i = 0; i < original_incoming_length && MAX_STR_LEN; i++)
    {
      in_out[i] = ' ';
    }
  in_out[i] = '\0';
  i = 0;
  p = tmp;
  while (*p != '\0')
    {
      in_out[i] = *(p++);
      in_out[i + 1] = '\0';
      if (in_out[i] < 32 && i > 0)
	{
	  if (in_out[i] == 8)
	    {
	      i--;
	    }
	  else if (in_out[i] == 9)
	    {
	      in_out[i++] = ' ';
	    }
	  else if (in_out[i] == '\t')
	    {
	      for (i++; i % 5; i++);
	    }
	  else if (in_out[i] >= 10 && in_out[i] <= 13)
	    {
	      break;
	    }
	  else
	    {
	      i--;
	    }
	}
      else
	{
	  i++;
	}
    }
  in_out[i] = '\0';
/*  for(i=strlen(in_out); i>0 && in_out[i-1]<=32; i--) {in_out[i-1]='\0';} */
}



/*************************************************************************
 * trim_empty_quotes() -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:   If there is a quotation mark - " - either side of a string *
 *            then remove the leftmost and rightmost quotation mark      *
 * Called by: ...                                                        *
 * Params:    incoming                           incoming/outgoing str.  *
 * Returns:   value of incoming [changed]                                *
 *************************************************************************/
char *
trim_empty_quotes (char *incoming)
{
	/** buffer *******************************************************/
  static char outgoing[MAX_STR_LEN];

	/** end vars ****************************************************/
  if (incoming[0] == '\"' && incoming[strlen (incoming) - 1] == '\"')
    {
      strcpy (outgoing, incoming + 1);
      outgoing[strlen (outgoing) - 1] = '\0';
    }
  else
    {
      strcpy (outgoing, incoming);
    }
  return (outgoing);
}





/*************************************************************************
 * truncate_to_drive_name() -- Philippe De Muyter 		         *
 *                                                                      *
 * Purpose:                                                              *
 * Called by:                                                            
 *
 * returns:                                                              
 *
 *************************************************************************/
char *
truncate_to_drive_name (char *partition)
{
		/** int **************************************************************/
  int i;

  for (i = strlen (partition); isdigit (partition[i - 1]); i--)
    continue;
  if (partition[i - 1] == 'p' && isdigit (partition[i - 2]))
    {
      i--;
    }
  partition[i] = '\0';
  return partition;
}






/*************************************************************************
 * turn_raid_level_number_to_string() -- Hugo Rabson                     *
 *                                                                       *
 * Purpose:  Turn RAID level (number) into a friendly string             *
 * Called by:                                                            *
 * Params:   raid_level           0, 1, 2, ...                           *
 * Returns:  static string containing friendly string                    *
 *************************************************************************/
char *
turn_raid_level_number_to_string (int raid_level)
{

	/** buffer ***********************************************************/
  static char output[MAX_STR_LEN];



  if (raid_level >= 0)
    {
      sprintf (output, " RAID %-2d ", raid_level);
    }
  else
    {
      sprintf (output, "Linear RAID");
    }
  return (output);
}









#define MNT_RESTORING "/mnt/RESTORING"
/*************************************************************************
 * severity_of_difference() -- Hugo Rabson                               *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Params:                                                               *
 * Returns:                                                              *
 *************************************************************************/
int severity_of_difference(char *fn, char *out_reason)
{
  int sev;
  char reason[MAX_STR_LEN];
  char filename[MAX_STR_LEN];

  if (!strncmp(fn, MNT_RESTORING, strlen(MNT_RESTORING)))
    { strcpy(filename, fn+strlen(MNT_RESTORING)); }
  else if (fn[0]!='/')
    { sprintf(filename, "/%s", fn); }
  else
    { strcpy(filename, fn); }

  sev = 3;
  sprintf(reason, "Changed since backup. Consider running a differential backup in a day or two.");
  if (!strncmp(filename, "/var/", 5)) { sev = 2; sprintf(reason, "/var's contents will change regularly, inevitably."); }
  if (!strncmp(filename, "/home", 5)) { sev = 2; sprintf(reason, "It's in your /home partiton. Therefore, it is important."); }
  if (!strncmp(filename, "/usr/", 5)) { sev = 3; sprintf(reason, "You may have installed/removed software during the backup."); }
  if (!strncmp(filename, "/etc/", 5)) { sev = 3; sprintf(reason, "Do not edit config files while backing up your PC."); }
  if (!strcmp(filename, "/etc/adjtime") || !strcmp(filename, "/etc/mtab")) { sev = 1; sprintf(reason, "This file changes all the time. It's OK."); }
  if (!strncmp(filename, "/root/", 6)) { sev = 3; sprintf(reason, "Were you compiling/editing something in /root?"); }
  if (!strncmp(filename, "/root/.", 7)) { sev = 2; sprintf(reason, "Temp or 'dot' files changed in /root."); }
  if (!strncmp(filename, "/var/lib/", 9)) { sev = 2; sprintf(reason, "Did you add/remove software during backing?"); }
  if (!strncmp(filename, "/var/lib/rpm", 12)) { sev = 3; sprintf(reason, "Did you add/remove software during backing?"); }
  if (!strncmp(filename, "/var/lib/slocate", 16)) { sev = 1; sprintf(reason, "The 'update' daemon ran during backup. This does not affect the integrity of your backup."); }
  if (!strncmp(filename, "/var/log/", 9) || strstr(filename,"/.xsession") || !strcmp(filename+strlen(filename)-4, ".log")) { sev = 1; sprintf(reason, "Log files change frequently as the computer runs. Fret not."); }
  if (!strncmp(filename, "/var/spool", 10)) { sev = 1; sprintf(reason, "Background processes or printers were active. This does not affect the integrity of your backup."); }
  if (!strncmp(filename, "/var/spool/mail", 10)) { sev = 2; sprintf(reason, "Mail was sent/received during backup."); }
  if (filename[strlen(filename)-1]== '~')
    { sev = 1; sprintf(reason, "Backup copy of another file which was modified recently."); }
  if (strstr(filename, "cache"))
    { sev = 1; sprintf(reason, "Part of a cache of data. Caches change from time to time. Don't worry."); }
  if (!strncmp(filename, "/var/run/", 9) || !strncmp(filename, "/var/lock", 8) || strstr(filename, "/.DCOPserver") || strstr(filename, "/.MCOP") || strstr(filename, "/.Xauthority"))
    { sev = 1; sprintf(reason, "Temporary file (a lockfile, perhaps) used by software such as X or KDE to register its presence."); }
  if (out_reason) { strcpy(out_reason,reason); }
  return(sev);
}




/*************************************************************************
 * compare_two_filelist_entries() -- Hugo Rabson                               *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Params:                                                               *
 * Returns:                                                              *
 *************************************************************************/
int compare_two_filelist_entries(void*va,void*vb)
{
  static int res;
  struct s_filelist_entry *fa, *fb;

  fa = (struct s_filelist_entry*)va;
  fb = (struct s_filelist_entry*)vb;
  res = strcmp(fa->filename, fb->filename);
  return(res);
}








/*************************************************************************
 * percent_media_full_comment() -- Hugo Rabson                           *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/
char *
percent_media_full_comment (struct s_bkpinfo *bkpinfo)
{
	/** int ************************************************/
  int percentage;
  int j;

	/** buffers ********************************************/
  static char outstr[MAX_STR_LEN];
  char pos_w_commas[MAX_STR_LEN], tmp[MAX_STR_LEN];

  sprintf(tmp, "%lld", g_tape_posK);
  strcpy(pos_w_commas, commarize(tmp));



  if (bkpinfo->media_size[g_current_media_number]<=0)
//    { fatal_error( "percentage_media_full_comment() - unknown media size"); }
    {
      sprintf( outstr, "Volume %d: %s kilobytes archived so far", g_current_media_number, pos_w_commas);
      return( outstr );
    }

/* update screen */
  if (IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type))
    {
      percentage = (int) (g_tape_posK / 10 / bkpinfo->media_size[g_current_media_number]);
      if (percentage > 100)
	{
	  percentage = 100;
	}
      sprintf (outstr, "Volume %d: [", g_current_media_number);
    }
  else
    {
      percentage =
	(int) (space_occupied_by_cd (bkpinfo->scratchdir) * 100 / 1024 /
	       bkpinfo->media_size[g_current_media_number]);
      sprintf (outstr, "CD %d: [", g_current_media_number);
    }
  for (j = 0; j < percentage; j += 5)
    {
      strcat (outstr, "*");
    }
  for (; j < 100; j += 5)
    {
      strcat (outstr, ".");
    }
  j = strlen (outstr);
  sprintf (outstr + j, "] %d%% used", percentage);
  return (outstr);
}







