/*
 * Copyright (C), 2000-2005 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif


#include "monitor.h"
#include "alert.h"
#include "event.h"
#include "process.h"


/**
 * Implementation of the event interface.
 *
 * @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 * @author Martin Pala <martinp@tildeslash.com>
 * @version \$Id: event.c,v 1.50 2005/09/25 20:59:49 martinp Exp $
 * @file
 */


/* ------------------------------------------------------------- Definitions */

EventTable_T Event_Table[]= {
  {EVENT_CHANGED,    "Changed",                "Changed not"},
  {EVENT_CHECKSUM,   "Checksum failed",        "Checksum passed"},
  {EVENT_CONNECTION, "Connection failed",      "Connection passed"},
  {EVENT_DATA,       "Data access error",      "Data access succeeded"},
  {EVENT_EXEC,       "Execution failed",       "Execution succeeded"},
  {EVENT_GID,        "GID failed",             "GID passed"},
  {EVENT_ICMP,       "ICMP failed",            "ICMP passed"},
  {EVENT_INVALID,    "Invalid type",           "Type passed"},
  {EVENT_MATCH,      "Regex match",            "No regex match"},
  {EVENT_NONEXIST,   "Does not exist",         "Exists"},
  {EVENT_PERMISSION, "Permission failed",      "Permission passed"},
  {EVENT_RESOURCE,   "Resource limit matched", "Resource limit passed"},
  {EVENT_SIZE,       "Size failed",            "Size passed"},
  {EVENT_TIMEOUT,    "Timeout",                "Timeout recovery"},
  {EVENT_TIMESTAMP,  "Timestamp failed",       "Timestamp passed"},
  {EVENT_UID,        "UID failed",             "UID passed"},
  /* Virtual events */
  {EVENT_NULL,       "No Event",               "No Event"},
};


/* -------------------------------------------------------------- Prototypes */


static void handle_event(Event_T);
static int  handle_action(Event_T, Action_T);


/* ------------------------------------------------------------------ Public */


/**
 * Post a new Event
 * @param service The Service the event belongs to
 * @param id The event identification
 * @param state The event state
 * @param action Description of the event action
 * @param s Optional message describing the event
 */
void Event_post(Service_T service, long id, short state, EventAction_T action,
  char *s, ...) {

  Event_T e = service->eventlist;

  ASSERT(service);
  ASSERT(action);
  ASSERT(state == STATE_FAILED || state == STATE_PASSED);

  if(e == NULL)
  {
    /* Only first failed event can initialize the queue for given event type,
     * thus passed events are ignored until first error. However, in the case
     * that the error flag is set for the passed event, we will allow it (i.e.
     * event queue was flushed during monit reload and the service was in
     * failed state before reload) */
    if(state != STATE_FAILED && !(service->error & id))
      return;

    /* Initialize event list and add first event. */
    NEW(e);
    e->id = id;
    e->collected = time(NULL);
    e->source = service;
    e->state = STATE_INIT;
    e->state_map = state;
    e->action = action;
    if(s)
    {
      long l;
      va_list ap;

      va_start(ap, s);
      e->message = Util_formatString(s, ap, &l);
      va_end(ap);
    }
    pthread_mutex_init(&e->mutex, NULL);
    service->eventlist = e;
  }
  else
  {
    /* Try to find the event with the same origin and type identification.
     * Each service and each test have its own custom actions object, so
     * we share actions object address to identify event source. */
    do
    {
      if(e->action == action && e->id == id)
      {
        LOCK(e->mutex)
          e->collected = time(NULL);

          /* Shift the existing event flags to the left
           * and set the first bit based on actual state */
          e->state_map <<= 1;
          e->state_map |= state;

          /* Update the message */
          if(s)
          {
            long l;
            va_list ap;

            FREE(e->message);
            va_start(ap, s);
            e->message = Util_formatString(s, ap, &l);
            va_end(ap);
          }

        END_LOCK;
	break;
      }
      e = e->next;
    }
    while(e);

    if(!e)
    {
      /* Only first failed event can initialize the queue for given event type,
       * thus passed events are ignored until first error */
      if(state != STATE_FAILED)
        return;

      /* Event was not found in the pending events list, we will add it. */
      NEW(e);
      e->id = id;
      e->collected = time(NULL);
      e->source = service;
      e->state = STATE_INIT;
      e->state_map = state;
      e->action = action;
      if(s)
      {
        long l;
        va_list ap;

        va_start(ap, s);
        e->message = Util_formatString(s, ap, &l);
        va_end(ap);
      }
      pthread_mutex_init(&e->mutex, NULL);
      e->next = service->eventlist;
      service->eventlist = e;
    }
  }

  e->state_changed = Event_check_state(e, state);

  /* In the case that the state changed, update it and reset the counter */
  if(e->state_changed)
  {
    e->state = state;
    e->count = 1;
  }
  else
  {
    e->count++;
  }

  LOCK(e->mutex)
    handle_event(e);
  END_LOCK;

}


/* -------------------------------------------------------------- Properties */


/**
 * Get the Service where the event orginated
 * @param E An event object
 * @return The Service where the event orginated
 */
Service_T Event_get_source(Event_T E) {

  ASSERT(E);

  return E->source;

}


/**
 * Get the Event timestamp
 * @param E An event object
 * @return The Event timestamp
 */
time_t Event_get_collected(Event_T E) {

  ASSERT(E);
  
  return E->collected;

}


/**
 * Get the Event raw state
 * @param E An event object
 * @return The Event raw state
 */
short Event_get_state(Event_T E) {

  ASSERT(E);
  
  return E->state;

}


/**
 * Return the actual event state based on event state bitmap
 * and event ratio needed to trigger the state change
 * @param E An event object
 * @param S Actual posted state
 * @return The Event raw state
 */
short Event_check_state(Event_T E, short S) {

  int       i;
  int       count = 0;
  Action_T  action;
  long long flag;

  ASSERT(E);

  /* Only the true failed state condition can change the initial state */
  if(S == STATE_PASSED && E->state == STATE_INIT)
  {
    return FALSE;
  }

  action = (S == STATE_PASSED)?E->action->passed:E->action->failed;

  /* Compare as many bits as cycles able to trigger the action */
  for(i = 0; i < action->cycles; i++)
  {
    /* Check the state of the particular cycle given by the bit position */
    flag = (E->state_map >> i) & 0x1;

    /* Count occurences of the posted state */
    if(flag == S)
    {
      count++;
    }
  }

  if(count >= action->count && S != E->state)
  {
    return TRUE;
  }
  
  return FALSE;

}


/**
 * Get the Event type
 * @param E An event object
 * @return The Event type
 */
int Event_get_id(Event_T E) {

  ASSERT(E);
  
  return E->id;

}


/**
 * Get the optionally Event message describing why the event was
 * fired.
 * @param E An event object
 * @return The Event message. May be NULL
 */
const char *Event_get_message(Event_T E) {

  ASSERT(E);

  return E->message;

}


/**
 * Get a textual description of actual event type. For instance if the
 * event type is possitive EVENT_TIMESTAMP, the textual description is
 * "Timestamp error". Likewise if the event type is negative EVENT_CHECKSUM
 * the textual description is "Checksum recovery" and so on.
 * @param E An event object
 * @return A string describing the event type in clear text. If the
 * event type is not found NULL is returned.
 */
const char *Event_get_description(Event_T E) {

  EventTable_T *et= Event_Table;

  ASSERT(E);

  while((*et).id)
  {
    if(E->id == (*et).id)
    {
      return E->state?(*et).description_failed:(*et).description_passed;
    }
    et++;
  }
  
  return NULL;

}


/**
 * Get an event action id.
 * @param E An event object
 * @return An action id
 */
short Event_get_action(Event_T E) {

  short id;
  Action_T A;

  ASSERT(E);

  A = E->state?E->action->failed:E->action->passed;

  /* In the case of passive mode we replace the description of start, stop
   * or restart action for alert action, because these actions are passive in
   * this mode */
  id= (E->source->mode == MODE_PASSIVE &&
       ((A->id == ACTION_START)||
        (A->id == ACTION_STOP) ||
        (A->id == ACTION_RESTART))
      )?ACTION_ALERT:A->id;

  return id;

}


/**
 * Get a textual description of actual event action. For instance if the
 * event type is possitive EVENT_NONEXIST, the textual description of
 * failed state related action is "restart". Likewise if the event type is
 * negative EVENT_CHECKSUM the textual description of recovery related action
 * is "alert" and so on.
 * @param E An event object
 * @return A string describing the event type in clear text. If the
 * event type is not found NULL is returned.
 */
const char *Event_get_action_description(Event_T E) {

  ASSERT(E);

  return actionnames[Event_get_action(E)];

}


/* ----------------------------------------------------------------- Private */


/*
 * Handle the event
 * @param E An event
 */
static void handle_event(Event_T E) {

  ASSERT(E);
  ASSERT(E->action);
  ASSERT(E->action->failed);
  ASSERT(E->action->passed);

  /* We will handle only first passed event, recurrent passed events
   * or insufficient passed events during failed service state are
   * ignored. Failed events are handled each time. */
  if(!E->state_changed && E->state == STATE_PASSED)
  {
    return;
  }

  if(E->message)
  {
    log("%s\n", E->message);
  }

  if(E->state == STATE_FAILED)
  {
    E->source->error |= E->id;
    handle_action(E, E->action->failed);
  }
  else
  {
    E->source->error &= ~E->id;
    handle_action(E, E->action->passed);
  }

  /* Possible event state change was handled so we will reset the flag. */
  E->state_changed = FALSE;

}


static int handle_action(Event_T E, Action_T A) {

  ASSERT(E);
  ASSERT(A);

  if(A->id == ACTION_IGNORE)
  {
    return TRUE;
  }

  /* Alert and collector event notification are common actions */
  handle_alert(E);
  handle_collector(E);

  switch(A->id)
  {
  case ACTION_ALERT:
      /* Already handled. */
      break;

  case ACTION_EXEC:
      spawn(E->source, A->exec, EVENT_DESCRIPTION(E));
      break;

  case ACTION_RESTART:
      if(E->source->def_timeout)
        E->source->nstart++;
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "restart");
      }
      return FALSE;

  case ACTION_START:
      if(E->source->def_timeout)
        E->source->nstart++;
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "start");
      } 
      return FALSE;

  case ACTION_STOP:
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "stop");
      } 
      return FALSE;

  case ACTION_UNMONITOR:
      control_service(E->source->name, "unmonitor");
      return FALSE;

  default:
      log("'%s' error -- unknown failure action: [%d]\n", E->source->name,
        A->id);
      break;

  }
  return TRUE;
}

