type error =
  | FORK   	(** fork failed due to lack of memory *)
  | READ   	(** read or select on pipes failed *)
  | CHDIR  	(** changing to working dir failed *)
  | ACCES  	(** [execv()] returned EACCES *)
  | PERM   	(** [execv()] returned EPERM *)
  | TOOBIG   	(** [execv()] returned E2BIG *)
  | NOEXEC 	(** [execv()] returned ENOEXEC *)
  | NAMETOOLONG (** [execv()] returned ENAMETOOLONG *)
  | NOENT       (** [execv()] returned ENOENT *)
  | NOMEM       (** [execv()] returned ENOMEM *)
  | NOTDIR      (** [execv()] returned ENOTDIR *)
  | LOOP        (** [execv()] returned ELOOP   *)
  | TXTBUSY     (** [execv()] returned ETXTBUSY *)
  | IO          (** [execv()] returned EIO *)
  | NFILE       (** [execv()] returned ENFILE *)
  | MFILE       (** [execv()] returned EMFLE *)
  | INVAL       (** [execv()] returned EINVAL *)
  | ISDIR       (** [execv()] returned EISDIR *)
  | LIBBAD      (** [execv()] returned ELIBBAD *)
  | FAILED      (** other fatal failure *)
exception Error of error * string
val init : unit

type spawn_flags =
  [ `CHILD_INHERITS_STDIN
  | `DO_NOT_REAP_CHILD
  | `FILE_AND_ARGV_ZERO
  | `LEAVE_DESCRIPTORS_OPEN
  | `SEARCH_PATH
  | `STDERR_TO_DEV_NULL
  | `STDOUT_TO_DEV_NULL 
  | `PIPE_STDIN
  | `PIPE_STDOUT
  | `PIPE_STDERR ]

type pid
external int_of_pid : pid -> int = "ml_int_of_pid" (** @raise Failure on Windows *)

type child = {
  pid             : pid option;
  standard_input  : int option;
  standard_output : int option;
  standard_error  : int option;
}

val async_with_pipes :
  ?working_directory:string ->
  ?environment:string list ->
  ?child_setup:(unit -> unit) ->
  flags:spawn_flags list ->
  string list -> child
(** @raise Error if the spawn fails
    @raise Invalid_argument if some [flags] are incompatible *)

type status =
  | EXITSTATUS of int
  | TERMSIG of int
  | STOPSIG of int

external sync :
  ?working_directory:string ->
  ?environment:string list  ->
  ?child_setup:(unit -> unit) ->
  flags:spawn_flags list ->
  string list -> status * string * string = "ml_g_spawn_sync"
(** @raise Error if the spawn fails *)

external close_pid : pid -> unit = "ml_g_spawn_close_pid"

external command_line_sync :
  string -> status * string * string = "ml_g_spawn_command_line_sync"
(** @raise Error if the spawn fails *)

external command_line_async : string -> unit = "ml_g_spawn_command_line_async"
(** @raise Error if the spawn fails *)

type source_id
external add_child_watch : ?prio:int -> pid -> (int -> unit) -> source_id = "ml_g_add_child_watch_full"
external remove_watch : source_id -> unit = "ml_g_source_remove"
