type db
type stmt
type argument

type sql_type = [ `BLOB | `FLOAT | `INTEGER | `NULL | `TEXT ]
type sql_value =
  [ `BLOB of string
  | `FLOAT of float
  | `INT of int
  | `INT64 of int64
  | `NULL
  | `TEXT of string
  | `VALUE of argument ]

type error_code =
    ERROR
  | INTERNAL
  | PERM
  | ABORT
  | BUSY
  | LOCKED
  | NOMEM
  | READONLY
  | INTERRUPT
  | IOERR
  | CORRUPT
  | NOTFOUND
  | FULL
  | CANTOPEN
  | PROTOCOL
  | EMPTY
  | SCHEMA
  | TOOBIG
  | CONSTRAINT
  | MISMATCH
  | MISUSE
  | NOLFS
  | AUTH
  | FORMAT
  | RANGE
  | NOTADB
exception Error of error_code * string

val init : unit

external open_db : string -> db = "ml_sqlite3_open"
val close_db : db -> unit

external interrupt : db -> unit = "ml_sqlite3_interrupt"
external is_complete : string -> bool = "ml_sqlite3_complete"
val version : string
external last_insert_rowid : db -> int64 = "ml_sqlite3_last_insert_rowid"
external changes : db -> int = "ml_sqlite3_changes"
external total_changes : db -> int = "ml_sqlite3_total_changes"
(* external get_autocommit : db -> bool = "ml_sqlite3_get_autocommit" *)
external sleep : int -> unit = "ml_sqlite3_sleep"

external busy_set : db -> (int -> [ `FAIL | `RETRY ]) -> unit
  = "ml_sqlite3_busy_handler"
external busy_unset : db -> unit = "ml_sqlite3_busy_handler_unset"
external busy_timeout : db -> int -> unit = "ml_sqlite3_busy_timeout"

external trace_set : db -> (string -> unit) -> unit = "ml_sqlite3_trace"
external trace_unset : db -> unit = "ml_sqlite3_trace_unset"

external progress_handler_set : db -> int -> (unit -> unit) -> unit
  = "ml_sqlite3_progress_handler"
external progress_handler_unset : db -> unit
  = "ml_sqlite3_progress_handler_unset"


external finalize_stmt : stmt -> unit = "ml_sqlite3_finalize_noerr"
val prepare_one : db -> string -> stmt

external reset : stmt -> unit = "ml_sqlite3_reset"
external expired : stmt -> bool = "ml_sqlite3_expired"
external step : stmt -> [ `DONE | `ROW ] = "ml_sqlite3_step"

external bind : stmt -> int -> sql_value -> unit = "ml_sqlite3_bind"
external bind_parameter_count : stmt -> int
  = "ml_sqlite3_bind_parameter_count"
external bind_parameter_index : stmt -> string -> int
  = "ml_sqlite3_bind_parameter_index"
external bind_parameter_name : stmt -> int -> string
  = "ml_sqlite3_bind_parameter_name"
external clear_bindings : stmt -> unit = "ml_sqlite3_clear_bindings"
external transfer_bindings : stmt -> stmt -> unit
  = "ml_sqlite3_transfer_bindings"

external column_blob : stmt -> int -> string = "ml_sqlite3_column_blob"
external column_double : stmt -> int -> float = "ml_sqlite3_column_double"
external column_int : stmt -> int -> int = "ml_sqlite3_column_int"
external column_int64 : stmt -> int -> int64 = "ml_sqlite3_column_int64"
external column_text : stmt -> int -> string = "ml_sqlite3_column_text"
external column_type : stmt -> int -> sql_type = "ml_sqlite3_column_type"
external data_count : stmt -> int = "ml_sqlite3_data_count"
external column_count : stmt -> int = "ml_sqlite3_column_count"
external column_name : stmt -> int -> string = "ml_sqlite3_column_name"
external column_decltype : stmt -> int -> string
  = "ml_sqlite3_column_decltype"

external value_blob : argument -> string = "ml_sqlite3_value_blob"
external value_double : argument -> float = "ml_sqlite3_value_double"
external value_int : argument -> int = "ml_sqlite3_value_int"
external value_int64 : argument -> int64 = "ml_sqlite3_value_int64"
external value_text : argument -> string = "ml_sqlite3_value_text"
external value_type : argument -> sql_type = "ml_sqlite3_value_type"

val create_fun_N : db -> string -> (argument array -> sql_value) -> unit
val create_fun_0 : db -> string -> (unit -> sql_value) -> unit
val create_fun_1 : db -> string -> (argument -> sql_value) -> unit
val create_fun_2 :
  db -> string -> (argument -> argument -> sql_value) -> unit
val create_fun_3 :
  db -> string -> (argument -> argument -> argument -> sql_value) -> unit

external delete_function : db -> string -> unit
  = "ml_sqlite3_delete_function"

val fold_rows : ('a -> stmt -> 'a) -> 'a -> stmt -> 'a

val exec : db -> string -> unit
val exec_f : db -> ('a, unit, string, unit) format4 -> 'a

val fetch :
  db ->
  string ->
  ?column_names:string array ref -> ('a -> string array -> 'a) -> 'a -> 'a
val fetch_f :
  db ->
  ?column_names:string array ref ->
  ('a -> string array -> 'a) -> 'a -> ('b, unit, string, 'a) format4 -> 'b
