# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: cpustats.tcl,v 2.17 2004/01/11 02:00:52 jfontain Exp $


package provide cpustats [lindex {$Revision: 2.17 $} 1]
package require network 1

namespace eval cpustats {

    array set data {
        updates 0
        0,label CPU 0,type ascii 0,message {CPU number (0 for all or a single CPUs)}
        1,label user 1,type real 1,message {percentage spent in user mode}
        2,label system 2,type real 2,message {percentage spent in system mode}
        3,label nice 3,type real 3,message {percentage spent in nice mode}
        4,label idle 4,type real 4,message {percentage spent in idle mode}
        5,label iowait 5,type real 5,message {percentage spent waiting for I/O to complete}
        6,label irq 6,type real 6,message {percentage spent servicing interrupts}
        7,label softirq 7,type real 7,message {percentage spent servicing software interrupts}
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open cpustats.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable remote
        variable data
        variable statisticsFile

        set lookup [expr {![info exists options(-n)] && ![info exists options(--numeric)]}]          ;# host or network names lookup
        if {![catch {set locator $options(--remote)}] || ![catch {set locator $options(-r)}]} {                 ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) cpustats($remote(host))
            if {[string equal $::tcl_platform(platform) unix]} {
                set remote(command) "$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/stat"
            } else {                                                                                                      ;# windows
                set remote(command) "plink -batch $remote(host) cat /proc/stat"                    ;# host is rather a putty session
            }
            set file [open "| $remote(command)"]
            fileevent $file readable {set ::cpustats::remote(busy) 0}
            vwait ::cpustats::remote(busy)
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message] || [catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set statisticsFile [open /proc/stat]                                      ;# keep local file open for better performance
        }
    }

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable statisticsFile
        variable data
        variable last

        if {[info exists remote]} {
            if {![info exists statisticsFile]} {                           ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| $remote(command)"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::cpustats::statisticsFile $file; ::cpustats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $statisticsFile 0                                                                  ;# rewind before retrieving data
        }
        gets $statisticsFile firstLine
        set lines {}
        while {([gets $statisticsFile line] >= 0) && [string match cpu* $line]} {                        ;# only save CPU data lines
            lappend lines $line
        }
        if {[info exists remote]} {
            read $statisticsFile                             ;# avoid write on pipe with no readers errors by reading remaining data
            if {[catch {close $statisticsFile} message]} {                               ;# communication error can be detected here
                flashMessage "error: $message"
                unset firstLine                                                                ;# consider data corrupted as a whole
            }
            unset statisticsFile
            set remote(busy) 0
        }
        if {[info exists firstLine]} {
            set index 0                                                 ;# first line is for all CPUs, data is in 100ths of a second
            set columns [scan $firstLine {cpu %u %u %u %u %u %u %u} user nice system idle iowait irq softirq]
            if {$columns < 7} {                                                                                        ;# kernel 2.4
                set iowait {}; set irq {}; set softirq {}
            }                                                                                           ;# else kernel 2.6 and above
            updateRow $index $user $nice $system $idle $iowait $irq $softirq
            if {[llength $lines] > 1} {                               ;# display per CPU statistics only if there is more than 1 CPU
                foreach line $lines {
                    if {[scan $line {cpu%*u %u %u %u %u %u %u %u} user nice system idle iowait irq softirq] >= 4} {
                        updateRow [incr index] $user $nice $system $idle $iowait $irq $softirq
                    }
                }
            }
        } else {                                                                                                ;# data is corrupted
            catch {array unset data {[0-9]*,[0-9]*}}
            catch {unset last}
            # only display first row (for all CPUs) with unknown values
            array set data {0,0 0 0,1 ? 0,2 ? 0,3 ? 0,4 ? 0,5 ? 0,6 ? 0,7 ?}
        }
        incr data(updates)
    }

    proc updateRow {index user nice system idle iowait irq softirq} {                          ;# the last 3 parameters can be empty
        variable last
        variable data

        set extended [string length $iowait]                                    ;# boolean, implies irq and softirq are also defined
        if {[info exists last($index,user)]} {
            set userDelta [expr {$user - $last($index,user)}]
            set niceDelta [expr {$nice - $last($index,nice)}]
            set systemDelta [expr {$system - $last($index,system)}]
            set idleDelta [expr {$idle - $last($index,idle)}]
            # force floating point calculations:
            if {$extended} {
                set iowaitDelta [expr {$iowait - $last($index,iowait)}]
                set irqDelta [expr {$irq - $last($index,irq)}]
                set softirqDelta [expr {$softirq - $last($index,softirq)}]
                set divider [expr {\
                    ($userDelta + $niceDelta + $systemDelta + $idleDelta + $iowaitDelta + $irqDelta + $softirqDelta) / 100.0}\
                ]
            } else {
                set divider [expr {($userDelta + $niceDelta + $systemDelta + $idleDelta) / 100.0}]
            }
            if {$divider == 0} {                                                                              ;# should never happen
                array set data [list $index,1 0 $index,2 0 $index,3 0 $index,4 100]
                if {$extended} {
                    array set data [list $index,5 0 $index,6 0 $index,7 0]
                }
            } else {
                array set data [list\
                    $index,1 [format %.1f [expr {$userDelta / $divider}]] $index,2 [format %.1f [expr {$systemDelta / $divider}]]\
                    $index,3 [format %.1f [expr {$niceDelta / $divider}]] $index,4 [format %.1f [expr {$idleDelta / $divider}]]\
                ]
                if {$extended} {
                    array set data [list\
                        $index,5 [format %.1f [expr {$iowaitDelta / $divider}]]\
                        $index,6 [format %.1f [expr {$irqDelta / $divider}]]\
                        $index,7 [format %.1f [expr {$softirqDelta / $divider}]]\
                    ]
                }
            }
        } else {                                                                       ;# first pass: CPU usage cannot be determined
            set data($index,0) $index
            array set data [list $index,1 ? $index,2 ? $index,3 ? $index,4 ? $index,5 ? $index,6 ? $index,7 ?]
        }
        set last($index,user) $user
        set last($index,system) $system
        set last($index,nice) $nice
        set last($index,idle) $idle
        if {$extended} {
            set last($index,iowait) $iowait
            set last($index,irq) $irq
            set last($index,softirq) $softirq
        }
    }

}
