# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: disks.tcl,v 1.5 2004/01/01 11:39:06 jfontain Exp $


package provide disks [lindex {$Revision: 1.5 $} 1]
package require network 1

namespace eval disks {

    variable nextIndex 0                                                                                ;# unique index for new rows

    # the table key is obviously the device name:
    array set data {
        updates 0
        0,label name 0,type dictionary 0,message {device name}
        1,label type 1,type ascii 1,message {type of device}
        2,label media 2,type ascii 2,message {media of device}
        3,label size 3,type real 3,message {size in megabytes}
        4,label model 4,type ascii 4,message {model description} 4,anchor left
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }

    set file [open disks.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file
    unset file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable remote
        variable data
        variable partitionsFile

        if {![catch {set locator $options(--remote)}] || ![catch {set locator $options(-r)}]} {                 ;# remote monitoring
            set data(pollTimes) {120 10 20 30 60 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) disks($remote(host))
            # check that file is actually readable by user:
            if {[string equal $::tcl_platform(platform) unix]} {
                set command "$remote(protocol) -n -l $remote(user) $remote(host) head -1 /proc/partitions"
            } else {                                                                                                      ;# windows
                set command "plink -batch $remote(host) head -1 /proc/partitions"                  ;# host is rather a putty session
            }
            set file [open "| $command"]
            fileevent $file readable {set ::disks::remote(busy) 0}
            vwait ::disks::remote(busy)                                                                ;# do not hang user interface
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message] || [catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
            # do as much pre-processing as possible on the remote host and return only the required data to optimize performance on
            # slow links. skip first 2 lines: header and blank separator.
            set remote(command) {(read; read; while read major minor blocks name rest; do if [[ $name == *[^0-9] ]]; then echo $blocks $name; fi done) < /proc/partitions}
        } else {
            set data(pollTimes) {60 5 10 20 30 120 300 600}
            set partitionsFile [open /proc/partitions]                                ;# keep local file open for better performance
        }
    }

    proc update {} {                                                                             ;# invoked periodically by the core
        variable remote

        if {[info exists remote]} {
            if {!$remote(busy)} remoteUpdate                                                             ;# avoid concurrent updates
        } else {
            localUpdate
        }
    }

    proc localUpdate {} {                                                                          ;# data resides in the local host
        variable partitionsFile

        seek $partitionsFile 0                                                                      ;# rewind before retrieving data
        gets $partitionsFile; gets $partitionsFile                                          ;# skip headers and blank separator line
        set lines {}
        while {[gets $partitionsFile line] >= 0} {
            foreach {major minor blocks name} $line break
            if {[string match {*[0-9]} $name]} continue                         ;# if names ends by a number, it must be a partition
            lappend lines [list $blocks $name]
        }
        process $lines
    }

    proc remoteUpdate {} {                                                               ;# initiate data retrieval from remote host
        variable remote

        set remote(busy) 1
        if {[string equal $::tcl_platform(platform) unix]} {
            set channel [open [list | $remote(protocol) -n -l $remote(user) $remote(host) $remote(command)]]
        } else {                                                                                                          ;# windows
            set channel [open [list | plink -batch $remote(host) $remote(command)]]
        }
        fileevent $channel readable "disks::remoteUpdated $channel"          ;# do not hang user interface and other modules updates
    }

    proc remoteUpdated {channel} {                                                         ;# data is now available from remote host
        variable remote

        set lines {}
        while {[gets $channel line] >= 0} {
            lappend lines $line
        }
        read $channel                                        ;# avoid write on pipe with no readers errors by reading remaining data
        if {[catch {close $channel} message]} {                                                                  ;# an error occured
            flashMessage "error: $message"
            set lines {}                                                                       ;# consider data corrupted as a whole
        }
        process $lines
        set remote(busy) 0
    }

    # sample /proc/partitions output (rio columns and after are optional):
    #
    # major minor #blocks  name rio  rmerge rsect  ruse   wio  wmerge wsect wuse  running use    aveq
    #
    #   22     0   8257032 hdc  1    3      8      10     0    0      0     0     -2      643770 41676482
    #   22     1   8249346 hdc1 0    0      0      0      0    0      0     0     0       0      0
    #   22    64    674978 hdd  0    0      0      0      0    0      0     0     -14     629370 34141112
    #    3     0   2062368 hda  182  1035   1228   460    2    2      4     20    0       480    480
    #    3     1   1798240 hda1 180  1032   1212   400    2    2      4     20    0       420    420
    #    3     2    264096 hda2 1    0      8      30     0    0      0     0     0       30     30
    #    3    64  40209120 hdb  5049 9492   116292 153170 2763 2575   42832 92630 -3      642110 41278532
    #    3    65  40209088 hdb1 5048 9489   116284 153170 2763 2575   42832 92630 0       30310  245870

    # process partitions file entries including the following fields only: blocks name rio ruse wio wuse aveq
    proc process {lines} {
        variable data
        variable last                                                                          ;# last values for delta calculations
        variable index                                                                                 ;# name to row number mapping
        variable nextIndex

        foreach line $lines {
            if {[scan $line {%u %s} blocks name] != 2} continue                                        ;# only process valid entries
            if {[catch {set row $index($name)}]} {                                                                      ;# new entry
                set row [set index($name) $nextIndex]
                incr nextIndex
                set data($row,0) $name                                                                     ;# initialize static data
                foreach [list data($row,1) data($row,2) data($row,4)] [data $name] {}
            }
            set value [expr {$blocks / 1024.0}]                               ;# block size is always 1024 bytes in the Linux kernel
            # note: disk size is usually static, but update it at each poll in case disk can be dynamically resized
            if {$value < 100} {                                                                           ;# less than 100 megabytes
                set data($row,3) [format %.1f $value]
            } else {
                set data($row,3) [expr {round($value)}]
            }
            set current($name) {}
        }
        cleanupEntriesData current
        incr data(updates)
    }

    proc readFile {name} {                                                                          ;# return whole contents of file
        variable remote

        set data {}
        if {[info exists remote]} {
            if {[string equal $::tcl_platform(platform) unix]} {
                set command "$remote(protocol) -n -l $remote(user) $remote(host) cat $name"
            } else {                                                                                                      ;# windows
                set command "plink -batch $remote(host) cat $name"                                 ;# host is rather a putty session
            }
            set file [open "| $command"]
            fileevent $file readable {set ::disks::remote(busy) 1}                               ;# detect readability but stay busy
            vwait ::disks::remote(busy)                                                                ;# do not hang user interface
            if {[catch {set data [read -nonewline $file]} message] || [catch {close $file} message]} {
                flashMessage "error: on remote host $remote(host) as user $remote(user): $message"
                return {}                                                       ;# make sure to return empty but clean data on error
            } else {
                return $data
            }
        } else {
            if {[catch {set file [open $name]} message]} {
                flashMessage "error: $message"
                return {}                                                       ;# make sure to return empty but clean data on error
            } else {
                set data [read -nonewline $file]
                close $file
                return $data
            }
        }
    }

    # sample /proc/scsi/scsi output:
    #
    # Attached devices: 
    # Host: scsi0 Channel: 00 Id: 01 Lun: 00
    #   Vendor: SEAGATE  Model: ST39173N         Rev: 5764
    #   Type:   Direct-Access                    ANSI SCSI revision: 02
    # Host: scsi0 Channel: 00 Id: 04 Lun: 00
    #   Vendor: CWS ORB2 Model: -SI U ID 4       Rev: D33 
    #   Type:   Direct-Access                    ANSI SCSI revision: 02
    # Host: scsi1 Channel: 00 Id: 01 Lun: 00
    #   Vendor: TOSHIBA  Model: DVD-ROM SD-M1201 Rev: 1R04
    #   Type:   CD-ROM                           ANSI SCSI revision: 02
    # Host: scsi1 Channel: 00 Id: 02 Lun: 00
    #   Vendor: EPSON    Model: SCANNER GT-9600  Rev: 1.03
    #   Type:   Processor                        ANSI SCSI revision: 02
    proc data {device} {                                                            ;# return device type, media and model from name
        set type {}; set media {}; set model {}                                                                           ;# unknown
        switch [string index $device 0] {
            h {
                set type IDE
                set media [readFile /proc/ide/$device/media]
                set model [readFile /proc/ide/$device/model]
            }
            s {
                set type SCSI
                set number [expr {[scan [string index $device end] %c] - [scan a %c]}]      ;# device index: sda -> 0, sdb -> 1, ...
                set index 0
                # skip header line, each entry has 3 lines of data
                foreach {line(0) line(1) line(2)} [lrange [split [readFile /proc/scsi/scsi] \n] 1 end] {
                    regexp {Type:(.*)ANSI SCSI revision:} $line(2) dummy media
                    set media [string trim $media]
                    # mountable devices from drivers/scsi/scsi.c kernel source
                    switch $media {
                        Direct-Access - WORM - CD-ROM - {Medium Changer} {}
                        default {
                            continue                                                                     ;# skip unmountable devices
                        }
                    }
                    if {$index == $number} {
                        regexp {Vendor:(.*)Model:(.*)Rev:(.*)$} $line(1) dummy vendor model revision
                        set model "[string trim $vendor] [string trim $model] [string trim $revision]"
                        break                                                            ;# found the device corresponding to number
                    }
                    incr index
                }
            }
        }
        return [list $type $media $model]
    }

    proc cleanupEntriesData {currentName} {
        upvar $currentName current
        variable index
        variable data

        foreach {name row} [array get index] {                                                        ;# cleanup disappeared entries
            if {[info exists current($name)]} continue
            unset index($name) data($row,0) data($row,1) data($row,2) data($row,3) data($row,4)
        }
    }

}
