# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: record.tcl,v 2.50 2004/01/01 11:39:06 jfontain Exp $


class record {

    proc record {this args} switched {$args} {
        switched::complete $this
    }

    proc ~record {this} {
        variable ${this}data

        catch {unset ${this}data}                                                                                   ;# if old format
        if {[info exists ($this,root)]} {
            dom::destroy $($this,root)                                                                           ;# cleanup XML data
        }
    }

    proc options {this} {
        return [list\
            [list -file {} {}]\
        ]
    }

    proc set-file {this value} {}

if {$global::withGUI} {                                                                                      ;# used only for saving

    # flag viewers requiring initialization configuration special treatment for options that actually are lists:
    array set series {
        ::store,comments {} ::thresholds,addresses {} ::dataTable,columnwidths {} ::freeText,cellindices {}
        ::summaryTable,cellrows {} ::summaryTable,columns {} ::summaryTable,columnwidths {} ::data2DPieChart,cellcolors {}
        ::data3DPieChart,cellcolors {} ::dataGraph,cellcolors {} ::dataStackedGraph,cellcolors {} ::dataBarChart,cellcolors {}
        ::dataSideBarChart,cellcolors {} ::dataStackedBarChart,cellcolors {} ::dataOverlapBarChart,cellcolors {}
    }
    # Note: this method is more appropriate than using special names or characters in configuration switches, since that would imply
    # that the object would have to know that its configuration is to be saved in a special way. It is better to assume that the
    # code responsible for saving data knows about the nature of the data to be saved.
    # Note: all viewers options with switches ending with "text" are stored in text form so that embedded new lines are preserved

    proc write {this} {                     ;# save current configuration in XML form (synchronize code with currentConfiguration{})
        variable series

        if {[string length $switched::($this,-file)] == 0} {
            error {-file option undefined}
        }
        set file [open $switched::($this,-file) w+]                                                           ;# create or overwrite

        set document [dom::create]
        set root [dom::document createElement $document moodssConfiguration]
        dom::document createTextNode [dom::document createElement $root version] $global::applicationVersion
        set seconds [clock seconds]
        dom::document createTextNode [dom::document createElement $root date] [clock format $seconds -format %D]
        dom::document createTextNode [dom::document createElement $root time] [clock format $seconds -format %T]
        set node [dom::document createElement $root configuration]
        foreach name [configuration::variables 0] {
            if {[string equal $name viewerColors]} continue                                                            ;# skip lists
            dom::element setAttribute $node $name [set ::global::$name]
        }
        nodeFromList $node viewerColors $::global::viewerColors                                                    ;# list of colors
        # main window coordinates are not saved as it would be bad manners to force initial window placement
        # use main window size to ignore tool bar presence interference:
        dom::document createTextNode [dom::document createElement $root width] [winfo width .]
        dom::document createTextNode [dom::document createElement $root height] [winfo height .]
        dom::document createTextNode [dom::document createElement $root pollTime] $global::pollTime
        if {[info exists databaseInstances::singleton]} {                                                   ;# database history mode
            set node [dom::document createElement $root databaseRange]
            foreach {from to} [databaseInstances::cursorsRange] {}
            dom::element setAttribute $node from $from
            dom::element setAttribute $node to $to
            set node [dom::document createElement $root databaseViewer]
            set path $widget::($databaseInstances::singleton,path)
            foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $path] {}
            foreach {xIcon yIcon} [canvasWindowManager::iconCoordinates $global::windowManager $path] {}
            dom::element setAttribute $node x $x
            dom::element setAttribute $node y $y
            dom::element setAttribute $node width $width
            dom::element setAttribute $node height $height
            dom::element setAttribute $node xIcon $xIcon
            dom::element setAttribute $node yIcon $yIcon
        }
        set modules [dom::document createElement $root modules]
        foreach instance $modules::(instances) {                             ;# note: in modules list, modules are in creation order
            set namespace $modules::instance::($instance,namespace)
            set module [dom::document createElement $modules module]
            dom::element setAttribute $module namespace $namespace
            dom::document createTextNode [dom::document createElement $module arguments] $modules::instance::($instance,arguments)
            set tables [dom::document createElement $module tables]
            foreach table $dataTable::(list) {                                 ;# note: in tables list, tables are in creation order
                # filter other module tables
                if {![string equal $namespace [namespace qualifiers [composite::cget $table -data]]]} continue
                foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $widget::($table,path)] {}
                set level [canvasWindowManager::getStackLevel $global::windowManager $widget::($table,path)]
                foreach {xIcon yIcon} [canvasWindowManager::iconCoordinates $global::windowManager $widget::($table,path)] {}
                # note: icon coordinates are empty if table is not minimized
                set node [dom::document createElement $tables table]
                dom::element setAttribute $node x $x
                dom::element setAttribute $node y $y
                dom::element setAttribute $node width $width
                dom::element setAttribute $node height $height
                dom::element setAttribute $node level $level
                dom::element setAttribute $node xIcon $xIcon
                dom::element setAttribute $node yIcon $yIcon
                set list [dataTable::initializationConfiguration $table]
                if {[llength $list] > 0} {
                    set options [dom::document createElement $node configuration]
                    foreach {switch value} $list {
                        set switch [string trimleft $switch -]      ;# remove heading dash (invalid name start) and restore it later
                        if {[info exists series(::dataTable,$switch)]} {                                    ;# it is actually a list
                            nodeFromList $options $switch $value                                      ;# so store as an encoded list
                        } else {
                            dom::element setAttribute $options $switch $value
                        }
                    }
                }
            }
        }
        set viewers [dom::document createElement $root viewers]
        foreach viewer $viewer::(list) {                                     ;# note: in viewers list, viewers are in creation order
            if {![viewer::saved $viewer]} continue                                               ;# viewer does not want to be saved
            set node [dom::document createElement $viewers viewer]
            set class [classof $viewer]
            dom::element setAttribute $node class $class
            if {[viewer::manageable $viewer]} {      ;# some viewers, such as the thresholds viewer, handle their display themselves
                foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $widget::($viewer,path)] {}
                set level [canvasWindowManager::getStackLevel $global::windowManager $widget::($viewer,path)]
                dom::element setAttribute $node x $x
                dom::element setAttribute $node y $y
                dom::element setAttribute $node width $width
                dom::element setAttribute $node height $height
                dom::element setAttribute $node level $level
            }
            nodeFromList $node cells [viewer::cells $viewer]                                                        ;# list of cells
            set list [viewer::initializationConfiguration $viewer]
            if {[llength $list] > 0} {
                if {[string equal [lindex $list 0] -configurations]} {
                    # some viewers, such as thresholds, need to save several arguments lists instead of just one, so they pass them
                    # as a list of lists under the -configurations switch, which is reserved for that particular usage
                    foreach sublist [lindex $list end] {            ;# use one entry per list otherwise it looks ugly in a XML sense
                        # use configurations reserved word as a flag so the list of lists can be regenerated at read time:
                        set options [dom::document createElement $node configurations]
                        foreach {switch value} $sublist {
                            set switch [string trimleft $switch -]  ;# remove heading dash (invalid name start) and restore it later
                            if {[info exists series($class,$switch)]} {                                     ;# it is actually a list
                                nodeFromList $options $switch $value                                  ;# so store as an encoded list
                            } elseif {[string match -nocase *text $switch]} {                        ;# text with embedded new lines
                                dom::document createTextNode [dom::document createElement $options $switch] $value
                            } else {
                                dom::element setAttribute $options $switch $value
                            }
                        }
                    }
                } else {
                    set options [dom::document createElement $node configuration]
                    foreach {switch value} $list {
                        set switch [string trimleft $switch -]      ;# remove heading dash (invalid name start) and restore it later
                        if {[info exists series($class,$switch)]} {                                         ;# it is actually a list
                            nodeFromList $options $switch $value                                      ;# so store as an encoded list
                        } elseif {[string match -nocase *text $switch]} {                            ;# text with embedded new lines
                            dom::document createTextNode [dom::document createElement $options $switch] $value
                        } else {
                            dom::element setAttribute $options $switch $value
                        }
                    }
                }
            }
        }
        set data [serialize $document]
        dom::destroy $root                                                                                                ;# cleanup

        puts $file $data
        close $file
    }

}

    proc read {this} {
        variable ${this}data

        if {[string length $switched::($this,-file)] == 0} {
            error {-file option undefined}
        }
        if {[catch {set file [open $switched::($this,-file)]} message]} {
            puts stderr $message
            exit 1
        }
        set line [gets $file]                                                                                 ;# retrieve first line
        seek $file 0                                                                                                       ;# rewind
        if {[regexp {^(version [1-9]\d*\.\d+(\.\d+)*)$} $line]} {                           ;# old data format (before version 16.8)
            array set ${this}data [::read $file]
        } elseif {[catch {set ($this,root) [dom::parse [::read $file]]} message]} {                                    ;# XML format
            puts stderr "file $switched::($this,-file) is not a valid moodss configuration file:\n$message"
            exit 1
        }
        close $file
    }

    proc modules {this} {
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            array set data [set ${this}data(modules)]
            # indexed modules must be sorted so that they are created in the correct order:
            return [lsort -dictionary [array names data]]
        }
        set list {}
        foreach node [dom::selectNode $($this,root) /moodssConfiguration/modules/module] {   ;# modules were saved in creation order
            lappend list [dom::element getAttribute $node namespace]
        }
        return $list
    }

    proc modulesWithArguments {this {validateCommand {}}} {  ;# validate command allows filtering out some modules, such as instance
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            set list {}
            foreach module [modules $this] {
                moduleData $this $module data
                eval lappend list $module $data(arguments)
            }
            return $list
        }
        set list {}
        foreach node [dom::selectNode $($this,root) /moodssConfiguration/modules/module] {
            # not evaluated because namespace may contain interpreted characters, such as ;, $, ...:
            set namespace [dom::element getAttribute $node namespace]
            if {([string length $validateCommand] > 0) && ![uplevel #0 $validateCommand $namespace]} continue
            lappend list $namespace
            eval lappend list [dom::node stringValue [dom::selectNode $node arguments]]
        }
        return $list                                            ;# format: module [-option [value] -option ...] module [-option ...]
    }

    proc pollTime {this} {
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            return [set ${this}data(pollTime)]
        }
        return [dom::node stringValue [dom::selectNode $($this,root) /moodssConfiguration/pollTime]]
    }

    proc sizes {this} {
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            return "[set ${this}data(width)] [set ${this}data(height)]"
        }
        return [list\
            [dom::node stringValue [dom::selectNode $($this,root) /moodssConfiguration/width]]\
            [dom::node stringValue [dom::selectNode $($this,root) /moodssConfiguration/height]]\
        ]
    }

    # Note: all viewers options with switches ending with "text" were stored in text form so that embedded new lines are preserved,
    # so apply corresponding special processing.
    proc viewersData {this} {
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            array set data [set ${this}data(viewers)]
            set list {}
            foreach id [lsort -integer [array names data]] {
                catch {unset viewer}
                array set viewer {x {} y {} width {} height {}}             ;# for example, thresholds viewer display is not managed
                set viewer(level) {}    ;# for backward compatibility when level was not stored in save file (versions 5.1 and down)
                array set viewer $data($id)
                set options {}                              ;# gather viewer specific switched options in a switch / value pair list
                foreach {name value} [array get viewer -*] {
                    lappend options $name $value
                }
                lappend list\
                    $viewer(class) $viewer(cells) $viewer(x) $viewer(y) $viewer(width) $viewer(height) $viewer(level) $options
            }
            return $list
        }
        set list {}
        foreach viewerNode [dom::selectNode $($this,root) /moodssConfiguration/viewers/viewer] {
            # note: coordinates, sizes and level may be empty (for thresholds viewer for example)
            lappend list [dom::element getAttribute $viewerNode class] [listFromNode $viewerNode cells]\
                [dom::element getAttribute $viewerNode x] [dom::element getAttribute $viewerNode y]\
                [dom::element getAttribute $viewerNode width] [dom::element getAttribute $viewerNode height]\
                [dom::element getAttribute $viewerNode level]
            set node [dom::selectNode $viewerNode configuration]
            if {[string length $node] > 0} {                                                          ;# simple viewer configuration
                set options {}
                foreach {name value} [array get [dom::node cget $node -attributes]] {
                    lappend options -$name $value                                       ;# heading dashes were stripped at save time
                }
                foreach node [dom::selectNode $node *] {                ;# if there are children, they are list or text type options
                    set name [dom::node cget $node -nodeName]
                    if {[string match -nocase *text $name]} {                                        ;# text with embedded new lines
                        lappend options -$name [dom::node stringValue $node]
                    } else {                                                                                                 ;# list
                        lappend options -$name [listFromNode $node]
                    }
                }
                lappend list $options
            } else {                                     ;# could be a viewer (such as thresholds) with multiple configuration lists
                set lists {}
                foreach node [dom::selectNode $viewerNode configurations] {          ;# then the special configurations name is used
                    set options {}
                    foreach {name value} [array get [dom::node cget $node -attributes]] {
                        lappend options -$name $value                                   ;# heading dashes were stripped at save time
                    }
                    foreach node [dom::selectNode $node *] {                         ;# if there are children, they are list options
                        set name [dom::node cget $node -nodeName]
                        if {[string match -nocase *text $name]} {                                    ;# text with embedded new lines
                            lappend options -$name [dom::node stringValue $node]
                        } else {                                                                                             ;# list
                            lappend options -$name [listFromNode $node]
                        }
                    }
                    lappend lists $options
                }
                if {[llength $lists] > 0} {                                                       ;# there were saved configurations
                    lappend list [list -configurations $lists]
                } else {
                    lappend list {}
                }
            }                                                               ;# else there may not be any switched configuration data
        }
        return $list
    }

    proc moduleData {this module dataName} {                                                             ;# used for old format only
        variable ${this}data
        upvar 1 $dataName data

        array set all [set ${this}data(modules)]
        array set data $all($module)
    }

    # used only for old format:
    proc tableData {this module creationIndex dataName} {              ;# index is module data table creation index, starting with 0
        variable ${this}data
        upvar 1 $dataName data

        moduleData $this $module moduleData
        array set tablesData $moduleData(tables)
        unset moduleData
        set data(level) {}              ;# for backward compatibility when level was not stored in save file (versions 5.1 and down)
        set data(xIcon) {}             ;# for backward compatibility when icon was not stored in save file (versions 15.11 and down)
        set data(yIcon) {}
        # use table identifier at the specified index in creation order sorted table indices:
        set index [lindex [lsort -integer [array names tablesData]] $creationIndex]
        if {[string length $index] > 0} {    ;# table entry may not exist if new views were added to module after the file was saved
            array set data $tablesData($index)
            return 1
        } else {
            return 0
        }
    }

    proc tableNode {this module creationIndex} {      ;# used for new format only, index is module data table creation index, from 0
        set node [dom::selectNode $($this,root) /moodssConfiguration/modules/module\[@namespace=\"$module\"\]]
        if {[string length $node] == 0} {error {internal error: please report to author}}
        # note: table entry may not exist if new views were added to module after the file was saved
        return [lindex [dom::selectNode $node tables/table] $creationIndex]                    ;# tables are saved in creation order
    }

    proc tableWindowManagerData {this module creationIndex} {          ;# index is module data table creation index, starting with 0
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            if {![tableData $this $module $creationIndex data]} {
                return {}
            }
            return [list $data(x) $data(y) $data(width) $data(height) $data(level) $data(xIcon) $data(yIcon)]
        }
        if {[string length [set node [tableNode $this $module $creationIndex]]] == 0} {
            return {}
        }
        array set data [array get [dom::node cget $node -attributes]]
        return [list $data(x) $data(y) $data(width) $data(height) $data(level) $data(xIcon) $data(yIcon)]
    }

    proc tableOptions {this module creationIndex} {                    ;# index is module data table creation index, starting with 0
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            if {![tableData $this $module $creationIndex data]} {
                return {}
            }
            set options {}                                   ;# gather table specific switched options in a switch / value pair list
            foreach {name value} [array get data -*] {
                lappend options $name $value
            }
            return $options
        }
        if {[string length [set node [tableNode $this $module $creationIndex]]] == 0} {
            return {}
        }
        set options {}
        set node [dom::selectNode $node configuration]
        if {[string length $node] > 0} {                                               ;# some switched configuration data was saved
            foreach {name value} [array get [dom::node cget $node -attributes]] {
                lappend options -$name $value                                           ;# heading dashes were stripped at save time
            }
            foreach node [dom::selectNode $node *] {                                 ;# if there are children, they are list options
                lappend options -[dom::node cget $node -nodeName] [listFromNode $node]
            }
        }
        return $options
    }

    proc configurationData {this} {                                                             ;# return a global name / value list
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            return [set ${this}data(configuration)]
        }
        set node [dom::selectNode $($this,root) /moodssConfiguration/configuration]
        set list [array get [dom::node cget $node -attributes]]
        lappend list viewerColors [listFromNode $node viewerColors]
        return $list
    }

    proc version {this} {                                      ;# return the version of the application that generated the save file
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            return [set ${this}data(version)]
        }
        return [dom::node stringValue [dom::selectNode $($this,root) /moodssConfiguration/version]]
    }

    proc databaseRange {this} {             ;# return a list of 2 integers: from and to in seconds (was not available in old format)
        set node [dom::selectNode $($this,root) /moodssConfiguration/databaseRange]
        if {[string length $node] == 0} {return {}}                                       ;# that must be a real-time type dashboard
        array set data [array get [dom::node cget $node -attributes]]
        return [list $data(from) $data(to)]
    }

    proc databaseViewerWindowManagerData {this} {
        set node [dom::selectNode $($this,root) /moodssConfiguration/databaseViewer]
        if {[string length $node] == 0} {return {}}                                       ;# that must be a real-time type dashboard
        array set data [array get [dom::node cget $node -attributes]]
        return [list $data(x) $data(y) $data(width) $data(height) $data(xIcon) $data(yIcon)]
    }

if {$global::withGUI} {                                                                                      ;# used only for saving

    proc pageLabels {this} {                                                                   ;# useful only for moodss before 16.2
        variable ${this}data

        if {[info exists ${this}data]} {                                                                               ;# old format
            set tabs {}
            catch {set tabs [set ${this}data(tabs)]}                                              ;# in case there were not any tabs
            return $tabs
        }
        return {}
    }

}

    proc currentConfiguration {} {       ;# current configuration in a high performance data storage (synchronize code with write{})
        set root [new container]
        # ignore version, data and time which always change between snapshots
        container::bind $root [set container [new container configuration]]
        foreach name [configuration::variables 0] {
            container::set $container $name [set ::global::$name]
        }
        container::set $root width [winfo width .]
        container::set $root height [winfo height .]
        container::set $root pollTime $global::pollTime
        if {[info exists databaseInstances::singleton]} {                                                   ;# database history mode
            container::bind $root [set container [new container databaseRange]]
            foreach {from to} [databaseInstances::cursorsRange] {}
            container::set $container from $from
            container::set $container to $to
            container::bind $root [set container [new container databaseViewer]]
            set path $widget::($databaseInstances::singleton,path)
            foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $path] {}
            foreach {xIcon yIcon} [canvasWindowManager::iconCoordinates $global::windowManager $path] {}
            container::set $container x $x
            container::set $container y $y
            container::set $container width $width
            container::set $container height $height
            container::set $container xIcon $xIcon
            container::set $container yIcon $yIcon
        }
        container::bind $root [set modules [new container modules]]
        foreach instance $modules::(instances) {                             ;# note: in modules list, modules are in creation order
            set namespace $modules::instance::($instance,namespace)
            container::bind $modules [set module [new container module]]
            container::set $module namespace $namespace
            container::set $module arguments $modules::instance::($instance,arguments)
            container::bind $module [set tables [new container tables]]
            foreach table $dataTable::(list) {                                 ;# note: in tables list, tables are in creation order
                # filter other module tables
                if {![string equal $namespace [namespace qualifiers [composite::cget $table -data]]]} continue
                foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $widget::($table,path)] {}
                set level [canvasWindowManager::getStackLevel $global::windowManager $widget::($table,path)]
                foreach {xIcon yIcon} [canvasWindowManager::iconCoordinates $global::windowManager $widget::($table,path)] {}
                # note: icon coordinates are empty if table is not minimized
                container::bind $tables [set container [new container table]]
                container::set $container x $x
                container::set $container y $y
                container::set $container width $width
                container::set $container height $height
                container::set $container level $level
                container::set $container xIcon $xIcon
                container::set $container yIcon $yIcon
                set list [dataTable::initializationConfiguration $table]
                if {[llength $list] > 0} {
                    container::bind $container [set options [new container configuration]]
                    foreach {switch value} $list {
                        container::set $options $switch $value
                    }
                }
            }
        }
        container::bind $root [set viewers [new container viewers]]
        foreach viewer $viewer::(list) {                                     ;# note: in viewers list, viewers are in creation order
            if {![viewer::saved $viewer]} continue                                               ;# viewer does not want to be saved
            container::bind $viewers [set container [new container viewer]]
            container::set $container class [classof $viewer]
            if {[viewer::manageable $viewer]} {      ;# some viewers, such as the thresholds viewer, handle their display themselves
                foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $widget::($viewer,path)] {}
                set level [canvasWindowManager::getStackLevel $global::windowManager $widget::($viewer,path)]
                container::set $container x $x
                container::set $container y $y
                container::set $container width $width
                container::set $container height $height
                container::set $container level $level
            }
            container::set $container cells [viewer::cells $viewer]
            set list [viewer::initializationConfiguration $viewer]
            if {[llength $list] > 0} {
                container::bind $container [set options [new container configuration]]
                foreach {switch value} $list {
                    container::set $options $switch $value
                }
            }
        }
        return $root
    }

    proc snapshot {} {                                                                             ;# remember current configuration
        if {[info exists (data)]} {delete $(data)}
        set (data) [currentConfiguration]
    }

    proc changed {} {                                                        ;# see if configuration has changed since last snapshot
        if {[info exists (data)]} {
            set container [currentConfiguration]
            set equal [container::equal $(data) [currentConfiguration]]
            delete $container
            return [expr {!$equal}]
        } else {    ;# snapshot was not yet taken, assume there was no change (can happen when user closes window right after start)
            return 0
        }
    }

}
