# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: sensors.tcl,v 1.11 2004/01/01 11:39:06 jfontain Exp $

package provide sensors [lindex {$Revision: 1.11 $} 1]
package require network 1


namespace eval sensors {

    variable nextIndex 0

    array set data {
        updates 0
        0,label data 0,type dictionary 0,message {monitored data identification}
        1,label value 1,type real 1,message {current value}
        2,label unit 2,type dictionary 2,message {data unit}
        3,label minimum 3,type real 3,message {data suggested safe minimum value}
        4,label maximum 4,type real 4,message {data suggested safe maximum value}
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open sensors.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable remote
        variable data

        if {![catch {set locator $options(--remote)}] || ![catch {set locator $options(-r)}]} {                 ;# remote monitoring
            set data(pollTimes) {60 20 30 120 300 600}                                   ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) sensors($remote(host))
            if {[string equal $::tcl_platform(platform) unix]} {
                set remote(command) "$remote(protocol) -n -l $remote(user) $remote(host) sensors"
            } else {                                                                                                      ;# windows
                set remote(command) "plink -batch $remote(host) sensors"                           ;# host is rather a putty session
            }
            set file [open "| $remote(command)"]
            fileevent $file readable {set ::sensors::remote(busy) 0}
            vwait ::sensors::remote(busy)                                                              ;# do not hang user interface
        } else {
            set data(pollTimes) {30 10 20 60 120 300 600}
            set file [open {| sensors}]
        }
        # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
        if {[catch {read $file} message] || [catch {close $file} message]} {
            if {[info exists remote]} {
                set message "on remote host $remote(host) as user $remote(user): $message"
            }
            error $message
        }
    }

    proc update {} {
        variable remote

        if {[info exists remote]} {
            if {$remote(busy)} return
            set remote(busy) 1
            set file [open "| $remote(command)"]
            fileevent $file readable "sensors::process $file"                ;# do not hang user interface and other modules updates
        } else {
            process [open {| sensors}]
        }
    }

    proc process {file} {
        variable index
        variable nextIndex
        variable data
        variable remote

        while {[gets $file line] >= 0} {
            lappend lines $line
        }
        if {[info exists remote]} {
            read $file                                       ;# avoid write on pipe with no readers errors by reading remaining data
            if {[catch {close $file} message]} {
                flashMessage "error: $message"
                set lines {}                                                                   ;# consider data corrupted as a whole
            }
        } else {
            close $file
        }
        foreach line $lines {
            if {[string match *: $line]} {                                                            ;# reconstruct bi-line entries
                set first $line
                continue                                                                                          ;# get second part
            } elseif {[info exists first]} {
                set line "$first $line"                                                                                ;# join parts
                unset first
            }
            set list [parse $line]
            if {[llength $list] == 0} continue                                                                       ;# invalid data
            foreach {name value unit minimum maximum} $list {}
            if {[catch {set row $index($name)}]} {                                                                      ;# new entry
                set row [set index($name) $nextIndex]
                incr nextIndex
                set data($row,0) $name                                                                     ;# initialize static data
                set data($row,2) $unit
                set data($row,3) $minimum
                set data($row,4) $maximum
            } elseif {[info exists valid($name)]} {                    ;# multiple instances of identical names: keep the last entry
                set data($row,2) $unit
                set data($row,3) $minimum
                set data($row,4) $maximum
            }
            set data($row,1) $value
            set valid($name) {}
        }
        foreach name [array name index] {                             ;# display unknown values for data that is no longer available
            if {![info exists valid($name)]} {
                set data($index($name),1) ?
            }
        }
        if {[info exists remote]} {
            set remote(busy) 0
        }
        incr data(updates)
    }

    proc parse {line} {
        set name1 {}; set value1 {}; set name2 {}; set value2 {}                 ;# in case entry (such as vid) has no limit defined
        if {[scan $line {%[^:]: %f%s (%s = %f%*[^,], %s = %f%*[^)]} name value unit name1 value1 name2 value2] >= 3} {
            set minimum ?; set maximum ?
            if {[string equal $name1 min]} {
                set minimum $value1
            }
            if {[string equal $name1 limit]} {
                set maximum $value1
            }
            if {[string equal $name2 max]} {
                set maximum $value2
            }
            return [list $name $value $unit $minimum $maximum]
        } else {
            return {}
        }
    }

}
