# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: snmp.tcl,v 2.42 2004/01/01 11:39:06 jfontain Exp $


package provide snmp [lindex {$Revision: 2.42 $} 1]
if {[lsearch -exact $auto_path /usr/lib] < 0} {                       ;# in case Tcl/Tk is somewhere else than in the /usr hierarchy
    lappend auto_path /usr/lib
}
set version [package require Tnm]
if {[package vcompare $version 2.1.10] < 0} {
    error {Tnm version 2.1.10 or above is required}
}
set tnm3 [expr {[package vcompare $version 3] >= 0}]                                                            ;# tested on windows
if {$tnm3} {                                                                                  ;# commands are in their own namespace
    namespace import Tnm::*
    package require miscellaneous
}

proc listRemovedDuplicates {list} {
    set return {}
    foreach element $list {
        if {[lsearch -exact $return $element] < 0} {
            lappend return $element
        }
    }
    return $return
}

namespace eval snmp {

    variable nextRow 0

    array set data {
        updates 0
        switches {
            -a 1 --address 1 --community 1 --context 1 --delay 1 --directory 1 -i 1 --identifiers 1 --mibs 1 --port 1 --password 1
            --retries 1 -t 1 --table 1 --timeout 1 --title 1 --trace 0 --trim 1 --user 1 --version 1 --window 1
        }
        pollTimes {10 5 20 30 60 120 300}
    }
    set file [open snmp.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

if {$::tnm3} {

    proc mib {args} {                                      ;# override original command since -exact switch has disappeared in Tnm 3
        if {[string equal [lindex $args 0] -exact]} {
            # emulate the -exact function for the following sub-command used in this module (add more if necessary):
            switch [lindex $args 1] {children - index - parent - syntax {
                if {[string length [lindex [Tnm::mib split [lindex $args 2]] end]] > 0} {
                    error "instance identifier not allowed: $args"
                    error {instance identifier not allowed}
                }
            }}
            return [eval Tnm::mib [lrange $args 1 end]]
        } else {
            return [eval Tnm::mib $args]
        }
    }

}

    proc reportError {identifier message} {
        if {[string length $identifier] > 0} {
            set text "mib: [mib file $identifier]\n"
        }
        append text $message
        error $text
    }

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable data
        variable session
        variable trace

        catch {set trace $options(--trace)}
        set directory {}; catch {set directory $options(--directory)}                                ;# optional MIB files directory
        set mibs {}; catch {set mibs [split $options(--mibs) ,]}                           ;# comma separated list of MIB file names
        foreach file [mibFiles $directory $mibs] {
            if {![file isfile $file]} {
                error "$file: not a regular file"
            }
            if {[catch {mib load $file} message]} {
                reportError {} $message
            }
        }
        catch {set table $options(-t)}
        catch {set table $options(--table)}                                                                     ;# favor long option
        set string {}
        catch {set string $options(-i)}
        catch {set string $options(--identifiers)}                                                              ;# favor long option
        set trim {}
        catch {set trim $options(--trim)}
        if {[info exists table]} {
            processTableIdentifiers $table $string $trim
        } else {
            if {[string length $string] == 0} {
                reportError {} {table and/or list of identifiers must be specified}
            }
            processIdentifiers $string $trim
        }
        set arguments {}
        if {![info exists options(-a)]} {set options(-a) 127.0.0.1}                                                    ;# by default
        catch {set arguments [list -address $options(-a)]}
        catch {set arguments [list -address $options(--address)]}                                               ;# favor long option
        foreach switch {community context delay port password retries timeout window} {
            catch {lappend arguments -$switch $options(--$switch)}
        }
        if {[info exists options(--version)]} {                                                     ;# default is 1 (SNMP version 1)
            switch [string tolower $options(--version)] {
                1 {}
                2c {lappend arguments -version SNMPv2C}
                2u {lappend arguments -version SNMPv2U}
                default {
                    reportError {} {version must be 1, 2C or 2U}
                }
            }
        }
        if {$::tnm3} {
            set session [eval snmp generator $arguments]
        } else {
            set session [eval snmp session $arguments]
        }
        if {[info exists options(--title)]} {
            set address $options(-a)
            if {[string match *t* $options(--title)] && ![info exists table]} {
                reportError {} {--title option including table name but no table was specified (-t or --table option)}
            }
            switch $options(--title) {
                a {set data(identifier) snmp($address)}
                t {set data(identifier) snmp($table)}
                at {set data(identifier) snmp($address,$table)}
                ta {set data(identifier) snmp($table,$address)}
                default {
                    reportError {} {--title option must be a combination of the 'a' and 't' letters}
                }
            }
        }
    }

    proc validateIdentifier {name {exact 1}} {                          ;# report an error if identifier is not part of a loaded MIB
        if {$exact} {
            set invalid [catch {mib -exact syntax $name} message]
        } else {
            set invalid [catch {mib syntax $name} message]
        }
        if {$invalid} {
            reportError {} $message
        }
    }

    proc processIdentifiers {string trim} {
        # string format is identifier,identifier,...,identifier,,identifier,identifier,... with , to separate columns and ,, tables
        variable data
        variable requestIdentifiers
        variable indexColumns
        variable delta

        regsub -all ,, $string | string
        if {$::tnm3} {
            set command children
        } else {
            set command successor
        }
        foreach list [split $string |] {
            set identifiers {}
            foreach identifier [listRemovedDuplicates [split $list ,]] {
                validateIdentifier $identifier 0                            ;# no strict lookup since identifier can be instantiated
                if {[llength [mib index $identifier]] > 0} {
                    reportError $identifier "$identifier is a table: use the -t (--table) switch"
                }
                if {[string equal [mib access $identifier] not-accessible]} {                               ;# object may be a group
                    set parent $identifier
                    set invalid 1
                    foreach identifier [mib $command $parent] {                                  ;# lookup immediate successors only
                        if {![string equal [mib access $identifier] not-accessible]} {                 ;# ignore tables, groups, ...
                            if {$::tnm3} {
                                lappend identifiers [mib label $identifier]                                     ;# remove MIB header
                            } else {
                                lappend identifiers $identifier
                            }
                            set invalid 0
                        }
                    }
                    if {$invalid} {
                        reportError $parent "$parent contains no accessible immediate successors"
                    }
                } else {
                    lappend identifiers $identifier
                }
            }
            lappend views $identifiers
        }
        set requestIdentifiers {}
        # use an empty hidden column as index since there is only a single row
        array set data {0,label {} 0,type ascii 0,message {} 0,0 {}}
        set nextColumn 1
        foreach list $views {
            set columns {}
            foreach identifier $list {
                set index [lsearch -exact $requestIdentifiers $identifier]                      ;# index in request identifiers list
                if {$index < 0} {                                                                                 ;# not yet in list
                    set index [llength $requestIdentifiers]
                    if {[regexp {\.\d+$} $identifier]} {                                                             ;# instanciated
                        lappend requestIdentifiers $identifier
                    } else {
                        lappend requestIdentifiers $identifier.0                     ;# eventually complete with instance identifier
                    }
                }
                if {[catch {set column $identifierColumn($index)}]} {         ;# identifiers thus columns may be duplicated in views
                    set column $nextColumn
                    set identifierColumn($index) $column
                    incr nextColumn
                }
                lappend indexColumns($index) $column                                      ;# there can be identical columns in views
                regsub ^$trim $identifier {} data($column,label)            ;# eventually trim string from left side of column title
                # no strict lookup since identifier can be instantiated
                set data($column,type) [identifierType $identifier delta($index) 0]
                set data($column,message) [identifierMessage $identifier $delta($index)]
                switch $data($column,type) {
                    integer - real {
                        # display numeric values centered
                    }
                    default {
                        set data($column,anchor) left                                                   ;# and others left justified
                    }
                }
                lappend columns $column
            }
            lappend viewsColumns $columns
        }
        foreach columns $viewsColumns {
            lappend data(views) [list visibleColumns $columns swap 1]       ;# use a swapped display since data table has only 1 row
        }
    }

    proc processTableIdentifiers {table format trim} {
        # format is identifier,identifier,...,identifier,,identifier,identifier,... with , to separate columns and ,, tables
        variable data
        variable requestIdentifiers
        variable requestLength
        variable indexLength
        variable indexColumns                            ;# correspondance of identifier index (in the request) to displayed columns
        variable requestOids
        variable delta
        variable accessible                                                      ;# whether the table index as a whole is accessible

        # check that this is a proper table
        validateIdentifier $table
        set indexes [mib -exact index $table]
        if {$::tnm3} {
            set list {}
            foreach identifier $indexes {
                lappend list [mib label $identifier]                                                    ;# use readable name not oid
            }
            set indexes $list
        } elseif {([llength $indexes] == 1) && [string equal [mib syntax $indexes] SEQUENCE]} {
            # if table is an extension, Tnm 2 returns the entry as index, so go one level deeper to get at the index(es)
            set indexes [mib -exact index $indexes]
        }
        set indexLength [llength $indexes]
        if {($indexLength == 0) || ![string equal [mib syntax $table] {SEQUENCE OF}]} {
            reportError $table "$table: not a table"                                       ;# must have an index but not be an entry
        }
        if {$::tnm3} {set command children} else {set command successor}
        set entry [mib -exact $command $table]
        if {[llength $entry] > 1} {
            reportError $table "$table has several successors: please report case to jfontain@free.fr"
        }

        # see whether table index is accessible as a whole
        set accessible 1
        foreach identifier $indexes {
            if {[string equal [mib access $identifier] not-accessible]} {
                set accessible 0
                break
            }
        }
        set children [mib -exact $command $entry]                         ;# columns (including indexes) are the successors of entry
        # include indexes initialization, which may not be part of children when table is an extension
        foreach identifier [concat $indexes $children] {
            if {$::tnm3} {set identifier [mib label $identifier]}                                               ;# remove MIB header
            set notAccessible($identifier) [string equal [mib access $identifier] not-accessible]
        }
        if {!$accessible} {                      ;# in any index column is not accessible, consider all index columns not accessible
            foreach identifier $indexes {
                set notAccessible($identifier) 1
            }
        }

        # generate view(s) in the form of list(s) of object identifiers and internally generated columns
        if {[string length $format] == 0} {                                            ;# no identifiers format: display all columns
            # process index(es) first
            if {$indexLength > 1} {                                                                         ;# multiple column index
                # include a row number column so that, for example, different views can be identically sorted
                set identifiers (number)
                if {$accessible} {
                    eval lappend identifiers $indexes
                }
            } else {
                set identifiers $indexes    ;# always include index column first (if not accessible, index is first object instance)
            }
            # process remaining identifiers
            foreach identifier $children {                                                               ;# table column identifiers
                if {$::tnm3} {set identifier [mib label $identifier]}                                           ;# remove MIB header
                if {[lsearch -exact $identifiers $identifier] >= 0} continue                          ;# already in identifiers list
                if {$notAccessible($identifier)} continue
                lappend identifiers $identifier
            }
            set views [list $identifiers]                                                                             ;# single view
        } else {                                                                                           ;# generate list of views
            set views {}
            regsub -all ,, $format { } format                          ;# make multiple views format a list of comma separated views
            foreach list $format {
                if {$indexLength > 1} {                                                                     ;# multiple column index
                    # include a row number column so that, for example, different views can be identically sorted
                    set identifiers (number)
                } else {                                        ;# make sure single index column object is placed first in all views
                    set identifiers $indexes       ;# include index column first (if not accessible, index is first object instance)
                }
                foreach identifier [split $list ,] {                                                            ;# check identifiers
                    if {[lsearch -exact $identifiers $identifier] >= 0} continue                      ;# already in identifiers list
                    validateIdentifier $identifier
                    if {![string equal $entry [mib -exact parent $identifier]] && ([lsearch -exact $indexes $identifier] < 0)} {
                        # note: index identifiers can belong to another table
                        reportError $identifier "$identifier is not a column of table $table"
                    }
                    if {$notAccessible($identifier)} continue                                ;# never display not accessible objects
                    lappend identifiers $identifier
                }
                lappend views $identifiers
            }
        }

        # generate representation table columns data (include all columns, as views are used to select actual displayed data)
        set column 0
        foreach identifiers $views {
            foreach identifier $identifiers {
                switch $identifier {
                    (number) {                                                                                     ;# special column
                        set data($column,label) {}
                        set data($column,type) integer
                        set data($column,message) {row creation order}
                    }
                    default {                                                                                   ;# normal identifier
                        regsub ^$trim $identifier {} data($column,label)    ;# eventually trim string from left side of column title
                        set type [identifierType $identifier counter($identifier)]   ;# remember whether the identifier is a counter
                        set data($column,type) $type
                        set data($column,message) [identifierMessage $identifier $counter($identifier)]
                        # display numeric values centered and others left justified:
                        switch $type {
                            integer - real {}
                            default {set data($column,anchor) left}
                        }
                    }
                }
                lappend identifierColumns($identifier) $column
                incr column
            }
        }

        # generate request object identifiers and their mapping to displayed columns
        for {set index 0} {$index < $indexLength} {incr index} {
            set delta($index) 0                                                              ;# index identifiers cannot be counters
        }
        set requestIdentifiers {}
        set requestOids {}
        set index 0
        set column 0
        catch {unset processed}
        foreach identifiers $views {
            foreach identifier $identifiers {
                if {[info exists processed($identifier)]} continue
                set processed($identifier) {}
                if {[string match (*) $identifier] || $notAccessible($identifier)} {
                    set indexColumns(key) $identifierColumns($identifier)
                } else {
                    lappend requestIdentifiers $identifier                                      ;# append identifier to request list
                    lappend requestOids [mib oid $identifier]
                    set indexColumns($index) $identifierColumns($identifier)
                    set delta($index) $counter($identifier)                                   ;# whether the identifier is a counter
                    incr index
                }                                                            ;# else special column or not accessible: not requested
                incr column
            }
        }
        if {$index == 0} {
            error {no accessible columns}
        }
        set requestLength [incr index]                                                                    ;# including system uptime

        # generate view(s) containing visible columns, a subset of all representation table columns
        if {[string length $format] > 0} {                                                   ;# no views needed for whole table dump
            set column 0
            foreach identifiers $views {
                set columns {}
                foreach identifier $identifiers {
                    lappend columns $column
                    incr column
                }
                # always sort single index or number column
                lappend data(views) [list visibleColumns $columns sort [list [lindex $columns 0] increasing]]
            }
        }
    }

    proc identifierType {name counterName {exact 1}} {                                                 ;# must be a valid identifier
        upvar 1 $counterName counter

        set counter 0
        if {$exact} {
            set syntax [mib -exact syntax $name]
        } else {
            set syntax [mib syntax $name]
        }
        switch -glob [string tolower $syntax] {
            *gauge* - *integer* - *unsigned* {
                # (includes Gauge, Gauge32, INTEGER, Integer32 and Unsigned32)
                # note: integers can be enumerations, such as ifOperStatus, in which case they are converted to names by the SNMP
                # library, which used to create problems with viewers requiring numeric values, but which now can handle invalid
                # data, and furthermore, the user is unlikely to drop what looks like non numeric data in graph, pie, ... viewers
                return real                  ;# instead of integer since Tcl cannot handle positive integers greater than 0x7FFFFFFF
            }
            *counter* {
                # (includes Counter, Counter32 and Counter64)
                set counter 1
                return real                                                            ;# since it is displayed as per second values
            }
            default {
                return dictionary                             ;# (includes OCTET STRING, OBJECT IDENTIFIER, IpAddress and TimeTicks)
            }
        }
    }

    proc identifierMessage {name delta} {
        regsub -all {\r} [mib description $name] {} message                                           ;# clean up DOS formatted text
        if {$delta} {
            return "(per second for the last polling period)\n$message"
        } else {
            return $message
        }
    }

    proc update {} {
        variable requestIdentifiers
        variable indexLength
        variable currentRow
        variable busy

        if {[info exist busy]} return                                        ;# wait till request is complete before sending another
        set busy {}
        if {[info exists indexLength]} {                                                                                    ;# table
            catch {unset currentRow}
            getBulk $requestIdentifiers
        } else {
            get $requestIdentifiers
        }
    }

    proc get {identifiers} {
        variable session
        variable trace

        set identifiers [concat 1.3.6.1.2.1.1.3.0 $identifiers]                                          ;# insert sysUpTime.0 first
        if {[info exists trace]} {
            puts ">>> request(get-request):[formattedIdentifiers $identifiers]"
        }
        $session get $identifiers {::snmp::processResponse %E %I [list %V]}
    }

    proc getBulk {identifiers} {
        variable session
        variable trace

        set identifiers [concat 1.3.6.1.2.1.1.3 $identifiers]                                              ;# insert sysUpTime first
        if {[info exists trace]} {
            puts ">>> request(get-bulk-request):[formattedIdentifiers $identifiers]"
        }
        $session getbulk 1 1 $identifiers {::snmp::processBulkResponse %E [list %V]}
    }

    proc formattedIdentifiers {list} {
        set string {}
        foreach identifier $list {
            append string " [mib name $identifier]"
        }
        return $string
    }

    proc formattedObjects {list} {                                                                               ;# list of varbinds
        set string {}
        foreach object $list {
            foreach {identifier type value} $object {}
            append string " [mib name $identifier]($value)"
        }
        return $string
    }

    proc secondsFromSystemUptime {object} {
        foreach {identifier type value} $object {}
        if {$::tnm3} {                                                                                   ;# value is in centiseconds
            return [expr {$value / 100.0}]
        }
        scan $value {%ud %u:%u:%f} days hours minutes seconds
        return [expr {($days * 86400) + ($hours * 3600) + ($minutes * 60) + $seconds}]         ;# calculate system uptime in seconds
    }

    # objects are a list of varbinds (see snmp manual page) belonging to the same row
    proc processResponse {status errorIndex objects} {
        variable session
        variable indexColumns
        variable last
        variable data
        variable delta
        variable trace
        variable busy

        unset busy
        if {[info exists trace]} {
            puts "<<< response($status):[formattedObjects $objects]"
        }
        if {![string equal $status noError]} {
            processError $status $errorIndex $objects
        }
        # objects list could be empty or system uptime value missing (may happen in "no such name" error cases):
        catch {set time [secondsFromSystemUptime [lindex $objects 0]]}
        if {![info exists time]} {
            for {set column 0} {1} {incr column} {                                                  ;# make data disappear from view
                if {[catch {unset data(0,$column)}]} break
            }
            catch {unset last(time)}
        } else {
            catch {set period [expr {$time - $last(time)}]}
            set objects [lrange $objects 1 end]                                                              ;# remove system uptime
            set index 0
            foreach object $objects {                                                                     ;# now fill row data cells
                foreach {identifier type value} $object {}
                if {$delta($index)} {
                    set current $value
                    if {[info exists period] && ($period > 0) && [info exists last($index)]} {
                        # display deltas per second for counters
                        set value [format %.2f [expr {int($current - $last($index)) / $period}]]
                    } else {
                        set value ?                                                   ;# need at least 2 values to make a difference
                    }
                    set last($index) $current
                } elseif {[string equal $type {OBJECT IDENTIFIER}]} {
                    catch {set value [mib name $value]}                                      ;# attempt conversion to readable value
                    if {$::tnm3 && [string equal [mib syntax $identifier] {OBJECT IDENTIFIER}]} {
                        set value [mib label $value]                                                    ;# do not display MIB header
                    }
                } elseif {[string equal $type TimeTicks]} {
                    set value [formattedTimeTicks $value]
                }
                foreach column $indexColumns($index) {
                    set data(0,$column) $value                                                                ;# there is only 1 row
                }
                incr index
            }
            set last(time) $time
        }
        incr data(updates)
    }

    proc processBulkResponse {status objects} {   ;# objects are a list of varbinds (see snmp manual page) belonging to the same row
        variable requestLength
        variable session
        variable requestOids
        variable indexLength
        variable cachedIndexRow
        variable indexRow
        variable nextRow
        variable currentRow
        variable indexColumns
        variable last
        variable data
        variable delta
        variable trace
        variable busy
        variable accessible

        if {[info exists trace]} {
            puts "<<< response($status):[formattedObjects $objects]"
        }
        set error [string compare $status noError]
        if {$error || ([llength $objects] != $requestLength)} {                         ;# number of responses differs from requests
            cleanup
            incr data(updates)
            if {$error} {
                flashMessage "error: $status from [$session cget -address]"
            }
            unset busy
            return                                                                                                           ;# done
        }
        set time [secondsFromSystemUptime [lindex $objects 0]]
        set objects [lrange $objects 1 end]                                                                  ;# remove system uptime

        set index 0
        foreach object $objects requested $requestOids {
            catch {unset value}
            foreach {identifier type value} $object {}
            if {($index == 0) && (([string first $requested $identifier] != 0) || ![info exists value])} {
                # end of table passed if first oid is not a successor of the request oid or object incomplete (for resiliency sake)
                cleanup
                incr data(updates)
                unset busy
                return                                                                                                       ;# done
            }
            if {$index == 0} {                                                                      ;# use first oid instance as key
                regexp {\.(.*)$} [mib name $identifier] dummy key
            }
            incr index
        }
        if {[catch {set row $indexRow($key)}]} {                                                                          ;# new row
            if {[catch {set row $cachedIndexRow($key)}]} {         ;# try to use old index so eventual viewers can resume displaying
                set row $nextRow
                set cachedIndexRow($key) $row
                incr nextRow
            }
            set indexRow($key) $row
            if {$indexLength > 1} {
                set value [expr {$row + 1}]                                                     ;# creation order column starts at 1
                foreach column $indexColumns(key) {set data($row,$column) $value}
            } elseif {!$accessible} {                                                          ;# not accessible single column index
                foreach column $indexColumns(key) {set data($row,$column) $key}
            }
        } else {
            set period [expr {$time - $last($row,time)}]
        }
        set currentRow($row) {}                                                            ;# keep track of current rows for cleanup
        set index 0
        set identifiers {}
        foreach object $objects requested $requestOids {                                                  ;# now fill row data cells
            foreach {identifier type value} $object {}
            lappend identifiers $identifier
            if {([string first $requested $identifier] != 0) || ![info exists value]} {
                # oid is not a successor of the request oid or object incomplete (for resiliency sake): use void value for type
                switch $data([lindex $indexColumns($index) 0],type) {
                    integer - real {set value ?}
                    default {set value {}}
                }
            } elseif {$delta($index)} {
                set current $value
                if {[info exists period] && ($period > 0)} {                               ;# display deltas per second for counters
                    set value [format %.2f [expr {int($current - $last($row,$index)) / $period}]]
                } else {
                    set value ?                                                       ;# need at least 2 values to make a difference
                }
                set last($row,$index) $current
            } elseif {[string equal $type {OBJECT IDENTIFIER}]} {
                catch {set value [mib name $value]}                                          ;# attempt conversion to readable value
                if {$::tnm3 && [string equal [mib syntax $identifier] {OBJECT IDENTIFIER}]} {
                    set value [mib label $value]                                                        ;# do not display MIB header
                }
            } elseif {[string equal $type TimeTicks]} {
                set value [formattedTimeTicks $value]
            }
            foreach column $indexColumns($index) {
                set data($row,$column) $value                                                ;# fill the columns for that identifier
            }
            incr index
        }
        set last($row,time) $time
        getBulk $identifiers                                                                         ;# keep going till end of table
    }

    proc cleanup {} {                                                                 ;# remove disappeared rows and associated data
        variable currentRow
        variable indexRow
        variable data
        variable last

        foreach {key row} [array get indexRow] {
            if {[info exists currentRow($row)]} continue
            unset indexRow($key)
            array unset data $row,\[0-9\]*
            array unset last $row,*
        }
        catch {unset currentRow}
    }

    proc processError {status errorIndex objects} {
        variable session

        set message "error: $status from [$session cget -address]"
        if {$errorIndex > 0} {
            foreach {identifier type value} [lindex $objects [expr {$errorIndex - 1}]] {}                 ;# error index starts at 1
            if {[info exists identifier]} {
                append message " for [mib name $identifier] identifier"
            } else {
                append message " at index $errorIndex"
            }
        }
        flashMessage $message
    }

if {$::tnm3} {
 
    proc formattedTimeTicks {value} {                                                                    ;# value is in centiseconds
        set integer [expr {$value / 100}]
        return [formattedTime $integer][format %02u [expr {$value - ($integer * 100)}]]
    }

} else {
 
    proc formattedTimeTicks {value} {
        set centiseconds 0                                                                         ;# in case it is zero and not set
        scan $value {%ud %u:%u:%u.%u} days hours minutes seconds centiseconds
        return [format %ud%02uh%02um%02us%02u $days $hours $minutes $seconds $centiseconds]
    }

}

    proc mibFiles {directory names} {
        if {[string length $directory] > 0} {
            if {![file isdirectory $directory]} {
                error "$directory: not a directory"
            }
            if {[llength $names] == 0} {                     ;# no specific MIBs specified: use all found in the specified directory
                set names [glob -nocomplain -directory $directory *.mib]
            }
        }
        set files {}
        foreach name $names {
            if {[string length [file extension $name]] == 0} {
                append name .mib                                           ;# complete with default file name extension if necessary
            }
            if {([string length $directory] > 0) && [string equal [file dirname $name] .]} {
                set name [file join $directory $name]             ;# eventually join file without directory with specified directory
            }
            lappend files $name
        }
        return $files
    }

    proc terminate {} {
        variable session

        catch {$session destroy}                                                                ;# try to clean up and ignore errors
    }

}
