# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: utility.tcl,v 1.70 2004/01/01 11:39:06 jfontain Exp $


proc commaSeparatedString {words} {
    for {set index 0} {$index < ([llength $words] - 1)} {incr index} {
        append string "[lindex $words $index], "
    }
    append string [lindex $words $index]
    return $string
}

proc startGatheringPackageDirectories {} {
    catch {rename source _source}
    proc source {file} {
        if {![string equal [file tail $file] pkgIndex.tcl]} {
            # all moodss modules must use pkgIndex.tcl files for package management
            return [uplevel 1 _source [list $file]]
        }
        foreach name [package names] {
            set versions($name) [package versions $name]
        }
        uplevel 1 _source [list $file]
        set directory [file dirname $file]
        foreach name [package names] {                                                        ;# see what new packages are available
            set available [package versions $name]
            if {[info exists versions($name)]} {
                if {[llength $available] > [llength $versions($name)]} {     ;# there exists another version of that package already
                    set ::package(exact,$name) {}                   ;# the exact version is thus required to load the chosen package
                    if {![info exists ::package(moodss,$name)]} {
                        set ::package(directory,$name) $directory
                        set ::package(version,$name) [lindex $available end]               ;# the new package version is always last
                    }                       ;# else there is a confirmea moodss module package by that name already so load that one
                }
            } else {                                                                              ;# first time this package is seen
                set ::package(directory,$name) $directory
                set ::package(version,$name) $available
                if {[string match *moodss* $directory]} {                                         ;# must be a moodss module package
                    set ::package(moodss,$name) {}
                }
            }
        }
    }
}

proc temporaryFileName {{extension {}} {identifier {}}} {
    if {[string length $identifier] == 0} {
        set identifier [pid]                                                              ;# use process identifier as unique string
    }
    switch $::tcl_platform(platform) {
        macintosh {
            error {not implemented yet}
        }
        unix {
            foreach directory {/var/tmp /usr/tmp /tmp} {                                             ;# assume /tmp is always usable
                if {[file isdirectory $directory] && [file writable $directory]} break
            }
        }
        windows {
            set directory c:/windows/temp
            catch {set directory $::env(TEMP)}
        }
    }
    set name [file join $directory moodss$identifier]
    if {[string length $extension] > 0} {
        append name .$extension
    }
    return $name
}

proc linesCount {string} {
    return [llength [split $string \n]]
}

if {$global::withGUI} {                                                                        ;# tkTable related utility procedures

proc adjustTableColumns {table} {                               ;# automatically set column widths according to column cells content
    upvar #0 [$table cget -variable] data

    if {[array size data] == 0} return
    update idletasks                                                 ;# make sure table and its labels is completely drawn and sized
    set label [label .temporary]        ;# use a temporary label for precise measurements, instead of using the font measure command
    set firstRow [$table cget -roworigin]
    set lastRow [expr {$firstRow + [$table cget -rows]}]
    set column [$table cget -colorigin]
    set lastColumn [expr {$column + [$table cget -cols]}]
    set defaultFont [$table cget -font]
    for {} {$column < $lastColumn} {incr column} {
        set maximum 0
        for {set row $firstRow} {$row < $lastRow} {incr row} {
            if {[string length [$table hidden $row,$column]] > 0} continue                         ;# take hidden cell width as null
            if {[catch {set window [$table window cget $row,$column -window]}]} {
                set font $defaultFont
                if {[$table tag includes title $row,$column] && ([string length [$table tag cget title -font]] > 0)} {
                    set font [$table tag cget title -font]
                }
                set text {}; catch {set text $data($row,$column)}                                              ;# data may not exist
                $label configure -font $font -text $text
                set width [expr {[winfo reqwidth $label] + (2 * [$table cget -padx])}]
            } else {
                set width [expr {[winfo reqwidth $window] + (2 * [$table window cget $row,$column -padx])}]
            }
            if {$width > $maximum} {
                set maximum $width
            }
        }
        $table width $column -$maximum
    }
    destroy $label
}

proc drawTableLimits {path lastColumn {embeddedWindowsCommand {}}} {
    set previous [$path tag row lastrow]
    if {[llength $previous] > 0} {                            ;# eventually reset last row aspect in case number of rows has changed
        $path tag row {} $previous
        if {[string length $embeddedWindowsCommand] > 0} {
            uplevel #0 $embeddedWindowsCommand $previous {{1 0 1 0}}
        }
    }
    catch {$path tag cell {} [$path tag cell lastcell]}      ;# eventually reset last cell aspect in case number of rows has changed
    set row [$path index end row]
    if {$row < 0} {                                                                                                  ;# no data rows
        $path configure -borderwidth {1 0 1 1}                             ;# so that title row bottom is delimited by a thin border
        $path window configure -1,$lastColumn -borderwidth 1
    } else {
        $path configure -borderwidth {1 0 1 0}                            ;# only draw a thin dark line on top and left of each cell
        $path window configure -1,$lastColumn -borderwidth {1 1 1 0}         ;# so that the right side is delimited by a thin border
        $path tag row lastrow $row                                               ;# so that the bottom is delimited by a thin border
        $path tag cell lastcell [$path index end]
        if {[string length $embeddedWindowsCommand] > 0} {
            uplevel #0 $embeddedWindowsCommand $row {{1 0 1 1}}
        }
    }
}

}

proc compareClocks {value1 value2} {
    return [expr {[clock scan $value1 -base 0] - [clock scan $value2 -base 0]}]
}

proc emailAddressError {string} {                                          ;# requires tcllib mime package (tested with version 1.2)
    foreach list [mime::parseaddress $string] {
        array set array $list
    }
    return $array(error)                                                                                   ;# empty if valid address
}

proc sendTextEmail {from to subject text} {
    set token [mime::initialize -canonical text/plain -string $text]
    lappend headers -servers [list $global::smtpServers]
    lappend headers -header [list From $from]
    foreach address $to {
        lappend headers -header [list To $address]
    }
    lappend headers -header [list Subject $subject]
    eval smtp::sendmessage $token $headers
}

if {$global::withGUI} {

# returns true if the 2 rectangles intersect, false otherwise.
proc intersect {rectangle1 rectangle2} {     ;# a rectangle is a list: left top right bottom (compatible with canvas bounding boxes)
    foreach {left1 top1 right1 bottom1} $rectangle1 {left2 top2 right2 bottom2} $rectangle2 {}
    return [expr {!(($right1 < $left2) || ($left1 > $right2) || ($bottom1 < $top2) || ($top1 > $bottom2))}]
}

proc serialize {document} {            ;# common XML document serialization so that formatting is common to all moodss related files
    return [dom::serialize $document -indent 0 -indentspec {2 {{} {}}}]                           ;# do not replace spaces with tabs
}

proc nodeFromList {parentNode name values} {
    set node [dom::document createElement $parentNode $name]
    foreach value $values {
        dom::document createTextNode [dom::document createElement $node item] $value        ;# use item as generic list element name
    }
    return $node
}

}

# From path pointing to node with values (see nodeFromList{}), return values list.
# If path is not specified, assume node is pointing to named list with item children.
proc listFromNode {parentNode {path {}}} {
    if {[string length $path] > 0} {
        append path /                                                                                       ;# separator is required
    }
    append path item                                                                    ;# item is used as generic list element name
    set values {}
    foreach node [dom::selectNode $parentNode $path] {
        lappend values [dom::node stringValue $node]
    }
    return $values
}

if {$global::withGUI} {

proc busy {set {paths {}} {cursor watch}} {                         ;# make widgets busy with special mouse cursor for user feedback
    static lifo

    if {[llength $paths] == 0} {                                                                   ;# cover all toplevels by default
        set paths .
        foreach path [winfo children .] {
            if {[string equal [winfo class $path] Toplevel]} {
                lappend paths $path
            }
        }
    }
    if {$set} {
        foreach path $paths {
            if {![info exists lifo($path)]} {
                set lifo($path) [new lifo]
            }
            xifo::in $lifo($path) [$path cget -cursor]
            $path configure -cursor $cursor
        }
        update idletasks
    } else {
        foreach path $paths {
            if {[catch {set stack $lifo($path)}]} continue                                                               ;# user bug
            catch {$path configure -cursor [xifo::out $stack]}                                                 ;# widget may be gone
            if {[xifo::isEmpty $stack]} {
                delete $stack
                unset lifo($path)
            }
        }
    }
    if {[string equal $::tcl_platform(platform) windows]} update                       ;# so that busy cursor really becomes visible
}

proc centerMessage {path text {background {}} {foreground {}}} {                                  ;# use empty text to destroy label
    set label $path.centeredMessage
    if {[string length $text] == 0} {
        catch {destroy $label}                                                                                ;# label may not exist
    } else {
        if {![winfo exists $label]} {
            label $label
        }
        $label configure -text $text -background $background -foreground $foreground
        place $label -relx 0.5 -rely 0.5 -anchor center
    }
}

proc 3DBorders {path background} {                                                               ;# algorithm stolen from tkUnix3d.c
    set intensity 65535                                                                                         ;# maximum intensity
    foreach {red green blue} [winfo rgb $path $background] {}
    if {(($red * 0.5 * $red) + ($green * 1.0 * $green) + ($blue * 0.28 * $blue)) < ($intensity * 0.05 * $intensity)} {
        set dark [format {#%04X%04X%04X}\
            [expr {($intensity + (3 * $red)) / 4}] [expr {($intensity + (3 * $green)) / 4}] [expr {($intensity + (3 * $blue)) / 4}]\
        ]
    } else {
        set dark [format {#%04X%04X%04X} [expr {(60 * $red) / 100}] [expr {(60 * $green) / 100}] [expr {(60 * $blue) / 100}]]
    }
    if {$green > ($intensity * 0.95)} {
        set light [format {#%04X%04X%04X} [expr {(90 * $red) / 100}] [expr {(90 * $green) / 100}] [expr {(90 * $blue) / 100}]]
    } else {
        set tmp1 [expr {(14 * $red) / 10}]
        if {$tmp1 > $intensity} {set tmp1 $intensity}
        set tmp2 [expr {($intensity + $red) / 2}]
        set lightRed [expr {($tmp1 > $tmp2)? $tmp1: $tmp2}]
        set tmp1 [expr {(14 * $green) / 10}]
        if {$tmp1 > $intensity} {set tmp1 $intensity}
        set tmp2 [expr {($intensity + $green) / 2}]
        set lightGreen [expr {($tmp1 > $tmp2)? $tmp1: $tmp2}]
        set tmp1 [expr {(14 * $blue) / 10}]
        if {$tmp1 > $intensity} {set tmp1 $intensity}
        set tmp2 [expr {($intensity + $blue) / 2}]
        set lightBlue [expr {($tmp1 > $tmp2)? $tmp1: $tmp2}]
        set light [format {#%04X%04X%04X} $lightRed $lightGreen $lightBlue]
    }
    return [list $dark $light]
}

proc setupTextBindings {path} {
    bind $path <Control-x> [bind Text <<Cut>>]
    bind $path <Control-c> [bind Text <<Copy>>]
    bind $path <Control-v> [bind Text <<Paste>>]
}

proc vectors {left top width height} {                   ;# from rectangle, return its vectors (coordinates and size) in a flat list
    return [list\
        $left $top $width 0 $left [expr {$top + $height}] $width 0 $left $top 0 $height [expr {$left + $width}] $top 0 $height\
    ]
}

}
