#!/usr/bin/perl

package Mooix::CallStack;
require XSLoader;
XSLoader::load('Mooix::CallStack');
use Mooix::Thing;

=head1 NAME

Mooix::CallStack - interface to mooix callstacks

=head1 SYNOPSIS

  use Mooix::CallStack;

=head1 DESCRIPTION

This is a perl interface to libmoocallstack. 
As such, it can create, load, save, and manipulate callstacks.

This module will mostly be used by mooix methods that need to do some extra
checking of who called them for security reasons.

You can access (nearly) all of the C functions in libmoocallstack
directly via this module. For example:

	my $stack = Mooix::CallStack::load($filename, 0);
	print Mooix::CallStack::size($stack);

If you prefer an object oriented interface, you can do that too:

	print $stack->size;

=head1 UTILITY METHODS

Note: You have to call these line Mooix::CallStack::method(), NOT
like Mooix::CallStack->method() and NOT like $stack->method().

=over 4

=cut

=item new

Creates a new stack and returns it. You must include 2 parameters: The
directory of the mooix object to go on top of the stack, and the method name.
(Similar to the push method.)

=cut
# implemented in XS

=item get

Find and loads up the callstack of the currently running method. 
Returns a Mooix::CallStack object for the callstack or undef on error.

=cut

sub get {
	# So, what goes on here is load returns a stack *, which is
	# typedefed to Mooix_CallStack in the XS binding, and is
	# typemapped into a T_PTROBJ, which is a blessed Mooix::CallStack
	# object. Hallelujah!
	return load(file($<), 0);
}

=item calledby

Pass it two parameters: A Mooix::Thing and a method name. Returns true iff the
current stackless method was originally run by that method of that
Mooix::Thing. If a third parameter is passed and is true, then the
Mooix::Thing's method must be the last method on the callstack.

This is useful when there is a stackless method (like avatar->login) that
must only ever be run by a known-good method of a known-good object.

This also allows there to be some additional stack frames in between the
topmost stack frame and the frame with the Mooix::Thing in it. These frames
must belong to child objects that are running the same method that is
currently being run. This allows child objects to override the method with
stackless methods of their own, and then call the super method, and everything
still work.

If an empty string is given for the method name, any method will do.

=cut

sub calledby {
	my $callindex=shift()->index;
	my $callmeth=shift;
	my $caller_last=shift || 0;

	my $ret=0;
	my $stack=get();
	my $this=$stack->index;
	my $thismeth=$stack->basemethod;
	$stack=$stack->nextsegment; # no need to check current method
	while ($stack) {
		if ($stack->index eq $callindex &&
		    (! length $callmeth || $stack->basemethod eq $callmeth)) {
			# found it..
			$ret=1;
			last unless $caller_last;
		}
		elsif (! $ret && $stack->index == $this &&
		       $stack->basemethod eq $thismeth) {
		       # ignore child method before $callindex
		}
		else {
			# something unknown, so fail.
			return 0;
		}
		$stack=$stack->nextsegment;
	}
	return $ret;
}

=back

=head1 METHODS

This modules makes available some extra methods that are not in the
C library:

=over 4

=item index

The object index, in a format identical to that used by Mooix::Thing's
index method.

=cut

sub index {
	my $this=shift;
	return $this->dev.".".$this->inode;
}

=item basemethod

The basename of the method.

=cut

sub basemethod {
	my $this=shift;
	my ($basemethod) = $this->method =~ m!(?:.*/)?(.*)!;
	return $basemethod;
}

=item nextsegment

Skips ahead to the next segment of the stack, and returns the item after that
stack boundry. Use to jump to the next method call on the stack.

=cut

sub nextsegment {
	my $stack=shift()->next;
	while ($stack) {
		return $stack if $stack->boundry;
		$stack=$stack->next;
	}
}

=back

=head1 FIELDS

Fields of Mooix::CallStack objects may be accessed in the regular OO manner.
The fields are named dev, inode, method, boundry, and next.

=head1 ITERATOR

To iterate over all the callstacks, rather than using callstack_walk(), which
is not implemented in this binding, use the built-in iterator:

  my $iterator = Mooix::CallStack::iterator;
  while (my $n = $iterator->next) {
  	my $stack=Mooix::CallStack::load(Mooix::CallStack::file($n));
	# etc
  }

=cut

sub iterator {
	return Mooix::CallStack::Iter::start();
}
  
=head1 BUGS

No garbage collection is ever done. Leaks.

=head1 SEE ALSO

moocallstack.h, L<Mooix::Thing(1)>, and =security-model= in mooix programmer's
online help.

=head1 COPYRIGHT

Copyright 2002-2003 by Joey Hess <joey@mooix.net>
under the terms of the modified BSD license given in full in the file
COPYRIGHT.

=head1 AUTHOR

Joey Hess <joey@mooix.net>

=cut

1
