#!/bin/sh
#
# Mooix shell language binding. Allows object oriented access to mooix
# objects from POSIX shell scripts.
#
# How to use it:
#
# 	#!/bin/sh
# 	. /usr/share/mooix/mooix.sh
#	this . field = value
#	this . list = "value" "value2" "value3"
#	this . counter = $(( $(this . counter) + 1 ))
#	result=$(this . method param param2 param3)
#	this . super
#	this . get otherobj /some/other/object
#	this . field = $(otherobj . field)
#
# Not shown in the above example is how to get at named parameters passed
# to the method (on stdin). Call the namedparams method of the 'this'
# object, and they will be made available in $param_<name> variables.
# If the parameter is an object reference, an object named param_<name> is
# automatically set up.
#
# To return an object reference, just invoke the name of the object without
# any parameters. To return text, echo the text.
#
# Note that this is:
#   - slow
#   - insecure (it should never, ever, be used in a stackless method or
#               as a verb or other method that might get untrusted user
#               input)
# In other words, it's POSIX shell script. :-/

# Echos each parameter on a line of its own.
_echoparams () {
	while [ "$1" ]; do
		echo "$1"
		shift
	done
}

# All object field/method calls are handled in here.
_handle () {
	OBJ=$1
	USEDIR=$2
	FULLDIR=$3
	shift 3
	if [ -n "$2" ]; then
		WHAT=$2
		FIELDFILE="$USEDIR/$2"
		shift 2
	else
		# "Stringify" an object.
		echo "mooix:$FULLDIR"
		return
	fi
	
	case "$WHAT" in
		get)
			# Set up an object.
			TPWD=$(pwd)
			DIR=${2#mooix:}
			cd "$DIR"
			eval "$1 () { _handle $1 $DIR $(pwd) \"\$@\"; }"
			cd "$TPWD"
			return
		;;
		id)
			# Get the directory of an object.
			echo $FULLDIR
			return
		;;
		fieldfile)
			# Returns the filename where a field exists. Takes
			# inheritence into account.
			TPWD=$(pwd)
			cd "$USEDIR"
			if [ "$1" != parent ]; then
				DIR="./parent"
				while [ -e "$DIR/.mooix" ]; do
					if [ -e "$DIR/$1" ]; then
						echo "$DIR/$1"
						cd "$TPWD"
						return
					fi
					DIR="$DIR/parent"
				done
			fi

			# Mixin support.
			if [ "$1" != "${1%_*}" ]; then
				MIXIN=${1%_*}
				FIELD=${1#${1%_*}_}
				DIR=./
				# Try to find the mixin object, recursing
				# to parents as necessary.
				while : ; do
					if [ -e "$DIR/$MIXIN/.mooix" ]; then
						# Now find the field in the
						# mixin or one of its
						# parents.
						if [ -e "$DIR/$MIXIN/$FIELD" -o \
						     -L "$DIR/$MIXIN/$FIELD" ]; then
							echo "$DIR/$MIXIN/$FIELD"
							cd "$TPWD"
							return
						fi
						DIR="$DIR/$MIXIN/parent"
						while [ -e "$DIR/.mooix" ]; do
							if [ -e "$DIR/$FIELD" ]; then
								echo "$DIR/$FIELD"
								cd "$TPWD"
								return
							fi
							DIR="$DIR/$PARENT"
						done

						# If the mixin doesn't have
						# it, failure (don't recurse
						# to other parents).
						cd "$TPWD"
						return
					fi
				done
			fi
			
			return
		;;
		super)
			# Call parent's implementation of the currently
			# running method. $METHOD provides state about which
			# method is running now ($0 won't work, does not
			# have the real path to the current method).
			FIELDFILE=''
			if [ -z "$METHOD" ]; then
				echo "er, METHOD is not set" >&2
				exit 1
			fi
			BASE_METHOD=${METHOD#${METHOD%/*}/} # basename
			PREFIX=${METHOD%/*}/parent # dirname
			while [ -e "$PREFIX/.mooix" ]; do
				if [ -f "$PREFIX/$BASE_METHOD" ]; then
					FIELDFILE="$PREFIX/$BASE_METHOD"
					break
				fi
				PREFIX="$PREFIX/parent"
			done
			if [ -z "$FIELDFILE" ]; then
				return
			fi
			# Fall through to the code that actually runs the
			# method (or accesses the field, if super is a field..)
		;;
		namedparams)
			while : ; do
				read NAME || return 0
				# Strip quotes.
				NAME=${NAME%\"}
				NAME=${NAME#\"}
				read VALUE || return
				# See if it looks like a mooix: ref.
				if [ "$VALUE" != "${VALUE#mooix:}" ]; then
					_handle '' '' '' . get "param_$NAME" "$VALUE"
				fi
				VALUE=${VALUE%\"}
				VALUE=${VALUE#\"}
				# the single quotes delay evaluation until
				# the second go round and close a security
				# hole in $VALUE. $NAME is still explotable
				# though.
				eval param_$NAME='$VALUE'
			done
			return
		;;
	esac
	
	TTPWD=$(pwd)
	if [ ! -e "$FIELDFILE" ] && [ "z$1" != z= ] ; then
		# Get inherited field.
		FIELDFILE=$(_handle $OBJ $USEDIR $FULLDIR . fieldfile $WHAT)
		if [ -z "$FIELDFILE" ]; then
			return
		fi
		# Returned relative to object directory, so chdir.
		cd "$USEDIR"
	elif [ "z$1" = z= ]; then
		# Set field.
		shift
		if [ -x "$FIELDFILE" ] && [ ! -d "$FIELDFILE" ]; then
			# It was set like a field, but it's really a
			# method. That's fine, just run it like a method.
			# below.
			:
		elif [ -n "$THIS" ] && [ "$USEDIR" != "$THIS" ]; then
			# Use setfield method that has the perms to
			# change another object's field (mayebe).
			cd "$USEDIR"
			_handle $OBJ . $(pwd) . setfield $WHAT "$@"
			cd "$TTPWD"
			return
		elif [ "$1" = "${1#mooix:}" ]; then
			# Set a field of an object. Support array setting by
			# setting multiple lines.
			_echoparams "$@" > $USEDIR/$WHAT
		else
			# Set a field to refer to an object. Requires a
			# symlink.
			rm -f $USEDIR/$WHAT
			ln -sf "${1#mooix:}" $USEDIR/$WHAT
		fi
	fi
	
	if [ -d "$FIELDFILE" ]; then
		# For an object reference, return the filename the
		# directory or link points to, absolute path. There is no
		# way to instantiate the object and return that, sadly, in
		# shell script.
		TPWD=$(pwd)
		cd "$FIELDFILE"
		DIR=$(pwd)
		cd "$TPWD"
		echo mooix:$DIR
		cd "$TTPWD"
		return
	elif [ -x "$FIELDFILE" ]; then
		if [ "$THIS" ]; then
			_echoparams "$@" | "$FIELDFILE"
		else
			TTTPWD=$(pwd)
			_echoparams "$@" | runmeth "$FIELDFILE"
		fi
	else
		# Get field.
		cat "$FIELDFILE"
	fi
	cd "$TTPWD"
}

# When the language binding is sourced, if running inside mooix,
# create an object called "this" which is the object the method is running
# on. Also,
if [ ! -z "$THIS" ]; then
	# Go to the object directory, so relative paths can be used; that
	# is marginally faster and safer.
	cd "$THIS"
	# Set up the 'this' object.
	_handle '' '' '' . get this .
fi
