/*
 * All the object-specific stuff goes here. None of the rest of mood needs
 * to know anything about how objects work.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the GNU GPL.
 */

#include "mood.h"
#include <stdlib.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <syslog.h>
#include <assert.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

/* Returns true if the current directory is a mooix object. */
int in_mooix_obj (void) {
	static struct stat buf;
	return (stat(MOOFILE, &buf) == 0);
}

/* Given a path, checks to see if it's in the current object.
 *
 * It's useful to assume that good clients (like libmooproxy) will only pass
 * in paths that were obtained with basename. That eliminates the need to
 * deal with all the crazy but valid things like ".///.///./////..". Instead, 
 * it's sufficient to just look for / in the path, and fail if it's there.
 * Also fail if the string is "..", cause that's not part of this object!
 */
inline int is_object_file (const char *string) {
	if (strchr(string, '/') || strcmp(string, "..") == 0)
		return 0;
	else
		return 1;
}

/*
 * Checks to see if it's ok to run a method, and checks to see if it's a
 * stackless method. The method must be defined on current object, or on 
 * one of its parents, or on a mixin. As a side effect, securely cd's into
 * the object that defines the method.
 *
 * If the method is stackless, returns that, otherwise, just returns normal,
 * while invalid is returned on error.
 */
enum method_type get_method_type (const char *filename) {
	struct stat filestat, sbuf;
	char *methbuf, *method, *path, *s, *buf;
	int stackless_ok = 1;
	
	/* Get the method name. */
	methbuf = strdup(filename);
	assert(methbuf != NULL);
	method = basename(methbuf);
	
	/* This is used later to compare a found method with the one we
	 * were told to run. This is checked to prevent various spoofs like
	 * moving links around while we're traversing them. */
	if (stat(filename, &filestat) == -1) {
		free(methbuf);
		return invalid;
	}

	/* Check to make sure that the method is really defined in the
	 * object it is being run on, or a parent thereof, or in a mixin.
	 * Follows the path to the method, checking every step along the
	 * way. */
	buf = strdup(filename);
	assert(buf != NULL);
	path = strdup(dirname(buf));
	free(buf);
	s = strtok(path, "/");
	
	for (;;) {
		if (! in_mooix_obj)
			return invalid;
		
		if (s == NULL) { /* at the end of the path */
			/* See if there is a method with the same name, 
			 * and if it is the same one. */
			if (stat(method, &sbuf) != -1 &&
			    sbuf.st_dev == filestat.st_dev &&
			    sbuf.st_ino == filestat.st_ino) {
				free(path);
				free(methbuf);
				/* Is it stackless (sticky)? That's why all the
				 * paranoid code above is necessary; attackers
				 * can't be allowed to spoof mood into running a
				 * sticky method on an object that it is not part
				 * of. */
				if (stackless_ok && (sbuf.st_mode & S_ISVTX) == S_ISVTX)
					return stackless;
				else
					return normal;
			}
			free(path);
			free(methbuf);
			return invalid;
		}
		
		/* Move on down the path (skipping "." components, and
		 * empty components caused by "//"). */
		if (strcmp(s, "") != 0 && strcmp(s, ".") != 0) {
			/* We'll follow the path into non-parents (to support
			 * mixins), but stackless methods in them cannot be run.
			 * This is because there is currently no way to indicate
			 * that a link to an object is a mixin or not. */
			if (stackless_ok && strcmp(s, "parent") != 0)
				stackless_ok = 0;

			if (chdir(s) == -1) {
				free(methbuf);
				free(path);
				return invalid;
			}
		}

		s = strtok(NULL, "/");
	}
}

/*
 * Creates and returns an callstack that has the owner of the object and
 * all its parents on it. If there is no owner, returns NULL. 
 *
 * Should be passed a fd open to the object's directory.
 *
 * Note that the owner of an object is not inherited from parent objects,
 * so this function does not look for owners of parent objects if the
 * object has no owner of its own. This is done for speed; looking up all
 * those owners would be slow.
 */
struct callstack *object_owner_stack (int objectfd) {
	struct callstack *ret = NULL;
	int olddir;
	int res;
	int ownerfd;
	struct stat buf;
	
	olddir = open(".", O_RDONLY);
	assert(olddir != -1);
	
	res = fchdir(objectfd);
	assert(res != -1);

	if ((ownerfd = open("owner", O_RDONLY)) != -1) {
		if (fstat(ownerfd, &buf) == 0 && S_ISDIR(buf.st_mode)) {
			ret = object_parents_stack(ownerfd, 0);
		}
		close(ownerfd);
	}
	
	res = fchdir(olddir);
	assert(res != -1);
	
	return ret;
}

/*
 * Creates and returns a callstack that has the object on top, and all the
 * parents of the object underneath. Should be passed a fd open to the
 * object's directory.
 *
 * Pass a true value for nochdir, unless you don't care where the method
 * leaves you when it's done.
 */
struct callstack *object_parents_stack (int objectfd, int nochdir) {
	struct callstack *ret;
	int res;
	int thisdir, olddir = -1;
	int c=0;

	if (nochdir) {
		olddir = open(".", O_RDONLY);
		assert(olddir != -1);
	}
	
	res = fchdir(objectfd);
	assert(res != -1);

	ret = callstack_push(NULL, objectfd, NULL);
	
	for (;;) {
		if (chdir("parent") == -1)
			break;
		
		if (! in_mooix_obj()) {
			warn("a parent is lacking the .mooix file");
			exit(-1);
		}

		thisdir = open(".", O_RDONLY);
		ret = callstack_push(ret, thisdir, NULL);
		close(thisdir);
		
		if (c++ > 200) {
			warn("possible parent loop");
			exit(-1);
		}
	}

	if (nochdir) {
		res = fchdir(olddir);
		assert(res != -1);
	}
	
	return callstack_invert(ret); /* invert result, so object is on top */
}

/*
 * Checks to see if it's ok to make a file the given mode.
 * 
 * Creation of executable files with the suid or sgid bit set is never allowed.
 */
int valid_file_mode (stackinfo *info, mode_t mode) {
	if (((mode & S_ISGID) == S_ISGID) || ((mode & S_ISUID) == S_ISUID)) {
		warn("illegal mode %i: has suid/sgid bit set", mode);
		errno = EINVAL;
		return 0;
	}
		
	return 1;
}

/*
 * Checks to see if it's ok to make a directory the given mode.
 * 
 * Creation of setgid directories is never allowed, setuid directories are.
 */
int valid_dir_mode (stackinfo *info, mode_t mode) {
	if ((mode & S_ISGID) == S_ISGID) {
		warn("illegal mode: has sgid bit set");
		errno = EINVAL;
		return 0;
	}
	
	return 1;
}

/*
 * Checks to see if it's ok to open a file with the given flags and mode.
 *
 * The rules are these:
 *
 * 0. File creation checks must be handed off to can_create().
 * 1. World readable/writable fields can always be opened.
 * 2. "unimportant" fields (group read/writable) may be opened if the top
 *    of the stackinfo's stack is the object, or one of its parents (in
 *    practice, this is always the case)
 * 3. "important" fields (not group read/writable) may only be opened if
 *    the stackinfo's important_ok flag is set.
 */
int can_open (stackinfo *info, const char *file, int flags, mode_t mode) {
	struct stat buf;
	int res;
	
	res = stat(file, &buf);
	
	if ((flags & O_CREAT) == O_CREAT) {
		if (! valid_file_mode(info, mode)) {
			errno = EINVAL;
			return 0;
		}
		
		/* Does file not exist? If so the stat failed. */
		if (res == -1)
			return can_create(info, file);
		/* File exists, so ignore the O_CREAT flag. */
	}
	
	if ((flags & O_RDWR) == O_RDWR) {
		/* 1. word read and writable fields */
		if (((buf.st_mode & S_IROTH) == S_IROTH) &&
		    ((buf.st_mode & S_IWOTH) == S_IWOTH)) {
			return 1;
		}
		/* 2. group read and writable "unimportant" fields */
		else if (((buf.st_mode & S_IRGRP) == S_IRGRP) &&
		         ((buf.st_mode & S_IWGRP) == S_IWGRP)) {
			return 1;
		}
		/* 3. user read and writable "important" fields */
		else if (((buf.st_mode & S_IRUSR) == S_IRUSR) &&
		         ((buf.st_mode & S_IWUSR) == S_IWUSR) &&
		         important_ok(info) == true) {
			return 1;
		}
		else {
			errno = EACCES;
			return 0;
		}
	}
	else if ((flags & O_WRONLY) == O_WRONLY) {
		/* 1. world writable fields */
		if ((buf.st_mode & S_IWOTH) == S_IWOTH) {
			return 1;
		}
		/* 2. group writable fields */
		else if ((buf.st_mode & S_IWGRP) == S_IWGRP) {
			return 1;
		}
		/* 3. user writable "important" fields */
		else if (((buf.st_mode & S_IWUSR) == S_IWUSR) &&
		         important_ok(info) == true) {
			return 1;
		}
		else {
			errno = EACCES;
			return 0;
		}
	}
	/* Order is important. O_RDONLY is typically 0, and bitwise anding 
	 * 0 with anything yeilds 0, so do this last.. */
	else if ((flags & O_RDONLY) == O_RDONLY) {
		/* world readable fields */
		if ((buf.st_mode & S_IROTH) == S_IROTH) {
			return 1;
		}
		/* 2. group readable fields */
		else if ((buf.st_mode & S_IRGRP) == S_IRGRP) {
			return 1;
		}
		/* 3. user readable "important" fields */
		else if (((buf.st_mode & S_IRUSR) == S_IRUSR) &&
		         important_ok(info) == true) {
			return 1;
		}
		else {
			errno = EACCES;
			return 0;
		}
	}
	else {
		gripe("open: bad flags %i", flags);
		errno = EACCES;
		return 0;
	}
}

/*
 * Checks to see if it's ok to create a file, directory, or symlink in the
 * object in the current directory.
 *
 * The rules are these:
 *
 * 1. If the object is world writable, creation always succeeds.
 * 2. If the object is group writable, creation succeeds if the top
 *    of the stackinfo's stack is the object, or one of its parents (in
 *    practice, this is always the case).
 * 3. If the object is user writable, creation succeeds if the stackinfo's
 *    important_ok flag is set. 
 */
int can_create (stackinfo *info, const char *file) {
	struct stat buf;
        int res;

        res = stat(".", &buf);
	assert(res == 0);

	/* 1. world writable objects */
	if ((buf.st_mode & S_IWOTH) == S_IWOTH) {
		return 1;
	}
	/* 2. group writable objects */
	else if ((buf.st_mode & S_IWGRP) == S_IWGRP) {
		return 1;
	}
	/* 3. user writable objects, and important_ok */
	else if (((buf.st_mode & S_IWUSR) == S_IWUSR) &&
	         important_ok(info) == true) {
		return 1;
	}
	else {
		errno = EACCES;
		return 0;
	}
}

/* Checks to see if a file can be unlinked. This is actually the same set
 * of tests as creating a file. */
int can_unlink (stackinfo *info, const char *file) {
	return can_create(info, file);
}

/* Checks to see if a directory can be rmdired. This is actually the same set
 * of tests as creating it. */
int can_rmdir (stackinfo *info, const char *dir) {
	return can_create(info, dir);
}

/*
 * Checks to see if it's ok to fchmod a file of the object in the current
 * directory. For this to be allowed, the only objects on the stack must be
 * the object and its parents and owner. Luckily, the important_ok flag is 
 * set if that is the case.
 */
int can_chmod (stackinfo *info, const char *file, int mode) {
	if (! valid_file_mode(info, mode)) {
		return 0;
	}
	return (important_ok(info) == true);
}

/*
 * Checks to see if a method is allowed to send a signal to every/any running
 * method of its object. For this to be allowed, the only objects on the
 * stack must be the object and its parents and owner, so again the
 * important_ok flag tells all.
 */
int can_signal (stackinfo *info) {
	return (important_ok(info) == true);
}
