#!/usr/bin/perl -T
# This is a setuid helper program.
# It expects the directory of a mooix object as its parameter, and looks to
# see if that object is a tty session object with a realtty field that
# points to the tty connected to stdin. The object should also have a
# unixuser field that is the same as the real uid of this process.
# If all conditions are met, it should be safe to make a copy of the tty
# device inside the tty session object.
use strict;
use warnings;
use Mooix::Thing;
use Mooix::Root;
use Mooix::Conf;
use POSIX q{ttyname};

# Use the SAFEPATH, so tainting is happy.
$ENV{PATH}=$Mooix::Conf::field{safepath};
# Make %ENV safer
delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   

if (! @ARGV) {
	die "need a session dir";
}

my $sessdir=shift;
if (! -d $sessdir) {
	die "bad session";
}

my $uid=getpwnam($Mooix::Conf::field{mooadmin});
if ((stat($sessdir))[4] != $uid) { # 4 = user
	die "bad session owner";
}

my $session = Mooix::Thing->get($sessdir);
if (! ref $session) {
	die "bad session";
}
$session->untaint;

unless ($session->isa($Mooix::Root->sessions->tty)) {
	die "bad session parent";
}
unless ($session->defines("realtty")) {
	die "no realtty field";
}
my $tty = ttyname(0) || die "cannot find your tty: $!";
unless (readlink($session->id."/realtty") eq $tty) {
	die "tty mismatch";
}
unless ($session->defines("unixuser")) {
	die "no unixuser field";
}
my $uuid = getpwnam($session->unixuser);
if ($uuid != $<) {
	die "bad user";
}

if ($session->defines("tty")) {
	die "session already has a tty";
}

my $dest=$session->id."/tty";
# TODO something a mite more portable.. mknod?
if (system("cp", "-a", $tty, $dest) != 0) {
	die("Unable to copy $tty to $dest; perhaps that directory is mounted nodev?");
}
# The tty must be owned by the mooadmin, and group writable (so ttysession
# methods can always read/write it).
chown($uid, -1, $dest) || die "chown $dest: $!";
chmod(0664, $dest) || die "chmod $dest: $!";

exit 0;
