#!/usr/bin/perl -T
# This takes care of actually registering someone in the moo. It runs suid.
# It expects to find PASSWORD and EMAIL set in the environment. It them
# checks the reglog for an item matching the email and password, and
# validates the item. If it likes what it sees, it sets up the account, and
# prints out the unix username. Otherwise, it'll return 1 if the user and
# password were not found, and 2 if there was some other error.
#
# Note that this is a suid program (or run by one), and any regular user can
# run it. That will do them noo good though, unless they know the email and
# password of an item in the reglog. Since the reglog's items are all readably
# only by mooadmin (and root), and it can only be written to by root and
# mooadmin (outside the moo) and mooadminobj and guest users (inside the moo),
# a fair degree of safety is assured.

# Don't allow interruption.
$SIG{INT}=$SIG{TERM}=$SIG{HUP}=sub {};

# This is set when run inside the moo, and perl's taint code chokes on
# stuff Mooix::Thing does if it's set.
BEGIN { delete $ENV{THIS} };
# Make %ENV safer
delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   

# Become fully root. Makes adduser work..
$<=$>;

use warnings;
use strict;
use Mooix::Thing;
use Mooix::Root;
use Mooix::Conf;
use Fcntl qw{:flock};

sub abort ($) {
	print STDERR shift()."\n";
	exit 2;
}

# Use the SAFEPATH, so -T is happy. This also ensures that the sbindir is
# in the path.
$ENV{PATH}=$Mooix::Conf::field{safepath};

if (@ARGV) {
	abort "no arguments, please";
}

# Get parameters and untaint.
my $email = $ENV{EMAIL};
my $password = $ENV{PASSWORD};
if (! defined $email || ! defined $password) {
	abort "must be called with EMAIL and PASSWORD";
}
# detaint
if ($email =~ /^(.+@.+\..+)$/) {
	$email = $1;
}
if ($password =~ /(.*)/) {
	$password = $1;
}
if (! length $email || ! length $password) {
	abort "must be called with EMAIL and PASSWORD";
}

my $reglog = $Mooix::Root->system->reglog;
if (! ref $reglog) {
	abort "bad reglog";
}
my $users = $Mooix::Root->users;
if (! ref $users) {
	abort "bad users";
}
my $parent = Mooix::Thing->get($Mooix::Conf::field{parentavatar});
if (! ref $parent) {
	abort "bad parentavatar";
}
my $sessionmanager = $Mooix::Root->system->sessionmanager;
if (! ref $sessionmanager) {
	abort "bad sessionmanager";
}

# Only one of these should be running at a time, so lock the reglog.
open(LOCK, $reglog->id."/.mooix") || abort "unable to lock reglog";
flock(LOCK, LOCK_EX) || abort "unable to lock reglog";

my $id = $reglog->find(password => $password, email => $email);
if (! $id) {
	exit 1;
}
my %item = $reglog->loginfo(id => $id);
if (! %item) {
	abort "unable to read item";
}

# Paranoid checks of everything.
if ($item{activated}) {
	abort "already activated";
}
if ($item{email} !~ /^.+@.+\..+$/) {
	abort "bad email address: $email ($item{email}) $id";
}
if (length $item{password} < 8) {
	abort "bad password";
}

# Come up with a unique unix username based on the name.
my $uname = lc $item{name};
$uname =~ s/[^a-z0-9]//g;
if (! length $item{name}) {
	abort "bad name";
}
$uname = $Mooix::Conf::field{moouserprefix}.$uname;
$uname = substr($uname, 0, 8);
# Get all the users from the password file.
my %users;
while (my @ent = getpwent()) {
	$users{$ent[0]}=1;
}
# To make the name unique, add a number on the end if necessary.
my $c=1;
my $origuname=$uname;
while (exists $users{$uname}) {
	$uname=substr($origuname, 0, 8 - length($c)).$c;
	$c++;
}

# Create directory for user. This serves as their home directory.
my $homedir = $users->newid(mkdir => 1, hint => $uname);
if (! length $homedir) {
	abort "unable to crete home directory"; 
}

# Add user (with locked password).
my $pid = fork();
if (! $pid) {
	close STDIN;
	close STDOUT;
	exec "addmoouser", $homedir, "/usr/bin/moologin", $uname;
	exit;
}
wait();
if ($? != 0) {
	abort "adding user failed";
}

# Make them own their home directory, and set up the object.
if (system("chown", "-R", "$uname:$uname", $homedir) != 0) {
	abort "problem chowning homedir";
}
my $obj = $parent->new(id => $homedir, name => $item{name}, users => $uname);
if (! ref $obj) {
	abort "unable to set up avatar";
}

# Add the object to the sessionmanager's avatars list.
$sessionmanager->avatars->add(object => $obj);

# Set the user's password. This is rather disgusting, but I cannot think of
# a better method, except chpasswd, which may not be widely available.
$pid = open(PASSWD, "|-");
if ($pid) {
	select PASSWD;
	$|=1;
	select STDOUT;
	print PASSWD $password."\n";
	sleep 1; # passwd has a slight pause
	print PASSWD $password."\n";
	close PASSWD || abort "running passwd failed (parent)";
}
else {
	close STDOUT;
	close STDERR;
	exec "/usr/bin/passwd", $uname || abort "running passwd failed (child)";
}

close LOCK;

# Mark item as activated and record username.
if (! $reglog->edit(id => $id, activated => 1, username => $uname)) {
	abort "edting item failed";
}

# Print out username.
print $uname."\n";
