/*
 * Client communications.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the GNU GPL.
 */

#include "mood.h"
#include <errno.h>
#include <sys/socket.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>

/* Free a command. */
void freecommand (struct command_t *cmd) {
	free(cmd->buf);
	free(cmd);
}

/* Read a command from the client. */
struct command_t *getcommand (int client) {
	struct command_t *cmd = malloc(sizeof(struct command_t));
	int i;
	char *p;
	char *oldp;
	
	/* Read raw structure from client, which gives us all the static
	 * sized bits. */
	cmd->type = 0;
	i = read(client, cmd, sizeof(struct command_t));
	
	if (i <= 0) {
		//debug("read %i from client; error: %s", i, strerror(errno));
		close(client);
		return NULL;
	}
	else if (i < sizeof(struct command_t)) {
		int total = 0;
		void *cmdp;
		
		//debug("short getcommand read %i (expected %i); taking contermeasures", i, sizeof(struct command_t));
		while (total < sizeof(struct command_t)) {
			if (i > 0)
				total += i;
			cmdp = (void *) cmd + total;
			i = read(client, cmdp, sizeof(struct command_t) - total);
			if (i <= 0) {
				close(client);
				return NULL;
			}
			//debug("read %i this time around error: %s ; total: %i", i, strerror(errno), total);
		}
	}
	
	/* Read the argv vector components from the client, in the form of a
	 * single long buffer. */
	rassert((cmd->buf = malloc((cmd->len + 1) * sizeof(char))) != NULL);
	rassert(read(client, cmd->buf, cmd->len) == cmd->len);
	cmd->buf[cmd->len] = '\0'; /* client is not required to do this */
	
	/* Now make argv point to the pieces of the buffer. */
	rassert((cmd->argv = malloc(cmd->argc * sizeof(char *))) != NULL);
	oldp = p = cmd->buf;
	for (i = 0; i < cmd->argc; i++) {
		rassert((p = memchr(p, '\0', cmd->len + 1 - (p - cmd->buf))) != NULL);
		p++; /* skip over \0 */
		cmd->argv[i] = oldp;
		oldp=p;
	}
	return cmd;
}

/* Return a result structure to the client. */
void result (int client, int ret, int err) {
	int c;
	struct result_t r;
	r.ret=ret;
	r.err=err;
	c = write(client, &r, sizeof(r));
	if (c != -1 && c < sizeof(r)) {
		//debug("short write to client (wrote %i of %i)", c, sizeof(r));
	}
}

/*
 * The rassert macro uses this function.
 */
int _rassert (int client, const char *expr, int ret, const char *func, const char *file, int line) {
	if (! ret) {
		if (errno)
			warn("%s:%i: %s: Assertion '%s' failed (%s).",
				file, line, func, expr, strerror(errno));
		else
			warn("%s:%i: %s: Assertion '%s' failed.",
				file, line, func, expr);
		result(client, ret, errno ? errno : ECANCELED);
		close(client);
		shutdown(client, SHUT_RDWR);
		exit(1);
	}
	return ret;
}
