/* Tty prompt method. Written in C for speed. */

#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <moomethod.h>

#include "safetext.c"

int use_history=1;

/* Global file handle. */
int tty;
FILE *ttyf;

/* Get an exclusive lock on the tty. */
void locktty () {
	if (flock(tty, LOCK_EX) == -1) {
		perror("failed to lock tty");
		exit(1);
	}
}

/* Write pending text to the tty. */
void dowrite () {
	int fd, count;
	char buf[128];
	/* Before printing text to the tty, we must clear out the line(s)
	 * used by readline. Otherwise, if the text to be printed is
	 * shorter than what readline has displayed, ugliness will ensue on
	 * the tty. The simplest way I can think of to accomplish this is
	 * to set readline's line buffer to empty (backing up what was
	 * there), and force a redisplay, which clears the old buffer
	 * display. Then we must merely clear out the prompt, and it's ok
	 * to write.
	 */
	char *real_buffer = rl_line_buffer;
	int real_point = rl_point;
	int real_end = rl_end;
	int real_mark = rl_mark;
	rl_line_buffer = "";
	rl_point = 0;
	rl_end = 0;
	rl_mark = 0;
	rl_save_prompt();
	rl_redisplay();

	/* The .pending file is where the text we're to write is stored.
	 * Lock it to prevent races, then read from it, then zero it out.
	 */
	if ((fd = open(".pending", O_RDWR)) != -1) {
		flock(fd, LOCK_EX);
		while ((count = read(fd, buf, 127)) > 0)
			write(tty, buf, count);
		ftruncate(fd, 0);
		close(fd);
	}
	
	/* Put back everything the way it was, and let readline know
	 * that it's been moved to a new line, and so it needs to
	 * redisplay the input area. */
	rl_restore_prompt();
	rl_line_buffer = real_buffer;
	rl_point = real_point;
	rl_end = real_end;
	rl_mark = real_mark;
	rl_on_new_line();
	rl_redisplay();
}

/* This function will be called by readline when it has a line of text. */
void process_line(char *input) {
	rl_callback_handler_remove();
	
	if (input != NULL) {
		if (use_history)
			add_history(input);
		printf("%s\n", escape(input));
		
		if (use_history) {
			/* Save out history. */
			stifle_history(20);
			write_history("history");
		}
	}
	else { /* ctrl-d */
		/* Just for kicks. */
		fprintf(ttyf, "\n");
	}

	exit(0);
}

/* Default text. */
char *def;
/* Shove the default text into readline's buffer. */
int setdef () {
	rl_insert_text(def);
	return 1;
}

/* This string holds possible completions, separated by bars. */
char *completions;
/* This holds a parsed version of the completions, in an array, NULL
 * terminated. */
char **comparray;
/* This holds more completions, from the avatar's completions method. */
char **avcomparray;
/* Generator function for completion. */
char *match_completions(const char *text, int state) {
	static int list_index, avlist_index, len;
	char *name;
	FILE *compf;
	char *methparams[3];
	
	if (! state) {
		/* New word. */
		if (! comparray) {
			/* Fill comparray, parsing the completions string. */
			int i = 0, size = 32;
			char *p = completions, *q;
			
			if (! completions) return NULL;
			comparray=malloc(size * sizeof(char *));
			while ((q = strchr(p, '|'))) {
				q[0]='\0';
				comparray[i] = p;
				i++;
				if (i == size - 2) {
					size *= 2;
					comparray=realloc(comparray, size * sizeof(char *));
				}
				p=q+1;
			}
			comparray[i] = p;
			comparray[i+1] = NULL;
		}

		/* Free up existing array. */
		if (avcomparray) {
			int i = 0;
			while (avcomparray[i]) {
				free(avcomparray[i]);
				i++;
			}
			free(avcomparray);
		}
		
		/* Get more completions from the avatar's completions
		 * method. */
		methparams[0] = "text";
		methparams[1] = escape(text);
		methparams[2] = NULL;
		compf = runmethod(getobj("avatar"), "completions", methparams);
		free(methparams[1]);
		if (compf) {
			avcomparray = fgetallvals(compf);
			fclose(compf);
		}
		else {
			avcomparray = NULL;
		}
		
		list_index = 0;
		avlist_index = 0;
		len=strlen(text);
	}
	
	/* Return the next item which partially matches from comparray. */
	while ((name = comparray[list_index])) {
		list_index++;
		if (strncasecmp(name, text, len) == 0)
			return strdup(name);
	}

	/* Or avcomparray. */
	while ((name = avcomparray[avlist_index])) {
		avlist_index++;
		if (strncasecmp(name, text, len) == 0)
			return strdup(name);
	}
	
	return NULL;
}

/* This is the tab completion function for readline. */
char **completor (const char *text, int start, int end) {
	/* Tell readline not to do its regular filename completion. */
	rl_attempted_completion_over = 1;
	
	return rl_completion_matches(text, match_completions);
}

int main (int argc, char **argv) {
	param *p;
	char *prompt = "> ";
	int pending;
	static struct stat buf;
	fd_set fds;
	struct timeval tv;
	completions = NULL;
	comparray = NULL;
	avcomparray = NULL;
	
	methinit();
	
	/* Get TERM type and tell readline. */
	rl_terminal_name = getfield("term");

	/* Set some readline parameters to more sane defaults for the moo
	 * environment. */
	rl_variable_bind("bell-style", "none");
	rl_variable_bind("show-all-if-ambiguous", "on");
	
	/* Parse parameters. */
	while ((p = getparam())) {
		if (strcmp(p->name, "prompt") == 0) {
			prompt = safen(p->value);
		}
		else if (strcmp(p->name, "default") == 0) {
			def = safen(p->value);
			rl_startup_hook = setdef;
		}
		else if (strcmp(p->name, "completions") == 0) {
			completions = safen(p->value);
		}
		else if  (strcmp(p->name, "nohistory") == 0) {
			use_history = ! atoi(p->value);
			freeparam(p);
		}
		else {
			freeparam(p);
		}
	}
	
	if (use_history) {
		/* Load up history. */
		using_history();
		read_history("history");
	}
	
	/* Open the tty device and lock it. */
	if ((tty = open("tty", O_RDWR)) == -1) {
		perror("open tty for locking");
		exit(1);
	}
	locktty();

	/* Set up readline to use the tty. */
	ttyf = fdopen(tty, "r+");
	if (ttyf == NULL) {
		perror("fdopen tty");
		exit(1);
	}
	rl_instream = rl_outstream = ttyf;
	rl_attempted_completion_function = completor;

	/* This filehandle is used as a semophore. */
	if ((pending = open(".pending", O_WRONLY)) == -1) {
		perror("open .pending");
		exit(1);
	}

	rl_callback_handler_install(prompt, process_line);

	while (1) {
		/* Check for pending text to display. */
		fstat(pending, &buf);
		if (buf.st_size > 0)
			dowrite();

		/* Check for user input. */
		FD_ZERO(&fds);
		FD_SET(tty, &fds);
		tv.tv_sec = 0;
		tv.tv_usec = 300000; /* 0.3 seconds */
		if (select(tty + 1, &fds, NULL, NULL, &tv) < 0 &&
			   errno != EINTR) {
			/* EINTR happens on resize.. */
			perror("select");
			exit(1);
		}

		if (FD_ISSET(tty, &fds) ) {
			rl_callback_read_char();
		}
	}
}
