/*	alg.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2001 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include "conf.h"
#include "alg.h"
#include "event.h"

#define MAX(x, y) (abs(x) > abs(y) ? (x) : (y))

/* locate the center of the movement. */
struct coord alg_locate_center (struct images *imgs, int width, int height)
{
	char *out=imgs->out;
	int *labels=imgs->labels;
	int x, y, centc=0;
	struct coord cent;

	cent.x=0;
	cent.y=0;
	
	/* If Labeling enabled - locate center of largest label */
	if (imgs->labelsize_max) {
		/* Locate largest label */
		for (y=0; y<height; y++) {
			for (x=0; x<width; x++) {
				if (*(labels++)==imgs->largest_label) {
					cent.x+=x;
					cent.y+=y;
					centc++;
				}
			}
		}
	} else {
		/* Locate movement */
		for (y=0; y<height; y++) {
			for (x=0; x<width; x++) {
				if (*(out++)) {
					cent.x+=x;
					cent.y+=y;
					centc++;
				}
			}
		}
	}
	if (centc) {
		cent.x=cent.x/centc;
		cent.y=cent.y/centc;
	}
	return (cent);
}

/* draw a box around the movement */
void alg_locate (struct coord *cent, struct images *imgs, int width, int height, char *new)
{
	char *out=imgs->out;
	int *labels=imgs->labels;
	int x, y, maxx=0, maxy=0, minx=width, miny=height;
	int centc=0, xdist=0, ydist=0;

	centc=0;
	/* If Labeling enabled draw box around largest label instead */
	if (imgs->labelsize_max) {
		for (y=0; y<height; y++) {
			for (x=0; x<width; x++) {
				if (*(labels++)==imgs->largest_label) {
					if (x > cent->x)
						xdist+=x-cent->x;
					else if (x < cent->x)
						xdist+=cent->x-x;
					if (y > cent->y)
						ydist+=y-cent->y;
					else if (y < cent->y)
						ydist+=cent->y-y;
					centc++;
				}
			}	
		}
	} else {
		for (y=0; y<height; y++) {
			for (x=0; x<width; x++) {
				if (*(out++)) {
					if (x > cent->x)
						xdist+=x-cent->x;
					else if (x < cent->x)
						xdist+=cent->x-x;
					if (y > cent->y)
						ydist+=y-cent->y;
					else if (y < cent->y)
						ydist+=cent->y-y;
					centc++;
				}
			}	
		}
	}
	if (centc) {
		minx=cent->x-xdist/centc*2;
		maxx=cent->x+xdist/centc*2;
		/* Make the box a litle bigger in y direction to make sure the
		 * heads fit in */
		miny=cent->y-ydist/centc*3;
		maxy=cent->y+ydist/centc*2;
	}
	if (maxx > width-1)
		maxx=width-1;
	else if (maxx < 0)
		maxx=0;
	if (maxy > height-1)
		maxy=height-1;
	else if (maxy < 0)
		maxy=0;
	if (minx > width-1)
		minx=width-1;
	else if (minx < 0)
		minx=0;
	if (miny > height-1)
		miny=height-1;
	else if (miny < 0)
		miny=0;
	cent->width=maxx-minx;
	cent->height=maxy-miny;
	out=imgs->out;
	
	/* Draw a box around the movement */
	for (x=minx; x<=maxx; x++) {
		new[x+width*miny]=~new[x+width*miny];
		new[x+width*maxy]=~new[x+width*maxy];
		out[x+width*miny]=~out[x+width*miny];
		out[x+width*maxy]=~out[x+width*maxy];
	}
	for (y=miny; y<=maxy; y++) {
		new[minx+y*width]=~new[minx+y*width];
		new[maxx+y*width]=~new[maxx+y*width];
		out[minx+y*width]=~out[minx+y*width];
		out[maxx+y*width]=~out[maxx+y*width];
	}
}

#define NORM               100
#define ABS(x)             ((x)<0 ? -(x) : (x))
#define DIFF(x, y)         (ABS((x)-(y)))
#define NDIFF(x, y)        (ABS(x)*NORM/(ABS(x)+2*DIFF(x,y)))
#define NRES(in, min, max) ((in)<(min)?NDIFF(min,in):((in)>(max)?NDIFF(max,in):100))

int alg_predict_eval_set(struct predict *min, struct predict *max, struct predict *in)
{
	return (
		NRES(in->x, min->x, max->x)+
		NRES(in->y, min->y, max->y)+
		NRES(in->dx, min->dx, max->dx)+
		NRES(in->dy, min->dy, max->dy)+
		NRES(in->width, min->width, max->height)+
		NRES(in->height, min->height, max->height)+
		NRES(in->diffs, min->diffs, max->diffs)+
		NRES(in->length, min->length, max->length)
	)/8;
}

void alg_predict_eval(struct context *cnt)
{
	struct predict_set *set=cnt->predict_sets;
	int high=0, i;
	struct predict_set *highset=set;

	while (set) {
		i=alg_predict_eval_set(&set->min, &set->max, &cnt->predict);
		printf("%s: %d\n", set->name, i);
		if (i>high) {
			high=i;
			highset=set;
		}
		set=set->next;
	}
	if (highset && high>=cnt->conf.predict_threshold) {
		printf("Best guess: %s (%d%%)\n", highset->name, high);
	} else {
		printf("Unknown type of movement\n");
	}
}

void alg_predict(struct coord *box, struct context *cnt, int diffs, int first)
{
	struct predict *avg=&cnt->predict;
	struct coord *lastbox=&cnt->predict_lastbox;

	if (first) {
		memcpy(&cnt->predict_lastbox, box, sizeof(struct coord));
		cnt->last_diffs=diffs;
		memset(avg, 0, sizeof(struct predict));
		avg->length=1;
		avg->x=box->x;
		avg->y=box->y;
		avg->width=box->width;
		avg->height=box->height;
		avg->diffs=diffs;
		return;
	}
	avg->length++;
	avg->x+=box->x;
	avg->y+=box->y;
	avg->width+=box->width;
	avg->height+=box->height;
	avg->dx+=box->x-lastbox->x;
	avg->dy+=box->y-lastbox->y;
	avg->diffs+=diffs;

	memcpy(&cnt->predict_lastbox, box, sizeof(struct coord));
	cnt->last_diffs=diffs;
	return;
}

void alg_predict_result(struct context *cnt, char *filename)
{
	struct predict *avg=&cnt->predict;
	FILE *fddesc;

	if (!avg->length)
		return;
	avg->x/=avg->length;
	avg->y/=avg->length;
	avg->width/=avg->length;
	avg->height/=avg->length;
	avg->diffs/=avg->length;
	printf("length: %d\t", avg->length);
	printf("x: %d\ty: %d\twidth: %d\theight: %d\tdiffs: %d\t", avg->x, avg->y, avg->width, avg->height, avg->diffs);
	printf("dx:%d\tdy:%d\n", avg->dx, avg->dy);
	alg_predict_eval(cnt);
	if ((fddesc=myfopen(filename, "w"))==NULL) {
		syslog(LOG_ERR, "Error writing motion description file %s: %m", filename);
		return;
	}
	fprintf(fddesc, "Auto generated description\n");
	fprintf(fddesc, "%d\t", avg->length);
	fprintf(fddesc, "%d\t%d\t%d\t%d\t%d\t", avg->x, avg->y, avg->width, avg->height, avg->diffs);
	fprintf(fddesc, "%d\t%d\n", avg->dx, avg->dy);
	fprintf(fddesc, "%d\t", avg->length);
	fprintf(fddesc, "%d\t%d\t%d\t%d\t%d\t", avg->x, avg->y, avg->width, avg->height, avg->diffs);
	fprintf(fddesc, "%d\t%d\n", avg->dx, avg->dy);
	fclose(fddesc);
}


void alg_noise_tune (struct context *cnt, char *new)
{
	struct images *imgs=&cnt->imgs;
	int i;
	char *ref=imgs->ref;
	int diff, sum=0, count=0;
	unsigned char *mask=imgs->mask;

	i=imgs->motionsize;
	for (; i>0; i--) {
		diff=*ref-*new;
		if (mask)
			diff=((diff**mask++)/255);
		if (diff) {
			sum+=ABS(diff);
			count++;
		}
		ref++;
		new++;
	}
	sum/=count/4;
	cnt->noise=(cnt->noise+sum)/2;
	printf("New noise level: %d\n", cnt->noise);
}

void alg_threshold_tune (struct context *cnt, int diffs, int motion)
{
	int i;
	int sum=0, top=diffs;

	if (!diffs)
		return;

	printf("%d\n", diffs);
	if (motion)
		diffs=cnt->threshold/4;

	for (i=0; i<THRESHOLD_TUNE_LENGTH-1; i++)
	{
		sum+=cnt->diffs_last[i];
		if (cnt->diffs_last[i+1] && !motion)
			cnt->diffs_last[i]=cnt->diffs_last[i+1];
		else
			cnt->diffs_last[i]=cnt->threshold/4;
		if (cnt->diffs_last[i]>top)
			top=cnt->diffs_last[i];
	}
	sum+=cnt->diffs_last[i];
	cnt->diffs_last[i]=diffs;

	sum/=THRESHOLD_TUNE_LENGTH/4;
	if (sum<top*2)
		sum=top*2;
	if (sum < cnt->conf.max_changes)
			cnt->threshold=(cnt->threshold+sum)/2;
	printf("New threshold: %d\n", cnt->threshold);
}

/*
Labeling by Joerg Weber. Based on an idea from Hubert Mara.
Floodfill enhanced by Ian McConnel based on code from
http://www.acm.org/pubs/tog/GraphicsGems/
http://www.codeproject.com/gdi/QuickFill.asp
*/

/*
 * Filled horizontal segment of scanline y for xl<=x<=xr.
 * Parent segment was on line y-dy.  dy=1 or -1
 */

#define MAXS 10000               /* max depth of stack */

#define PUSH(Y, XL, XR, DY)     /* push new segment on stack */ \
        if (sp<stack+MAXS && Y+(DY) >= 0 && Y+(DY) < height) \
        {sp->y = Y; sp->xl = XL; sp->xr = XR; sp->dy = DY; sp++;}

#define POP(Y, XL, XR, DY)      /* pop segment off stack */ \
        {sp--; Y = sp->y+(DY = sp->dy); XL = sp->xl; XR = sp->xr;}

typedef struct {short y, xl, xr, dy;} Segment;


static int iflood(int x, int y,       /* seed point */
                  int width, int height, char *out, int *labels, int newvalue)
{
	int l, x1, x2, dy;
	Segment stack[MAXS], *sp = stack;    /* stack of filled segments */
	int count = 0;

	if (x < 0 || x >= width || y < 0 || y >= height)
		return 0;

	PUSH(y, x, x, 1);             /* needed in some cases */
	PUSH(y+1, x, x, -1);          /* seed segment (popped 1st) */

	while (sp > stack) {
		/* pop segment off stack and fill a neighboring scan line */
		POP(y, x1, x2, dy);
		/*
		 * segment of scan line y-dy for x1<=x<=x2 was previously filled,
		 * now explore adjacent pixels in scan line y
		 */
		for (x = x1; x >= 0 && out[y*width+x] != 0 && labels[y*width+x] == 0; x--) {
			labels[y*width+x] = newvalue;
			count++;
		}
		
		if (x >= x1)
			goto skip;
		
		l = x + 1;
		
		if (l < x1)
			PUSH(y, l, x1-1, -dy);  /* leak on left? */
		
		x = x1 + 1;
		
		do {
			for (; x < width && out[y*width+x] != 0 && labels[y*width+x]==0; x++){
				labels[y*width+x] = newvalue;
				count++;
			}
			PUSH(y, l, x-1, dy);
			if (x > x2+1)
				PUSH(y, x2+1, x-1, -dy);  /* leak on right? */
			skip:
			for (x++; x <= x2 && !(out[y*width+x] != 0 && labels[y*width+x]==0); x++);
			l = x;
		} while (x <= x2);
	}
	return count;
}

int alg_labeling (struct context *cnt)
{
	struct images *imgs=&cnt->imgs;
	char *out=imgs->out;
	int *labels=imgs->labels;
	int i, ix, iy, pixelpos;
	int width=imgs->width;
	int height=imgs->height;
	int labelsize=0;
	int current_label=2;
	imgs->total_labels=0;
	imgs->labelsize_max=0;

	/* init: 0 means no label set / not checked */
	for( i=0; i<width*height; i++ ) {
		labels[i]=0;
	}	
	for( ix=0; ix<width-1; ix++ ) {
		for( iy=0; iy<height-1; iy++ ) {
			pixelpos=ix+iy*width;
			/* no motion - no label */
			if( out[pixelpos] == 0 ) {
				labels[pixelpos]=1;
				continue;
			/* already visited by iflood */
			if (labels[pixelpos] > 0)
				continue;
			}
			labelsize=iflood(ix, iy, width, height, out, labels, current_label);
			if( labelsize > 0 ) {
				//printf( "Label: %i (%i) Size: %i (%i,%i)\n", current_label, imgs->total_labels, labelsize, ix, iy );//jw
				if( imgs->labelsize_max < labelsize ){
					imgs->labelsize_max=labelsize;
					imgs->largest_label=current_label;
				}
				imgs->total_labels++;
				current_label++;
			}
		}
	}	
	//printf( "%i Labels found. Largest connected Area: %i Pixel(s). Largest Label: %i\n", imgs->total_labels, imgs->labelsize_max, imgs->largest_label);
	return imgs->labelsize_max;
}

/* Dilates a 3x3 box */
static int dilate9(char *img, int width, int height, void *buffer)
{
	int y, i, sum = 0;
	char *Row1,*Row2,*Row3;
	Row1 = buffer;
	Row2 = Row1 + width;
	Row3 = Row1 + 2*width;
	memset(Row2, 0, width);
	memcpy(Row3, img, width);
	for (y = 0; y < height; y++) {
		memcpy(Row1, Row2, width);
		memcpy(Row2, Row3, width);
		if (y == height-1)
			memset(Row3, 0, width);
		else
			memcpy(Row3, img+(y+1)*width, width);

		for (i = width-2; i >= 1; i--) {
			int blob;
			blob = MAX(Row1[i-1], Row1[i]);
			blob = MAX(blob, Row1[i+1]);
			blob = MAX(blob, Row2[i-1]);
			blob = MAX(blob, Row2[i]);
			blob = MAX(blob, Row2[i+1]);
			blob = MAX(blob, Row3[i-1]);
			blob = MAX(blob, Row3[i]);
			blob = MAX(blob, Row3[i+1]);
			if (blob != 0) {
				img[y*width+i] = blob;
				sum++;
			}
		}
		img[y*width] = img[y*width+width-1] = 0;
	}
	return sum;
}


/* Dilates a + shape */
static int dilate5(char *img, int width, int height, void *buffer)
{
	int y, i, sum = 0;
	char *Row1,*Row2,*Row3;
	Row1 = buffer;
	Row2 = Row1 + width;
	Row3 = Row1 + 2*width;
	memset(Row2, 0, width);
	memcpy(Row3, img, width);
	for (y = 0; y < height; y++) {
		memcpy(Row1, Row2, width);
		memcpy(Row2, Row3, width);
		if (y == height-1)
			memset(Row3, 0, width);
		else
			memcpy(Row3, img+(y+1)*width, width);

		for (i = width-2; i >= 1; i--) {
			int blob;
			blob = MAX(Row1[i], Row2[i-1]);
			blob = MAX(blob, Row2[i]);
			blob = MAX(blob, Row2[i+1]);
			blob = MAX(blob, Row3[i]);
			if (blob != 0) {
				img[y*width+i] = blob;
				sum++;
			}
		}
		img[y*width] = img[y*width+width-1] = 0;
	}
	return sum;
}


/* Erodes a 3x3 box */
static int erode9(char *img, int width, int height, void *buffer)
{
	int y, i, sum = 0;
	char *Row1,*Row2,*Row3;
	Row1 = buffer;
	Row2 = Row1 + width;
	Row3 = Row1 + 2*width;
	memset(Row2, 0, width);
	memcpy(Row3, img, width);
	for (y = 0; y < height; y++) {
		memcpy(Row1, Row2, width);
		memcpy(Row2, Row3, width);
		if (y == height-1)
			memset(Row3, 0, width);
		else
			memcpy(Row3, img+(y+1)*width, width);

		for (i = width-2; i >= 1; i--) {
			if (Row1[i-1] == 0 ||
				Row1[i]   == 0 ||
				Row1[i+1] == 0 ||
				Row2[i-1] == 0 ||
				Row2[i]   == 0 ||
				Row2[i+1] == 0 ||
				Row3[i-1] == 0 ||
				Row3[i]   == 0 ||
				Row3[i+1] == 0)
				img[y*width+i] = 0;
			else
				sum++;
		}
		img[y*width] = img[y*width+width-1] = 0;
	}
	return sum;
}

/* Erodes in a + shape */
static int erode5(char *img, int width, int height, void *buffer)
{
	int y, i, sum = 0;
	char *Row1,*Row2,*Row3;
	Row1 = buffer;
	Row2 = Row1 + width;
	Row3 = Row1 + 2*width;
	memset(Row2, 0, width);
	memcpy(Row3, img, width);
	for (y = 0; y < height; y++) {
		memcpy(Row1, Row2, width);
		memcpy(Row2, Row3, width);
		if (y == height-1)
			memset(Row3, 0, width);
		else
			memcpy(Row3, img+(y+1)*width, width);

		for (i = width-2; i >= 1; i--) {
			if (Row1[i]   == 0 ||
				Row2[i-1] == 0 ||
				Row2[i]   == 0 ||
				Row2[i+1] == 0 ||
				Row3[i]   == 0)
				img[y*width+i] = 0;
			else
				sum++;
		}
		img[y*width] = img[y*width+width-1] = 0;
	}
	return sum;
}


/* 
 * Despeckling routine to remove noisy detections.
 */
int alg_despeckle(struct context *cnt)
{
	int diffs = 0;
	char *out = cnt->imgs.out;
	int width = cnt->imgs.width;
	int height= cnt->imgs.height;
	int i, len = strlen(cnt->conf.despeckle);
	void *buffer = malloc(3*width);

	if (!buffer) return 0;

	for (i = 0; i < len; i++) {
		switch (cnt->conf.despeckle[i]) {
		case 'E':
			if ((diffs = erode9(out, width, height, buffer)) == 0) i=len;
			break;
		case 'e':
			if ((diffs = erode5(out, width, height, buffer)) == 0) i=len;
			break;
		case 'D':
			diffs = dilate9(out, width, height, buffer);
			break;
		case 'd':
			diffs = dilate5(out, width, height, buffer);
			break;
		/* no further despeckle after labeling! */
		case 'l':
			diffs = alg_labeling(cnt);
			i=len;
			break;
		}
	}
	free(buffer);
	return diffs;
}


/*	The heart of the motion detector
 *	Basicly: ref-new and count the pixels that are over a noise threshold
 */
int alg_diff_standard (struct context *cnt, char *new)
{
	struct images *imgs=&cnt->imgs;
	int i, diffs=0;
	long int level=0;
	int noise=cnt->noise;
	char *ref=imgs->ref;
	char *out=imgs->out;
	unsigned char *mask=imgs->mask;


	/* If the average level of the picture is to low, compensate by 
	 * lowering the noise threshold
	 */
	if (cnt->conf.nightcomp) {
		i=imgs->motionsize;
		for (i--; i>=0; i--) {
			level+=(unsigned char)new[i];
		}
		level/=imgs->motionsize;
		if (level < noise*2)
			noise/=2;
	}

	memset(out, 0, imgs->size);
	i=imgs->motionsize;
	for (; i>0; i--) {
		register char curdiff=*ref-*new; /* using a temp variable is 12% faster */

		if (mask)
			curdiff=((int)((char)curdiff**mask++)/255);

		if (curdiff > noise || curdiff < -noise) {
			*out=*new;
			diffs++;
		}

		out++;
		ref++;
		new++;
	}

	return diffs;
}

/*
	Very fast diff function, does not do nightcompensation or mask
	overlaying.
*/
char alg_diff_fast (struct context *cnt, int max_n_changes, char *new)
{
	struct images *imgs=&cnt->imgs;
	int i, diffs=0, step=imgs->motionsize/10000;
	int noise=cnt->noise;
	char *ref=imgs->ref;

	if (!step%2)
		step++;
	/* we're checking only 1 of several pixels */
	max_n_changes /= step;

	i=imgs->motionsize;
	for (; i>0; i-=step) {
		char curdiff=*ref-*new; /* using a temp variable is 12% faster */
		if (curdiff >  noise || curdiff < -noise ) {
			diffs++;
			if (diffs > max_n_changes)
				return 1;
		}
		ref+=step;
		new+=step;
	}

	return 0;
}

/* alg_diff uses diff_fast to quickly decide if there is anything worth
 * sending to diff_standard.
*/
int alg_diff (struct context *cnt, char *new)
{
	int diffs=0;
	
	if (alg_diff_fast(cnt, cnt->conf.max_changes/2, new))
		diffs=alg_diff_standard(cnt, new);

	return diffs;
}

/* Detect a sudden massive change in the picture.
   It is assumed to be the light being switched on or a camera displacement.
   In any way the user doesn't think it is worth capturing.
 */
int alg_lightswitch (struct context *cnt, int diffs)
{
	struct images *imgs=&cnt->imgs;
	
	if (cnt->conf.lightswitch < 0)
		cnt->conf.lightswitch = 0;
	if (cnt->conf.lightswitch > 100)
		cnt->conf.lightswitch = 100;
	
	/* is lightswitch percent of the image changed?  */
	if (diffs > (imgs->motionsize * cnt->conf.lightswitch / 100))
		return 1;
	
	return 0;
}

int alg_switchfilter (struct context *cnt, int diffs, struct coord *cent, char *newimg)
{
	int linediff=diffs/cnt->conf.height;
	char *out=cnt->imgs.out;
	int y, x, line;
	int lines=0, vertlines=0;

	for (y=0; y<cnt->conf.height; y++) {
		line=0;
		for (x=0; x<cnt->conf.width; x++) {
			if (*(out++)) {
				line++;
			}
		}
		if (line>cnt->conf.width/18) {
			vertlines++;
		}
		if (line>linediff*2) {
			lines++;
		}
	}
//	printf("%d %d %d %d  %d\n", lines, vertlines, linediff, lines-vertlines, diffs);
	if (vertlines>cnt->conf.height/10 && lines<vertlines/3 &&
	    (vertlines>cnt->conf.height/4 || lines-vertlines>lines/2)) {
		if (cnt->conf.text_changes) {
			char tmp[80];
			sprintf(tmp, "%d %d", lines, vertlines);
			draw_text(newimg, cnt->conf.width-10, 20, cnt->conf.height, cnt->conf.width, tmp);
			return diffs;
		}
	}
	return 0;
}
