
/*
 *	netcam.c
 *
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU Public License Version 2
 *	See also the file 'COPYING'.
 *
 */
 
#ifdef HAVE_CURL

#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include <curl/curl.h>
#include <jpeglib.h>
#include <setjmp.h>

/* for rotation */
#include "rotate.h"

struct my_error_mgr {
	struct jpeg_error_mgr pub;
	jmp_buf setjmp_buffer;
};

void my_error_exit (j_common_ptr cinfo)
{
	struct my_error_mgr *myerr=(struct my_error_mgr *) cinfo->err;

	(*cinfo->err->output_message) (cinfo);

	longjmp (myerr->setjmp_buffer, 1);
}

int netcam_start (struct context *cnt)
{
	CURL *curl;
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
	FILE *jpegfile;
	int capture=1;

	if ((jpegfile=tmpfile())==NULL)
		return -1;

	while (capture) {
		curl=curl_easy_init();
		if (curl) {
			curl_easy_setopt(curl, CURLOPT_FILE, jpegfile);
			curl_easy_setopt(curl, CURLOPT_URL, cnt->conf.netcam_url);
			if (cnt->conf.netcam_userpass) {
				curl_easy_setopt(curl, CURLOPT_USERPWD, cnt->conf.netcam_userpass);
			}	
			if (curl_easy_perform(curl)) {
				syslog(LOG_ERR, "Error getting netcam image");
				rewind(jpegfile);
				curl_easy_cleanup(curl);
				continue;
			}
			capture=0;		

			rewind(jpegfile);	

			cinfo.err=jpeg_std_error(&jerr);
			jpeg_create_decompress(&cinfo);
			jpeg_stdio_src(&cinfo, jpegfile);
			jpeg_read_header(&cinfo, TRUE);
		
			if((cnt->conf.rotate_deg == 90) || (cnt->conf.rotate_deg == 270)) {
				/* Swap width and height if rotation is 90 or 270 degrees. */
				cnt->conf.width=cinfo.image_height;
				cnt->conf.height=cinfo.image_width;
			} else {
				cnt->conf.width=cinfo.image_width;
				cnt->conf.height=cinfo.image_height;
			}
			cnt->imgs.width=cinfo.image_width;
			cnt->imgs.height=cinfo.image_height;
			cnt->imgs.size=cnt->imgs.width*cnt->imgs.height*3/2;
			cnt->imgs.motionsize=cnt->imgs.width*cnt->imgs.height;
			cnt->imgs.type=VIDEO_PALETTE_YUV420P;

			curl_easy_cleanup(curl);
		}
	}
	fclose(jpegfile);
	return 0;
}

unsigned char *netcam_next (struct context *cnt, char *image)
{
	CURL *curl;
	struct jpeg_decompress_struct cinfo;
	struct my_error_mgr jerr;
	FILE *jpegfile;
	unsigned char *pic, *upic, *vpic;
	JSAMPARRAY line;
	int i, line_size, y;
	JSAMPROW row[1];
	int capture=1;
	int width, height; /* for rotation */

	if ((jpegfile=tmpfile())==NULL)
		return NULL;

	while (capture) {
		curl=curl_easy_init();
		if (curl) {
			curl_easy_setopt(curl, CURLOPT_FILE, jpegfile);
			curl_easy_setopt(curl, CURLOPT_URL, cnt->conf.netcam_url);
			if (cnt->conf.netcam_userpass) {
				curl_easy_setopt(curl, CURLOPT_USERPWD, cnt->conf.netcam_userpass);
			}
			if (curl_easy_perform(curl))
				syslog(LOG_ERR, "Error getting netcam image");
			else
				capture=0;
			curl_easy_cleanup(curl);
		}
		rewind(jpegfile);
	}

	cinfo.err=jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	if (setjmp(jerr.setjmp_buffer)) {
		jpeg_destroy_decompress(&cinfo);
		fclose(jpegfile);
		return NULL;
	}
	
	jpeg_create_decompress(&cinfo);
	jpeg_stdio_src(&cinfo, jpegfile);
	jpeg_read_header(&cinfo, TRUE);
	
	cinfo.out_color_space=JCS_YCbCr;
	
	/* Width and height get the correct values, because the ones in the conf
	 * struct (may) have been swapped earlier.
	 */
	if((cnt->conf.rotate_deg == 90) || (cnt->conf.rotate_deg == 270)) {
		width = cnt->conf.height;
		height = cnt->conf.width;
	} else {
		width = cnt->conf.width;
		height = cnt->conf.height;
	}
	
	jpeg_start_decompress(&cinfo);
	line_size=width*3;
	
	line=(*cinfo.mem->alloc_sarray) ((j_common_ptr) &cinfo, JPOOL_IMAGE, cinfo.output_width*cinfo.output_components, 1);
	pic=image;
	upic=pic+width*height;
	vpic=upic+(width*height)/4;
	row[0]=(unsigned char *)line;
	y=0;
	while(cinfo.output_scanline<height) {
		jpeg_read_scanlines(&cinfo, row, 1);
		for (i=0; i<line_size; i+=3) {
			pic[i/3]=((unsigned char *)line)[i];
			if (i & 1) {
				upic[(i/3)/2]=((unsigned char *)line)[i+1];
				vpic[(i/3)/2]=((unsigned char *)line)[i+2];
			}
		}
		pic+=line_size/3;
		if (y++ & 1) {
			upic+=width/2;
			vpic+=width/2;
		}
	}
	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	fclose(jpegfile);

	/* Finally rotate the picture if specified. The size passed is the
	 * size of a YUV 4:2:0 image.
	 */
	if(cnt->conf.rotate_deg)
		rotate_map(image, (width * height * 3) / 2, width, height, cnt->conf.rotate_deg);

	return pic;
}
#endif /* HAVE_CURL */
