// Copyright 2012-2013 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::error::Error;
use std::fmt;
use std::result;
use std::str;

use Version;
use version::Identifier;
use semver_parser;

#[cfg(feature = "serde")]
use serde::ser::{Serialize, Serializer};
#[cfg(feature = "serde")]
use serde::de::{self, Deserialize, Deserializer, Visitor};

use self::Op::{Ex, Gt, GtEq, Lt, LtEq, Tilde, Compatible, Wildcard};
use self::WildcardVersion::{Major, Minor, Patch};
use self::ReqParseError::*;

/// A `VersionReq` is a struct containing a list of predicates that can apply to ranges of version
/// numbers. Matching operations can then be done with the `VersionReq` against a particular
/// version to see if it satisfies some or all of the constraints.
#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
pub struct VersionReq {
    predicates: Vec<Predicate>,
}

impl From<semver_parser::range::VersionReq> for VersionReq {
    fn from(other: semver_parser::range::VersionReq) -> VersionReq {
        VersionReq { predicates: other.predicates.into_iter().map(From::from).collect() }
    }
}

#[cfg(feature = "serde")]
impl Serialize for VersionReq {
    fn serialize<S>(&self, serializer: S) -> result::Result<S::Ok, S::Error>
        where S: Serializer
    {
        // Serialize VersionReq as a string.
        serializer.collect_str(self)
    }
}

#[cfg(feature = "serde")]
impl<'de> Deserialize<'de> for VersionReq {
    fn deserialize<D>(deserializer: D) -> result::Result<Self, D::Error>
        where D: Deserializer<'de>
    {
        struct VersionReqVisitor;

        /// Deserialize `VersionReq` from a string.
        impl<'de> Visitor<'de> for VersionReqVisitor {
            type Value = VersionReq;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("a SemVer version requirement as a string")
            }

            fn visit_str<E>(self, v: &str) -> result::Result<Self::Value, E>
                where E: de::Error
            {
                VersionReq::parse(v).map_err(de::Error::custom)
            }
        }

        deserializer.deserialize_str(VersionReqVisitor)
    }
}

#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
enum WildcardVersion {
    Major,
    Minor,
    Patch,
}

#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
enum Op {
    Ex, // Exact
    Gt, // Greater than
    GtEq, // Greater than or equal to
    Lt, // Less than
    LtEq, // Less than or equal to
    Tilde, // e.g. ~1.0.0
    Compatible, // compatible by definition of semver, indicated by ^
    Wildcard(WildcardVersion), // x.y.*, x.*, *
}

impl From<semver_parser::range::Op> for Op {
    fn from(other: semver_parser::range::Op) -> Op {
        use semver_parser::range;
        match other {
            range::Op::Ex => Op::Ex,
            range::Op::Gt => Op::Gt,
            range::Op::GtEq => Op::GtEq,
            range::Op::Lt => Op::Lt,
            range::Op::LtEq => Op::LtEq,
            range::Op::Tilde => Op::Tilde,
            range::Op::Compatible => Op::Compatible,
            range::Op::Wildcard(version) => {
                match version {
                    range::WildcardVersion::Major => Op::Wildcard(WildcardVersion::Major),
                    range::WildcardVersion::Minor => Op::Wildcard(WildcardVersion::Minor),
                    range::WildcardVersion::Patch => Op::Wildcard(WildcardVersion::Patch),
                }
            }
        }
    }
}

#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
struct Predicate {
    op: Op,
    major: u64,
    minor: Option<u64>,
    patch: Option<u64>,
    pre: Vec<Identifier>,
}

i