/**
   The command-line version of MRIConvert.
**/

#include <string>
#include <iostream>
#include <vector>

#include <wx/app.h>
#include <wx/cmdline.h>
#include <wx/string.h>
#include <wx/dir.h>
#include <wx/config.h>
#include <wx/stdpaths.h>

#include "McVerter.h"
#include "OutputFactory.h"
#include "Converter.h"
#include "OutputterBase.h"
#include "StringConvert.h"
#include "version_string.h"

using namespace std;
using namespace jcs;

int main(int argc, char** argv)
{
  // First make sure we can use wx stuff.
  wxInitializer initializer;
  if ( !initializer )
  {
    cerr << "Failed to initialize the wxWidgets library, aborting.";
    return -1;
  }

  // Set up to parse command line arguments.

  // Get the list of permitted output formats.
  wxString allowed_formats_string(_T("Output format: "));
  int type = 0;
  while(type < (OutputFactory::GetNumberOfTypes() - 1) ) {
    allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
    allowed_formats_string.append(_T(", "));
    ++type;
  }
  allowed_formats_string.append(_T("or "));
  allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
  
  // Define what parameters the parser will process.
  static const wxCmdLineEntryDesc cmdLineDesc[] = {
    
    { wxCMD_LINE_OPTION, _T("o"), _T("output"),
      _T("Output directory for converted files"),
      wxCMD_LINE_VAL_STRING, wxCMD_LINE_OPTION_MANDATORY  },
    { wxCMD_LINE_OPTION, _T("f"), _T("format"), 
      allowed_formats_string,
      wxCMD_LINE_VAL_STRING, wxCMD_LINE_OPTION_MANDATORY },
    { wxCMD_LINE_SWITCH, _T("x"), _T(""), _T("Save each series to a separate directory")},
    { wxCMD_LINE_SWITCH, _T("d"), _T(""), _T("Save output volumes as 4D files")}, 
    { wxCMD_LINE_SWITCH, _T("a"), _T(""), _T("Save header only (metaimage only)")},
    { wxCMD_LINE_SWITCH, _T("n"), _T("nii"), _T("Save files as .nii (nifti/fsl only)")},
    { wxCMD_LINE_SWITCH, _T("b"), _T("v16"), _T("Save .v16 files (BrainVoyager only)")},
    { wxCMD_LINE_SWITCH, _T("u"), _T(""), _T("Use patient id instead of patient name for output file")},
    { wxCMD_LINE_OPTION, _T("s"), _T("skip"), _T("Skip first n volumes"), wxCMD_LINE_VAL_NUMBER},
    { wxCMD_LINE_OPTION, _T("m"), _T("match"), 
      _T("Only convert files whose series descriptions include this string")},
    { wxCMD_LINE_OPTION, _T("F"), _T("fnformat"), 
      _T("Use this format for name of output file: +/- PatientName, \
PatientId, \
SeriesDate, \
SeriesTime, \
StudyId, \
StudyDescription, \
SeriesNumber, \
SequenceName, \
ProtocolName, \
SeriesDescription")},
    { wxCMD_LINE_SWITCH, _T("r"), _T("rescale"), _T("Apply rescale slope and intercept to data")},
    { wxCMD_LINE_SWITCH, _T("v"), _T("verbose"), _T("Verbose mode")},
    { wxCMD_LINE_SWITCH, _T("i"), _T("version"), _T("Version info")},
    { wxCMD_LINE_SWITCH, _T("h"), _T("help"), _T("Display help"),
      wxCMD_LINE_VAL_NONE, wxCMD_LINE_OPTION_HELP},
    { wxCMD_LINE_PARAM, NULL, NULL, _T("Input directory or file(s)"),
      wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_MULTIPLE  },
    { wxCMD_LINE_NONE }
  };
  
  wxCmdLineParser parser(argc, argv);
  
  parser.SetDesc(cmdLineDesc);
  
  // Now parse the parameters given.
  if (parser.Parse() > 0) return 0;

  // Bail-out cases.
  if (parser.Found(_T("i"))) {
    cout << "mcverter " << VERSION_STRING << endl;
    return 0;
  }
  if (parser.Found(_T("help"))) return 0;
  
  // Selector for output format.
  wxString format;
  parser.Found(_T("format"), &format);
  int format_type = 0;
  int found = wxNOT_FOUND;
  while (format_type < OutputFactory::GetNumberOfTypes()) {
    found = format.Find(wxString(OutputFactory::GetShortDescription(format_type), wxConvLocal));
    if (found != wxNOT_FOUND) break;
    ++format_type;
  }
  if (found == wxNOT_FOUND) {
    cerr << "Invalid format type: " << format << endl;
    cerr << allowed_formats_string << endl;
    return 0;
  }
  
  // Selector for output directory.
  wxString outputDir;
  parser.Found(_T("o"), &outputDir);
  if (!wxDir::Exists(outputDir)) {
    if (!wxFileName::Mkdir(outputDir, 0777, wxPATH_MKDIR_FULL)) {
      cout << "Unable to create directory: " << outputDir << endl;
      return 0;
    }
  }
  
  // We know enough to create our converter and outputter.
  Converter* converter = new Converter(format_type);
  OutputterBase* outputter = converter->GetOutputter();
  outputter->mOutputList.rootDir = outputDir;
  outputter->SetOption("split", parser.Found(_T("x")));
  outputter->SetOption("dim", parser.Found(_T("d")) ? 4 : 3);
  outputter->SetOption("ho", parser.Found(_T("a")));
  outputter->SetOption("nii", parser.Found(_T("n")));
  outputter->SetOption("v16", parser.Found(_T("b")));
  outputter->SetOption("rescale", parser.Found(_T("r")));
  
  // Handle request for using PatientId rather than PatientName.
  if (parser.Found(_T("u"))) {
    outputter->defaultNameFields[OutputterBase::PatientName] 
      = OutputterBase::NameField("PatientName", false);
    outputter->defaultNameFields[OutputterBase::PatientId]
      = OutputterBase::NameField("PatientId", true);
  } else {
    outputter->defaultNameFields[OutputterBase::PatientName] 
      = OutputterBase::NameField("PatientName", true);
    outputter->defaultNameFields[OutputterBase::PatientId]
      = OutputterBase::NameField("PatientId", false);
  }
  
  // Handle request for alternate output filename format.
  wxString fnformat;
  parser.Found(_T("F"), &fnformat);
  std::string stdfnformat = std::string (fnformat.mb_str());
  OutputterBase::FieldMap::iterator it = outputter->defaultNameFields.begin();
  OutputterBase::FieldMap::iterator it_end = outputter->defaultNameFields.end();
  while (it != it_end) {
    if (stdfnformat.find("+" + it->second.name) != -1) {
      outputter->defaultNameFields[it->second.idx].value = true;
    }
    if (stdfnformat.find("-" + it->second.name) != -1) {
      outputter->defaultNameFields[it->second.idx].value = false;
    }
    it++;
  }
  
  // Set number of volumes to skip.
  long skip = 0;
  parser.Found(_T("s"), &skip);
  outputter->SetOption("skip", static_cast<int> (skip));
  
  // Set chattiness.
  verbose = parser.Found(_T("v"));
  
  // Does user want only certain series?
  wxString match;
  parser.Found(_T("m"), &match);
  
  // We've processed all options and switches,
  // now add the requested files.
  if (verbose) cout << "Adding files..." << endl;
  for (unsigned int i = 0; i < parser.GetParamCount(); ++i) {
    wxFileName filename(parser.GetParam(i));
    if (filename.FileExists()) {
      converter->AddFile(filename.GetFullPath(), match);
    }
    else if (filename.DirExists()) {
      wxArrayString files;
      wxDir::GetAllFiles(filename.GetFullPath(), &files);
      unsigned int n_files = files.GetCount();
      for (unsigned int i = 0; i < files.GetCount(); ++i) {
        converter->AddFile(files.Item(i), match);
      }
    }
    else {
      cout << "File or directory " << parser.GetParam(i).mb_str() << " not found. Add trailing directory separator and try again." << endl;
    }
  }

  if (verbose) {
    MessageList::iterator ml_it = converter->messages.begin();
    while (ml_it != converter->messages.end()) {
      Message::iterator it = ml_it->begin();
      while (it != ml_it->end()) {
        cout << it->mb_str(wxConvLocal) << " ";
        ++it;
      }
      cout << endl;
      ++ml_it;
    }
  }
  converter->messages.clear();
  converter->UpdateAllOutput();

  if (verbose) cout << "Converting...." << endl;

  // Let the real work begin.
  converter->ConvertAll();

  if (verbose) cout << "Finished" << endl;

  delete converter;

  return 0;
}
