#ifndef BASIC_3D_OUTPUTTER_TXX_
#define BASIC_3D_OUTPUTTER_TXX_

#include <string>
#include <map>

#include <wx/log.h>

#include "Volume.h"
#include "OutputterBase.h"
#include "BasicVolumeFormat.h"

namespace jcs {

    ///
    /**
     */
    template <class T>
    Basic3DConversion<T>::Basic3DConversion(Basic3DOutputter* outputter, SeriesHandler* handler)
	: mOutputter(outputter), mHandler(handler)
    {
    }


    ///
    /**
     */
    template <class T> int
    Basic3DConversion<T>::GetSkip()
    {
	std::string series_uid = mHandler->GetSeriesUid();
	int nVolumes = mHandler->GetNumberOfVolumes();
	int skip = mOutputter->GetSkip(series_uid);
	if (nVolumes > skip)
	    return skip;
	else
	    return 0;
    }


    ///
    /**
     */
    template <class T> int
    Basic3DConversion<T>::GetNumberOfVolumes()
    {
	int nVolumes = mHandler->GetNumberOfVolumes();
	return nVolumes - GetSkip();
    }


    /// The main reason for being.
    /** This is the heart of MRIConvert.
     */
    template <class T> void
    Basic3DConversion<T>::Convert()
    {
	//typedef std::map <VolId, Volume<T> > vMapType;
	vMapType volumes;
	mHandler->rescale = mOutputter->rescale;
	mHandler->GetVolumes(volumes);

	GetHeaderForSeries();

	std::string series_uid = mHandler->GetSeriesUid();
	int dimensionality = mOutputter->GetDimensionality(series_uid);

	if (dimensionality == 4) {
		
	    wxFileName file = mOutputter->GetFileName(series_uid);
	    if (file.GetName() == _T("error")) {
		wxLogError(_T("File name error"));
		return;
	    }
	    file.SetExt(_T(""));
	    BasicVolumeFormat* outputVolume = mOutputter->GetOutputVolume(file.GetFullPath().mb_str(wxConvLocal));

	    typename vMapType::iterator it = volumes.begin();
	    typename vMapType::iterator it_end = volumes.end();
	    CompleteHeaderForVolume(*it);

	    outputVolume->WriteHeader(GetHeader());

	    for (int i = 0; i < GetSkip(); ++i) ++it;

	    for (;it != it_end; it++) {
		ProcessSlicesAndAppend(outputVolume, it);
		wxTheApp->Yield();
	    }
	    delete outputVolume;

	} else {

	    typename vMapType::iterator it = volumes.begin();
	    typename vMapType::iterator it_end = volumes.end();

	    for (int i = 0; i < GetSkip(); ++i) ++it;

	    for (;it != it_end; it++) {

		CompleteHeaderForVolume(*it);

		wxFileName file = mOutputter->GetFileNameFromVolId(it->first);
		if (file.GetName() == _T("error")) {
		    wxLogError(_T("File name error"));
		    break;
		}

		file.SetExt(_T(""));
		BasicVolumeFormat* outputVolume = mOutputter->GetOutputVolume(file.GetFullPath().mb_str(wxConvLocal));

		outputVolume->WriteHeader(GetHeader());

		ProcessSlicesAndAppend(outputVolume, it);

		delete outputVolume;
		wxTheApp->Yield();
	    }

	}

	if (this->mHandler->IsMoCo()) WriteMoCoFiles();
	WriteStringInfo();
    }


    ///
    /**
       \param outputVolume Pointer to BasicVolumeFormat structure containing volumes to write.
       \param it Iterator through vMapType structure.
     */
    template <class T> void
    Basic3DConversion<T>::ProcessSlicesAndAppend(BasicVolumeFormat* outputVolume, typename vMapType::iterator it)
    {
	typedef std::map<float, std::vector<T> > SliceMap;
	typename SliceMap::iterator slice_it = it->second.begin();
	typename SliceMap::iterator slice_end = it->second.end();
	for (; slice_it != slice_end; slice_it++) {
	    ProcessSlice(slice_it->second);
	    outputVolume->AppendRawData(reinterpret_cast<char*> (&slice_it->second.front()),
					slice_it->second.size() * sizeof(slice_it->second.front()));
	}
    }


    ///
    /**
     */
    template <class T> void
    Basic3DConversion<T>::WriteMoCoFiles()
    {
	std::vector<std::string> moco;
	this->mHandler->GetImageComments(moco);

	if (moco.size() != 0) {
	    std::string key = this->mHandler->GetSeriesUid() + this->mOutputter->moco_postfix;
	    wxFileName fileName = this->mOutputter->mOutputList.GetFullFileName(key);
	    IterativeWrite(fileName, moco);
	}
    }


    template <class T> void
    Basic3DConversion<T>::WriteStringInfo()
    {
	std::string key = this->mHandler->GetSeriesUid() + "_info";
	wxFileName fileName = this->mOutputter->mOutputList.GetFullFileName(key);
	IterativeWrite(fileName, this->mHandler->GetStringInfo());
    }


    ///
    /** Writes each string in the given vector to file.
	\param fileName Name of file to write.
	\param src Vector of strings to write.
    */
    template <class T> void
    Basic3DConversion<T>::IterativeWrite(wxFileName fileName, std::vector<std::string> src)
    {
	if (fileName.GetName() != _T("error")) {
	    wxFileName::Mkdir(fileName.GetPath(wxPATH_GET_VOLUME), 0777, wxPATH_MKDIR_FULL);
    
	    std::ofstream output;
	    output.open(fileName.GetFullPath().mb_str(wxConvLocal));
	    std::vector<std::string>::iterator it = src.begin();
	    std::vector<std::string>::iterator it_end = src.end();
	    for (; it < it_end; it++)
		output << *it << std::endl;  
	    output.close();
	}
	else {
	    wxLogError(_T("File name error"));
	}
    }

} // end namespace jcs

#endif
