#ifndef SERIES_HANDLER_H
#define SERIES_HANDLER_H

#include <string>
#include <vector>
#include <set>
#include <map>

#include "Volume.h"
#include "DicomFile.h"

class wxFileName;

namespace jcs {

enum AoCode {
	ASCENDING,
	DESCENDING,
	INTERLEAVED_ODD,
	INTERLEAVED_EVEN,
	UNKNOWN,
	OTHER
};


// Things constant for a 3D volume
//typedef std::vector<std::string> VolId;
struct VolId {
	typedef std::vector<std::string> IdType;
	IdType ids;
	
	bool operator<  (const VolId &rhs) const
	{
		return (ids < rhs.ids);
	}

	bool operator==  (const VolId &rhs) const
	{
		return (
			(ids == rhs.ids) &&
			(orientNumber == rhs.orientNumber));
	}

	int orientNumber;


};


// Stuff used for automatic generation of filenames & directories
// save to avoid unneccessary file reads

struct SeriesInfo {
	std::string subject_name;
	std::string subject_id;
//	std::string study_uid;
	std::string study_number;
	std::string series_number;
	std::string StudyDate;
	std::string SeriesDate;
	std::string study_description;
	std::string series_description;
};

struct GradientInfo {
	std::vector<double> values;
	std::vector<double> xGrads;
	std::vector<double> yGrads;
	std::vector<double> zGrads;
};

void Normalize(GradientInfo& info);
void RotateGradInfo(GradientInfo& info, std::vector<double>& r);

class SeriesHandler
{

protected:

	typedef std::string string;

public :

	SeriesHandler(const std::string& seriesUid);
	virtual ~SeriesHandler();

	const std::string& GetSeriesUid() const { return mSeriesUid; }

	template <class T> int Find(DicomElement e, T& value) const
	{
		if (mFileMap.empty()) return 0;
		DicomFile dicomFile((*mFileMap.begin()).first.c_str());
		return dicomFile.Find(e, value);
	}

	template <class T> int Find(const std::string& s, T& value) const
	{
		if (mFileMap.empty()) return 0;
		DicomFile dicomFile((*mFileMap.begin()).first.c_str());
		return dicomFile.Find(s, value);
	}

	template <class T> int Find(const std::string& s, T& value,
		const VolId& id) const
	{
		std::string filename;
		if (!GetFirstFilename(id, filename)) return 0;
		DicomFile dicomFile(filename.c_str());
		return dicomFile.Find(s, value);
	}


	virtual std::vector<std::string> GetStringInfo();

	virtual void GetVolumes (std::map <VolId, Volume <wxInt16> >& v)
	{ mGetVolumes<wxInt16> (v); } 
	
	virtual void GetVolumes (std::map <VolId, Volume <wxUint16> >& v)
	{ mGetVolumes<wxUint16> (v); } 
	
	virtual void GetVolumes (std::map <VolId, Volume <wxInt8> >& v)
	{ mGetVolumes<wxInt8> (v); }

	virtual void GetVolumes (std::map <VolId, Volume <wxUint8> >& v)
	{ mGetVolumes<wxUint8> (v); } 

	int AddFile(const char* filename);
	
	int GetNumberOfFiles() const { return mFileMap.size(); }
	virtual int GetNumberOfSlices() const;
	virtual int GetNumberOfVolumes() const;
	virtual std::vector<double> GetVoxelSize();
	virtual int GetRows();
	virtual int GetColumns();
	virtual int GetFrames() const;

	virtual double GetSliceDuration() const;
	virtual double GetVolumeInterval() const;
	virtual AoCode GetAcquisitionOrder() const { return UNKNOWN; }

//	virtual bool IsFmri() const { return false; }
	virtual bool IsMosaic() const { return false; }
	bool IsBigEndian() const;


	std::set<VolId> GetVolIds() const;
	std::map<VolId, std::string> GetVolIdsAndFiles() const;
	SeriesInfo seriesInfo;

	std::vector<double> GetIppForFirstSlice(const VolId& id);

	std::vector<double> GetRotationMatrix(const VolId& id);

	virtual GradientInfo GetGradientInfo() { return GradientInfo(); };
	virtual bool IsDti() const {return false; }

	virtual bool IsMoCo() const {return false; }
	void GetImageComments(std::vector<string>& comments) const;

	std::vector<std::string> GetFilenames() const;
	// Only used by dicomtree class, which only needs a list of files -- change!
//	FileMapType::iterator begin() { return mFileMap.begin(); }
//	FileMapType::iterator end()	{ return mFileMap.end(); }

	bool rescale;

protected :
	typedef std::vector<VolId> VolListType;

	virtual VolListType ReadVolIds(DicomFile& file);
	typedef std::map <std::string, VolListType >  FileMapType;
	FileMapType mFileMap;
//	std::vector<double> mIpp1;
//	std::vector<double> rotation;

	std::map<VolId, std::vector<double> > position; // ipp for slice 1, calculated in GetVolumes
	std::vector<std::vector<double> > orientations;

	virtual std::string GetImagePositionPatient(DicomFile& dfile, int frame);

private :
	SeriesHandler& operator=(const SeriesHandler& rhs);

	bool GetFirstFilename(const VolId& id, std::string& str) const;

	// moved over from DicomSeries
	const std::string mSeriesUid;
	std::set<std::string> mInstanceUids;

	// todo: see if some of this can be moved out of the template?
	template <class T> void
	mGetVolumes(std::map<VolId, Volume<T> >& v);


};

class SyngoHandler: public  SeriesHandler
{
public:
	SyngoHandler(const std::string& seriesUid);

	virtual double GetSliceDuration() const;
	virtual AoCode GetAcquisitionOrder() const;
	virtual double GetVolumeInterval() const;
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const;
	virtual bool IsMoCo() const;


protected:
	int	ReadCSAImageHeader(const std::string& tag, std::string& value) const;
	int	ReadCSASeriesHeader(const std::string& tag, std::string& value) const;

	virtual VolListType ReadVolIds(DicomFile& file);

};

class NumarisMosaicHandler : public SeriesHandler
{
public :
	NumarisMosaicHandler(const std::string& seriesUid);

	virtual int GetNumberOfSlices() const;
	virtual std::vector<double> GetVoxelSize();
	virtual int GetRows();
	virtual int GetColumns();
	virtual AoCode GetAcquisitionOrder() const;
	virtual double GetVolumeInterval() const;

	virtual std::vector<std::string> GetStringInfo();

	virtual void GetVolumes (std::map<VolId, Volume <wxInt16> >& v)
	{ mGetVolumes<wxInt16> (v); } 
	
	virtual void GetVolumes(std::map<VolId, Volume <wxUint16> >& v)
	{ mGetVolumes<wxUint16> (v); } 
	
	virtual void GetVolumes (std::map<VolId, Volume <wxInt8> >& v)
	{ mGetVolumes<wxInt8> (v); }

	virtual void GetVolumes (std::map<VolId, Volume <wxUint8> >& v)
	{ mGetVolumes<wxUint8> (v); } 

	virtual bool IsMosaic() const { return true; }

protected :
	virtual std::vector<double> GetSliceOrder();
	virtual VolListType ReadVolIds(DicomFile& file);

private:

	template <class T> void
	mGetVolumes(std::map<VolId, Volume<T> >& v);

};


class SyngoMosaicHandler : public SyngoHandler
{
public :
	SyngoMosaicHandler(const std::string& seriesUid);

	virtual int GetNumberOfSlices() const;
	virtual int GetRows();
	virtual int GetColumns();

//	virtual AoCode GetAcquisitionOrder() const;
//	virtual double GetSliceDuration() const;

	virtual std::vector<std::string> GetStringInfo();

	virtual void GetVolumes (std::map<VolId, Volume <wxInt16> >& v)
	{ mGetVolumes<wxInt16> (v); } 
	
	virtual void GetVolumes(std::map<VolId, Volume <wxUint16> >& v)
	{ mGetVolumes<wxUint16> (v); } 
	
	virtual void GetVolumes (std::map<VolId, Volume <wxInt8> >& v)
	{ mGetVolumes<wxInt8> (v); }

	virtual void GetVolumes (std::map<VolId, Volume <wxUint8> >& v)
	{ mGetVolumes<wxUint8> (v); } 

	virtual bool IsMosaic() const { return true; }

protected:
//	virtual std::vector<double> GetSliceOrder();
	virtual VolListType ReadVolIds(DicomFile& file);
//	virtual std::vector<double> GetIppForFirstSlice(const VolId& id);

private :
	double GetPhaseFov() const;
	double GetRoFov() const;
	template <class T> void
	mGetVolumes(std::map<VolId, Volume<T> >& v);


};

class GeEpiHandler: public  SeriesHandler
{
public:
	GeEpiHandler(const std::string& seriesUid);

protected:
	virtual VolListType ReadVolIds(DicomFile& file);

};


class GeDtiRbHandler: public  SeriesHandler
{
public:
	GeDtiRbHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const {return true; }


protected:
	virtual VolListType ReadVolIds(DicomFile& file);

};

class GeDti2Handler: public GeEpiHandler
{
public:
	GeDti2Handler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const {return true; }


protected:
	virtual VolListType ReadVolIds(DicomFile& file);

private:
	std::vector<std::vector<double> > gradients;
};

class EnhancedMrHandler: public SeriesHandler
{
public:
	EnhancedMrHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const;
	virtual std::vector<std::string> GetStringInfo();
	virtual std::vector<double> GetVoxelSize();

protected:
	virtual VolListType ReadVolIds(DicomFile& file);
	virtual std::string GetImagePositionPatient(DicomFile& dfile, int frame);

private:
	std::vector<std::string> ipps;
	std::map<VolId, std::string> bvecMap;
	std::map<VolId, std::string> bvalMap;
	std::map<VolId, int> volFrameMap;
	std::string pixelSpacing;

};

class AchievaDtiHandler: public  SeriesHandler
{
public:
	AchievaDtiHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const {return true; }


protected:
	virtual VolListType ReadVolIds(DicomFile& file);

private:
	std::vector<std::vector<double> > gradients;


};
}

#include "SeriesHandler.txx"

#endif
