/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include "CFormRawMap.h"

#include "xRawLogViewerMain.h"

#include <wx/msgdlg.h>
#include <wx/filedlg.h>
#include <wx/progdlg.h>
#include <wx/app.h>

//(*InternalHeaders(CFormRawMap)
#include <wx/string.h>
#include <wx/intl.h>
#include <wx/font.h>
#include <wx/bitmap.h>
#include <wx/image.h>
#include <wx/artprov.h>
//*)

#include <wx/numdlg.h>

// General global variables:
#include <mrpt/core.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::opengl;
using namespace mrpt::system;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::gui;
using namespace std;

// The map built from laser & odometry.
CMultiMetricMap			theMap;
CPose3DInterpolator		robot_path;


//(*IdInit(CFormRawMap)
const long CFormRawMap::ID_STATICTEXT7 = wxNewId();
const long CFormRawMap::ID_STATICTEXT6 = wxNewId();
const long CFormRawMap::ID_STATICTEXT5 = wxNewId();
const long CFormRawMap::ID_STATICTEXT1 = wxNewId();
const long CFormRawMap::ID_SLIDER1 = wxNewId();
const long CFormRawMap::ID_SPINCTRL1 = wxNewId();
const long CFormRawMap::ID_STATICTEXT3 = wxNewId();
const long CFormRawMap::ID_SLIDER2 = wxNewId();
const long CFormRawMap::ID_SPINCTRL2 = wxNewId();
const long CFormRawMap::ID_STATICTEXT10 = wxNewId();
const long CFormRawMap::ID_SLIDER3 = wxNewId();
const long CFormRawMap::ID_SPINCTRL3 = wxNewId();
const long CFormRawMap::ID_BUTTON2 = wxNewId();
const long CFormRawMap::ID_BUTTON6 = wxNewId();
const long CFormRawMap::ID_BUTTON5 = wxNewId();
const long CFormRawMap::ID_BUTTON1 = wxNewId();
const long CFormRawMap::ID_BUTTON3 = wxNewId();
const long CFormRawMap::ID_BUTTON7 = wxNewId();
const long CFormRawMap::ID_BUTTON8 = wxNewId();
const long CFormRawMap::ID_BUTTON9 = wxNewId();
const long CFormRawMap::ID_STATICTEXT8 = wxNewId();
const long CFormRawMap::ID_BITMAPBUTTON1 = wxNewId();
const long CFormRawMap::ID_BUTTON4 = wxNewId();
const long CFormRawMap::ID_TEXTCTRL1 = wxNewId();
const long CFormRawMap::ID_PANEL1 = wxNewId();
const long CFormRawMap::ID_CUSTOM2 = wxNewId();
const long CFormRawMap::ID_PANEL3 = wxNewId();
//*)

BEGIN_EVENT_TABLE(CFormRawMap,wxDialog)
    //(*EventTable(CFormRawMap)
    //*)
END_EVENT_TABLE()

CFormRawMap::CFormRawMap(wxWindow* parent,wxWindowID id)
{
    //(*Initialize(CFormRawMap)
    wxStaticBoxSizer* StaticBoxSizer2;
    wxFlexGridSizer* FlexGridSizer8;
    wxFlexGridSizer* FlexGridSizer11;
    wxFlexGridSizer* FlexGridSizer7;
    wxFlexGridSizer* FlexGridSizer4;
    wxFlexGridSizer* FlexGridSizer9;
    wxFlexGridSizer* FlexGridSizer6;
    wxFlexGridSizer* FlexGridSizer10;
    wxFlexGridSizer* FlexGridSizer5;
    wxStaticBoxSizer* StaticBoxSizer1;

    Create(parent, wxID_ANY, _("Creation of \"raw map & paths\" from scans & odometry"), wxDefaultPosition, wxDefaultSize, wxCAPTION|wxDEFAULT_DIALOG_STYLE|wxSYSTEM_MENU|wxRESIZE_BORDER|wxMAXIMIZE_BOX, _T("wxID_ANY"));
    FlexGridSizer1 = new wxFlexGridSizer(2, 1, 0, 0);
    FlexGridSizer1->AddGrowableCol(0);
    FlexGridSizer1->AddGrowableRow(1);
    Panel2 = new wxPanel(this, ID_PANEL1, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL, _T("ID_PANEL1"));
    FlexGridSizer2 = new wxFlexGridSizer(1, 2, 0, 0);
    FlexGridSizer2->AddGrowableCol(1);
    FlexGridSizer2->AddGrowableRow(0);
    FlexGridSizer4 = new wxFlexGridSizer(0, 1, 0, 0);
    FlexGridSizer3 = new wxFlexGridSizer(4, 3, 0, 0);
    FlexGridSizer3->AddGrowableCol(1);
    StaticText5 = new wxStaticText(Panel2, ID_STATICTEXT7, wxEmptyString, wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT7"));
    FlexGridSizer3->Add(StaticText5, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText4 = new wxStaticText(Panel2, ID_STATICTEXT6, _("Select which indexes to process:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT6"));
    FlexGridSizer3->Add(StaticText4, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText2 = new wxStaticText(Panel2, ID_STATICTEXT5, wxEmptyString, wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT5"));
    FlexGridSizer3->Add(StaticText2, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText1 = new wxStaticText(Panel2, ID_STATICTEXT1, _("First entry:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT1"));
    FlexGridSizer3->Add(StaticText1, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    slFrom = new wxSlider(Panel2, ID_SLIDER1, 0, 0, 100, wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_SLIDER1"));
    FlexGridSizer3->Add(slFrom, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 1);
    edFirst = new wxSpinCtrl(Panel2, ID_SPINCTRL1, _T("0"), wxDefaultPosition, wxDefaultSize, 0, 0, 100, 0, _T("ID_SPINCTRL1"));
    edFirst->SetValue(_T("0"));
    FlexGridSizer3->Add(edFirst, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText3 = new wxStaticText(Panel2, ID_STATICTEXT3, _("Last entry:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT3"));
    FlexGridSizer3->Add(StaticText3, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    slTo = new wxSlider(Panel2, ID_SLIDER2, 0, 0, 100, wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_SLIDER2"));
    FlexGridSizer3->Add(slTo, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 1);
    edLast = new wxSpinCtrl(Panel2, ID_SPINCTRL2, _T("0"), wxDefaultPosition, wxDefaultSize, 0, 0, 100, 0, _T("ID_SPINCTRL2"));
    edLast->SetValue(_T("0"));
    FlexGridSizer3->Add(edLast, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText6 = new wxStaticText(Panel2, ID_STATICTEXT10, _("Decimation:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT10"));
    FlexGridSizer3->Add(StaticText6, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    slDecimate = new wxSlider(Panel2, ID_SLIDER3, 0, 1, 200, wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_SLIDER3"));
    FlexGridSizer3->Add(slDecimate, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    edDec = new wxSpinCtrl(Panel2, ID_SPINCTRL3, _T("1"), wxDefaultPosition, wxDefaultSize, 0, 1, 200, 1, _T("ID_SPINCTRL3"));
    edDec->SetValue(_T("1"));
    FlexGridSizer3->Add(edDec, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer4->Add(FlexGridSizer3, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer5 = new wxFlexGridSizer(3, 1, 0, 0);
    FlexGridSizer5->AddGrowableCol(0);
    FlexGridSizer5->AddGrowableCol(1);
    FlexGridSizer6 = new wxFlexGridSizer(0, 1, 0, 0);
    FlexGridSizer6->AddGrowableCol(0);
    FlexGridSizer6->AddGrowableRow(0);
    StaticBoxSizer1 = new wxStaticBoxSizer(wxHORIZONTAL, Panel2, _("Generate..."));
    FlexGridSizer8 = new wxFlexGridSizer(0, 3, 0, 0);
    FlexGridSizer8->AddGrowableCol(0);
    btnGenerate = new wxButton(Panel2, ID_BUTTON2, _("Map from odometry"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON2"));
    btnGenerate->SetDefault();
    FlexGridSizer8->Add(btnGenerate, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnGenerateRTK = new wxButton(Panel2, ID_BUTTON6, _("Map from RTK GPS"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON6"));
    btnGenerateRTK->SetDefault();
    FlexGridSizer8->Add(btnGenerateRTK, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnGeneratePaths = new wxButton(Panel2, ID_BUTTON5, _("Random paths..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON5"));
    btnGeneratePaths->SetDefault();
    FlexGridSizer8->Add(btnGeneratePaths, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer1->Add(FlexGridSizer8, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer6->Add(StaticBoxSizer1, 1, wxALL|wxEXPAND|wxALIGN_TOP|wxALIGN_CENTER_HORIZONTAL, 3);
    FlexGridSizer5->Add(FlexGridSizer6, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer7 = new wxFlexGridSizer(0, 3, 0, 0);
    FlexGridSizer7->AddGrowableCol(0);
    FlexGridSizer7->AddGrowableRow(0);
    StaticBoxSizer2 = new wxStaticBoxSizer(wxHORIZONTAL, Panel2, _("Result"));
    FlexGridSizer9 = new wxFlexGridSizer(0, 1, 0, 0);
    FlexGridSizer11 = new wxFlexGridSizer(1, 3, 0, 0);
    FlexGridSizer11->AddGrowableCol(0);
    btnSaveTxt = new wxButton(Panel2, ID_BUTTON1, _("Save map as text..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON1"));
    FlexGridSizer11->Add(btnSaveTxt, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnSave3D = new wxButton(Panel2, ID_BUTTON3, _("Save map as 3D scene..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON3"));
    FlexGridSizer11->Add(btnSave3D, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnSavePath = new wxButton(Panel2, ID_BUTTON7, _("Save vehicle path..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON7"));
    FlexGridSizer11->Add(btnSavePath, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnSaveObsPath = new wxButton(Panel2, ID_BUTTON8, _("Save observations poses..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON8"));
    FlexGridSizer11->Add(btnSaveObsPath, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnView3D = new wxButton(Panel2, ID_BUTTON9, _("Preview map in 3D"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON9"));
    FlexGridSizer11->Add(btnView3D, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer9->Add(FlexGridSizer11, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    lbCount = new wxStaticText(Panel2, ID_STATICTEXT8, _("Point count=0 (No decimation)"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT8"));
    FlexGridSizer9->Add(lbCount, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer2->Add(FlexGridSizer9, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer7->Add(StaticBoxSizer2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer5->Add(FlexGridSizer7, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 3);
    FlexGridSizer10 = new wxFlexGridSizer(0, 3, 0, 0);
    FlexGridSizer10->AddGrowableCol(0);
    FlexGridSizer10->AddGrowableRow(0);
    FlexGridSizer10->Add(-1,-1,1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnHelp = new wxBitmapButton(Panel2, ID_BITMAPBUTTON1, wxArtProvider::GetBitmap(wxART_MAKE_ART_ID_FROM_STR(_T("wxART_INFORMATION")),wxART_BUTTON), wxDefaultPosition, wxDefaultSize, wxBU_AUTODRAW, wxDefaultValidator, _T("ID_BITMAPBUTTON1"));
    FlexGridSizer10->Add(btnHelp, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 4);
    btnClose = new wxButton(Panel2, ID_BUTTON4, _("Close"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON4"));
    FlexGridSizer10->Add(btnClose, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer5->Add(FlexGridSizer10, 1, wxALL|wxEXPAND|wxALIGN_TOP|wxALIGN_CENTER_HORIZONTAL, 5);
    FlexGridSizer4->Add(FlexGridSizer5, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer2->Add(FlexGridSizer4, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    edOpts = new wxTextCtrl(Panel2, ID_TEXTCTRL1, _("; ====================================================\n; MULTIMETRIC MAP CONFIGURATION\n; ====================================================\n[map]\n; Creation of maps:\noccupancyGrid_count = 0\ngasGrid_count = 0\nlandmarksMap_count = 0\nbeaconMap_count = 0\npointsMap_count = 1\nheightMap_count = 0\ncolourPointsMap_count=0\n\n; ====================================================\n; MULTIMETRIC MAP: PointsMap #00\n; ====================================================\n; Insertion Options for PointsMap 00:\n[map_pointsMap_00_insertOpts]\nminDistBetweenLaserPoints = 0.05\nisPlanarMap = 0\nalso_interpolate = 0\n\n; ====================================================\n; MULTIMETRIC MAP: HeightMap #00\n; ====================================================\n; Creation Options for HeightMap 00:\n[map_heightGrid_00_creationOpts]\nmapType = 0 \t\t; See CHeightGridMap2D::CHeightGridMap2D\nmin_x = -10\nmax_x = 10\nmin_y = -10\nmax_y = 10\nresolution = 0.10\n\n; ====================================================\n; MULTIMETRIC MAP: HeightMap #00\n; ====================================================\n; Insertion Options for HeightMap 00:\n[map_heightGrid_00_insertOpts]\nfilterByHeight = 0 ; 0/1: Do not/do filter.\nz_min =-0.10\nz_max = 0.10\n\n; ====================================================\n; MULTIMETRIC MAP: ColourPointsMap #00\n; ====================================================\n; Insertion Options for ColourPointsMap 00:\n[map_colourPointsMap_00_insertOpts]\nminDistBetweenLaserPoints = 0.05\nisPlanarMap = 0\nalso_interpolate = 0\n\n"), wxDefaultPosition, wxSize(247,94), wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB|wxTE_MULTILINE|wxHSCROLL, wxDefaultValidator, _T("ID_TEXTCTRL1"));
    wxFont edOptsFont(8,wxSWISS,wxFONTSTYLE_NORMAL,wxNORMAL,false,_T("Courier New"),wxFONTENCODING_DEFAULT);
    edOpts->SetFont(edOptsFont);
    FlexGridSizer2->Add(edOpts, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 1);
    Panel2->SetSizer(FlexGridSizer2);
    FlexGridSizer2->Fit(Panel2);
    FlexGridSizer2->SetSizeHints(Panel2);
    FlexGridSizer1->Add(Panel2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    Panel3 = new wxPanel(this, ID_PANEL3, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL, _T("ID_PANEL3"));
    BoxSizer1 = new wxBoxSizer(wxHORIZONTAL);
    plotMap = new mpWindow(Panel3,ID_CUSTOM2,wxDefaultPosition,wxSize(742,380),0);
    BoxSizer1->Add(plotMap, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    Panel3->SetSizer(BoxSizer1);
    BoxSizer1->Fit(Panel3);
    BoxSizer1->SetSizeHints(Panel3);
    FlexGridSizer1->Add(Panel3, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    SetSizer(FlexGridSizer1);
    FlexGridSizer1->Fit(this);
    FlexGridSizer1->SetSizeHints(this);
    Center();

    Connect(ID_SLIDER1,wxEVT_SCROLL_THUMBTRACK,(wxObjectEventFunction)&CFormRawMap::OnslFromCmdScrollThumbTrack);
    Connect(ID_SLIDER1,wxEVT_SCROLL_CHANGED,(wxObjectEventFunction)&CFormRawMap::OnslFromCmdScrollThumbTrack);
    Connect(ID_SLIDER2,wxEVT_SCROLL_THUMBTRACK,(wxObjectEventFunction)&CFormRawMap::OnslToCmdScrollThumbTrack);
    Connect(ID_SLIDER2,wxEVT_SCROLL_CHANGED,(wxObjectEventFunction)&CFormRawMap::OnslToCmdScrollThumbTrack);
    Connect(ID_SLIDER3,wxEVT_SCROLL_THUMBTRACK,(wxObjectEventFunction)&CFormRawMap::OnslDecimateCmdScrollThumbTrack);
    Connect(ID_SLIDER3,wxEVT_SCROLL_CHANGED,(wxObjectEventFunction)&CFormRawMap::OnslDecimateCmdScrollThumbTrack);
    Connect(ID_BUTTON2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnGenerateClick);
    Connect(ID_BUTTON6,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnGenerateFromRTK);
    Connect(ID_BUTTON5,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnGeneratePathsClick);
    Connect(ID_BUTTON1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnSaveTxtClick);
    Connect(ID_BUTTON3,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnSave3DClick);
    Connect(ID_BUTTON7,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnSavePathClick);
    Connect(ID_BUTTON8,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnSaveObsPathClick);
    Connect(ID_BUTTON9,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnView3DClick);
    Connect(ID_BUTTON4,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CFormRawMap::OnbtnCloseClick);
    //*)

    Maximize();
    FlexGridSizer1->RecalcSizes();
}

CFormRawMap::~CFormRawMap()
{
    //(*Destroy(CFormRawMap)
    //*)
}

void loadMapInto3DScene(COpenGLScene &scene)
{
	{
		mrpt::opengl::CGridPlaneXYPtr	obj = mrpt::opengl::CGridPlaneXY::Create( -200,200,-200,200,0,5 );
		obj->m_color_R =
		obj->m_color_G =
		obj->m_color_B = 0.3f;
		obj->m_color_A = 1;

		scene.insert( obj );
	}

	// The robot path:
	{
		mrpt::opengl::CSetOfLinesPtr  obj = mrpt::opengl::CSetOfLines::Create();

		obj->m_color_R = 1;
		obj->m_color_G = 0;
		obj->m_color_B = 0;
		obj->m_color_A = 0;

		obj->setLineWidth( 2 );

		double x0=0,y0=0,z0=0;
		for (CPose3DInterpolator::iterator it=robot_path.begin();it!=robot_path.end();it++)
		{
			CPose3D &p = it->second;
			obj->appendLine( x0,y0,z0, p.x,p.y,p.z );
			x0 = p.x;
			y0 = p.y;
			z0 = p.z;
		}

		scene.insert( obj );
	}

	opengl::CSetOfObjectsPtr objs = opengl::CSetOfObjects::Create();
	theMap.getAs3DObject( objs );
	scene.insert( objs );
}


// From slider moved:
void CFormRawMap::OnslFromCmdScrollThumbTrack(wxScrollEvent& event)
{
    int toVal = slTo->GetValue();
    int curVal = slFrom->GetValue();

    if (curVal>toVal) slFrom->SetValue(toVal);
    edFirst->SetValue( wxString::Format(_("%d"),slFrom->GetValue()) );
}

// To slider moved:
void CFormRawMap::OnslToCmdScrollThumbTrack(wxScrollEvent& event)
{
    int fromVal = slFrom->GetValue();
    int curVal = slTo->GetValue();

    if (curVal<fromVal) slTo->SetValue(fromVal);
    edLast->SetValue( wxString::Format(_("%d"),slTo->GetValue()) );
}

// Generate the map:
void CFormRawMap::OnbtnGenerateClick(wxCommandEvent& event)
{
    WX_START_TRY

    // Go: generate the map:
    size_t      i;
    CPose2D     curPose(0,0,0);

    size_t      first = edFirst->GetValue();
    size_t      last  = edLast->GetValue();
    size_t      decimate = edDec->GetValue();
    ASSERT_(first<=last);
    ASSERT_(last<=rawlog.size()-1);
    ASSERT_(decimate>0);

    // Create a memory "ini file" with the text in the window:
    CConfigFileMemory       configSrc( CStringList( std::string(edOpts->GetValue().mb_str()) ) );

	TSetOfMetricMapInitializers		lstMaps;
	lstMaps.loadFromConfigFile( configSrc, "map" );
	theMap.setListOfMaps( &lstMaps );

	CPointsMapPtr	thePntsMap;

	if( !theMap.m_pointsMaps.empty() )
		thePntsMap = theMap.m_pointsMaps[0];
	else if (theMap.m_colourPointsMap.present())
		thePntsMap = theMap.m_colourPointsMap;

    wxBusyCursor    waitCursor;

    wxProgressDialog    progDia(
        wxT("Creating raw map"),
        wxT("Working..."),
        (int)(last-first+1), // range
        this, // parent
        wxPD_CAN_ABORT |
        wxPD_APP_MODAL |
        wxPD_SMOOTH |
        wxPD_AUTO_HIDE |
        wxPD_ELAPSED_TIME |
        wxPD_ESTIMATED_TIME |
        wxPD_REMAINING_TIME);

    wxTheApp->Yield();  // Let the app. process messages
    size_t count = 0;

    vector_float    pathX,pathY;
    bool            abort = false;

    // An (aprox) estimate of the final size of the map (great improve in speed!)
	if (thePntsMap) thePntsMap->reserve( (last-first+1)*800 );

    for (i=first;!abort && i<=last;i++)
    {
    	switch( rawlog.getType(i) )
        {
		case CRawlog::etActionCollection:
			{
				CActionCollectionPtr    acts = rawlog.getAsAction(i);
				CPose2D                 poseIncrement;
				bool                    poseIncrementLoaded = false;

				for (size_t j=0;j<acts->size();j++)
				{
					CActionPtr act = acts->get(j);
					if (act->GetRuntimeClass() == CLASS_ID( CActionRobotMovement2D ))
					{
						CActionRobotMovement2DPtr mov = CActionRobotMovement2DPtr( act );

						// Load an odometry estimation, but only if it is the only movement
						//  estimation source: any other may be a better one:
						if ( !poseIncrementLoaded || mov->estimationMethod!= CActionRobotMovement2D::emOdometry )
						{
							poseIncrementLoaded=true;
							poseIncrement = mov->poseChange->getEstimatedPose();
						}
					}
				}

				if ( !poseIncrementLoaded && i<last )
					THROW_EXCEPTION_CUSTOM_MSG1("ERROR: Odometry not found at step %d!",(int)i);

				curPose = curPose + poseIncrement;
				pathX.push_back( curPose.x );
				pathY.push_back( curPose.y );
			}
			break;
		case CRawlog::etSensoryFrame:
			{
				if (( (i>>1) % decimate)==0)
				{
					CPose3D		dumPose(curPose);
					rawlog.getAsObservations(i)->insertObservationsInto( &theMap, &dumPose );
				}
			}
			break;
		case CRawlog::etObservation:
			{
				if (( (i>>1) % decimate)==0)
				{
					CPose3D		dumPose(curPose);
					theMap.insertObservation( rawlog.getAsObservation(i).pointer(), &dumPose );
				}
			}
			break;

        }; // end switch

        if ((count++ % 50)==0)
        {
            if (!progDia.Update((int)(i-first)))
                abort = true;
            wxTheApp->Yield();
        }
    } // end for i

    progDia.Update( (int)(last-first+1) );


    // Load into the graphs:
    // ----------------------------------
    plotMap->DelAllLayers(true,false);

    mpFXYVector *lyPoints = new mpFXYVector();
    mpFXYVector *lyPath   = new mpFXYVector();
    lyPath->SetPen( wxPen(wxColour(255,0,0),2) );
    lyPath->SetContinuity( true );
    lyPoints->SetPen( wxPen(wxColour(0,0,255),0) );

    plotMap->AddLayer( lyPoints );
    plotMap->AddLayer( lyPath );
    plotMap->EnableDoubleBuffer(true);

    lyPath->SetData( pathX,pathY );

	if (thePntsMap)
	{
		size_t nPts = thePntsMap->size();
		size_t decimation = 1;

		if (nPts>100000)
		{
			decimation = nPts / 100000;
		}

		vector_float    Xs,Ys;
		thePntsMap->getAllPoints(Xs,Ys, decimation );

		lyPoints->SetData(Xs,Ys);
		plotMap->LockAspect(false);
		plotMap->Fit();      // Update the window to show the new data fitted.
		plotMap->LockAspect(true);
		plotMap->AddLayer( new mpScaleX() );
		plotMap->AddLayer( new mpScaleY() );

		if (decimation>1)
				lbCount->SetLabel( wxString::Format(_("Point count=%u\n(Decimation=%u)"),unsigned(Xs.size()), unsigned(decimation) ) );
		else	lbCount->SetLabel( wxString::Format(_("Point count=%u\n(No decimation)"),unsigned(Xs.size()) ) );
	}


    WX_END_TRY
}

void CFormRawMap::OnbtnSaveTxtClick(wxCommandEvent& event)
{
    wxString caption = wxT("Save as text file...");
    wxString wildcard = wxT("Text files (*.txt)|*.txt|All files (*.*)|*.*");

    wxString defaultDir( _U( iniFile->read_string(iniFileSect,"LastDir",".").c_str() ) );

    wxString defaultFilename = _( "map.txt" );
    wxFileDialog dialog(this, caption, defaultDir, defaultFilename,wildcard, wxSAVE | wxFD_OVERWRITE_PROMPT);

    if (dialog.ShowModal() == wxID_OK)
    {
        wxString fileName = dialog.GetPath();
        try
        {
			theMap.saveMetricMapRepresentationToFile( std::string(fileName.mb_str()) );
        }
        catch (std::exception &e)
        {
            wxMessageBox( _U(e.what()), _("Exception"), wxOK, this);
        }
    }
}

// Save as a 3D scene:
void CFormRawMap::OnbtnSave3DClick(wxCommandEvent& event)
{
    wxString caption = wxT("Save as 3D scene file...");
    wxString wildcard = wxT("MRPT 3D scene files (*.3Dscene)|*.3Dscene|All files (*.*)|*.*");

    wxString defaultDir( _U( iniFile->read_string(iniFileSect,"LastDir",".").c_str() ) );

    wxString defaultFilename = _( "map.3Dscene" );
    wxFileDialog dialog(this, caption, defaultDir, defaultFilename,wildcard, wxSAVE | wxFD_OVERWRITE_PROMPT);

    if (dialog.ShowModal() == wxID_OK)
    {
        try
        {
			wxBusyCursor    waitCursor;

            CFileGZOutputStream fil( std::string(dialog.GetPath().mb_str()) );
            COpenGLScene		scene;

            loadMapInto3DScene(scene);

            fil << scene;
        }
        catch (std::exception &e)
        {
            wxMessageBox( _U(e.what()), _("Exception"), wxOK, this);
        }
    }
}

// Close:
void CFormRawMap::OnbtnCloseClick(wxCommandEvent& event)
{
    Close();
}

void CFormRawMap::OnslDecimateCmdScrollThumbTrack(wxScrollEvent& event)
{
    edDec->SetValue( wxString::Format(_("%d"),slDecimate->GetValue() ) );
}

// Generate random paths from the motion model:
void CFormRawMap::OnbtnGeneratePathsClick(wxCommandEvent& event)
{
    WX_START_TRY

    // Go: generate the map:
    size_t      i;


    long nPaths = ::wxGetNumberFromUser(
                      _("Each incremental pose will be drawn from its probabilistic motion model)"),
                      _("How many random samples of the path to generate?\n"),
                      _("Generation of random paths"),
                      100 /*Default*/, 1/*Min*/, 10000/*Max*/);

    if (nPaths<0) return;

    size_t      first = edFirst->GetValue();
    size_t      last  = edLast->GetValue();
    size_t      decimate = edDec->GetValue();
    ASSERT_(first<=last);
    ASSERT_(last<=rawlog.size()-1);
    ASSERT_(decimate>0);
    ASSERT_(nPaths>0 && nPaths<100000);

    // Load into the graphs:
    // ----------------------------------
    plotMap->DelAllLayers(true,false);

    wxBusyCursor    waitCursor;

    size_t     nComputationSteps = (last-first+1)*nPaths;

    wxProgressDialog    progDia(
        wxT("Generating paths"),
        wxT("Working..."),
        (int)nComputationSteps, // range
        this, // parent
        wxPD_CAN_ABORT |
        wxPD_APP_MODAL |
        wxPD_SMOOTH |
        wxPD_AUTO_HIDE |
        wxPD_ELAPSED_TIME |
        wxPD_ESTIMATED_TIME |
        wxPD_REMAINING_TIME);

    wxTheApp->Yield();  // Let the app. process messages
    size_t count = 0;

    bool            abort = false;

    CPosePDFParticles	pdfParts( nPaths );

    for (int pathIter = 0;!abort && pathIter<nPaths;pathIter++)
    {
        vector_float    pathX,pathY,pathPhi;
        CPose2D         curPose(0,0,0);

        // Initial pose:
        pathX.push_back(0);
        pathY.push_back(0);
        pathPhi.push_back(0);

        for (i=first;!abort && i<=last;i++)
        {
            switch ( rawlog.getType(i) )
            {
			case CRawlog::etActionCollection:
            	{
					CActionCollectionPtr    acts = rawlog.getAsAction(i);
					CPose2D                 poseIncrement;
					bool                    poseIncrementLoaded = false;

					for (size_t j=0;j<acts->size();j++)
					{
						CActionPtr act = acts->get(j);
						if (act->GetRuntimeClass() == CLASS_ID( CActionRobotMovement2D ))
						{
							CActionRobotMovement2DPtr mov = CActionRobotMovement2DPtr( act );

							// Load an odometry estimation, but only if it is the only movement
							//  estimation source: any other may be a better one:
							if ( !poseIncrementLoaded || mov->estimationMethod!= CActionRobotMovement2D::emOdometry )
							{
								poseIncrementLoaded=true;
								// Draw random pose:
								mov->drawSingleSample( poseIncrement );
							}
						}
					}

					if ( !poseIncrementLoaded )
						THROW_EXCEPTION_CUSTOM_MSG1("ERROR: Odometry not found at step %d!",(int)i);

					curPose = curPose + poseIncrement;
					pathX.push_back( curPose.x );
					pathY.push_back( curPose.y );
					pathPhi.push_back( curPose.phi );
				}
				break;

			default:
				{
					// It's an observation... nothing to do.
				}
            };

            if ((count++ % 50)==0)
            {
                if (!progDia.Update((int)count)) abort = true;
                wxTheApp->Yield();
            }
        } // end for i

        if (!abort)
        {
            // Save for the covariance:
            // ----------------------------
            *(pdfParts.m_particles[pathIter].d) = CPose2D(
                                                      pathX[pathX.size()-1],
                                                      pathY[pathX.size()-1],
                                                      pathPhi[pathX.size()-1] );

            // Add new layer with this path:
            // ----------------------------------
            mpFXYVector *lyPath   = new mpFXYVector();
            lyPath->SetPen( wxPen(wxColour(255,0,0),1) );
            lyPath->SetContinuity( true );

            // Load into the graphs:
            lyPath->SetData( pathX,pathY );

            plotMap->AddLayer( lyPath, false );
        }

    } // end for path samples

    progDia.Update( (int)nComputationSteps );

    // Add a layer with a covariance with the last pose:
    CMatrix COV ( pdfParts.getEstimatedCovariance() );
    CPose2D	meanPose ( pdfParts.getEstimatedPose() );
    mpCovarianceEllipse	*lyCov = new mpCovarianceEllipse( COV(0,0),COV(1,1),COV(0,1) );
    lyCov->SetContinuity( true );
    lyCov->SetCoordinateBase( meanPose.x, meanPose.y, 0 );
    plotMap->AddLayer( lyCov ,false);

    //
    plotMap->EnableDoubleBuffer(true);

    plotMap->Fit();      // Update the window to show the new data fitted.
    plotMap->LockAspect(true);
    plotMap->AddLayer( new mpScaleX() );
    plotMap->AddLayer( new mpScaleY() );

    lbCount->SetLabel( wxString::Format(_("%d paths"),nPaths) );

    WX_END_TRY
}

void CFormRawMap::OnGenerateFromRTK(wxCommandEvent& event)
{
    WX_START_TRY

    // Go: generate the map:
    size_t      i;
    CPose2D     curPose(0,0,0);

    size_t      first = edFirst->GetValue();
    size_t      last  = edLast->GetValue();
    size_t      decimate = slDecimate->GetValue();
    ASSERT_(first<=last);
    ASSERT_(last<=rawlog.size()-1);
    ASSERT_(decimate>0);

    // Create a memory "ini file" with the text in the window:
    CConfigFileMemory       configSrc( CStringList( std::string(edOpts->GetValue().mb_str()) ) );

	TSetOfMetricMapInitializers		lstMaps;
	lstMaps.loadFromConfigFile( configSrc, "map" );
	theMap.setListOfMaps( &lstMaps );

	CPointsMapPtr	thePntsMap;

	if( !theMap.m_pointsMaps.empty() )
		thePntsMap = theMap.m_pointsMaps[0];
	else if (theMap.m_colourPointsMap.present())
		thePntsMap = theMap.m_colourPointsMap;

	// An (aprox) estimate of the final size of the map (great improve in speed!)
	if (thePntsMap) thePntsMap->reserve( last-first+1 );

    wxBusyCursor    waitCursor;


    size_t count = 0;

	robot_path.clear();
	robot_path.setMaxTimeInterpolation(3.0);	// Max. seconds of GPS blackout not to interpolate.

    vector_float			pathX,pathY;

    bool		abort = false;

	double		ref_lon=0, ref_lat=0, ref_alt=0;
	bool		ref_valid = false;

	// ------------------------------------------
	// Look for the 2 observations:
	// ------------------------------------------
    wxProgressDialog    progDia(
        wxT("Building map"),
        wxT("Getting GPS observations..."),
        (int)(last-first+1), // range
        this, // parent
        wxPD_CAN_ABORT |
        wxPD_APP_MODAL |
        wxPD_SMOOTH |
        wxPD_AUTO_HIDE |
        wxPD_ELAPSED_TIME |
        wxPD_ESTIMATED_TIME |
        wxPD_REMAINING_TIME);

	// The list with all time ordered gps's in valid RTK mode
	typedef std::map< mrpt::system::TTimeStamp, std::vector<CObservationGPSPtr> > TListGPSs;
	TListGPSs	list_gps_obs;

    for (i=first;!abort && i<=last;i++)
    {
        switch ( rawlog.getType(i) )
		{
		default:
			break;

        case CRawlog::etObservation:
            {
                CObservationPtr o = rawlog.getAsObservation(i);

				if (o->GetRuntimeClass()==CLASS_ID(CObservationGPS))
				{
					CObservationGPSPtr obs = CObservationGPSPtr(o);

					if (obs->has_GGA_datum && obs->GGA_datum.fix_quality==4)
					{
						// Add to the list:
						TListGPSs::iterator q = list_gps_obs.find(obs->timestamp);
						if (q!=list_gps_obs.end())
						{
							q->second.push_back( obs );
						}
						else
						{
							list_gps_obs[obs->timestamp].push_back( obs );
						}
					}
                }
            }
            break;
        } // end switch type

		// Show progress:
        if ((count++ % 100)==0)
        {
            if (!progDia.Update((int)(i-first)))
                abort = true;
            wxTheApp->Yield();
        }
    } // end for i

    progDia.Update( (int)(last-first+1) );


	// ------------------------------------------
	// Look for the 2 observations:
	// ------------------------------------------
	int N_GPSs = list_gps_obs.size();

	if (N_GPSs)
	{
		wxProgressDialog    progDia3(
			wxT("Building map"),
			wxT("Estimating 6D path..."),
			N_GPSs, // range
			this, // parent
			wxPD_CAN_ABORT |
			wxPD_APP_MODAL |
			wxPD_SMOOTH |
			wxPD_AUTO_HIDE |
			wxPD_ELAPSED_TIME |
			wxPD_ESTIMATED_TIME |
			wxPD_REMAINING_TIME);


		int	idx_in_GPSs = 0;
		for (TListGPSs::iterator i=list_gps_obs.begin();i!=list_gps_obs.end();i++, idx_in_GPSs++)
		{
			// Now check if we have 2 gps with the same time stamp:
			if (i->second.size()>=2)
			{
				int idx1,idx2;

				if (i->second[0]->sensorPose.x==0 && i->second[0]->sensorPose.y==0 && i->second[0]->sensorPose.z==0)
				{
					idx1=0;
					idx2=1;
				}
				else if (i->second[1]->sensorPose.x==0 && i->second[1]->sensorPose.y==0 && i->second[1]->sensorPose.z==0)
				{
					idx1=1;
					idx2=0;
				}
				else
				{
					THROW_EXCEPTION("TODO: Allow GPSs to be at arbitrary positions on the vehicle.")
				}

				CObservationGPSPtr gps1 = i->second[idx1];	// For now, this one is fixed at (0,0,0)
				CObservationGPSPtr gps2 = i->second[idx2];

				// They have different labels: Compute relative angles on the vehicle:
				// Compute the yaw/pitch relative to the vehicle:
				double Asx = gps2->sensorPose.x - gps1->sensorPose.x;
				double Asy = gps2->sensorPose.y - gps1->sensorPose.y;
				double Asz = gps2->sensorPose.z - gps1->sensorPose.z;
				double Asxy = sqrt( square(Asx) + square(Asy) );
				ASSERT_(Asxy!=0);

				double delta_yaw   = atan2( Asy , Asx );
				double delta_pitch = atan2( -Asz, Asxy );

				// Process them:
				if (gps1->GGA_datum.UTCTime == gps2->GGA_datum.UTCTime)
				{
					// It's better if we have a sensor at (0,0,0)
					bool gps1_is_origin = false;
					if (gps1->sensorPose.x!=0 || gps1->sensorPose.y!=0 || gps1->sensorPose.z!=0)
					{
						if (gps2->sensorPose.x==0 && gps2->sensorPose.y==0 && gps2->sensorPose.z==0)
						{
							std::swap(*gps1,*gps2);
							gps1_is_origin = true;
						}
					}
					else gps1_is_origin=true;

					// get the reference lat/lon?
					if (!ref_valid)
					{
						ref_valid 	= true;
						ref_lon		= gps1->GGA_datum.longitude_degrees;
						ref_lat		= gps1->GGA_datum.latitude_degrees;
						ref_alt		= gps1->GGA_datum.altitude_meters;
					}

					// Compute the XYZ coordinates of both sensors:
					double	x1,y1,z1;
					mrpt::topography::coordinatesTransformation_WGS84(
						gps1->GGA_datum.longitude_degrees,
						gps1->GGA_datum.latitude_degrees,
						gps1->GGA_datum.altitude_meters,
						x1,y1,z1,
						ref_lon, ref_lat, ref_alt );

					double	x2,y2,z2;
					mrpt::topography::coordinatesTransformation_WGS84(
						gps2->GGA_datum.longitude_degrees,
						gps2->GGA_datum.latitude_degrees,
						gps2->GGA_datum.altitude_meters,
						x2,y2,z2,
						ref_lon, ref_lat, ref_alt );


					// Get the absolute yaw/pitch between the two XYZ points:
					double Ax = x2-x1;
					double Ay = y2-y1;
					double Az = z2-z1;
					double Axy = sqrt( square(Ax) + square(Ay) );
					ASSERT_(Axy!=0);

					double sensors_yaw   = atan2( Ay , Ax );
					double sensors_pitch = atan2( -Az, Axy );

					// Vehicle yaw & pitch:
					double veh_yaw   = sensors_yaw   - delta_yaw;
					double veh_pitch = sensors_pitch - delta_pitch;
					double veh_roll	 = 0;

					// ------------------------------------------------------------------------------------------------
					// If we have IMU readings, use them to obtain a better estimate of pitch & specially, the roll.
					// ------------------------------------------------------------------------------------------------
					TListTimeAndObservations	lstIMUs;
					rawlog.findObservationsByClassInRange(
						gps1->timestamp - mrpt::system::secondsToTimestamp(0.05),
						gps1->timestamp + mrpt::system::secondsToTimestamp(0.05),
						CLASS_ID(CObservationIMU),
						lstIMUs );

					if (lstIMUs.size())
					{
						// Average the sensor roll & pitch:
						size_t nIMU = lstIMUs.size();
						vector_double	pitchs(nIMU), rolls(nIMU);

						size_t k;
						TListTimeAndObservations::iterator toi;

						for (k=0,toi=lstIMUs.begin();k<nIMU;k++,toi++)
						{
							CObservationIMUPtr o = CObservationIMUPtr( toi->second );
							pitchs[k] = o->rawMeasurements[IMU_PITCH];
							rolls[k]  = o->rawMeasurements[IMU_ROLL];
						}

						//double avrg_pitch = mrpt::math::mean( pitchs );
						//double avrg_roll  = mrpt::math::mean( rolls );

						//double imu_vehicle_pitch = mrpt::math::wrapToPi( avrg_roll + M_PI );
						double imu_vehicle_roll  = 0; //-avrg_pitch;

						veh_roll = imu_vehicle_roll;
					}

					// Estimate vehicle 3D position:
					/*CPoint3D	inv_As = CPoint3D(0,0,0) - CPose3D(
						gps1->sensorPose.x,gps1->sensorPose.y,gps1->sensorPose.z,
						veh_yaw,veh_pitch,veh_roll);
					*/
					CPoint3D	gps1_global_xyz(x1,y1,z1);
					CPoint3D	veh_xyz= gps1_global_xyz; // + inv_As;	TODO!!!

					// Final vehicle pose:
					CPose3D		veh_pose( veh_xyz.x,veh_xyz.y,veh_xyz.z,veh_yaw,veh_pitch,veh_roll );

			/*		cout << "sensor pose: " << gps1->sensorPose << endl;
					cout << "sensor: " << gps1_global_xyz << endl;
					cout << "veh_xyz: " << veh_xyz << endl << endl;
					pause();
					cout << "veh: " << veh_pose << endl;*/

					// Add to the interpolator:
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.x );
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.y );
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.z );
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.yaw );
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.pitch );
					MRPT_CHECK_NORMAL_NUMBER( veh_pose.roll );

					robot_path.insert( gps1->timestamp, veh_pose );

					// For the graphs:
					pathX.push_back(veh_pose.x);
					pathY.push_back(veh_pose.y);
				}
			}

			// Show progress:
			if ((count++ % 100)==0)
			{
				if (!progDia3.Update(idx_in_GPSs))
					abort = true;
				wxTheApp->Yield();
			}
		} // end for i

	} // end step generate 6D path

	if (!robot_path.size())
	{
		::wxMessageBox(_("Sorry, no valid RTK, multiple antennas GPS data found"));
		return;
	}

	// ---------------------------------------------------
	//   Now, build the map using the interpolator
    // --------------------------------------------------

    wxProgressDialog    progDia2(
        wxT("Building map"),
        wxT("Populating the map with observations..."),
        (int)(last-first+1), // range
        this, // parent
        wxPD_CAN_ABORT |
        wxPD_APP_MODAL |
        wxPD_SMOOTH |
        wxPD_AUTO_HIDE |
        wxPD_ELAPSED_TIME |
        wxPD_ESTIMATED_TIME |
        wxPD_REMAINING_TIME);

	progDia2.SetSize( 400,progDia.GetSize().GetHeight() );
    wxTheApp->Yield();  // Let the app. process messages

    for (i=first;!abort && i<=last;i++)
    {
        switch ( rawlog.getType(i) )
		{
		default:
			break;

        case CRawlog::etObservation:
            {
                CObservationPtr o = rawlog.getAsObservation(i);

				// Interpolate:
				CPose3D		p;
				bool		valid_interp;
				robot_path.interpolate( o->timestamp, p, valid_interp);

				if (valid_interp)
				{
/*					try
					{
						MRPT_CHECK_NORMAL_NUMBER(p.x);
						MRPT_CHECK_NORMAL_NUMBER(p.y);
						MRPT_CHECK_NORMAL_NUMBER(p.z);
						MRPT_CHECK_NORMAL_NUMBER(p.yaw);
						MRPT_CHECK_NORMAL_NUMBER(p.pitch);
						MRPT_CHECK_NORMAL_NUMBER(p.roll);
					}
					catch( std::exception &)
					{
						THROW_EXCEPTION_CUSTOM_MSG1("Error interpolating point at iter=%u",(unsigned int)i);
					}
*/

					// Decimation counters for each sensor label:
					static std::map<string, size_t>	decim_count;

					size_t &dec_cnt = decim_count[o->sensorLabel];

					if ( (++dec_cnt % decimate)==0)
					{
						theMap.insertObservation( o.pointer(), &p );
					}
				}
            }
            break;
        } // end switch type

		// Show progress:
        if ((count++ % 100)==0)
        {
            if (!progDia2.Update((int)(i-first)))
                abort = true;
            wxTheApp->Yield();
        }
    } // end for i

    progDia2.Update( (int)(last-first+1) );


    // Load into the graphs:
    // ----------------------------------
    plotMap->DelAllLayers(true,false);

    mpFXYVector *lyPoints = new mpFXYVector();
    mpFXYVector *lyPath   = new mpFXYVector();
    lyPath->SetPen( wxPen(wxColour(255,0,0),2) );
    lyPath->SetContinuity( true );
    lyPoints->SetPen( wxPen(wxColour(0,0,255),0) );

    plotMap->AddLayer( lyPoints );
    plotMap->AddLayer( lyPath );
    plotMap->EnableDoubleBuffer(true);

    lyPath->SetData( pathX,pathY );

	if (thePntsMap)
	{
		size_t nPts = thePntsMap->size();
		size_t decimation = 1;

		if (nPts>100000)
		{
			decimation = nPts / 100000;
		}

		vector_float    Xs,Ys;
		thePntsMap->getAllPoints(Xs,Ys, decimation );

		lyPoints->SetData(Xs,Ys);
		plotMap->LockAspect(false);
		plotMap->Fit();      // Update the window to show the new data fitted.
		plotMap->LockAspect(true);
		plotMap->AddLayer( new mpScaleX() );
		plotMap->AddLayer( new mpScaleY() );

		if (decimation>1)
				lbCount->SetLabel( wxString::Format(_("Point count=%u\n(Decimation=%u)"),unsigned(Xs.size()), unsigned(decimation) ) );
		else	lbCount->SetLabel( wxString::Format(_("Point count=%u\n(No decimation)"),unsigned(Xs.size()) ) );
	}

    WX_END_TRY
}

void CFormRawMap::OnbtnSavePathClick(wxCommandEvent& event)
{
	if (!robot_path.size())
	{
		wxMessageBox(_("The robot path is empty."));
		return;
	}

    wxString caption = wxT("Save path as txt file...");
    wxString wildcard = wxT("Text files (*.txt)|*.txt|All files (*.*)|*.*");

    wxString defaultDir( _U( iniFile->read_string(iniFileSect,"LastDir",".").c_str() ) );

    wxString defaultFilename = _( "robot_path.txt" );
    wxFileDialog dialog(this, caption, defaultDir, defaultFilename,wildcard, wxSAVE | wxFD_OVERWRITE_PROMPT);

    if (dialog.ShowModal() != wxID_OK) return;

	if (!robot_path.saveToTextFile( string(dialog.GetPath().mb_str()) ) )
		::wxMessageBox(_("Error creating file."));
}

void CFormRawMap::OnbtnSaveObsPathClick(wxCommandEvent& event)
{
}


void CFormRawMap::OnbtnView3DClick(wxCommandEvent& event)
{
	COpenGLScene		scene;

	loadMapInto3DScene(scene);

	win3Dmap = CDisplayWindow3DPtr( new CDisplayWindow3D("Raw-map 3D preview") );
	COpenGLScenePtr the_scene = win3Dmap->get3DSceneAndLock();
	*the_scene = scene;
	win3Dmap->unlockAccess3DScene();
	win3Dmap->repaint();
}
