/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>

#define BOOST_ALL_DYN_LINK
#include <boost/program_options.hpp>


using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::opengl;
using namespace mrpt::math;
using namespace std;



namespace po = boost::program_options;


//#define SAVE_SOG_GRID
bool 	SAVE_SOG_3DSCENE;
bool	SAVE_SOG_ALL_MAPS_OVERLAP_HYPOTHESES;
bool	SAVE_CORR_AND_NONCORR_DISTS;
bool	IS_VERBOSE = false;

// Mode of operation
bool is_match, is_detect_test;

string      	RESULTS_DIR("GRID-MATCHING_RESULTS");
string 			fil_grid1, fil_grid2;
string 			OUTPUT_FIL;
string 			CONFIG_FIL;


double STD_NOISE_XY=0, STD_NOISE_PHI=0;
double GT_Ax,GT_Ay, GT_Aphi;

bool	NOISE_IN_LASER		= false;
bool	NOISE_IN_POSE		= false;

#define N_ITERS					1
#define START_NOISE_STD			0.00f
#define END_NOISE_STD			0.00f
#define NOISE_STEPS				0.01f

/* ------------------------------------------------------------------------
					do_grid_align
   ------------------------------------------------------------------------ */
void do_grid_align()
{
	CMRPTImage				img1,img2;
	CMRPTImageFloat			imgf1,imgf2;

	mrpt::random::Randomize();

	// Grid or simplemaps?
	if (is_match)
	{
		string ext1 = mrpt::system::extractFileExtension(fil_grid1,true);
		string ext2 = mrpt::system::extractFileExtension(fil_grid2,true);
		if (ext1!=ext2 || ext1!="simplemap")
		{
			cerr << "Both file extensions must be 'simplemap'" << endl;
			return;
		}
	}
	if (is_detect_test)
	{
		string ext1 = mrpt::system::extractFileExtension(fil_grid1,true);
		if (ext1!="simplemap")
		{
			cerr << "The file extension must be 'simplemap'" << endl;
			return;
		}
	}

	CMultiMetricMap			the_map1,the_map2;

	TSetOfMetricMapInitializers	map_inits;
	{
		TMetricMapInitializer	map_init;
		map_init.metricMapClassType = CLASS_ID( COccupancyGridMap2D );
		map_init.occupancyGridMap2D_options.resolution = 0.05;
		//map_init.m_disableSaveAs3DObject = true;

		map_init.occupancyGridMap2D_options.insertionOpts.maxOccupancyUpdateCertainty = 0.8;
		map_init.occupancyGridMap2D_options.insertionOpts.maxDistanceInsertion = 30;

		map_inits.push_back(map_init);
	}
	{
		TMetricMapInitializer	map_init;
		map_init.metricMapClassType = CLASS_ID( CSimplePointsMap );
		map_init.pointsMapOptions_options.insertionOpts.minDistBetweenLaserPoints = 0.10;
		map_inits.push_back(map_init);
	}

	float	noiseStd;


	mrpt::system::deleteFiles(format("%s/*.*",RESULTS_DIR.c_str()));
	mrpt::system::createDirectory(RESULTS_DIR);

	FILE	*f_log= os::fopen(format("%s/__log.txt",RESULTS_DIR.c_str()).c_str(),"wt");

	CFileStream			f_out_log;
	if (!OUTPUT_FIL.empty())
		f_out_log.open(OUTPUT_FIL, fomAppend );

	CGridMapAligner					gridAlign;
	CGridMapAligner::TReturnInfo	info;
	float							tim;

	gridAlign.options.methodSelection = CGridMapAligner::amRobustMatch;
//	gridAlign.options.methodSelection = CGridMapAligner::amCorrelation;

	// ---------------------------------------------
	//		Options: Feature extraction
	// ---------------------------------------------
	gridAlign.options.featsPerSquareMeter		= 0.02f;
	gridAlign.options.feats_DirectionSectors	= 32;
	gridAlign.options.feats_RangeSectors		= 8;
	gridAlign.options.feats_startDist			= 0.10f;
	gridAlign.options.feats_endDist				= 3.00f;

	// ---------------------------------------------
	//		Options: Feature matching
	// ---------------------------------------------
	gridAlign.options.thresholdStdTimes			= 1.0f;

	// ---------------------------------------------
	//				Options: RANSAC
	// ---------------------------------------------
	gridAlign.options.ransac_mahalanobisDistanceThreshold = 6.0f;
	gridAlign.options.ransac_maxSetSize			= 100;
	gridAlign.options.ransac_minSetSize			= 5;
	gridAlign.options.ransac_SOG_sigma_m		= 0.05; // grid1->getResolution() * 2;
	gridAlign.options.ransac_nSimulations		= 2500;

	if (!CONFIG_FIL.empty())
	{
		// Load options for the grid-matching algorithm:
		CConfigFile cfg(CONFIG_FIL);
		gridAlign.options.loadFromConfigFile( cfg, "grid-match" );

		// load alternative config. for the metric maps:
		if (cfg.sectionExists("metric_maps"))
		{
			map_inits.loadFromConfigFile(cfg,"metric_maps");
		}
	}

	// Dump params:
	if (IS_VERBOSE)
	{
		gridAlign.options.dumpToConsole();
		map_inits.dumpToConsole();
	}

	// Create the map with a points & grid-map within:
	the_map1.setListOfMaps( &map_inits );
	the_map2.setListOfMaps( &map_inits );

	ASSERT_( the_map1.m_gridMaps.size()>=1 );
	ASSERT_( the_map2.m_gridMaps.size()>=1 );

	COccupancyGridMap2DPtr	grid1 = the_map1.m_gridMaps[0];
	COccupancyGridMap2DPtr	grid2 = the_map2.m_gridMaps[0];

	CSensFrameProbSequence 	map1,map2, map2noisy;

	// Load maps:
	CFileGZInputStream(fil_grid1) >> map1;

	// If it's detect_test only load one map:
	if (is_match)
	{
		CFileGZInputStream(fil_grid2) >> map2;
	}

	// Generate the_map1 now:
	the_map1.loadFromProbabilisticPosesAndObservations( map1 );



	grid1->saveAsBitmapFile(format("%s/the_map1.png",RESULTS_DIR.c_str()));
	CLandmarksMap							lm1;
	grid1->extractPanoramicFeatures(
		lm1,
		(int)round((grid1->getSizeX()*grid1->getSizeY())*square(grid1->getResolution())* gridAlign.options.featsPerSquareMeter),
		gridAlign.options.feats_DirectionSectors,
		gridAlign.options.feats_RangeSectors,
		gridAlign.options.feats_startDist,
		gridAlign.options.feats_endDist );

	grid1->saveAsBitmapFileWithLandmarks( format("%s/grid1_LM.png",RESULTS_DIR.c_str()) ,&lm1);

	// ADDITIVE WHITE GAUSSIAN NOISE:
	// ----------------------------------------
	for (noiseStd=START_NOISE_STD;noiseStd<=END_NOISE_STD;noiseStd+=NOISE_STEPS)
	{
		vector_float	stats_covDet, stats_stdPhi;
		vector_float	stats_errorXY, stats_errorPhi;
		vector_float	stats_bruteErrorXY, stats_bruteErrorPhi;
		vector_float	stats_ratio;
		vector_float	stats_GT_likelihood;

		for (int iter=0;iter<N_ITERS;iter++)
		{
			// Insert noise into the laser scans of map2:
			// ------------------------------------------------
			map2noisy = is_match ? map2 : map1;

			for (unsigned int q=0;q<map2noisy.size();q++)
			{
				CPose3DPDFPtr 		PDF;
				CSensoryFramePtr 	SF;

				map2noisy.get(q,PDF,SF);

				// If it's detect_test, translate the map2 by a fixed, known quantity:
				if (is_detect_test)
				{
					CPose2D  gt(GT_Ax,GT_Ay, GT_Aphi);
					gt = CPose2D(0,0,0) - gt;
					PDF->changeCoordinatesReference(  CPose3D( gt ) );
				}

				if (NOISE_IN_LASER)
				{
					CObservation2DRangeScanPtr obs = SF->getObservationByClass<CObservation2DRangeScan>();
					if (obs)
					{
						for (unsigned int k=0;k<obs->scan.size();k++)
						{
							obs->scan[k] += mrpt::random::RandomNormal(0,noiseStd);
							if (obs->scan[k]<0) obs->scan[k] = 0;
						}
					}
				} // end of NOISE_IN_LASER

				if (NOISE_IN_POSE)
				{
					CPosePDFGaussianPtr newPDF = CPosePDFGaussianPtr( new CPosePDFGaussian(
						PDF->getEstimatedPose(),
						PDF->getEstimatedCovariance() ) );

					// Change the pose:
					newPDF->mean.x += mrpt::random::RandomNormal(0,noiseStd);
					newPDF->mean.y += mrpt::random::RandomNormal(0,noiseStd);
					//newPDF->mean.phi += 0.10f*CRandomGenerator::RandomNormal(0,noiseStd);

					// Change into the map:
					map2noisy.set( q, newPDF, CSensoryFramePtr() );

				} // end of NOISE_IN_POSE
			}

			the_map2.loadFromProbabilisticPosesAndObservations( map2noisy );

			grid2->resetPanoramicFeaturesCache();

			// Save maps:
			grid2->saveAsBitmapFile( format("%s/grid2_noise_%f.png",RESULTS_DIR.c_str(),noiseStd) );

			CLandmarksMap							lm2;
			grid2->extractPanoramicFeaturesCached(
				lm2,
				(int)round((grid2->getSizeX()*grid2->getSizeY())*square(grid2->getResolution())* gridAlign.options.featsPerSquareMeter),
				gridAlign.options.feats_DirectionSectors,
				gridAlign.options.feats_RangeSectors,
				gridAlign.options.feats_startDist,
				gridAlign.options.feats_endDist );
			grid2->saveAsBitmapFileWithLandmarks( format("%s/grid2_LM_noise_%f.png",RESULTS_DIR.c_str(),noiseStd) ,&lm2);

			// --------------------------
			//        DO ALIGN
			// --------------------------
			CPosePDFPtr	parts = gridAlign.Align(
				//grid2.pointer(), grid1.pointer(),
				&the_map1, &the_map2,
				CPose2D(0,0,0),
				&tim,
				&info );

			// STATS:
			CPose2D		estimateMean( parts->getEstimatedPose() );
			CMatrix		estimateCOV = parts->getEstimatedCovariance();
			float		stdPhi = sqrt(estimateCOV(2,2));
			estimateCOV.setSize(2,2);
			float		stdXY  = sqrt(estimateCOV.det());

			estimateCOV = parts->getEstimatedCovariance();

			float		Axy			= estimateMean.distance2DTo(GT_Ax,GT_Ay);
			float		Aphi		= fabs( math::wrapToPi(estimateMean.phi - (float)DEG2RAD( GT_Aphi )));
			float		AxyBrute	= info.noRobustEstimation.distance2DTo(GT_Ax,GT_Ay);
			float		AphiBrute	= fabs( math::wrapToPi(info.noRobustEstimation.phi - (float)DEG2RAD( GT_Aphi )));
			float		ratio		= 1;

			printf("Done in %.03fms\n", 1000.0f*tim);

			std::cout << "Mean pose:\n\t " << estimateMean << "\n";
			std::cout << "Estimate covariance::\n" << estimateCOV << "\n";

			// Save particles:
			if (IS_CLASS(parts, CPosePDFParticles))
			{
				CPosePDFParticlesPtr partsPdf = CPosePDFParticlesPtr( parts );

				partsPdf->saveToTextFile( format("%s/particles_noise_%.03f.txt", RESULTS_DIR.c_str(), noiseStd) );

				printf("Goodness: %.03f%%\n", 100 * info.goodness);
				std::cout << partsPdf->particlesCount() << " particles\n";
				std::cout << "Covariance:\n\t " << parts->getEstimatedCovariance() << "\n";

				ratio = partsPdf->particlesCount() / (float)gridAlign.options.ransac_nSimulations;
			}
			else
			if (IS_CLASS(parts,CPosePDFSOG))
			{
				CPosePDFSOGPtr pdf_SOG= CPosePDFSOGPtr( parts );
				printf("SoG has %u modes\n",(unsigned int)pdf_SOG->m_modes.size());

				pdf_SOG->normalizeWeights();

				stats_GT_likelihood.push_back( (float)pdf_SOG->evaluatePDF(CPose2D(GT_Ax,GT_Ay,(float)DEG2RAD(GT_Aphi)),true) );
				ratio = (float)pdf_SOG->m_modes.size();

				pdf_SOG->saveToTextFile("_debug_SoG.txt");

				if (SAVE_SOG_3DSCENE)
				{
					COpenGLScene				scene3D;
					opengl::CSetOfObjectsPtr	thePDF3D = opengl::CSetOfObjects::Create();
					pdf_SOG->getAs3DObject( thePDF3D );
					opengl::CGridPlaneXYPtr	gridXY	= opengl::CGridPlaneXY::Create(-10,10,-10,10,0,1);
					scene3D.insert( gridXY	);
					scene3D.insert( thePDF3D );
					CFileGZOutputStream("_out_SoG.3Dscene") << scene3D;
				}

				// Save all the maps overlap hypotheses:
				if (SAVE_SOG_ALL_MAPS_OVERLAP_HYPOTHESES)
				{
					std::deque<CPosePDFSOG::TGaussianMode>::iterator	it;
					size_t								nNode;
					CMRPTImageFloat						imgGrid1, imgCanvas;
					grid1->resizeGrid( min(grid1->getXMin(),-60.0f), max(grid1->getXMax(),60.0f), min(-40.0f,grid1->getYMin()), max(30.0f,grid1->getYMax()) );
					grid1->getAsImage( imgGrid1, true );
					int			imgGrid1LY = imgGrid1.getHeight();
					const CPose2D	nullPose(0,0,0);
					CPoint2D	p1( grid2->getXMin(), grid2->getYMin() );
					CPoint2D	p2( grid2->getXMin(), grid2->getYMax() );
					CPoint2D	p3( grid2->getXMax(), grid2->getYMax() );
					CPoint2D	p4( grid2->getXMax(), grid2->getYMin() );
					for (nNode=0,it = pdf_SOG->m_modes.begin();it!=pdf_SOG->m_modes.end();it++,nNode++)
					{
						CPose2D		x = it->mean;
						CPoint2D	pp1( x + p1 );
						CPoint2D	pp2( x + p2 );
						CPoint2D	pp3( x + p3 );
						CPoint2D	pp4( x + p4 );

						// Draw the background = the_map1:
						imgCanvas = imgGrid1;

						// Draw the overlaped the_map2:
						imgCanvas.line( grid1->x2idx(pp1.x),imgGrid1LY-1-grid1->y2idx(pp1.y),grid1->x2idx(pp2.x),imgGrid1LY-1-grid1->y2idx(pp2.y),0x00);
						imgCanvas.line( grid1->x2idx(pp2.x),imgGrid1LY-1-grid1->y2idx(pp2.y),grid1->x2idx(pp3.x),imgGrid1LY-1-grid1->y2idx(pp3.y),0x00);
						imgCanvas.line( grid1->x2idx(pp3.x),imgGrid1LY-1-grid1->y2idx(pp3.y),grid1->x2idx(pp4.x),imgGrid1LY-1-grid1->y2idx(pp4.y),0x00);
						imgCanvas.line( grid1->x2idx(pp4.x),imgGrid1LY-1-grid1->y2idx(pp4.y),grid1->x2idx(pp1.x),imgGrid1LY-1-grid1->y2idx(pp1.y),0x00);

						imgCanvas.saveToFile( format("%s/_OVERLAP_MAPS_SOG_MODE_%04u.png",RESULTS_DIR.c_str(), (unsigned int)nNode  ) );

						// Save as 3D scene:
						COpenGLScene	scene;
						CPointsMap::COLOR_3DSCENE_R = 0;
						CPointsMap::COLOR_3DSCENE_G = 0;
						CPointsMap::COLOR_3DSCENE_B = 1;
						CSetOfObjectsPtr obj1 = CSetOfObjects::Create();
						the_map1.getAs3DObject( obj1 );

						CPointsMap::COLOR_3DSCENE_R = 1;
						CPointsMap::COLOR_3DSCENE_G = 0;
						CPointsMap::COLOR_3DSCENE_B = 0;
						CSetOfObjectsPtr obj2 = CSetOfObjects::Create();
						the_map2.getAs3DObject( obj2 );

						obj2->setPose(x);

						scene.insert(obj1);
						scene.insert(obj2);

						CFileGZOutputStream( format("%s/_OVERLAP_MAPS_SOG_MODE_%04u.3Dscene",RESULTS_DIR.c_str(), (unsigned int)nNode)) << scene;
					}


					if (f_out_log.fileOpenCorrectly())
					{
						f_out_log.printf( "%i  %f %f %f \n",
							static_cast<int>( pdf_SOG->size() ),
							estimateMean.x, estimateMean.y, estimateMean.phi );
					}

				} // end if it's a SOG


				// Save the descriptor distances for corresponding and non-corresponding features
				//  (only known if we are in the "is_detect_test" mode!)
				if (SAVE_CORR_AND_NONCORR_DISTS)
				{
					CLandmarksMapPtr lm1 = info.landmarks_map1;
					CLandmarksMapPtr lm2 = info.landmarks_map2;

					// GT transformation:
					const CPose2D  GT_Ap( GT_Ax, GT_Ay, GT_Aphi);
					CMetricMap::TMatchingPairList	gt_corrs;

					float minDist, minAng;

					CFileOutputStream	fout_CORR("GT_EXP_CORR.txt", true);
					CFileOutputStream	fout_NCORR("GT_EXP_NCORR.txt", true);


					// Compute the distances:
					//for (size_t q=0;q<info.correspondences_dists_maha.size();q++)
					for (size_t i1=0;i1<lm1->landmarks.size();i1++)
					{
						vector_double   D(lm2->landmarks.size());
						size_t i2;
						size_t gt_corr_of_i1=0;

						for (i2=0;i2<lm2->landmarks.size();i2++)
						{
							CLandmark *l1 = lm1->landmarks.get( i1 ); //info.correspondences_dists_maha[q].idx_this );
							CLandmark *l2 = lm2->landmarks.get( i2 ); //info.correspondences_dists_maha[q].idx_other );

							CPoint2D  P1 = CPoint2D( l1->pose_mean_x,l1->pose_mean_y );
							CPoint2D  P2 = GT_Ap + CPoint2D( l2->pose_mean_x,l2->pose_mean_y );

							const double dErr = P1.distanceTo(P2);

							if (dErr<0.30)
							{
								//cout << dErr << " ";
								gt_corrs.push_back( CMetricMap::TMatchingPair(
									i1,i2,
									l1->pose_mean_x,l1->pose_mean_y,l1->pose_mean_z,
									l2->pose_mean_x,l2->pose_mean_y,l2->pose_mean_z ) );
								gt_corr_of_i1 = i2;
							}

							CGridMapAligner::landmarksMatchingCorrelation(l1,l2,minDist,minAng);
							D[i2] = minDist;
						}

						double m,s; // Mean & Std:
						mrpt::math::meanAndStd(D,m,s);

						D = Abs( (D-m)/s );

						// The output files:
						for (i2=0;i2<lm2->landmarks.size();i2++)
						{
							if (i2==gt_corr_of_i1)
									fout_CORR.printf("%f\n",D[i2]);
							else	fout_NCORR.printf("%f\n",D[i2]);
						}

						//mrpt::system::pause();
					} // end for i1

					COccupancyGridMap2D::saveAsBitmapTwoMapsWithCorrespondences(
						"GT_corrs.png",
						grid1.pointer(),
						grid2.pointer(),
						gt_corrs );

				}

#ifdef SAVE_SOG_GRID
				// Save grid evaluation of the SOG:
				CMatrix		gridLimits(1,4);
				gridLimits(0,0) = estimateMean.x - 0.10f;
				gridLimits(0,1) = estimateMean.x + 0.10f,
				gridLimits(0,2) = estimateMean.y - 0.10f;
				gridLimits(0,3) = estimateMean.y + 0.10f;
				gridLimits.saveToTextFile( format("%s/SOG_grid_limits_noise_%f.txt",RESULTS_DIR.c_str() ,noiseStd) );

				CMatrixD	evalGrid;
				pdf_SOG->evaluatePDFInArea(
					gridLimits(0,0), gridLimits(0,1),
					gridLimits(0,2), gridLimits(0,3),
					0.002f,
					0,
					evalGrid,
					true			// Sum over all phis
					);

				evalGrid.saveToTextFile( format("%s/SOG_grid_noise_%f.txt",RESULTS_DIR.c_str() ,noiseStd) );
#endif
			}

			// STATS:
			stats_covDet.push_back( stdXY);
			stats_stdPhi.push_back(stdPhi);
			stats_errorXY.push_back(Axy);
			stats_errorPhi.push_back(Aphi);
			stats_bruteErrorXY.push_back(AxyBrute);
			stats_bruteErrorPhi.push_back(AphiBrute);
			stats_ratio.push_back( ratio );

			printf("\n------------------------------------------------------------------------\n");
			printf("Noise:%f -> ratio:%.02f%%  ERRs:(%.03f,%.03f) STDs:(%.03f %.03f)\n",
				noiseStd,
				ratio*100,
				Axy,
				Aphi,
				stdXY,
				stdPhi );
			printf("------------------------------------------------------------------------\n\n");


		} // end iter

		fprintf(f_log,"%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %e\n",
			noiseStd,
			math::mean(stats_covDet), math::stddev(stats_covDet),
			math::mean(stats_stdPhi), math::stddev(stats_stdPhi),
			math::mean(stats_errorXY), math::stddev(stats_errorXY),
			math::mean(stats_errorPhi), math::stddev(stats_errorPhi),
			math::mean(stats_bruteErrorXY), math::stddev(stats_bruteErrorXY),
			math::mean(stats_bruteErrorPhi), math::stddev(stats_bruteErrorPhi),
			math::mean(stats_ratio), math::stddev(stats_ratio),
			math::mean(stats_GT_likelihood)
			);


	} // For each noise

	fclose(f_log);
}


// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main(int argc, char **argv)
{
    try
    {
		// Declare the supported options.
		po::options_description desc("Allowed options");
		desc.add_options()
			("match", "Operation: match two maps")
			("detect-test", "Operation: Quality of match with one map")

			("map1", po::value<string>(&fil_grid1), "Map #1 to align (*.simplemap)")
			("map2", po::value<string>(&fil_grid2), "Map #2 to align (*.simplemap)")

			("out", po::value<string>(&OUTPUT_FIL)->default_value("gridmatching_out.txt"), "Output file for the results")
			("config", po::value<string>(&CONFIG_FIL), "Optional config. file with more params")

			("save-sog-3d", "Save a 3D view of all the SOG modes")
			("save-sog-all", "Save all the map overlaps")
			("save-corr-dists", "Save corr & non-corr distances")

			("noise-xy", po::value<double>(&STD_NOISE_XY)->default_value(0), "In detect-test mode,std. noise in XY")
			("noise-phi", po::value<double>(&STD_NOISE_PHI)->default_value(0), "In detect-test mode,std. noise in PHI (deg)")

			("Ax", po::value<double>(&GT_Ax)->default_value(4), "In detect-test mode, displacement in X")
			("Ay", po::value<double>(&GT_Ay)->default_value(2), "In detect-test mode, displacement in Y")
			("Aphi", po::value<double>(&GT_Aphi)->default_value(30), "In detect-test mode, displ. in PHI (deg)")

			("verbose", "verbose output")
			("nologo", "skip the logo at startup")
			("help", "produce help message")
		;

		// Parse arguments:
		po::variables_map vm;
		po::store(po::parse_command_line(argc, argv, desc), vm);
		po::notify(vm);


		SAVE_SOG_3DSCENE = 0!=vm.count("save-sog-3d");
		SAVE_SOG_ALL_MAPS_OVERLAP_HYPOTHESES= 0!=vm.count("save-sog-all");
		IS_VERBOSE = 0!=vm.count("verbose");
		SAVE_CORR_AND_NONCORR_DISTS = 0!=vm.count("save-corr-dists");

		if (1!=vm.count("nologo"))
		{
			printf(" grid-matching version 0.1 - Part of the MRPT\n");
			printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
		}

		if (vm.count("help"))
		{
			cout << desc << endl;
			return 1;
		}

		is_match = 1==vm.count("match");
		is_detect_test = 1==vm.count("detect-test");

		/*if (SAVE_CORR_AND_NONCORR_DISTS && !is_detect_test)
		{
			cout << "Error: --save-corr-dists can only work with --detect-test" << endl;
			return 1;
		}*/

		if ( (!is_match && !is_detect_test) || (is_match && is_detect_test) )
		{
			cout << endl << "Error: One operation mode 'match' or 'detect-test' must be selected." << endl;
			cout << desc << endl;
			return 1;
		}

		if ( is_match )
		{
			// maps:
			if ( 1!=vm.count("map1") && 1!=vm.count("map2") )
			{
				cout << endl << "Error: Two maps must be passed: --map1=xxx and --map2=xxx" << endl;
				cout << desc << endl;
				return 1;
			}
		}

		// pass deg 2 rads:
		GT_Aphi= DEG2RAD(GT_Aphi);

		// Invoke method:
		do_grid_align();

		return 0;
	}
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		return -1;
	}
	catch(...)
	{
		cerr << "Untyped exception." << endl;
		return -1;
	}

}
