/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  MRPT_MATH_H
#define  MRPT_MATH_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/math/vector_ops.h>
#include <mrpt/math/CHistogram.h>

#include <numeric>
#include <cmath>

/*---------------------------------------------------------------
		Namespace
  ---------------------------------------------------------------*/
namespace mrpt
{
	/** This base provides a set of functions for maths stuff.
	 */
	namespace math
	{
	    using namespace mrpt::utils;

		/** Loads one row of a text file as a numerical std::vector.
		  * \return false on EOF or invalid format.
		  * The body of the function is implemented in MATH.cpp
			*/
		bool MRPTDLLIMPEXP loadVector( utils::CFileStream &f, std::vector<int> &d);

		/** Loads one row of a text file as a numerical std::vector.
		  * \return false on EOF or invalid format.
		  * The body of the function is implemented in MATH.cpp
			*/
		bool MRPTDLLIMPEXP loadVector( utils::CFileStream &f, std::vector<double> &d);

		/** Returns true if value is Not-a-number (NAN)
	      */
	    bool MRPTDLLIMPEXP isNan(float v);

	    /** Returns true if value is Not-a-number (NAN)
	      */
	    bool MRPTDLLIMPEXP isNan(double v);

	    /** Returns true if value is finite
	      */
	    bool MRPTDLLIMPEXP isFinite(float v);

	    /** Returns true if value is finite
	      */
	    bool MRPTDLLIMPEXP isFinite(double v);

		/** A template for counting how many elements in an array are non-Zero.
			*/
		template <class T>
		size_t  countNonZero(const std::vector<T> &a)
		{
			typename std::vector<T>::const_iterator	it_a;
			size_t		count=0;
			for (it_a=a.begin(); it_a!=a.end(); it_a++) if (*it_a) count++;
			return count;
		}

		/** Finds the maximum value (and the corresponding zero-based index) from a given vector.
		  */
		template<class T>
		T  maximum(const std::vector<T> &v, unsigned int *maxIndex = NULL)
		{
			typename std::vector<T>::const_iterator maxIt = std::max_element(v.begin(),v.end());
			if (maxIndex) *maxIndex = static_cast<unsigned int>( std::distance(v.begin(),maxIt) );
			return *maxIt;
		}

		/** Compute the norm-infinite of a vector ($f[ ||\mathbf{v}||_\infnty $f]), ie the maximum absolute value of the elements.
		  */
		template<class T>
		T  norm_inf(const std::vector<T> &v, unsigned int *maxIndex = NULL)
		{
			double	M=0;
			int		i,M_idx=-1;
			typename std::vector<T>::const_iterator	it;
			for (i=0, it=v.begin(); it!=v.end();it++,i++)
			{
				double	it_abs = fabs( static_cast<double>(*it));
				if (it_abs>M || M_idx==-1)
				{
					M = it_abs;
					M_idx = i;
				}
			}
			if (maxIndex) *maxIndex = M_idx;
			return static_cast<T>(M);
		}


		/** Compute the 2-norm of the vector (the Euclidean distance to the origin).
		  */
		template<class T>
		T  norm(const std::vector<T> &v)
		{
			T	total=0;
			typename std::vector<T>::const_iterator	it;
			for (it=v.begin(); it!=v.end();it++)
				total += square(*it);
			return ::sqrt(total);
		}

		/** Finds the maximum value (and the corresponding zero-based index) from a given vector.
		  * \sa maximum, minimum_maximum
		  */
		template <class T>
		T  minimum(const std::vector<T> &v, unsigned int *minIndex = NULL)
		{
			typename std::vector<T>::const_iterator minIt = std::min_element(v.begin(),v.end());
			if (minIndex) *minIndex = static_cast<unsigned int>( std::distance(v.begin(),minIt) );
			return *minIt;
		}

		/** Compute the minimum and maximum of a vector at once
		  * \sa maximum, minimum
		  */
		template<class T>
		void minimum_maximum(const std::vector<T> &v, T& out_min, T& out_max, unsigned int *minIndex = NULL,unsigned int *maxIndex = NULL)
		{
			size_t N = v.size();
			if (N)
			{
				out_max = out_min = v[0];
				unsigned int min_idx=0,max_idx=0;
				for (size_t i=0;i<N;i++)
				{
					if (v[i]<out_min)
					{
						out_min=v[i];
						min_idx = i;
					}
					if (v[i]>out_max)
					{
						out_max=v[i];
						max_idx = i;
					}
				}
				if (minIndex) *minIndex = min_idx;
				if (maxIndex) *maxIndex = max_idx;
			}
		}

		/** Computes the mean value of a vector
		  * \sa math::stddev,math::meanAndStd
		  */
		template<class T>
			double  mean(const std::vector<T> &v)
		{
			if (v.empty())
					return 0;
			else	return static_cast<double>( std::accumulate(v.begin(),v.end(), static_cast<T>(0) ) ) / v.size();
		}

		/** Computes the sum of all the elements of a vector
		  * \sa cumsum
		  */
		template<class T>
			T  sum(const std::vector<T> &v)
		{
			return std::accumulate(v.begin(),v.end(), static_cast<T>(0) );
		}

		/** Generates an equidistant sequence of numbers given the first one, the last one and the desired number of points. */
		template<typename T,typename K>
		void linspace(T first,T last, size_t count, std::vector<K> &out_vector)
		{
			if (count<2)
			{
				out_vector.assign(1,last);
				return;
			}
			else
			{
				out_vector.resize(count);
				const T incr = (last-first)/T(count-1);
				T c = first;
				for (size_t i=0;i<count;i++,c+=incr)
					out_vector[i] = K(c);
			}
		}

		/** Generates an equidistant sequence of numbers given the first one, the last one and the desired number of points. */
		template<class T>
		std::vector<T> linspace(T first,T last, size_t count)
		{
			std::vector<T> ret;
			mrpt::math::linspace(first,last,count,ret);
			return ret;
		}

		/** Generates a vector of all ones of the given length. */
		template<class T>
		std::vector<T> ones(size_t count)
		{
			return std::vector<T>(count,1);
		}

		/** Generates a vector of all zeros of the given length. */
		template<class T>
		std::vector<T> zeros(size_t count)
		{
			return std::vector<T>(count,0);
		}

		/** Normalize a vector, such as its norm is the unity.
		  *  If the vector has a null norm, the output is a null vector.
		  */
		template<class T>
			void normalize(const std::vector<T> &v, std::vector<T> &out_v)
		{
			T	total=0;
			typename std::vector<T>::const_iterator it;
			for (it=v.begin(); it!=v.end();it++)
				total += square(*it);
			total = ::sqrt(total);
			if (total)
			{
				out_v.resize(v.size());
				typename std::vector<T>::iterator q;
				for (it=v.begin(),q=out_v.begin(); q!=out_v.end();it++,q++)
					*q = *it / total;
			}
			else out_v.assign(v.size(),0);
		}

		/** Computes the cumulative sum of all the elements of a vector
		  * \sa sum
		  */
		template<class T>
			std::vector<T>  cumsum(const std::vector<T> &v)
		{
			T			     last = 0;
			std::vector<T>	 ret(v.size());
			typename std::vector<T>::const_iterator it;
			typename std::vector<T>::iterator 		it2;
			for (it = v.begin(),it2=ret.begin();it!=v.end();it++,it2++)
				last = (*it2) = last + (*it);
			return ret;
		}

		/** Computes the cumulative sum of all the elements of a vector, saving the result in a given vector.
		  * \sa sum
		  */
		template<class T>
			void cumsum(const std::vector<T> &v, std::vector<T> &out_cumsum)
		{
			T			     last = 0;
			out_cumsum.resize(v.size());
			typename std::vector<T>::const_iterator it;
			typename std::vector<T>::iterator 		it2;
			for (it = v.begin(),it2=out_cumsum.begin();it!=v.end();it++,it2++)
				last = (*it2) = last + (*it);
		}

		/** Computes the standard deviation of a vector
		  * \param v The set of data
		  * \param unbiased If set to true or false the std is normalized by "N-1" or "N", respectively.
		  * \sa math::mean,math::meanAndStd
		  */
		template<class T>
			double  stddev(const std::vector<T> &v, bool unbiased = true)
		{
			if (v.size()<2)
				return 0;
			else
			{
				// Compute the mean:
				typename std::vector<T>::const_iterator it;
				double					 vector_std=0,vector_mean = 0;
				for (it = v.begin();it!=v.end();it++) vector_mean += (*it);
				vector_mean /= static_cast<double>(v.size());
				// Compute the std:
				for (it = v.begin();it!=v.end();it++) vector_std += square((*it)-vector_mean);
				vector_std = sqrt(vector_std  / static_cast<double>(v.size() - (unbiased ? 1:0)) );

				return vector_std;
			}
		}

		/** Computes the standard deviation of a vector
		  * \param v The set of data
		  * \param out_mean The output for the estimated mean
		  * \param out_std The output for the estimated standard deviation
		  * \param unbiased If set to true or false the std is normalized by "N-1" or "N", respectively.
		  * \sa math::mean,math::stddev
		  */
		template<class T>
			void  meanAndStd(
				const std::vector<T>	&v,
				double			&out_mean,
				double			&out_std,
				bool			unbiased = true)
		{
			if (v.size()<2)
			{
				out_std = 0;
				if (v.size()==1)
					out_mean = v[0];
			}
			else
			{
				// Compute the mean:
				typename std::vector<T>::const_iterator it;
				out_std=0,out_mean = 0;
				for (it = v.begin();it!=v.end();it++) out_mean += (*it);
				out_mean /= static_cast<double>(v.size());

				// Compute the std:
				for (it = v.begin();it!=v.end();it++) out_std += square(static_cast<double>(*it)-out_mean);
				out_std = sqrt(out_std / static_cast<double>((v.size() - (unbiased ? 1:0)) ));
			}
		}

		/** Computes the weighted histogram for a vector of values and their corresponding weights.
		  *  \param values [IN] The N values
		  *  \param weights [IN] The weights for the corresponding N values
		  *  \param binWidth [IN] The desired width of the bins
		  *  \param out_binCenters [OUT] The centers of the M bins generated to cover from the minimum to the maximum value of "values" with the given "binWidth"
		  *  \param out_binValues [OUT] The ratio of values at each given bin, such as the whole vector sums up the unity.
		  */
		template<class T>
			void  weightedHistogram(
				const std::vector<T>		&values,
				const std::vector<T>		&weights,
				float				binWidth,
				std::vector<float>	&out_binCenters,
				std::vector<float>	&out_binValues )
			{
				MRPT_TRY_START;

				ASSERT_( values.size() == weights.size() );
				ASSERT_( binWidth > 0 );
				T				minBin = minimum( values );
				unsigned int	nBins = static_cast<unsigned>(ceil((maximum( values )-minBin) / binWidth));

				// Generate bin center and border values:
				out_binCenters.resize(nBins);
				out_binValues.clear(); out_binValues.resize(nBins,0);
				float		   halfBin = 0.5f*binWidth;;
				vector_float   binBorders(nBins+1,minBin-halfBin);
				for (unsigned int i=0;i<nBins;i++)
				{
					binBorders[i+1] = binBorders[i]+binWidth;
					out_binCenters[i] = binBorders[i]+halfBin;
				}

				// Compute the histogram:
				float	totalSum = 0;
				for (typename  std::vector<T>::const_iterator itVal = values.begin(), itW = weights.begin(); itVal!=values.end(); itVal++, itW++ )
				{
					int idx = round(((*itVal)-minBin)/binWidth);
					if (idx>=nBins) idx=nBins-1;
					ASSERT_(idx>=0);
					out_binValues[idx] += *itW;
					totalSum+= *itW;
				}

				if (totalSum)
					out_binValues = out_binValues / totalSum;


				MRPT_TRY_END;
			}


		/** Computes the factorial of an integer number and returns it as a 64-bit integer number.
		  */
		uint64_t MRPTDLLIMPEXP  factorial64(unsigned int n);

		/** Computes the factorial of an integer number and returns it as a double value (internally it uses logarithms for avoiding overflow).
		  */
		double MRPTDLLIMPEXP  factorial(unsigned int n);


		/** Modifies the given angle to translate it into the [0,2pi[ range.
		  * \note Take care of not instancing this template for integer numbers, since it only works for float, double and long double.
		  * \sa wrapToPi, wrapTo2Pi
		  */
		template <class T>
		void wrapTo2PiInPlace(T &a)
		{
			bool was_neg = a<0;
			a = fmod(a, static_cast<T>(M_2PI) );
			if (was_neg) a+=static_cast<T>(M_2PI);
		}

		/** Modifies the given angle to translate it into the [0,2pi[ range.
		  * \note Take care of not instancing this template for integer numbers, since it only works for float, double and long double.
		  * \sa wrapToPi, wrapTo2Pi
		  */
		template <class T>
		T wrapTo2Pi(T a)
		{
			wrapTo2PiInPlace(a);
			return a;
		}

		/** Modifies the given angle to translate it into the ]-pi,pi] range.
		  * \note Take care of not instancing this template for integer numbers, since it only works for float, double and long double.
		  * \sa wrapTo2Pi, wrapToPiInPlace
		  */
		template <class T>
		T wrapToPi(T a)
		{
			return wrapTo2Pi( a + static_cast<T>(M_PI) )-static_cast<T>(M_PI);
		}

		/** Modifies the given angle to translate it into the ]-pi,pi] range.
		  * \note Take care of not instancing this template for integer numbers, since it only works for float, double and long double.
		  * \sa wrapToPi,wrapTo2Pi
		  */
		template <class T>
		void wrapToPiInPlace(T &a)
		{
			a = wrapToPi(a);
		}

		/** Round up to the nearest power of two of a given number
		  */
		template <class T>
		T round2up(T val)
		{
			T n = 1;
			while (n < val)
			{
				n <<= 1;
				if (n<=1)
					THROW_EXCEPTION("Overflow!");
			}
			return n;
		}

		/** Round a decimal number up to the given 10'th power (eg, to 1000,100,10, and also fractions)
		  *  power10 means round up to: 1 -> 10, 2 -> 100, 3 -> 1000, ...  -1 -> 0.1, -2 -> 0.01, ...
		  */
		template <class T>
		T round_10power(T val, int power10)
		{
			long double F = ::pow((long double)10.0,-(long double)power10);
			long int t = round_long( val * F );
			return T(t/F);
		}

		/** Cholesky factorization: in = out' · out
		  *	Given a positive-definite symmetric matrix, this routine constructs its Cholesky decomposition.
		  * On input, only the upper triangle of "IN" need be given; it is not modified.
		  *  The Cholesky factorization is returned in "out" in the upper triangle.
		  *  (by AJOGD @ JAN-2007)
		  */
		template<class T>
		void  chol(const CMatrixTemplateNumeric<T> &in,CMatrixTemplateNumeric<T> &out)
		{
			if (in.getColCount() != in.getRowCount()) THROW_EXCEPTION("Cholesky factorization error, in matrix not square");
			size_t	i,j,k;
			T		sum;
			out.setSize(in.getRowCount(),in.getColCount());
			for (i=0;i<in.getRowCount();i++)
			{
				for (j=i;j<in.getColCount();j++)
				{
					sum=in(i,j);
					for (k=i-1;(k>=0)&(k<in.getColCount());k--)
					{
						sum -= out(k,i)*out(k,j);
					}
					if (i==j)
					{
						if (sum<0)
						{
							in.saveToTextFile("c:\\cholesky_non_defined_positive.txt");
							THROW_EXCEPTION("Cholesky factorization error, in matrix not defined-positive");
						}
						out(i,j)=sqrt(sum);
					}
					else
					{
						out(i,j)=sum/out(i,i);
						out(j,i)=0;
					}
				}
			}
		}

		/** Calculate the correlation between two matrices
		  *  (by AJOGD @ JAN-2007)
		  */
		template<class T>
		double  correlate_matrix(const CMatrixTemplateNumeric<T> &a1, const CMatrixTemplateNumeric<T> &a2)
		{
			if ((a1.getColCount()!=a2.getColCount())|(a1.getRowCount()!=a2.getRowCount()))
				THROW_EXCEPTION("Correlation Error!, images with no same size");

			int i,j;
			T x1,x2;
			T syy=0, sxy=0, sxx=0, m1=0, m2=0 ,n=a1.getRowCount()*a2.getColCount();

			//find the means
			for (i=0;i<a1.getRowCount();i++)
			{
				for (j=0;j<a1.getColCount();j++)
				{
					m1 += a1(i,j);
					m2 += a2(i,j);
				}
			}
			m1 /= n;
			m2 /= n;

			for (i=0;i<a1.getRowCount();i++)
			{
				for (j=0;j<a1.getColCount();j++)
				{
					x1 = a1(i,j) - m1;
					x2 = a2(i,j) - m2;
					sxx += x1*x1;
					syy += x2*x2;
					sxy += x1*x2;
				}
			}

			return sxy / sqrt(sxx * syy);
		}


		/**	Matrix QR decomposition. A = QR, where R is upper triangular and Q is orthogonal, that is, ~QQ = 1
		  * If A is a LxM dimension matrix, this function only return the LxL upper triangular matrix R instead of LxM pseudo-upper
		  * triangular matrix (been L<=M)
		  * This function has been extracted from "Numerical Recipes in C".
		  *	/param A is the original matrix to decompose
		  * /param c,Q. The orthogonal matrix Q is represented as a product of n-1 Householder matrices Q1,...Qn-1, where Qj = 1 - u[j] x u[j]/c[j]
		  *				The i'th component of u[j] is zero for i = 1,...,j-1 while the nonzero components are returned in Q(i,j) for i=j,...,n
		  *	/param R is the upper triangular matrix
		  *	/param sign returns as true (1) is singularity is encountered during the decomposition, but the decomposition is still complete
		  *				in this case; otherwise it returns false (0)
		  */

			template<class T>
			void MRPTDLLIMPEXP qr_decomposition(
				CMatrixTemplateNumeric<T>	&A,
				CMatrixTemplateNumeric<T>	&R,
				CMatrixTemplateNumeric<T>	&Q,
				CVectorTemplate<T>			&c,
				int								&sing);


		/**If R = CHOL(A) is the original Cholesky factorization of A, then R1 = CHOLUPDATE(R,X) returns the upper triangular
		  * Cholesky factor of A + X*X', where X is a column vector of appropriate length.
		  */
		template<class T>
			void MRPTDLLIMPEXP UpdateCholesky(
				CMatrixTemplateNumeric<T>	&chol,
				CVectorTemplate<T>			&r1Modification);

		/** Compute the two eigenvalues of a 2x2 matrix.
		  * \param in_matrx The 2x2 input matrix.
		  * \param min_eigenvalue (out) The minimum eigenvalue of the matrix.
		  * \param max_eigenvalue (out) The maximum eigenvalue of the matrix.
		  * by FAMD, MAR-2007
		  */
		void MRPTDLLIMPEXP  computeEigenValues2x2(
			const CMatrixFloat	&in_matrix,
			float								&min_eigenvalue,
			float								&max_eigenvalue );

		/** Computes the 'exp' of all the elements of a vector
		  * \sa math::Log
		  */
		template<class T>
			std::vector<T> Exp(const std::vector<T> &v)
		{
			std::vector<T>	ret(v.size());
			typename std::vector<T>::const_iterator it;
			typename std::vector<T>::iterator		it2;
			for (it = v.begin(),it2=ret.begin();it!=v.end();it++,it2++) *it2 = ::exp(*it);
			return ret;
		}

		/** Computes the 'log' of all the elements of a vector
		  * \sa math::Exp
		  */
		template<class T>
			std::vector<T> Log(const std::vector<T> &v)
		{
			std::vector<T>	ret(v.size());
			typename std::vector<T>::const_iterator it;
			typename std::vector<T>::iterator		it2;
			for (it = v.begin(),it2=ret.begin();it!=v.end();it++,it2++) *it2 = ::log(*it);
			return ret;
		}

		/** A numerically-stable method to compute average likelihood values with strongly different ranges (unweighted likelihoods: compute the arithmetic mean).
		  *  This method implements this equation:
		  *
		  *  \f[ return = - \log N + \log  \sum_{i=1}^N e^{ll_i-ll_{max}} + ll_{max} \f]
		  *
		  * See also the <a href="http://babel.isa.uma.es/mrpt/index.php/Averaging_Log-Likelihood_Values:Numerical_Stability">tutorial page</a>.
		  */
		double MRPTDLLIMPEXP  averageLogLikelihood( const vector_double &logLikelihoods );

		/** A numerically-stable method to average likelihood values with strongly different ranges (weighted likelihoods).
		  *  This method implements this equation:
		  *
		  *  \f[ return = \log \left( \frac{1}{\sum_i e^{lw_i}} \sum_i  e^{lw_i} e^{ll_i}  \right) \f]
		  *
		  * See also the <a href="http://babel.isa.uma.es/mrpt/index.php/Averaging_Log-Likelihood_Values:Numerical_Stability">tutorial page</a>.
		  */
		double MRPTDLLIMPEXP  averageLogLikelihood(
			const vector_double &logWeights,
			const vector_double &logLikelihoods );

		/** Generates a string with the MATLAB commands required to plot an confidence interval (ellipse) for a 2D Gaussian ('float' version)..
		  *  \param cov22 The 2x2 covariance matrix
		  *  \param mean  The 2-length vector with the mean
		  *  \param stdCount How many "quantiles" to get into the area of the ellipse: 2: 95%, 3:99.97%,...
		  *  \param style A matlab style string, for colors, line styles,...
		  *  \param nEllipsePoints The number of points in the ellipse to generate
		  */
		std::string MRPTDLLIMPEXP  MATLAB_plotCovariance2D(
			const CMatrixFloat  &cov22,
			const CVectorFloat  &mean,
			const float         &stdCount,
			const std::string   &style = std::string("b"),
			const size_t        &nEllipsePoints = 30 );

		/** Generates a string with the MATLAB commands required to plot an confidence interval (ellipse) for a 2D Gaussian ('double' version).
		  *  \param cov22 The 2x2 covariance matrix
		  *  \param mean  The 2-length vector with the mean
		  *  \param stdCount How many "quantiles" to get into the area of the ellipse: 2: 95%, 3:99.97%,...
		  *  \param style A matlab style string, for colors, line styles,...
		  *  \param nEllipsePoints The number of points in the ellipse to generate
		  */
		std::string MRPTDLLIMPEXP  MATLAB_plotCovariance2D(
			const CMatrixDouble  &cov22,
			const CVectorDouble  &mean,
			const float         &stdCount,
			const std::string   &style = std::string("b"),
			const size_t        &nEllipsePoints = 30 );


		/** Efficiently compute the inverse of a 4x4 homogeneous matrix by only transposing the rotation 3x3 part and solving the translation with dot products.
		  */
		void MRPTDLLIMPEXP  homogeneousMatrixInverse(
			const CMatrixDouble &M,
			CMatrixDouble &out_inverse_M);

		/** Counts the number of elements that appear in both vectors (comparison through the == operator)
		  *  It is assumed that no repeated elements appear within each of the vectors.
		  */
		template<class T>
		size_t  countCommonElements(
            const std::vector<T> &a,
            const std::vector<T> &b)
		{
		    size_t ret=0;
			typename std::vector<T>::const_iterator it1;
			typename std::vector<T>::const_iterator	it2;
			for (it1 = a.begin();it1!=a.end();it1++)
			    for (it2 = b.begin();it2!=b.end();it2++)
                    if ( (*it1) == (*it2) )
                         ret++;

			return ret;
		}

		/** Interpolate a data sequence "ys" ranging from "x0" to "x1" (equally spaced), to obtain the approximation of the sequence at the point "x".
		  *  If the point "x" is out of the range [x0,x1], the closest extreme "ys" value is returned.
		  */
		template <class T>
		T interpolate(
			const T					&x,
			const std::vector<T>	&ys,
			const T					&x0,
			const T					&x1 )
		{
			MRPT_TRY_START
			ASSERT_(x1>x0); ASSERT_(!ys.empty());
			const size_t N = ys.size();
			if (x<=x0)	return ys[0];
			if (x>=x1)	return ys[N-1];
			const T Ax = (x1-x0)/T(N);
			const size_t i = int( (x-x0)/Ax );
			if (i>=N-1) return ys[N-1];
			const T Ay = ys[i+1]-ys[i];
			return ys[i] + (x-(x0+i*Ax))*Ay/Ax;
			MRPT_TRY_END
		}

		/** Estimate the Jacobian of a multi-dimensional function around a point "x", using finite differences of a given size in each input dimension.
		  */
		void MRPTDLLIMPEXP  estimateJacobian(
			const vector_double				&x,
			utils::TFunctor_retVecDbl_inp2VecDbl	functor,
			const vector_double				&increments,
			const vector_double				&userParam,
			math::CMatrixDouble					&out_Jacobian );

		/** Interpolates the value of a function in a point "t" given 4 SORTED points where "t" is between the two middle points
		  *  If wrap2pi is true, output "y" values are wrapped to ]-pi,pi] (It is assumed that input "y" values already are in the correct range).
		  */
		double MRPTDLLIMPEXP  spline(const double t, const std::vector<double> &x, const std::vector<double> &y, bool wrap2pi = false);

		/** Computes the normalized or normal histogram of a sequence of numbers given the number of bins and the limits.
		  */
		template<class T>
			vector_double histogram(const std::vector<T> &v, double limit_min, double limit_max, size_t number_bins, bool do_normalization = false )
		{
			mrpt::math::CHistogram	H( limit_min, limit_max, number_bins );
			vector_double ret(number_bins);
			size_t i;
			for (i=0;i<v.size();i++)	H.add(static_cast<double>( v[i] ));
			for (i=0;i<number_bins;i++)	ret[i] = do_normalization ? H.getBinRatio(i) : H.getBinCount(i);
			return ret;
		}

		/** Assignment operator for initializing a std::vector from a C array (The vector will be automatically set to the correct size).
		  * \code
		  *	 vector_double  v;
		  *  const double numbers[] = { 1,2,3,5,6,7,8,9,10 };
		  *  loadVector( v, numbers );
		  * \endcode
		  * \note This operator performs the appropiate type castings, if required.
		  */
		template <typename T, typename At, size_t N>
		std::vector<T>& loadVector( std::vector<T> &v, At (&theArray)[N] )
		{
			MRPT_COMPILE_TIME_ASSERT(N!=0)
			v.resize(N);
			for (size_t i=0; i < N; i++)
				v[i] = static_cast<T>(theArray[i]);
			return v;
		}

		/** Absolute value of a vector. */
		template <class T>
		std::vector<T>  Abs(const std::vector<T> &a)
		{
			typename std::vector<T>	res(a.size());
			for (size_t i=0;i<a.size();i++)
				res[i] =  static_cast<T>( fabs( static_cast<double>( a[i] ) ) );
			return res;
		}

	} // End of MATH namespace

} // End of namespace

#endif
