/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef opengl_CMesh_H
#define opengl_CMesh_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/math/CMatrix.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/vision/utils.h>

namespace mrpt
{
	namespace opengl
	{
		class MRPTDLLIMPEXP CMesh;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CMesh, CRenderizable )

		/** A set of connected colored triangles.
		  *  This class can be used to draw any solid, arbitrarily complex object (without textures).
		  *  \sa opengl::COpenGLScene
		  */
		class MRPTDLLIMPEXP CMesh : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CMesh )
		protected:
			mrpt::utils::CMRPTImage		m_textureImage;

			bool						m_enableTransparency;
			bool						m_colorFromZ;
			bool						m_isWireFrame;

			math::CMatrix		Z;		//!< Z(x,y): Z-coordinate of the point (x,y)
			math::CMatrix		mask;
			math::CMatrix		U, V;	//!< Texture coordinates
			math::CMatrix		C;		//!< Color [0,1] for each cell, updated by updateColorsMatrix

			mrpt::vision::TColormap		m_colorMap; //!< Used when m_colorFromZ is true

			bool	m_modified_Z;		//!< Whether C is not up-to-date wrt to Z

			void updateColorsMatrix();	//!< Called internally to assure C is updated.

		public:

			void enableTransparency( bool v )	{ m_enableTransparency = v; }
			void enableWireFrame( bool v ) 		{ m_isWireFrame = v; }
			void enableColorFromZ( bool v, mrpt::vision::TColormap	colorMap = mrpt::vision::cmJET )
			{
				m_colorFromZ = v;
				m_colorMap   = colorMap;
			}

			/** This method sets the matrix of heights for each position (cell) in the mesh grid */
			void setZ( const mrpt::math::CMatrixTemplateNumeric<float> &in_Z );

			/** Returns a reference to the internal Z matrix, allowing changing it efficiently */
			mrpt::math::CMatrixFloat & getZ();

			/** Returns a reference to the internal mask matrix, allowing changing it efficiently */
			mrpt::math::CMatrixFloat & getMask();

			/** This method sets the boolean mask of valid heights for each position (cell) in the mesh grid */
			void setMask( const mrpt::math::CMatrixTemplateNumeric<float> &in_mask );

			/** Sets the (u,v) texture coordinates (in range [0,1]) for each cell */
			void setUV( const mrpt::math::CMatrixTemplateNumeric<float> &in_U, const mrpt::math::CMatrixTemplateNumeric<float> &in_V);

			float				xMin, xMax, yMin, yMax;

			/** Class factory  */
			static CMeshPtr Create(bool enableTransparency = false, float xMin = 0.0f, float xMax = 0.0f, float yMin = 0.0f, float yMax = 0.0f )
			{
				return CMeshPtr( new CMesh( enableTransparency, xMin ,xMax , yMin ,yMax ) );
			}

			/** Render
			  */
			void  render();

			/** Assigns a texture image, and disable transparency.
			  */
			void  assignImage(
				const utils::CMRPTImage&	img );

		private:
			/** Constructor
			  */
			CMesh( bool enableTransparency = false, float xMin = 0.0f, float xMax = 0.0f, float yMin = 0.0f, float yMax = 0.0f ) :
				m_textureImage(0,0),
				m_enableTransparency(enableTransparency),
				m_colorFromZ(false),
				m_isWireFrame(false),
				Z(0,0), mask(0,0), U(0,0), V(0,0), C(0,0),
				m_colorMap( mrpt::vision::cmJET ),
				m_modified_Z(true),
				xMin(xMin), xMax(xMax), yMin(yMin), yMax(yMax)
			{
				m_color_A = 1.0f;
				m_color_R = 0.0f;
				m_color_G = 0.0f;
				m_color_B = 0.6f;
			}
			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CMesh() { }

		};

	} // end namespace

} // End of namespace

#endif
