/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPose3DPDF_H
#define CPose3DPDF_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/math/CMatrixD.h>

namespace mrpt
{
namespace poses
{
	using namespace mrpt::math;

	class CPosePDF;

	DEFINE_SERIALIZABLE_PRE( CPose3DPDF )

	/** Declares a class that represents a Probability Density Function (PDF) of a 3D pose (6D actually).
	 *   This class is just the base class for unifying many diferent
	 *    ways this PDF can be implemented.
	 *
	 *  For convenience, a pose composition is also defined for any
	 *    PDF derived class, changeCoordinatesReference, in the form of a method rather than an operator.
     *
	 *  For a similar class for 3D points (without attitude), see CPointPDF
	 *
	 *
	 *  See also the tutorial on <a href="http://babel.isa.uma.es/mrpt/index.php/Probability_Density_Distributions_Over_Spatial_Representations">probabilistic spatial representations in the MRPT</a>.
	 *
	 * \sa CPose3D, CPosePDF, CPointPDF
	 */
	class MRPTDLLIMPEXP CPose3DPDF : public mrpt::utils::CSerializable
	{
		DEFINE_VIRTUAL_SERIALIZABLE( CPose3DPDF )

	 public:
		/** Returns the entropy of the estimated pose covariance matrix.
		  */
		double  getEstimatedCovarianceEntropy();

		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
		  */
		virtual CPose3D  getEstimatedPose() const = 0 ;

		 /** Returns an estimate of the pose covariance matrix (6x6 cov.matrix for x,y,z,yaw,pitch,and roll variables)
		  */
		virtual CMatrixD getEstimatedCovariance() const = 0;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  * \sa createFrom2D
		  */
		virtual void  copyFrom(const CPose3DPDF &o) = 0;

		/** This is a static transformation method from 2D poses to 3D PDFs, preserving the representation type (particles->particles, Gaussians->Gaussians,etc)
		  *  It returns a new object of any of the derived classes of CPose3DPDF. This object must be deleted by the user when not required anymore.
		  *  \sa copyFrom
		  */
		static CPose3DPDF* createFrom2D(const CPosePDF &o);

		/** Save the PDF to a text file. See derived classes for more information about the format of generated files.
		 */
		virtual void  saveToTextFile(const std::string &file) const = 0;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		virtual void  changeCoordinatesReference( const CPose3D &newReferenceBase ) = 0;

		/** Draws a single sample from the distribution
		  */
		virtual void  drawSingleSample( CPose3D &outPart ) const = 0;

		/** Draws a number of samples from the distribution, and saves as a list of 1x6 vectors, where each row contains a (x,y,z,yaw,pitch,roll) datum.
		  */
		virtual void  drawManySamples( size_t N, std::vector<vector_double> & outSamples ) const = 0;

		/** Bayesian fusion of two pose distributions, then save the result in this object (WARNING: Currently only distributions of the same class can be fused! eg, gaussian with gaussian,etc)
		  */
		virtual void  bayesianFusion( CPose3DPDF &p1, CPose3DPDF &p2 )  = 0 ;

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		virtual void  inverse(CPose3DPDF &o) const = 0;


	}; // End of class def.


	} // End of namespace
} // End of namespace

#endif
